 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

/*
   * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Data Segment Utility Routines
   >>>>
   >>>>   Static:
   >>>>			__kdms_create_segment()
   >>>>                 __kdms_destroy_segment()
   >>>>                 _reference_segment()
   >>>>			_copy_segment_data()
   >>>>
   >>>>  Private:
   >>>>                 _kdms_get_segment_info()
   >>>>                 _kdms_set_segment_info()
   >>>>
   >>>>                 _kdms_copy_pres_struct()
   >>>>
   >>>>			_kdms_create_segment()
   >>>>                 _kdms_destroy_segment()
   >>>>                 _kdms_destroy_segments()
   >>>>
   >>>>                 _kdms_reference_segments()
   >>>>                 _kdms_establish_segment()
   >>>>
   >>>>                 _kdms_lock_segment()
   >>>>                 _kdms_lock_segments()
   >>>>
   >>>>                 _kdms_set_segment()
   >>>>                 _kdms_get_segment()
   >>>>
   >>>>                 _kdms_update_segment()
   >>>>                 _kdms_initialize_segment()
   >>>>                 _kdms_migrate_segment_data()
   >>>>                 _kdms_rename_segment()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define NUM_POINTS(a,b) ((b[0] - a[0] + 1) * \
       (b[1] - a[1] + 1)*(b[2] - a[2] + 1)*(b[3] - a[3] + 1)*(b[4] - a[4] + 1))

/*-----------------------------------------------------------
|
|  routine name: _kdms_get_segment_info - returns the dsegment and/or
|                                    presentation attribute structure.
|
|       purpose: returns the dsegment and/or presentation attribute
|                structure.
|
|         input: object    - the data object in which the dsegment is
|                            being built
|                segment   - the name of the segment in which we want
|                            to return the segment->
|        output: dsegment     - returns the dsegment if not NULL
|                presentation - returns the presentation attributes if not NULL
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: mark young
|          date: nov 18, 1992 15:14
| modifications:
|
------------------------------------------------------------*/

int
_kdms_get_segment_info(kobject object, int segment, kpresentation ** presentation)
{
   klist *temp;

   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   /*
    * get the presentation attribute associated wit the string token.
    */
   if (presentation)
   {
      if ((temp = klist_locate(object->presentations,
                               (kaddr) ((long)segment))) != NULL)
      {
         *presentation = (kpresentation *) temp->client_data;
         return (TRUE);
      }
      else
      {
         return (FALSE);        /* NOT AN ERROR */
      }
   }
   _kdms_set_error(KDMS_EINTERNAL);
   return (FALSE);
}

/*-----------------------------------------------------------
|
|  routine name: _kdms_set_segment_info - initializes the dsegment and/or
|                                    presentation attribute structure.
|
|       purpose: initialized the dsegment and/or presentation attribute
|                structure.
|
|         input: object    - the data object in which the dsegment is
|                            being built
|                segment   - the name of the segment in which we want
|                            to return the segment->
|                presentation - the presentation attributes to be initialized
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: Mark Young, Jeremy Worley
|          date: Nov 18, 1992 15:14
| modifications:
|
------------------------------------------------------------*/

int
_kdms_set_segment_info(kobject object, int segment, kpresentation * presentation)
{

   /*
    * set the presentation attribute associated with the string token.
    */
   if (klist_locate(object->presentations, (kaddr) ((long)segment)))
      object->presentations = klist_delete(object->presentations,
                                           (kaddr) ((long)segment));

   object->presentations = klist_add(object->presentations,
                                     (kaddr) ((long)segment), presentation);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_pres_struct
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 26, 1993 14:18
| Modifications:
|
------------------------------------------------------------*/

int 
_kdms_copy_pres_struct(kpresentation * src, kpresentation * dest)
{
   int i;

   dest->dimension = src->dimension;

   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      dest->size[i] = src->size[i];
      dest->order[i] = src->order[i];
      dest->offset[i] = src->offset[i];
   }

   dest->ragged = src->ragged;

   dest->coupling = src->coupling;

   dest->complex_convert = src->complex_convert;

   dest->interp = src->interp;

   dest->pad_real = src->pad_real;
   dest->pad_imag = src->pad_imag;

   dest->scaling = src->scaling;

   dest->scale_factor = src->scale_factor;

   dest->scale_offset_real = src->scale_offset_real;
   dest->scale_offset_imag = src->scale_offset_imag;

   dest->norm_min = src->norm_min;
   dest->norm_max = src->norm_max;

   dest->datatype = src->datatype;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  routine name: (static)__kdms_create_segment - creates and initializes a new
|                                   data segment.
|
|       purpose: creates a new data segment->  the new data segment
|                is malloc and initialized.  the object that the
|                data segment was initialized for is placed at the
|                head of the list.
|
|         input: object - the object in which we are creating the data
|                         segment for.
|                segment - the segment in which we will be creating the
|                          data segment and attribute structure.
|
|        output: none
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: Jeremy Worley
|          date: Nov 13, 1992 17:56
| modifications:
|
------------------------------------------------------------*/

static int
__kdms_create_segment(
                        kobject object,
                        int segment,
                        kpresentation * template)
{
   kpresentation *pres;
   int i;

   /*
    * create the presentation structure
    */
   pres = (kpresentation *) kcalloc(1, sizeof(kpresentation));

   if (pres == NULL)
      return (FALSE);

   pres->id = segment;

   /*
    * this stuff oughta be done in a different way:
    * Oct 05, 1993 22:38
    */
   pres->size = kmalloc(KDMS_MAX_DIM * sizeof(int));
   pres->order = kmalloc(KDMS_MAX_DIM * sizeof(int));
   pres->offset = kmalloc(KDMS_MAX_DIM * sizeof(int));

   /*
    * if necessary, initialize the segment
    */
   if (template == NULL)
   {
      pres->segment = (kdsegment *) kcalloc(1, sizeof(kdsegment));

      if (pres->segment == NULL)
         return (FALSE);

      pres->segment->id = pres->id;

      pres->segment->size = kcalloc(KDMS_MAX_DIM, sizeof(int));
      pres->segment->order = kcalloc(KDMS_MAX_DIM, sizeof(int));
      pres->segment->begin = kcalloc(KDMS_MAX_DIM, sizeof(int));
      pres->segment->end = kcalloc(KDMS_MAX_DIM, sizeof(int));

#if 0
      pres->segment->size = kmalloc(KDMS_MAX_DIM * sizeof(int));
      pres->segment->order = kmalloc(KDMS_MAX_DIM * sizeof(int));
      pres->segment->begin = kmalloc(KDMS_MAX_DIM * sizeof(int));
      pres->segment->end = kmalloc(KDMS_MAX_DIM * sizeof(int));
#endif
      pres->segment->permanence = TRUE;
      pres->segment->buffer_threshold = -1;

      pres->segment->outerbound = kcalloc(KDMS_MAX_DIM, sizeof(int));

#if 0
      pres->segment->outerbound = kmalloc(KDMS_MAX_DIM * sizeof(int));
#endif
      
      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         pres->size[i] = 0;
         pres->offset[i] = 0;
         pres->order[i] = -1;
         pres->segment->size[i] = 0;
         pres->segment->order[i] = -1;
         pres->segment->begin[i] = 0;
         pres->segment->end[i] = 0;
         pres->segment->outerbound[i] = 0;
      }
      pres->dimension = -1;

      pres->ragged = KDMS_FIXED;

      pres->complex_convert = KREAL;
      pres->scaling = KCAST;
      pres->interp = KPAD;
      pres->pad_real = 0.0;
      pres->pad_imag = 0.0;

      pres->coupling = KCOUPLED;

      pres->segment->update = FALSE;
      pres->segment->locked = FALSE;
      pres->segment->list = NULL;
   }
   else
   {
      _kdms_copy_pres_struct(template, pres);
      pres->segment = template->segment;
   }

   /*
    * add the parent object to the segment's "user" list
    */
   pres->segment->list = klist_add(pres->segment->list, object, NULL);

   /*
    * install the segment
    */
   _kdms_set_segment_info(object, segment, pres);

   /*
    * if the object is already locked then that means that we are trying to
    * create a segment on an input object.  So, we must create it with a
    * temporary to begin with.
    */

   pres->segment->temporary = object->phys->locked | object->phys->temporary;

   pres->segment->file = NULL;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_create_segment
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 26, 1993 08:07
| Modifications:
|
------------------------------------------------------------*/


int
_kdms_create_segment(
                       kobject object,
                       int segment,
                       kpresentation * template)
{
   kpresentation *pres;

   if (!__kdms_create_segment(object, segment, template))
      return (FALSE);

   if (!_kdms_get_segment_info(object, segment, &pres) ||
       !pres)
      return (FALSE);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  routine name: (static)__kdms_destroy_segment - the real destroy routine
|
|       purpose: destroy a object from the object list.  
|
|                the one thing that this routine does not do
|                is remove the dead segment from the linked list.
|
|         input: object   -  the object to be destroyed from the object list
|                dsegment -  the current object list
|
|        output: none
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: Jeremy Worley
|          date: May 10, 1993 15:49
| modifications:
|
------------------------------------------------------------*/

static int
__kdms_destroy_segment(kobject object, int segment)
{
   kpresentation *pres;

   /*
    * get the presentation
    */
   if (!_kdms_get_segment_info(object, segment, &pres))
      return (FALSE);

   /*
    * access callbacks occur before actual delete
    */
   if (pres->segment->num_callbacks != 0)
      _kdms_process_data_callbacks(object, pres, NULL, NULL,
                                   KDMS_CALLBACK_DELETE);

   /*
    * if the list becomes empty then we can destroy the data, but destroy the
    * presentation structure since this is never shared.
    */
   pres->segment->list =
      klist_delete(pres->segment->list, object);

   /*
    * free up the callbacks associated with this object.  by the way.
    * ya better d*mn well be sure this is called before the segment
    * structure is freed!
    */
   _kdms_delete_object_callbacks(object, segment);

   /*
    * free the presentation attributes
    */
   kdms_free_pres_attributes(pres);

   /*
    * If the object was the only one referencing this segment, then we can
    * remove all of its resources. yay!
    */
   if (!(pres->segment->list))
   {

      kfree(pres->segment->data);

      kfclose(pres->segment->file);

      kdms_free_segment_attributes(pres);
      kfree(pres->segment->attributes);
      kfree(pres->segment->begin);
      kfree(pres->segment->end);
      kfree(pres->segment->size);
      kfree(pres->segment->order);
      kfree(pres->segment->outerbound);
      kfree(pres->segment);
   }

   /*
    * free up other memory
    */
   kfree(pres->size);
   kfree(pres->order);
   kfree(pres->offset);
   kfree(pres);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  routine name: _kdms_destroy_segment - destroy a data services object from
|                                   the data segment list.
|
|       purpose: destroy a object from the object list.  this is done by
|                deleting the object from the linked list.  the new
|                list is then passed back to the calling routine.
|
|         input: object   -  the object to be destroyed from the object list
|                dsegment -  the current object list
|
|        output: none
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: Jeremy Worley
|          date: Jan 04, 1994 16:02
| modifications:
|
------------------------------------------------------------*/

int
_kdms_destroy_segment(kobject object, int segment)
{
   /*
    * first call the real destroy routine
    */
   __kdms_destroy_segment(object, segment);

   /*
    * delete the data segment and presentation from the object's list.
    */
   object->presentations = klist_delete(object->presentations,
                                        (kaddr) ((long)segment));
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_destroy_segments
|                - destroys all the data segments from
|                                    the data services object
|
|       Purpose: Delete all the data segments from the data services
|                object by calling _kdms_destroy_segment() with each dsegment->
|
|         Input: object   -  The object to be destroyed from the object list
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Sep 15, 1992 16:44
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_destroy_segments(kobject object)
{
   klist *list,
     *temp;


   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   list = object->presentations;
   while (list != NULL)
   {
      temp = klist_next(list);
      __kdms_destroy_segment(object, klist_token(list));
      list = temp;
   }
   klist_free(object->presentations, NULL);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _reference_segment
|
|       Purpose: creates a presentation
|                that points to another presentation's segment->
|
|         Input: in_obj    - the input object to copy from
|                out_obj   - the output object to copy to
|                segment   - the name of the segment in which we want
|                            to add to the data segment list.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 23, 1993 10:25
| Modifications:
|
------------------------------------------------------------*/

static int
_reference_segment(kobject in_obj, int segment, kobject out_obj)
{
   kpresentation *pres1 = NULL,
     *pres2 = NULL;
   int i;

   /*
    * get the presentation associated with the first segment.
    */
   if (!_kdms_get_segment_info(in_obj, segment, &pres1) || !pres1)
   {
      return (FALSE);
   }

   /*
    * check validity of segment.  If certain attributes have not
    * been initialized, then generate a verbose warning message.
    */
   if (pres1->dimension <= 0 ||
       !_kdms_legal_datatype(pres1->datatype) ||
       !_kdms_legal_size(pres1->size,pres1->dimension)) 
      kinfo(KSYSLIB,"kdms_reference_segment:  The %s segment in the object "
	    "associated with %s has not been fully initialized.  The "
	    "datatype, size, index order and dimension should be set before "
	    "attempting to operate on this segment (index order and dimension "
	    "are initialized for you if you are using an application service. "
	    "A side effect of the "
	    "kdms_reference_segment function is that the reference object "
	    "is decoupled.  This means that the attributes mentioned above "
	    "should be set on the original segment.", 
	    ktoken_to_string(segment), in_obj->phys->filename);


   /*
    * try to get the presentation associated with the second segment. if it
    * exists then destroy it.
    */
   if (_kdms_get_segment_info(out_obj, segment, &pres2))
      _kdms_destroy_segment(out_obj, segment);

   /*
    * create the presentation structure
    */
   pres2 = (kpresentation *) kcalloc(1, sizeof(kpresentation));

   if (pres2 == NULL)
      return (FALSE);

   /*
    * call the update segment first
    */
   if (pres1->segment->update == TRUE)
      _kdms_update_segment(in_obj, pres1);

   /*
    * copy all of the attributes.
    */
   *pres2 = *pres1;


   /* copy all of the user defined attributes */
   pres2->attributes = NULL;
   if (pres1->attributes != NULL)
      for (i = 0; i < pres1->attributes->nattributes; i++)
         kdms_establish_attribute(&(pres2->attributes),
                                  pres1->attributes->attributes[i]);

   /*
    * this stuff oughta be done in a different way:
    * Oct 05, 1993 22:38
    */
   pres2->size = kmalloc(KDMS_MAX_DIM * sizeof(int));
   pres2->order = kmalloc(KDMS_MAX_DIM * sizeof(int));
   pres2->offset = kmalloc(KDMS_MAX_DIM * sizeof(int));
   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      pres2->size[i] = pres1->size[i];
      pres2->order[i] = pres1->order[i];
      pres2->offset[i] = pres1->offset[i];
   }

   /*
    * default coupling
    */
   pres2->coupling = KDEMAND;

   _kdms_set_segment_info(out_obj, segment, pres2);

   /*
    * very important to make sure that we add the reference to the list
    */
   pres1->segment->list = klist_add(pres1->segment->list, out_obj, NULL);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_reference_segments - creates presentations
|                that points to another presentation's segment->
|
|       Purpose:
|                WILL NOT WORK IF LIST IS CIRCULAR
|
|         Input: in_obj    - the input object to copy from
|                out_obj   - the output object to copy to
|                segment   - the name of the segment in which we want
|                            to add to the data segment list.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 23, 1993 10:25
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_reference_segments(kobject in_obj, kobject out_obj)
{
   klist *cur;

   for (cur = klist_head(in_obj->presentations); cur != NULL;
        cur = klist_next(cur))
      _reference_segment(in_obj, klist_token(cur), out_obj);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_establish_segment
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Sep 01, 1993 16:29
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_establish_segment(
                          kobject src_obj,
                          int src_segment,
                          kobject dest_obj,
                          int dest_segment)
{
   kpresentation *src_pres = NULL,
     *dest_pres = NULL;
   int i;

   if (!_kdms_get_segment_info(src_obj, src_segment, &src_pres)
       || !src_pres)
      return (FALSE);

   /*
    * if the destination object already has a segment, and it has a legal
    * data type and size, then we should leave the segment alone.
    * Otherwise, copy the storage related attributes only from the src to
    * the destination.
    */
   if (!_kdms_get_segment_info(dest_obj, dest_segment, &dest_pres)
       || !dest_pres)
   {
      _kdms_create_segment(dest_obj, dest_segment, NULL);
      _kdms_get_segment_info(dest_obj, dest_segment, &dest_pres);
      for (i = 0; i < src_pres->dimension; i++)
      {
         dest_pres->order[i] = src_pres->order[i];
         dest_pres->segment->order[i] = src_pres->segment->order[i];
         dest_pres->size[i] = src_pres->size[i];
         dest_pres->segment->size[i] = src_pres->segment->size[i];
      }
      dest_pres->datatype = dest_pres->segment->datatype = src_pres->datatype;
      dest_pres->dimension = dest_pres->segment->dimension =
         src_pres->dimension;
   }
   else
   {
      if (dest_pres->dimension <= 0)
         dest_pres->dimension = src_pres->dimension;

      if (!_kdms_legal_datatype(dest_pres->datatype))
         dest_pres->datatype = src_pres->datatype;

      if (!_kdms_legal_size(dest_pres->size,dest_pres->dimension))
      {
         for (i = 0; i < src_pres->dimension; i++)
            dest_pres->size[i] = src_pres->size[i];
         for (i = src_pres->dimension; i < KDMS_MAX_DIM; i++)
            dest_pres->size[i] = 1;
      }

      if (!_kdms_legal_order(dest_pres->order, dest_pres->dimension))
         for (i = 0; i < src_pres->dimension; i++)
            dest_pres->order[i] = src_pres->order[i];
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_lock_segment
|
|       Purpose: Locks the physical layer. Also deletes
|                all non-permanent segments.
|
|         Input: object
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 23, 1993 13:08
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_lock_segment(kobject object, kpresentation * pres)
{
   kdsegment *dsegment = pres->segment;
   int tmp[KDMS_MAX_DIM],
      old[KDMS_MAX_DIM],
      siz[KDMS_MAX_DIM],
      i;

   /*
    * don't wanna do this stuff if the segment is already locked.
    */
   if (dsegment->locked == TRUE)
      return (TRUE);

   if (dsegment->permanence == FALSE)
   {
      _kdms_destroy_segment(object, pres->id);
      return (FALSE);
   }

   /*
    * these things are important so that we how to open the output file.
    */
   if (pres->segment->update == TRUE)
      _kdms_update_segment(object, pres);

/*    _kdms_initialize_segment(object, pres); */

   dsegment->locked = TRUE;
   pres->coupling = KUNCOUPLED;

   /*
    * This stuff is totally obsolete.  It'll be deleted soon after 2.0
    */
   if (pres->id == kstring_to_token(KDMS_SEGMENT_MAP) || pres->id ==
       kstring_to_token(KDMS_SEGMENT_LOCATION))
   {
      for (i = 0; i < 5; i++)
      {
         old[i] = pres->order[i];
         siz[i] = pres->size[i];
      }

      _kdms_glue_order(object, ktoken_to_string(pres->id), pres->order);

      tmp[0] = _kdms_get_order(pres->order[0], old,
                               pres->size, pres->dimension, 1);
      tmp[1] = _kdms_get_order(pres->order[1], old,
                               pres->size, pres->dimension, 1);
      tmp[2] = _kdms_get_order(pres->order[2], old,
                               pres->size, pres->dimension, 1);
      tmp[3] = _kdms_get_order(pres->order[3], old,
                               pres->size, pres->dimension, 1);
      tmp[4] = _kdms_get_order(pres->order[4], old,
                               pres->size, pres->dimension, 1);
      for (i = 0; i < 5; i++)
         pres->size[i] = tmp[i];

      pres->segment->locked = FALSE;
      _kdms_update_segment(object, pres);
      pres->segment->locked = TRUE;

      for (i = 0; i < pres->dimension; i++)
      {
         pres->order[i] = old[i];
         pres->size[i] = siz[i];
      }
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_lock_segments
|
|       Purpose: Locks the physical layer
|
|         Input: object
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 23, 1993 13:01
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_lock_segments(kobject object)
{
   klist *list;
   kpresentation *pres = NULL;
   int nonbuffered = object->phys->routines->write == NULL;
   int i;
   
   if (object == NULL || object->type != KOBJ_DATA)
   {
      _kdms_set_error(KDMS_EOBJ_INVALID);
      return (FALSE);
   }

   list = object->presentations;

   while (list != NULL)
   {
      _kdms_get_segment_info(object, klist_token(list), &pres);

      if (nonbuffered && pres->segment->file != NULL)
	 _kdms_flush_segment(object, pres);

      _kdms_lock_segment(object, pres);

      /*
       * this strange chunk of code copes with problems related
       * to large data sets stored to non-bufferable file formats.
       * I hate this approach, but it fixes a potentially very
       * ugly bug reported by Hank van Bekkam (Thanks hank).
       */
      if (nonbuffered && pres->segment->file != NULL)
      {
	 kfree(pres->segment->data);
	 for (i = 0; i < pres->segment->dimension; i++)
	 {
	    pres->segment->begin[i] = 0;
	    pres->segment->end[i] = pres->segment->size[i] - 1;
	 }
	 
	 pres->segment->data = _kdms_transport_segment_data(
                                       pres->segment->file,
				       kfile_getmachtype(kfileno(pres->segment->file)),
				       kfread_generic,
				       kfseek,
				       0,
                                       pres->segment->datatype,
                                       pres->segment->dimension,
                                       pres->segment->size,
				       NULL,
                                       &pres->segment->data,
                                       pres->segment->begin, 
				       pres->segment->end);
      }
      
      list = klist_next(list);
   }

   /*
    * well, it looks like its time to lock the object.  why here? for some of
    * the file formats, the header must be locked in order to know where each
    * segment should be placed relative to the top of the file (Viff for
    * example).  Of course, if its already been done, don't do it again.
    */
   if (object->phys->resources != NULL)
      return (TRUE);

   if (!_kdms_glue_output(object, object->phys->filename, object->phys->flags))
      return (TRUE);

   _kdms_copy_temps(object);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_set_segment
|
|       Purpose: Sets the values of the data segment->  This should
|                be done by the file format glue layer (viff,tiff,etc).
|
|         Input: object     - the data object in which the dsegment is
|                             being built
|                segment    - the segment to set the data for
|                data       - the segment data
|                datatype   - the segment data type
|                order      - index order of the data.
|                size       - size of the data.
|                begin      - beginning corner of data (if data!=NULL)
|                end        - ending corner of data (if data!=NULL)
|                offset     - the data offset from the beginning of
|                             the transport stream
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 22, 1993 13:27
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_set_segment(
                    kobject object,
                    char *segment,
                    kaddr data,
                    int datatype,
                    int *order,
                    int *size,
                    int *begin,
                    int *end,
                    int dimension)
{
   kpresentation *pres;
   int token,
      i;

   if (!segment)
   {
      _kdms_set_error(KDMS_ESEG_VALID);
      return (FALSE);
   }
   token = kstring_to_token(segment);

   /*
    * create the segment.  We query first because if the segment
    * already exists, then we can reuse it.  This helps to facilitate
    * the kdms_reopen functionality.
    */
   if (!kdms_query_segment(object, segment))
      if (!_kdms_create_segment(object, token, NULL))
         return (FALSE);

   if (!_kdms_get_segment_info(object, token, &pres))
      return (FALSE);

   /*
    * make sure that the size is at least 1
    */
   if (size != NULL)
   {
      for (i = 0; i < dimension; i++)
         pres->size[i] = (size[i] > 0) ? size[i] : 1;
      for (i = dimension; i < KDMS_MAX_DIM; i++)
         pres->size[i] = 1;
   }
   else
   {
      pres->size[0] =
         pres->size[1] =
         pres->size[2] =
         pres->size[3] =
         pres->size[4] = -1;
   }

   /*
    * set the index order
    */
   pres->dimension = pres->segment->dimension = dimension;

   if (order != NULL)
   {
      for (i = 0; i < pres->dimension; i++)
         pres->order[i] = pres->segment->order[i] = order[i];
   }
   else
   {
      pres->order[0] =
         pres->order[1] =
         pres->order[2] =
         pres->order[3] =
         pres->order[4] = -1;
   }

   /*
    * set the data type
    */
   pres->datatype = datatype;

   /*
    * instantiate a the physical layer
    */
   pres->segment->initialized = TRUE;

   if (!object->phys->headerless)
   {
      pres->segment->update = TRUE;
      _kdms_lock_segment(object, pres);
   }

   if (pres->segment->data != NULL)
      kfree(pres->segment->data);
   
   pres->segment->data = data;

   if (data != NULL)
   {
      for (i = 0; i < pres->segment->dimension; i++)
      {
         pres->segment->begin[i] = begin[i];
         pres->segment->end[i] = end[i];
      }
   }
   else
   {
      for (i = 0; i < pres->segment->dimension; i++)
	 pres->segment->begin[i] = pres->segment->end[i] = 0;
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_get_segment
|
|       Purpose: sets the values of the data segment->  this should
|                be done by the file format glue layer (viff,tiff,etc).
|
|         Input: object     - the data object in which the dsegment info
|                             being retrieved
|                segment    - the segment to get the data for
|        Output: data       - the segment data
|                datatype   - the segment data type
|                order      - index order of data
|                size       - size of data
|                begin      - beginning corner of data
|                end        - ending corner of data
|                offset     - the data offset from the beginning of
|                             the transport stream
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 22, 1993 13:34
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_get_segment(
                    kobject object,
                    char *segment,
                    kaddr * data,
                    int *datatype,
                    int **order,
                    int **size,
                    int **begin,
                    int **end,
                    int *dimension)
{
   kdsegment *dsegment;
   kpresentation *pres;
   int token;

   token = kstring_to_token(segment);

   /*
    * if the segment does not exist, return FALSE...
    */
   if (klist_locate(object->presentations, (kaddr) ((long)token)) == NULL)
   {
      return (FALSE);           /* This is not an error */
   }

   if (!_kdms_get_segment_info(object, token, &pres))
      return (FALSE);

   dsegment = pres->segment;

   if (data && dsegment)
      *data = dsegment->data;
   if (datatype)
      if (dsegment)
         *datatype = dsegment->datatype;
      else
         *datatype = pres->datatype;

   /*
    * retrieve the desired sizes and current begin and end positions
    * we are intentionally returning a pointer to the actual structure.
    */
   if (size)
      *size = (dsegment->locked) ? dsegment->size : pres->size;

   /*
    * retrieve the desired index order information
    */
   if (order)
      *order = (dsegment->locked) ? dsegment->order : pres->order;

   if (begin)
      *begin = (dsegment->locked) ? dsegment->begin : *begin;

   if (end)
      *end = (dsegment) ? dsegment->end : *end;

   if (dimension)
      *dimension = (dsegment) ? dsegment->dimension : pres->dimension;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_update_segment
|
|       Purpose: This function is called when the segment
|                is not locked and a storage related attribute
|                has been changed.
|
|                What this does is flush the segment and
|                then create a new file descriptor and
|                push it through the get and put routines.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jun 01, 1993 13:23
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_update_segment(kobject object, kpresentation * pres)
{
   kpresentation ptmp;
   kdsegment stmp;
   int i;
   int order_match;

   pres->segment->update = FALSE;

   if (!(_kdms_legal_datatype(pres->segment->datatype) &&
         _kdms_legal_size(pres->segment->size, pres->segment->dimension) &&
         _kdms_legal_order(pres->segment->order, pres->segment->dimension)))
   {
      pres->segment->datatype = pres->datatype;
      pres->segment->dimension = pres->dimension;
      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         pres->segment->size[i] = pres->size[i];
         pres->segment->order[i] = pres->order[i];
      }
      return (TRUE);
   }

   /*
    * if everything is the same, don't waste your time! dammit!
    */
   order_match = (pres->segment->order[0] == pres->order[0] &&
		  pres->segment->order[1] == pres->order[1] &&
		  pres->segment->order[2] == pres->order[2] &&
		  pres->segment->order[3] == pres->order[3] &&
		  pres->segment->order[4] == pres->order[4]);

   if (pres->segment->size[0] == pres->size[0] &&
       pres->segment->size[1] == pres->size[1] &&
       pres->segment->size[2] == pres->size[2] &&
       pres->segment->size[3] == pres->size[3] &&
       pres->segment->size[4] == pres->size[4] && 
       order_match &&
       pres->segment->datatype == pres->datatype)
      return (TRUE);

   /*
    * If nothing has been written, then no point in updating.
    */
   if (pres->segment->data == NULL && pres->segment->file == NULL &&
      pres->segment->locked != TRUE)
   {
      pres->segment->datatype = pres->datatype;
      pres->segment->dimension = pres->dimension;
      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         pres->segment->size[i] = pres->size[i];
         pres->segment->order[i] = pres->order[i];
      }
      return (TRUE);
   }

#if 0

   /* -- not clear if this works for data type changes ... -- */
   /* -- kconvert seems to break ... -- */

   /* 
    *  If the sizes are almost the same except in the trailing 
    *  index dimension, then we're still cool and don't have to
    *  panic about stuff.
    */
   if (order_match)
   {
      for (i = 0, size_match = TRUE; i < pres->segment->dimension-1; i++)
	 size_match  &= (pres->segment->size[i] == pres->size[i]);

      if (size_match)
	 return TRUE;
   }
#endif

   /*
    * create a new temporary file for the segment, transfer the contents...
    */
   ptmp = *pres;

#if 0
   ptmp.scaling         = KNONE;
   ptmp.complex_convert = KREAL;
   ptmp.interp          = KPAD;
#endif
   
   /*
    * fixup the temporary presentation pointers
    */
   ptmp.order = (int *)kmalloc(KDMS_MAX_DIM * sizeof(int));
   kmemcpy(ptmp.order, pres->order, KDMS_MAX_DIM * sizeof(int));

   ptmp.size = (int *)kmalloc(KDMS_MAX_DIM * sizeof(int));
   kmemcpy(ptmp.size, pres->size, KDMS_MAX_DIM * sizeof(int));


   stmp = *(pres->segment);

   /* 
    * fixup the pointers in the temporary segment
    */
   stmp.order = (int *)kmalloc(KDMS_MAX_DIM * sizeof(int));
   kmemcpy(stmp.order, pres->segment->order, KDMS_MAX_DIM * sizeof(int));

   stmp.size = (int *)kmalloc(KDMS_MAX_DIM * sizeof(int));
   kmemcpy(stmp.size, pres->segment->size, KDMS_MAX_DIM * sizeof(int));

   stmp.begin = (int *)kmalloc(KDMS_MAX_DIM * sizeof(int));
   kmemcpy(stmp.begin, pres->segment->begin, KDMS_MAX_DIM * sizeof(int));

   stmp.end = (int *)kmalloc(KDMS_MAX_DIM * sizeof(int));
   kmemcpy(stmp.end, pres->segment->end, KDMS_MAX_DIM * sizeof(int));

   ptmp.segment = &stmp;

   pres->segment->file = NULL;
   pres->segment->datatype = pres->datatype;
   kmemcpy(pres->segment->size, pres->size, KDMS_MAX_DIM * sizeof(int));
   kmemcpy(pres->segment->order, pres->order, KDMS_MAX_DIM * sizeof(int));

   /*
    * zero out the begin and end markers and set the pointer to null.
    * this is done so that if you are sizing upward, the buffer gets
    * realloced correctly and there are no unexpected flushes.  Failure
    * do this causes problems if there is a data type change or size
    * increase
    */
   pres->segment->begin[0] =
      pres->segment->begin[1] =
      pres->segment->begin[2] =
      pres->segment->begin[3] =
      pres->segment->begin[4] =
      pres->segment->end[0] =
      pres->segment->end[1] =
      pres->segment->end[2] =
      pres->segment->end[3] =
      pres->segment->end[4] = 0;
   pres->segment->data = NULL;

   _kdms_migrate_segment_data(object, &ptmp, object, pres);

   kfclose(stmp.file);
   kfree(stmp.data);
   kfree(stmp.begin);
   kfree(stmp.end);
   kfree(stmp.order);
   kfree(stmp.size);
   kfree(ptmp.order);
   kfree(ptmp.size);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_initialize_segment
|
|       Purpose: This thing copies attributes from the
|                presentation to the segment.  It then
|                sets the initialized flag.
|
|         Input: object - object to act on.
|                pres - presentaiton to act on.
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jun 01, 1993 13:42
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_initialize_segment(kpresentation * pres)
{
   if (pres == NULL)
   {
      _kdms_set_error(KDMS_EINTERNAL);
      return (FALSE);
   }

   pres->segment->initialized = TRUE;

   pres->segment->datatype = pres->datatype;

   kmemcpy(pres->segment->size, pres->size, KDMS_MAX_DIM * sizeof(int));
   kmemcpy(pres->segment->order, pres->order, KDMS_MAX_DIM * sizeof(int));

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _copy_segment_data
|
|       Purpose: How many gotos can *you* use in a meaningful
|                way in less than 10 lines of code? (and still 
|                keep your sanity!)
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 06, 1993 09:45
| Modifications:
|
------------------------------------------------------------*/

static int
_copy_segment_data(
                     kobject object1,
                     kpresentation * pres1,
                     kobject object2,
                     kpresentation * pres2)
{
   int begin[KDMS_MAX_DIM] =
   {0, 0, 0, 0, 0};
   int end[KDMS_MAX_DIM] =
   {0, 0, 0, 0, 0};
   int chunk[KDMS_MAX_DIM] =
   {1, 1, 1, 1, 1};
   int i;
   kaddr data = NULL;

   /*
    * So, if the the whole data set is in memory, then there is no need
    * to go through an iterative get-put thing with a smaller chunk size.
    * Instead, for performance, just grab the physical buffer directly	
    * and do a put data.  This will only work, however, if the source 
    * physical data type is the same as the destination's presenation 
    * data type.  Its cheating, but it should cut the number of pipeline
    * executions in programs that repeatedly resize the physical layer of
    * an object.
    */
#if 0

   /* -- if this is happening as a result of a kdms_copy_object, the
         begin and end points on pres1->segment appear to not yet be
	 initialized -- */

   if (pres1->segment->file == NULL && 
       pres1->segment->datatype == pres2->datatype)
      return _kdms_put_data(object2, pres2, pres1->segment->begin, 
			    pres1->segment->end, pres1->segment->data);
#endif

   _kdms_compute_chunk_size(pres1->size, pres1->dimension, chunk,
			    pres1->datatype, 
			    pres1->segment->buffer_threshold);
   
      for (i = 0; i < KDMS_MAX_DIM; i++)
	 end[i] = chunk[i] - 1;
   
   /*
    * dont want to generate any callbacks during this operation.
    */
   _kdms_disable_callbacks();

   /*
    * whip through all of the data getting it from the source and
    * writing to the destination.
    */
   do
   {
      data = _kdms_get_data(object1, pres1, begin, end, data);
      _kdms_put_data(object2, pres2, begin, end, data);
   }
   while (_kdms_advance_position(begin, end, pres1->size, chunk,
				 pres1->dimension));

   kfree(data);

   _kdms_enable_callbacks();

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_migrate_segment_data
|
|       Purpose: This is used to copy data from one
|                segment to another without other
|                side effects.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1993 12:15
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_migrate_segment_data(
                             kobject src_obj,
                             kpresentation * pres1,
                             kobject dest_obj,
                             kpresentation * pres2)
{
   kpresentation temp;

   /*
    * flush the src object....the flush really doesn't do anything for read
    * only objects.  Its important to do this for read/write objects.
    * the flush will work with or without the conditional, however if
    * we have everything in memory already anyway, we can avoid the flush
    * to save major time.
    */
   if (pres1->segment->file != NULL)
      _kdms_flush_segment(src_obj, pres1);

   /*
    * this goofiness is here so that you can guarantee that if some user
    * changes the sizes of their data prior to doing a copy data, then it
    * will still work.
    */
   temp = *pres2;
   temp.scaling = KNONE;
   temp.segment = pres1->segment;

   /*
    * copy the data from the source object to the destination
    * object.
    */
   _copy_segment_data(src_obj, &temp, dest_obj, pres2);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_rename_segment
|       Purpose: Changes the name of a segment.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 14, 1994 18:07
| Modifications:
|
------------------------------------------------------------*/
 
int
_kdms_rename_segment(kobject object, char *old_name, char *new_name)
{
   klist *list;
   kpresentation *pres;
   int old_token = kstring_to_token(old_name);
   int new_token = kstring_to_token(new_name);

   if (!_kdms_get_segment_info(object, old_token, &pres) || !pres)
      return(FALSE);

   pres->segment->id = new_token;

   /*
    * update all of the reference objects by renaming them as well.
    */
   for (list = klist_head(object->phys->list); list != NULL;
        list = klist_next(list))
   {	
      if (_kdms_get_segment_info(list->identifier, old_token, &pres))
      {
	    pres->id = new_token;

	    object->presentations = klist_delete(object->presentations,
						 (kaddr) ((long)old_token));

	    object->presentations = klist_add(object->presentations,
					      (kaddr) ((long)new_token), pres);
      }
   }

   return (TRUE);
}
