 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                 Data Access Routines
   >>>>
   >>>>   Static:
   >>>>
   >>>>         __compute_skips
   >>>>         __compute_strides
   >>>>
   >>>>         __clip
   >>>>         _clip
   >>>>
   >>>>         __transpose
   >>>>         _transpose
   >>>>
   >>>>         _cast
   >>>>
   >>>>         _scale
   >>>>
   >>>>         _normalize
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <float.h>

#define Round(x) (kfloor(x + 0.5))

#define KDMS_MAX_NUMBER_POINTS 500000

extern int _kdms_buffer_threshold;
extern int _kdms_compiled;

/*
 * These static variables will eventually be replaced with variables
 * kept in a pipeline stage structure.
 */
static kaddr tmp1 = NULL;
static int tmp1_size = 0;

static kaddr tmp2 = NULL;
static int tmp2_size = 0;

static kaddr tmp3 = NULL;
static int tmp3_size = 0;

/*-----------------------------------------------------------
|
|  Routine Name: (static) kdms_optimize_bufsize
|
|       Purpose: Optimizes the size of the buffer to be
|		 maintained in memory.  The way it is
|		 done here should minimize the number of
|		 of seeks.  It first fills out of the
|		 lowest order then the next, etc. until
|		 a threshold size is exceeded.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 24, 1993 14:54
| Modifications:
|
------------------------------------------------------------*/

static int
kdms_optimize_bufsize(
			int *size,
			int datatype,
			int *begin,
			int *end,
			int dim,
			int buffer_threshold)
{
   int dsize = kdata_size(datatype);
   int zero[KDMS_MAX_DIM] =
   {0, 0, 0, 0, 0};
   int num;
   int sz,
      i;

   if (dsize == 0)
   {
      _kdms_set_error(KINVALID_DATATYPE);
      return (FALSE);
   }

   /*
    * this code twiddles the size of the buffer.  If the buffer_threshold
    * is -1, then use the constant or the environment variable.  The
    * Numer of points #define is there to make things a little smarter
    */
   if (buffer_threshold >= 0)
      num = kmax(buffer_threshold / dsize, 1);
   else
      num = kmax((_kdms_buffer_threshold / dsize), KDMS_MAX_NUMBER_POINTS);

   for (i = 0; i < dim; i++)
   {
      begin[i] = kmax(begin[i], 0);
      end[i] = kmax(0, kmin(end[i], size[i] - 1));
   }
   
   for (i = 0; i < dim; i++)
   {
      if (_kdms_num_elements(begin, end, dim) < num)
      {
	 sz = end[i] - begin[i] + 1;
	 if (sz <= 0) 
	 {
	    kerror("kdataman", "kdms_optimize_bufsize",
		   "Internal error. begin and end markers are illegal."
	           "This problem is so bad, that we're gonna assert here.");
	    return (FALSE);
	 }
	 
	 while (_kdms_num_elements(begin, end, dim) < num && 
		end[i] < size[i] - 1)
	    end[i] += sz;

	 end[i] = kmin(end[i], size[i] - 1);
      }
      else
      {
	 return (TRUE);
      }
   }

   /*
    * here's an exception that could make certain cases much faster.
    */
   if (_kdms_num_elements(zero, end, dim) <= num)
      for (i = 0; i < dim; i++)
	 begin[i] = 0;

   /*
    * onward...
    */
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  routine name: _kdms_flush_segment
|
|       purpose: this function flushes a segment of data.
|		 this routine is intended to be used to
|		 generically dump the contents of a certain
|		 segment's buffer to its output transport.
|
|		 the reason that this routine exists is
|		 that we probably do not want to strictly
|		 read and write the data in the same way
|		 that the programmer want to retrieve the
|		 data from data services.  for example,
|		 if the programmer want to get a single
|		 point, it would be stupid for us to have
|		 a buffer size of a single point.  we would
|		 really want to do lines or planes or
|		 something...furthermore, we may want to
|		 buffer sublines or tiles...this function
|		 can be as complicated as we want it to
|		 be...i suggest adding a field to each
|		 segment that describes how to buffer the
|		 stupid thing.
|
|         input: object   -
|		 dsegment -
|
|        output:
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: jeremy worley
|          date: oct 21, 1992 14:40
| modifications:
|
------------------------------------------------------------*/

int
_kdms_flush_segment(kobject object, kpresentation * pres)
{
   /*
    * If the segment has not been initialized, then this is the first attempt
    * to write.  In this case, instantiate the physical segment.
    */
   if (!(_kdms_legal_datatype(pres->segment->datatype) &&
       _kdms_legal_order(pres->segment->order, pres->segment->dimension) &&
       _kdms_legal_size(pres->segment->size, pres->segment->dimension)))
      if (!_kdms_initialize_segment(pres))
	 return (FALSE);

   /*
    * Aug 04, 1993 10:29
    * if the index order in the segment is different than the index order
    * in the presentation, then update the segment.  This calls this
    * routine again, so we must make sure that this same condition
    * doesn't exist in the second call.  This is managed by the update
    * call.
    */
   if (pres->segment->update == TRUE)
      return (_kdms_update_segment(object, pres));

   if (pres->segment->data != NULL && ((object->phys->flags & KOBJ_WRITE) ||
				       pres->segment->temporary == TRUE))
      _kdms_write_data(object, pres->segment,
		       pres->segment->begin, pres->segment->end);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_flush_segments - flush all the data segments from
|				     the data services object
|
|       Purpose: Flush all the data segments from the data services
|		 object by calling kdsegment_flush() with each dsegment->
|
|         Input: object   -  The object to flush from the object list
|
|        Output: none
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Sep 15, 1992 16:44
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_flush_segments(kobject object)
{
   klist *list;
   int status = TRUE;
   kpresentation *pres;

   /*
    * Need to flush all the data segments before closing the object.
    */
   list = object->presentations;
   while (list != NULL)
   {
      if (_kdms_get_segment_info(object, klist_token(list),
				 &pres) && pres)
      {
	 if (pres->segment->file == NULL)
	    return (TRUE);
	 if (!_kdms_flush_segment(object, pres))
	    status = FALSE;
	 kfflush(pres->segment->file);
      }
      list = klist_next(list);
   }

   return (status);
}
/*-----------------------------------------------------------
|
|  Routine Name: (static) __need_to_preset
|       Purpose: Determines if the data need to be preset to
|                the pad value.  This can occur when part of
|                the space requested is outside of the real
|                dataspace.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: May 07, 1994 09:57
| Modifications:
|
------------------------------------------------------------*/

static int
__need_to_preset(kpresentation * pres, int *begin, int *end)
{
   int temp1[KDMS_MAX_DIM],
      temp2[KDMS_MAX_DIM];

   if (!_kdms_in_pres_data_space(pres, begin, end))
      return (TRUE);

   _kdms_pres2phys(pres->segment, pres, begin, temp1, 0);
   _kdms_pres2phys(pres->segment, pres, end, temp2, 0);

   if (!_kdms_in_phys_data_space(pres->segment, temp1, temp2))
      return (TRUE);

   return (FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: __adjust_size
|       Purpose: Readjusts the size of a static array according
|                to the current needs of the mock pipeline
|                being executed.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 27, 1994 09:48
| Modifications:
|
------------------------------------------------------------*/

kaddr 
__adjust_size(
		kaddr * ptr,
		int *size,
		int num_pts,
		int datatype)
{
   register int total = num_pts * kdata_size(datatype);

   if (total <= *size)
      return (*ptr);

   *ptr = krealloc(*ptr, (unsigned)total);

   *size = (*ptr == NULL) ? 0 : total;

   return (*ptr);
}

/*-----------------------------------------------------------
|
|  Routine Name: _in_phys_buffer
|       Purpose: Returns true if current slab is in the
|                current physical buffer and false if any
|		 part of the data requested, that exists
|		 is not in the physical buffer.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 14:18
| Modifications:
|
------------------------------------------------------------*/

static int
__in_phys_buffer(kdsegment * segment, int *begin, int *size)
{
   register int i;

   if (segment->data == NULL)
      return (FALSE);		/* not  an error */

   for (i = 0; i < segment->dimension; i++)
      if (kmax(0, begin[i]) < segment->begin[i] ||
	  kmin(segment->size[i], begin[i] + size[i] + 1) > segment->end[i] + 1)
	 return (FALSE);	/* not an error */

   return (TRUE);
}
/*-----------------------------------------------------------
|
|  Routine Name: __same_order
|       Purpose: determines if two index orders are equivelant
|         Input:
|        Output:
|       Returns: TRUE if the index orders are equivalent, FALSE
|                if they are not equivalent
|    Written By: Jeremy Worley
|          Date: May 03, 1994 08:48
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED*/
static int
__same_order(
	       int *a,
	       int *asize,
	       int *b,
	       int *bsize,
	       int dimension)
{
   int i;
   int j;
   int status = TRUE;
   
   /*
    * if the index orders are exactly the same, forget all of the 
    * "smart" nonsense below. 
    */
   for (i = 0; i < dimension; i++)
      status &= (a[i] == b[i]);

   if (status)
      return (TRUE);

   /*
    * step through the dimensions in which the both axes are non-zero.
    * where non-zero, the two axes must have the same index order.
    * if they don't, then this thing returns false.
    */
   for (i = 0, j = 0; i < dimension && j < dimension;) 
   {
      while (i < dimension && asize[i] == 1)
	 i++;

      while (j < dimension && bsize[j] == 1)
	 j++;

      if (i >= dimension || j >= dimension)
	 break;
      
      if (a[i] != b[j])
	 return (FALSE);
      else
      {
	 i++;
	 j++;
      }
   }

   /*
    * at this point, either a or b has been completely checked.  Now
    * walk through the other case and make sure that any > 1 sizes
    * occur on indices in which the other array set has only size
    * 1 axes on the higher order.
    */
   while (i < dimension)
   {
      if (asize[i] > 1) 
      {
	 j = 0;
	 while (j < dimension && b[j] != a[i]) 
	    j++;
	 for (; j < dimension; j++)
	    if (bsize[j] > 1)
	       return(FALSE);
      }
      i++;
   }

   while (j < dimension)
   {
      if (bsize[j] > 1) 
      {
	 i = 0;
	 while (i < dimension && a[i] != b[j]) 
	    i++;
	 for (; i < dimension; i++)
	    if (asize[i] > 1)
	       return(FALSE);
      }
      j++;
   }
	
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __compute_skips
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:48
| Modifications:
|
------------------------------------------------------------*/

static int
__compute_skips(
		  int dimension,
		  int *src_stride,
		  int *dst_stride,
		  int *src_unit,
		  int *dst_unit,
		  int *src_skip,
		  int *dst_skip)
{
   register int i;

   /*
    * compute the skip factors from the strides and the unit size.
    * this is done because we allow access of subspaces in both the
    * input and output. 
    */
   src_skip[0] = src_stride[0];
   dst_skip[0] = dst_stride[0];

   for (i = 1; i < dimension; i++)
   {
      src_skip[i] = src_stride[i] - src_stride[i - 1] * src_unit[i - 1];
      dst_skip[i] = dst_stride[i] - dst_stride[i - 1] * dst_unit[i - 1];
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __compute_strides
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:48
| Modifications:
|
------------------------------------------------------------*/

static int
__compute_strides(
		    int datatype,
		    int dimension,
		    int *src_size,
		    int *dst_size,
		    int *src_stride,
		    int *dst_stride)
{
   register int i;
   register int datasize = kdata_size(datatype);

   /*
    * compute the striding into the data spaces.
    */
   if (datatype == KBIT) 
   {
      src_stride[0] = 0;
      dst_stride[0] = 0;
      src_stride[1] = (src_size[0] + 7) >> 3;
      dst_stride[1] = (dst_size[0] + 7) >> 3;
      
      for (i = 2; i < dimension; i++) 
      {
	 src_stride[i] = src_stride[i - 1] * src_size[i - 1];
	 dst_stride[i] = dst_stride[i - 1] * dst_size[i - 1];
      }
   }
   else 
   {
      for (i = 1, src_stride[0] = dst_stride[0] = datasize; i < dimension; i++)
      {
	 src_stride[i] = src_stride[i - 1] * src_size[i - 1];
	 dst_stride[i] = dst_stride[i - 1] * dst_size[i - 1];
      }
   }
   
      
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __clip
|       Purpose: a recursive orthogonal clip routine.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 20, 1994 11:34
| Modifications:
|
------------------------------------------------------------*/

static void 
__clip(
	 unsigned char *src_buf,
	 unsigned char *dst_buf,
	 int datasize,
	 int rank,
	 int *unit,
	 int *src_offset,
	 int *dst_offset,
	 int *src_skip,
	 int *dst_skip,
	 int linesize)
{
   register int i;
   register int len = unit[rank];
   register int src_skp = src_skip[rank];
   register int dst_skp = dst_skip[rank];

   if (rank > 0)
   {
      for (i = 0; i < len; i++, *src_offset += src_skp, *dst_offset += dst_skp)
	 __clip(src_buf, dst_buf, datasize, rank - 1, unit, src_offset,
		dst_offset, src_skip, dst_skip, linesize);
   }
   else
   {
      kmemcpy(dst_buf + *dst_offset, src_buf + *src_offset,
	      (unsigned long)linesize);
      *dst_offset += linesize;
      *src_offset += linesize;
   }
   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __bit_clip
|       Purpose: a recursive orthogonal clip routine.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 20, 1994 11:34
| Modifications:
|
------------------------------------------------------------*/


static void 
__bit_clip(
	 unsigned char *src_buf,
	 unsigned char *dst_buf,
	 int datasize,
	 int rank,
	 int *unit,
	 int *src_offset,
	 int *dst_offset,
	 int *src_skip,
	 int *dst_skip,
	 int linesize,
	 int src_shift,
	 int dst_shift)
{
   register int i;
   register int len = unit[rank];
   register int src_skp;
   register int dst_skp;

   src_skp = src_skip[rank];
   dst_skp = dst_skip[rank];

   if (rank > 0)
   {
      for (i = 0; i < len; i++, *src_offset += src_skp, *dst_offset += dst_skp)
	 __bit_clip(src_buf, dst_buf, datasize, rank - 1, unit, src_offset,
		    dst_offset, src_skip, dst_skip, linesize, src_shift, dst_shift);

   }
   else
   {
      unsigned char *dsttmp = dst_buf + *dst_offset;
      unsigned char *srctmp = src_buf + *src_offset;
      unsigned char himask[] ={0xff,0x7f,0x3f,0x1f,0x0f,0x07,0x03,0x01};
      unsigned char lomask[] ={0x0,0x80,0xc0,0xe0,0xf0,0xf8,0xfc,0xfe};
      unsigned char tmp;
	 
      if (src_shift == dst_shift)
      {
	 *dsttmp++ |= *srctmp++ & himask[src_shift];
	 
	 for (i = 0; i < linesize - 1; i++)
	    *dsttmp++ = *srctmp++;

	 if (src_shift != 0)
	    *dsttmp |= *srctmp & lomask[src_shift];
      }
      else if (src_shift > dst_shift)
      {
	 for (i = 0; i < linesize; i++)
	 {
	    *dsttmp |= (*srctmp++ << (src_shift - dst_shift)) &
	       lomask[8 - src_shift + dst_shift];
	    *dsttmp++   |= (*srctmp >> (8 - src_shift + dst_shift)) & 
	       himask[8 - src_shift + dst_shift];
	 }
      }
      else
      {
	 for (i = 0; i < linesize; i++)
	 {
	    *dsttmp++ |= (*srctmp >> (dst_shift - src_shift)) &
	       himask[dst_shift - src_shift];
	    *dsttmp |= (*srctmp++ << (8 - dst_shift + src_shift)) &
	       lomask[dst_shift - src_shift];
	 }
      }
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _clip
|       Purpose: Implements a 5 space orthogonal clip function.
|         Input: src_size - the total size of the source buffer
|                dst_size - the total size of the destination buffer.
|                dim      - dimension of the data set.
|                datatype - datatype of data in both src and dest.
|                src_begin - beginning corner of slice in src.
|                dst_begin - beginning corner of slice in dest.
|                unit      - size of slab to get.
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 20, 1994 09:17
| Modifications:
|
------------------------------------------------------------*/

static int
_clip(
	kaddr src_buf,
	kaddr dst_buf,
	int *src_size,
	int *dst_size,
	int dimension,
	int datatype,
	int *src_begin,
	int *dst_begin,
	int *unit)
{
   register int i;
#if 0
   register int j;
   register int k;
#endif
   register int src_factor;
   register int dst_factor;
   int src_offset = 0;
   int dst_offset = 0;
   int src_stride[KDMS_MAX_DIM];
   int dst_stride[KDMS_MAX_DIM];
   int src_skip[KDMS_MAX_DIM];
   int dst_skip[KDMS_MAX_DIM];
   int _unit[KDMS_MAX_DIM];
   int _src_size[KDMS_MAX_DIM];
   int _dst_size[KDMS_MAX_DIM];
   int datasize = kdata_size(datatype);
   int linesize;
   int src_shift = 0;
   int dst_shift = 0;
   
   /*
    * set up some temporary variables.  It is possible that these values
    * will change, and we can't afford any side effects.
    */
   for (i = 0; i < dimension; i++)
   {
      _unit[i] = unit[i];
      _src_size[i] = src_size[i];
      _dst_size[i] = dst_size[i];
   }

   /*
    * compute the starting offsets
    */
   if (datatype == KBIT)
   {
      src_offset = src_begin[0] >> 3;
      dst_offset = dst_begin[0] >> 3;
      src_shift = src_begin[0]  % 8;
      dst_shift = dst_begin[0]  % 8;
      for (i = 1, src_factor = ((_src_size[0] + 7) >> 3), 
	      dst_factor = ((_dst_size[0] + 7) >> 3);
	   i < dimension;
	   i++, src_factor *= _src_size[i - 1], dst_factor *= _dst_size[i - 1])
      {
	 src_offset += src_begin[i] * src_factor;
	 dst_offset += dst_begin[i] * dst_factor;
      }
   }
   else
   {
      for (i = 0, src_factor = dst_factor = datasize;
	   i < dimension;
	   i++, src_factor *= _src_size[i - 1], dst_factor *= _dst_size[i - 1])
      {
	 src_offset += src_begin[i] * src_factor;
	 dst_offset += dst_begin[i] * dst_factor;
      }
   }
   
   /*
    * the next chunk of code is to improve performance.
    */
#if 0
   DONT DO THIS AT ALL IF OPERATING ON BIT DATA!!!
   for (i = 0; i < dimension - 1; i++)
      if (src_size[i] == dst_size[i])
      {
	 _unit[i] *= _unit[i + 1];
	 _src_size[i] *= _src_size[i + 1];
	 _dst_size[i] *= _dst_size[i + 1];
	 for (j = i + 1; j < dimension - 1; j++)
	 {
	    _unit[j] = _unit[j + 1];
	    _src_size[j] = _src_size[j + 1];
	    _dst_size[j] = _dst_size[j + 1];
	 }
	 dimension--;		/* reduce dimensionality */
	 i--;			/* stay where we are */
      }
#endif
   /*
    * compute the strides into the source and destination data sets.
    */
   __compute_strides(datatype, dimension, src_size, dst_size,
		     src_stride, dst_stride);

   /*
    * compute the skip factors
    */
   __compute_skips(dimension, src_stride, dst_stride, _unit, _unit, 
		   src_skip, dst_skip);

   /*
    * obtain data.
    */
   if (datatype == KBIT)
   {
      linesize = (_unit[0] + 7) >> 3; /* computed here for efficiency */
      
      __bit_clip((unsigned char *)src_buf, (unsigned char *)dst_buf, datasize,
	     dimension - 1, _unit, &src_offset, &dst_offset, src_skip, 
	     dst_skip, linesize, src_shift, dst_shift);
   }
   else
   {
      linesize = _unit[0] * datasize;
      
      __clip((unsigned char *)src_buf, (unsigned char *)dst_buf, datasize,
	     dimension - 1, _unit, &src_offset, &dst_offset, src_skip, 
	     dst_skip, linesize);
   }
   
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: (static) __clipinterp
|       Purpose: a recursive orthogonal clipinterp routine.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jun 04, 1994 15:34
| Modifications:
|
------------------------------------------------------------*/

static void 
__clipinterp(
	 unsigned char *src_buf,
	 unsigned char *dst_buf,
	 int datasize,
	 int datatype,
	 int rank,
	 int dimension,
	 int *src_unit,
	 int *dst_unit,
	 int *src_size,
	 int *dst_size,
	 int *src_offset,
	 int *dst_offset,
         double *src_fskip,
         int *src_stride,
         int *dst_start,
	 int *src_skip,
	 int *dst_skip)
{
   register int i;
   register int len = dst_unit[rank];
   double f, b;

   if (rank > 0)
   {
      f = kabs(dst_start[rank] + 1) * src_fskip[rank];
      b = f - src_fskip[rank] + FLT_EPSILON;

      for (i = 0; i < len; i++, b = f, f += src_fskip[rank] + FLT_EPSILON)
      {
	 __clipinterp(src_buf, dst_buf, datasize, datatype, rank - 1, 
		      dimension,
		      src_unit, dst_unit, src_size, dst_size, src_offset, 
		      dst_offset, src_fskip, src_stride, dst_start,
		      src_skip, dst_skip);

	 *src_offset += ((int)f - (int)b) * src_stride[rank]
	     - src_size[rank - 1] * src_stride[rank - 1]; /* -src_unit */
      }
      *src_offset += (rank < dimension - 1) ? src_skip[rank+1] : 0;
      *dst_offset += (rank < dimension - 1) ? dst_skip[rank+1] : 0;
   }
   else
   {
      kdata_resample(src_buf + *src_offset, (int)(dst_unit[0] * src_fskip[0]),
		     dst_unit[0], datatype, KZERO_ORDER, KREAL,
		     src_fskip[0], 0.0, 0.0,
		     kfraction(src_fskip[0] * dst_start[0]), 0.0, 
		     dst_buf + *dst_offset);
      *dst_offset += dst_size[0] * datasize;
      *src_offset += src_size[0] * datasize;
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _bit_resample
|       Purpose: Interpolate lines of bit data.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Aug 02, 1994 09:00
| Modifications:
|
------------------------------------------------------------*/

static void 
_bit_resample(
   unsigned char *in, 
   int in_num_elem, 
   int out_num_elem,
   int bitshift,
   double offset,
   unsigned char *out)
{
   int i;
   register unsigned char mask;
   register unsigned char maskout;
   double istride = (double)in_num_elem/(double)out_num_elem;

   for (i = 0; i < (out_num_elem >> 3) + ((out_num_elem % 8) != 0); i++)
	    out[i] = 0;
   for (i = 0; i < out_num_elem; i++) {
      mask = 0x80 >> ((int) (i * istride + bitshift + offset) % 8);
      maskout = 0x80 >> (i % 8);
      out[(int) (i >> 3)] |=
	 ((in[((int)(i * istride + bitshift + offset) >> 3)] & mask) ? maskout : 0x0);
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __bit_clipinterp
|       Purpose: a recursive orthogonal clipinterp routine.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jun 04, 1994 15:34
| Modifications:
|
------------------------------------------------------------*/

static void 
__bit_clipinterp(
	 unsigned char *src_buf,
	 unsigned char *dst_buf,
	 int datasize,
	 int datatype,
	 int rank,
	 int dimension,
	 int *src_unit,
	 int *dst_unit,
	 int *src_size,
	 int *dst_size,
	 int *src_offset,
	 int *dst_offset,
         double *src_fskip,
         int *src_stride,
         int *dst_start,
	 int *src_skip,
	 int *dst_skip,
	 int shift,
	 int eol)
{
   register int i;
   register int len = dst_unit[rank];
   double f, b;

   if (rank > 0)
   {
      f = kabs(dst_start[rank] + 1) * src_fskip[rank];
      b = f - src_fskip[rank] + FLT_EPSILON;

      for (i = 0; i < len; i++, b = f, f += src_fskip[rank] + FLT_EPSILON)
      {
	 __bit_clipinterp(src_buf, dst_buf, datasize, datatype, rank - 1, 
			  dimension,
			  src_unit, dst_unit, src_size, dst_size, src_offset, 
			  dst_offset, src_fskip, src_stride, dst_start,
			  src_skip, dst_skip, shift, eol);

	 *src_offset += ((int)f - (int)b) * src_stride[rank]
	     - ((src_size[rank-1] +7) >> 3) * src_stride[rank - 1];
      }
      *src_offset += (rank < dimension - 1) ? src_skip[rank+1] : 0;
      *dst_offset += (rank < dimension - 1) ? dst_skip[rank+1] : 0;
   }
   else
   {
      _bit_resample(src_buf + *src_offset, src_unit[0], dst_unit[0],
		    shift, kfraction(src_fskip[0] * dst_start[0]), 
		    dst_buf + *dst_offset);
	 
      *dst_offset += (dst_size[0] + 7) >> 3;
      *src_offset += (src_size[0] + 7) >> 3;
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _clipinterp
|       Purpose: Implements an n-space orthogonal interpolating clip function.
|         Input: src_size - the total size of the source buffer
|                dst_size - the total size of the destination buffer.
|                dim      - dimension of the data set.
|                datatype - datatype of data in both src and dest.
|                src_begin - beginning corner of slice in src.
|                dst_begin - beginning corner of slice in dest.
|                unit      - size of slab to get.
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jun 04, 1994 15:34
| Modifications:
|
------------------------------------------------------------*/

static int
_clipinterp(
	kaddr src_buf,
	kaddr dst_buf,
	int *src_size,
	int *dst_size,
	int *src_total,
	int *dst_total,
	int dimension,
	int datatype,
	int *src_begin,
	int *dst_begin,
	int *src_unit,
	int *dst_unit,
        int *dst_start)
{
   register int i;
   register int src_factor;
   register int dst_factor;
   int src_offset = 0;
   int dst_offset = 0;
   int src_stride[KDMS_MAX_DIM];
   int dst_stride[KDMS_MAX_DIM];
   int datasize = kdata_size(datatype);
   double src_fskip[KDMS_MAX_DIM];
   int src_skip[KDMS_MAX_DIM];
   int dst_skip[KDMS_MAX_DIM];
   
   /*
    * compute the starting offsets
    */
   if (datatype == KBIT)
   {
      src_offset = src_begin[0] >> 3;
      dst_offset = dst_begin[0] >> 3;
      for (i = 1, src_factor = ((src_size[0] + 7) >> 3), 
	      dst_factor = ((dst_size[0] + 7) >> 3);
	   i < dimension;
	   i++, src_factor *= src_size[i - 1], dst_factor *= dst_size[i - 1])
      {
	 src_offset += src_begin[i] * src_factor;
	 dst_offset += dst_begin[i] * dst_factor;
      }
   }
   else
   {
      for (i = 0, src_factor = dst_factor = datasize; i < dimension;
	   i++, src_factor *= src_size[i - 1], dst_factor *= dst_size[i - 1])
      {
	 src_offset += src_begin[i] * src_factor;
	 dst_offset += dst_begin[i] * dst_factor;
      }
   }
   
   /*
    * compute the strides into the source and destination data sets.
    */
   __compute_strides(datatype, dimension, src_size, dst_size,
		     src_stride, dst_stride);

   /*
    * compute the skip factors
    */
   __compute_skips(dimension, src_stride, dst_stride, src_unit, dst_unit, 
		   src_skip, dst_skip);

   for (i = 0; i < dimension; i++)
      src_fskip[i] = (double)src_total[i] / (double)dst_total[i];

   /*
    * obtain data.
    */
   if (datatype == KBIT)
   {
      
      int shift = src_begin[0]  % 8;
      int eol = (src_unit[0] + src_begin[0] == src_size[0]);

      src_stride[0] = 1;
      
      __bit_clipinterp((unsigned char *)src_buf, (unsigned char *)dst_buf, 
		       datasize, datatype, dimension - 1, dimension, src_unit, 
		       dst_unit, src_size, dst_size,&src_offset, &dst_offset,
		       src_fskip, src_stride, dst_start, src_skip, dst_skip, 
		       shift, eol);
   }
   else
   {
      __clipinterp((unsigned char *)src_buf, (unsigned char *)dst_buf, 
		   datasize, datatype, dimension - 1, dimension, src_unit, dst_unit, 
		   src_size, dst_size,&src_offset, &dst_offset,
		   src_fskip, src_stride, dst_start, src_skip, dst_skip);
   }
   

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __transpose
|       Purpose: a recursive orthogonal transpose routine.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 20, 1994 15:08
| Modifications:
|
------------------------------------------------------------*/

static void 
__transpose(
	      unsigned char *src_buf,
	      unsigned char *dst_buf,
	      int datasize,
	      int rank,
	      int dimension,
	      int *unit,
	      int *src_offset,
	      int *dst_offset,
	      int *src_skip,
	      int *dst_skip)
{
   register int i;
   register int len = unit[rank];
   register int src_skp = src_skip[rank];
   register int dst_skp = dst_skip[rank];

   if (rank >= 0)
   {
      for (i = 0; i < len; i++, *src_offset += src_skp, *dst_offset += dst_skp)
      {
	 __transpose(src_buf, dst_buf, datasize, rank - 1, dimension, unit,
		     src_offset, dst_offset, src_skip, dst_skip);
      }
   }
   else
   {
      kmemcpy(dst_buf + *dst_offset, src_buf + *src_offset, 
	      (unsigned)datasize);
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __bit_transpose
|       Purpose: a recursive orthogonal transpose routine.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 20, 1994 15:08
| Modifications:
|
------------------------------------------------------------*/

static void 
__bit_transpose(
	      unsigned char *src_buf,
	      unsigned char *dst_buf,
	      int datasize,
	      int rank,
	      int dimension,
	      int *unit,
	      int *src_offset,
	      int *dst_offset,
	      int *src_skip,
	      int *dst_skip,
	      int shift)
{
   register int i;
   register int len = unit[rank];
   register int src_skp = src_skip[rank];
   register int dst_skp = dst_skip[rank];

   if (rank >= 0)
   {
      for (i = 0; i < len; i++, *src_offset += src_skp, *dst_offset += dst_skp)
      {
	 __bit_transpose(src_buf, dst_buf, datasize, rank - 1, dimension, unit,
		     src_offset, dst_offset, src_skip, dst_skip, shift);
      }
   }
   else
   {
      kmemcpy(dst_buf + *dst_offset, src_buf + *src_offset, 
	      (unsigned)datasize);
   }

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _transpose
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 20, 1994 13:01
| Modifications:
|
------------------------------------------------------------*/

static int
_transpose(
	     kaddr src_buf,
	     kaddr dst_buf,
	     int *src_size,
	     int *dst_size,
	     int dimension,
	     int datatype,
	     int *src_order,
	     int *dst_order)
{
   int src_stride[KDMS_MAX_DIM];
   int dst_stride[KDMS_MAX_DIM];
   int tmp[KDMS_MAX_DIM];
   int datasize = kdata_size(datatype);
   int src_skip[KDMS_MAX_DIM];
   int dst_skip[KDMS_MAX_DIM];
   int unit[KDMS_MAX_DIM];
   int src_offset = 0;
   int dst_offset = 0;

   /*
    * compute the strides into the source and destination data sets.
    */
   __compute_strides(datatype, dimension, src_size, dst_size, tmp, dst_stride);

   /*
    * rotate the source strides according to the index order.
    */
   _kdms_transpose_coords(src_order, tmp, dst_order, src_stride, dimension, 0);

   /*
    * compute the skip factors
    */
   __compute_skips(dimension, src_stride, dst_stride, dst_size, dst_size, 
		   src_skip, dst_skip);

   /*
    * make a copy of size, because its going to get munged with during the
    * recursion
    */
   kmemcpy(unit, dst_size, dimension * sizeof(int));

   /*
    * transpose the data
    */
   __transpose((unsigned char *)src_buf, (unsigned char *)dst_buf,
	       datasize, dimension - 1, dimension, unit, &src_offset,
	       &dst_offset, src_skip, dst_skip);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _cast
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:08
| Modifications:
|
------------------------------------------------------------*/

static int
_cast(
	kaddr src_buf,
	kaddr dst_buf,
	int *size,
	int dimension,
	int src_datatype,
	int dst_datatype)
{
   register int i;
   register int num = 1;
   int status = 1;
   
   if (src_datatype == KBIT) 
   {
      unsigned char *src = (unsigned char *)src_buf;
      unsigned char *dst = (unsigned char *)dst_buf;
      
      for (i = 1; i < dimension; i++)
	 num *= size[i];

      for (i = 0; i < num; i++) 
      {
	 status |= kdata_cast((kaddr)src, src_datatype, dst_datatype, size[0],
			      KREAL, 1.0, 1.0, dst);
	 src += (size[0] + 7) >> 3;
	 dst += size[0] * kdata_size(dst_datatype);
      }
   }
   else if (dst_datatype == KBIT)
   {
      unsigned char *src = (unsigned char *)src_buf;
      unsigned char *dst = (unsigned char *)dst_buf;
      
      for (i = 1; i < dimension; i++)
	 num *= size[i];

      for (i = 0; i < num; i++) 
      {
	 status |= kdata_cast((kaddr)src, src_datatype, dst_datatype, size[0],
			      KREAL, 1.0, 1.0, dst);
	 dst += (size[0] + 7) >> 3;
	 src += size[0] * kdata_size(src_datatype);
      }
   }
   else
   {
      for (i = 0; i < dimension; i++)
	 num *= size[i];

      status = kdata_cast(src_buf, src_datatype, dst_datatype, num,
			  KREAL, 1.0, 1.0, dst_buf);
   }

   return(status);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _complex_convert
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:08
| Modifications:
|
------------------------------------------------------------*/

static int
_complex_convert(
	kaddr src_buf,
	kaddr dst_buf,
	int *size,
	int dimension,
	int complex_convert,
	int in_datatype,
	int out_datatype,
	double rval,
	double ival)
{
   register int i;
   register int num = 1;
   int status;
   
   for (i = 0; i < dimension; i++)
      num *= size[i];

   status = kdata_cast(src_buf, in_datatype, out_datatype, num,
		       complex_convert, 1.0, 1.0, dst_buf);

   /*
    * Experiment for donna
    */
   if (out_datatype == KCOMPLEX)
   {
      kcomplex *tmp = dst_buf;
      
      switch (complex_convert)
      {
	 case KREAL:
	    for (i = 0; i < num; i++)
	       tmp[i] = kccomp(kcreal(tmp[i]),(float)ival);
	    break;
	 case KIMAGINARY:
	    for (i = 0; i < num; i++)
	       tmp[i] = kccomp((float)rval, kcimag(tmp[i]));
	    break;
	 default:
	    break;
      }
   }
   else if (out_datatype == KDCOMPLEX)
   {
      kdcomplex *tmp = dst_buf;
      
      switch (complex_convert)
      {
	 case KREAL:
	    for (i = 0; i < num; i++)
	       tmp[i] = kdccomp(kdcreal(tmp[i]),(double)ival);
	    break;
	 case KIMAGINARY:
	    for (i = 0; i < num; i++)
	       tmp[i] = kdccomp((double)rval, kdcimag(tmp[i]));
	    break;
	 default:
	    break;
      }
   }

   return (status);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _scale
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:34
| Modifications:
|
------------------------------------------------------------*/

static int
_scale(
	 kaddr data,
	 int *size,
	 int dimension,
	 int datatype,
	 double scale_factor,
	 double real_offset,
	 double imaginary_offset)
{
   register int i;
   register int num = 1;

   for (i = 0; i < dimension; i++)
      num *= size[i];

   return (kdata_scale(data, num, datatype, scale_factor, 0.0,
		       real_offset, imaginary_offset, data));
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _normalize
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:41
| Modifications:
|
------------------------------------------------------------*/


static int
_normalize(
	     kaddr data,
	     int *size,
	     int dimension,
	     int datatype,
	     double norm_minimum,
	     double norm_maximum)
{
   register int i;
   register int num = 1;
   double factor;
   double offset;

   for (i = 0; i < dimension; i++)
      num *= size[i];

   switch (datatype)
   {
      case KBIT:
	 /* the easy case...normalization doesn't make sense, so return */
	 return (TRUE);
      case KBYTE:
	 {
	    signed char *tmp = (signed char *)data;
	    signed char rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (signed char)(tmp[i] * factor + offset);
	 }
	 break;
      case KUBYTE:
	 {
	    unsigned char *tmp = (unsigned char *)data;
	    unsigned char rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -((double)rmin * factor) + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (unsigned char)(tmp[i] * factor + offset);
	 }
	 break;
      case KSHORT:
	 {
	    short *tmp = (short *)data;
	    short rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (short)(tmp[i] * factor + offset);
	 }
	 break;
      case KUSHORT:
	 {
	    unsigned short *tmp = (unsigned short *)data;
	    unsigned short rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -((double)rmin * factor) + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (unsigned short)(tmp[i] * factor + offset);
	 }
	 break;
      case KINT:
	 {
	    int *tmp = (int *)data;
	    int rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (int)(tmp[i] * factor + offset);
	 }
	 break;
      case KUINT:
	 {
	    unsigned int *tmp = (unsigned int *)data;
	    unsigned int rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -((double)rmin * factor) + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (unsigned int)(tmp[i] * factor + offset);
	 }
	 break;
      case KLONG:
	 {
	    long *tmp = (long *)data;
	    long rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (long)(tmp[i] * factor + offset);
	 }
	 break;
      case KULONG:
	 {
	    unsigned long *tmp = (unsigned long *)data;
	    unsigned long rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -((double)rmin * factor) + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (unsigned long)(tmp[i] * factor + offset);
	 }
	 break;
      case KFLOAT:
	 {
	    float *tmp = (float *)data;
	    float rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0.0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (float)(tmp[i] * factor + offset);
	 }
	 break;
      case KDOUBLE:
	 {
	    double *tmp = (double *)data;
	    double rmin,
	       rmax;
	    rmin = rmax = tmp[0];

	    for (i = 1; i < num; i++)
	    {
	       rmin = kmin(tmp[i], rmin);
	       rmax = kmax(tmp[i], rmax);
	    }

	    if (rmax - rmin != 0.0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = (double)(tmp[i] * factor + offset);
	 }
	 break;
      case KCOMPLEX:
	 {
	    kcomplex *tmp = (kcomplex *) data;
	    double rmin, rmax, imin, imax, mmin, mmax;
	    mmin = mmax = (double)kcmag(tmp[0]);
	    rmin = rmax = (double)kcreal(tmp[0]);	
	    imin = imax = (double)kcimag(tmp[0]);	
	    for (i = 1; i < num; i++)
	    {
	       if (kcmag(tmp[i]) < mmin) {
		  rmin = kcreal(tmp[i]);
		  imin = kcimag(tmp[i]);
		  mmin = kcmag(tmp[i]);	
	       }
	       if (kcmag(tmp[i]) > mmax) {
		  rmax = kcreal(tmp[i]);
		  imax = kcimag(tmp[i]);
		  mmax = kcmag(tmp[i]);	
	       }
	    }
	    if (rmax - rmin != 0.0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

kinfo(KSTANDARD,"(%g %g), (%g %g), [%g %g] = %g\n",rmin, imin, rmax, imax, mmin, mmax, factor);
	    for (i = 0; i < num; i++)
	       tmp[i] = kcadd(kcmult(tmp[i],
				     kccomp((double)factor, 0.0)),
			      kccomp((float)rmin, (float)imin));
	 }
	 break;
      case KDCOMPLEX:
	 {
	    kdcomplex *tmp = (kdcomplex *) data;
	    double rmin,
	       rmax;
	    rmin = rmax = (double)kdcmag(tmp[0]);

	    for (i = 1; i < num; i++)
	    {
	       rmin = (double)kmin(kdcmag(tmp[i]), rmin);
	       rmax = (double)kmax(kdcmag(tmp[i]), rmax);
	    }

	    if (rmax - rmin != 0.0)
	       factor = (norm_maximum - norm_minimum) / 
		  ((double)rmax - (double)rmin);
	    else
	       factor = (norm_maximum - norm_minimum);

	    offset = -rmin * factor + norm_minimum;
	    for (i = 0; i < num; i++)
	       tmp[i] = kdcadd(kdcmult(tmp[i], kdccomp((double)factor, 0.0)),
			       kdccomp((double)offset, 0.0));
	 }
	 break;
      default:
	 _kdms_set_error(KDMS_EINTERNAL);
	 return (FALSE);
   }
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _update_buffer
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 09:50
| Modifications:
|
------------------------------------------------------------*/

static int
_update_buffer(
		 kobject object,
		 kpresentation * pres,
		 int *start,
		 int *size,
		 int dimension)
{
   int tend[KDMS_MAX_DIM];
   int tbegin[KDMS_MAX_DIM];
   register int i;

   if (__in_phys_buffer(pres->segment, start, size))
      return (TRUE);

   /*
    * If the file is null, and the segment isn't locked, then the
    * whole thing is in memory (what there is of it.)  If this is the
    * case, then there is no point in mucking around with the code
    * below.  This check prevents an unwanted flush when only part of
    * the data set has been written to this segment, but the whole
    * thing is later asked for.  The clip stuff that is called later
    * will cope with an incomplete data set.
    */
   if (pres->segment->file == NULL && (!pres->segment->locked && !object->phys->headerless))
   {
      if (pres->segment->data != NULL)
	 return TRUE;
      else
	 return FALSE;
   }
   
   
   /*
    * gotta check to make sure that the begin marker is inside the
    * data space.  The end marker wil be properly trimmed by
    * kdms_optimize_bufsize, but the the assumption made by
    * kdms_optimize_bufsize is that begin and end somehow contain the
    * data space.  This is not an error state, but rather a simple a
    * statement that the data could not be loaded.  The calling
    * routine should know what to do with this.
    */
   for (i = 0; i < dimension; i++)
      if (start[i] >= pres->segment->size[i])
	 return FALSE;
   
   if (pres->segment->data != NULL && (pres->segment->temporary || 
       pres->segment->file != NULL))
      _kdms_flush_segment(object, pres);

   for (i = 0; i < dimension; i++)
   {
      tbegin[i] = start[i];
      tend[i] = start[i] + size[i] - 1;
   }

   for (i = dimension; i < KDMS_MAX_DIM; i++)
      tbegin[i] = tend[i] = 0;

   kdms_optimize_bufsize(pres->segment->size, pres->segment->datatype,
			 tbegin, tend, pres->dimension, 
			 pres->segment->buffer_threshold);

   /* 
    *if read buffer fails, then that is because the file
    * does not exist....so the data is new.
    */
   if (!_kdms_read_buffer(object, pres->segment, tbegin, tend))
      return (FALSE);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _update_transport
|       Purpose: 
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 28, 1994 08:38
| Modifications:
|
------------------------------------------------------------*/

static int
_update_transport(
		    kobject object,
		    kpresentation * pres,
		    int *start,
		    int *size)
{
   int tend[KDMS_MAX_DIM] = {0,0,0,0,0};
   int tbegin[KDMS_MAX_DIM] = {0,0,0,0,0};
   register int i;
   unsigned long num;
   
   if (__in_phys_buffer(pres->segment, start, size))
      return (TRUE);

   for (i = 0; i < pres->dimension; i++)
   {
      tbegin[i] = start[i];
      tend[i] = start[i] + size[i] - 1;
   }

#if 0
   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      tbegin[i] = start[i];
      tend[i] = start[i] + size[i] - 1;
   }
#endif

   if (!_kdms_flush_segment(object, pres))
      return (FALSE);

   /* 
    * note.  output objects cannot have optimized internal buffers.
    * total bummer.  Also, if this is a bit data set, then we gotta
    * back out and advance to byte boundaries on the lowest axis.
    */
   if (pres->segment->datatype == KBIT)
   {
      tbegin[0] = tbegin[0] - (tbegin[0] % 8);
      tend[0] = kmin((tend[0] + 7 - (tend[0] % 8)), pres->segment->size[0]-1);

      num = (tend[0] - tbegin[0] + 8) >> 3;
      for (i = 1; i < pres->segment->dimension; i++)
	 num *= (tend[i] - tbegin[i] + 1);
   }
   else
   {
      num = (unsigned long)_kdms_num_elements(tbegin, tend,
					      pres->segment->dimension);
   }
   
   /*
    * this malloc can potentially fail.  Check it out.
    */
   pres->segment->data = (kaddr) krealloc(pres->segment->data, num *
					  kdata_size(pres->segment->datatype));

   if (pres->segment->data == NULL)
      return (FALSE);

   if (pres->segment->datatype == KBIT)
      kmemset(pres->segment->data, 0, num);

   /*
    * if the data type is bit, then we gotta grab the the current contents
    * of the transport because it is possible that the byte boundary
    * corner markers are not the same as the real corner markers. 
    */
   if (pres->segment->datatype == KBIT && pres->segment->file != NULL)
   {
      if (!_kdms_read_buffer(object, pres->segment, tbegin, tend))
	 return (FALSE);
   }

   /*
    * update the corner markers
    */
   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      pres->segment->begin[i] = tbegin[i];
      pres->segment->end[i] = tend[i];
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __need_to_complex_convert
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 14, 1994 17:27
| Modifications:
|
------------------------------------------------------------*/

static int 
__need_to_complex_convert(kpresentation *pres)
{
   if ((pres->segment->datatype == KCOMPLEX || 
	pres->segment->datatype == KDCOMPLEX) && 
       (pres->datatype != KCOMPLEX && pres->datatype != KDCOMPLEX))
      return (TRUE);

   if ((pres->datatype == KCOMPLEX || pres->datatype == KDCOMPLEX) && 
       (pres->segment->datatype != KCOMPLEX && 
	pres->segment->datatype != KDCOMPLEX))
      return (TRUE);

   return (FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __need_to_precast
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 14, 1994 17:27
| Modifications:
|
------------------------------------------------------------*/

static int
__need_to_precast(kpresentation *pres, int get)
{
   if (get)
      return (pres->segment->datatype < pres->datatype);
   else
      return (pres->datatype < pres->segment->datatype);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __need_to_postcast
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 14, 1994 17:27
| Modifications:
|
------------------------------------------------------------*/

static int
__need_to_postcast(kpresentation *pres, int get)
{
   if (get)
      return (pres->segment->datatype > pres->datatype);
   else
      return (pres->datatype > pres->segment->datatype);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __need_to_scale
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 14, 1994 17:27
| Modifications:
|
------------------------------------------------------------*/

static int
__need_to_scale(kpresentation *pres)
{
   if (pres->scaling == KSCALE && 
       (pres->scale_factor != 1.0 || pres->scale_offset_real != 0.0 ||
	pres->scale_offset_imag != 0.0))
      return(TRUE);
   else
      return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) __outside_of_data_space
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 14, 1994 17:32
| Modifications:
|
------------------------------------------------------------*/

static int
__outside_of_data_space(
   kpresentation *pres, 
   int *begin, 
   int *end,
   int *_begin, 
   int *_end)
{
   int a;
   int b;
   int c;
   int d;
   int i;
   
   /*
    * first check to see if both are "below" the space
    */
   for (i = 0, a = 0, b = 0, c = 0, d = 0; i < pres->dimension; i++)
   {
      a |= (begin[i] < 0);
      b |= (end[i] < 0);
#if 0
      c |= (_begin[i] < 0);
      d |= (_end[i] < 0);
#endif
   }
   
   if ((a && b) || (c && d))
      return (TRUE);

   for (i = 0, a = 0, b = 0, c = 0, d = 0; i < pres->dimension; i++)
   {
      a |= (begin[i] > pres->size[i] - 1);
      b |= (end[i] > pres->size[i] - 1);
      c |= (_begin[i] > pres->segment->size[i] - 1);
      d |= (_end[i] > pres->segment->size[i] - 1);
   }
   
   if ((a && b) || (c && d))
      return (TRUE);

   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: __need_to_transpose
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 14, 1994 17:43
| Modifications:
|
------------------------------------------------------------*/

static int
__need_to_transpose(kpresentation *pres, int *phys_unit, int *pres_unit)
{
   return(!__same_order(pres->order, pres_unit, pres->segment->order, 
			phys_unit, pres->dimension));   
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_get_data
|       Purpose:
|
|		Here's a rundown of how the temporary buffer sharing
|		works in this routine.
|		0000
|		|||+---- cmplx conv
|		||+----- pre
|		|+------ trans
|		+------- post
|
|		0 - segment buffer
|		1 - tmp1
|		2 - tmp2
|		3 - tmp3
|		4 - tmp4
|		X - data
|
|			clip	cmplx	pre	trans	post
|		0000	0X				
|		0001	01	1X
|		0010	01		1X
|		0011	01	12	2X
|		0100	01			1X
|		0101	01	12		2X
|		0110	01		13	3X
|		0111	01	12	23	3X
|		1000	01				1X
|		1001	01	12			2X
|		1010	--
|		1011	--
|		1100	01			13	3X
|		1101	01	12		23	3X
|		1110	--
|		1111	--
|
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 21, 1994 14:39
| Modifications:
|
------------------------------------------------------------*/

kaddr
_kdms_get_data(
                 kobject object,
                 kpresentation * pres,
                 int *begin,
                 int *end,
                 kaddr data)
{
   int i;
   int num_points;
   kaddr src_buf;
   kaddr dst_buf;

   /*
    * this is the presentation unit size in the presentation index order
    * and spacial size.
    */
   int pres_unit[KDMS_MAX_DIM];

   /*
    * These are the physical layer begin, end, and unit sizes
    * converted to the destination spacial-size and index order. 
    */
   int phys_begin[KDMS_MAX_DIM];
   int phys_end[KDMS_MAX_DIM];
   int phys_unit[KDMS_MAX_DIM];

   /*
    * variables which are used to indicate whether an operation should
    * occur. 
    */
   int transpose;
   int interpolate;
   int scale;
   int normalize;
   int cmplx_convert;
   int precast;
   int postcast;
   int outofspace;
   int type;

   /*
    * ...............................................................
    */

   /*
    * compute the size of, and total number of points in the resulting
    * space, given the presentation begin and end markers. 
    */
   for (i = 0, num_points = 1; i < pres->dimension; i++)
   {
      pres_unit[i] = end[i] - begin[i] + 1;
      num_points *= pres_unit[i];
   }

   /*
    * compute physical layer tuples. phys_unit is the physical version
    * of pres_unit.  phys_begin is the physical version begin, and
    * phys_end is the physical version of phys_end.
    *
    * NOTE: phys_unit cannot be computed with pres to phys because if it
    * is too small, and we are interpolating up, then the result will
    * be 0 size.  We'll get much better results by computing phys_begin
    * and phys_end and then computing phys_unit by hand. 
    */
   _kdms_pres2phys(pres->segment, pres, begin, phys_begin, 0);
   _kdms_pres2phys(pres->segment, pres, pres_unit, phys_unit, 1);

   for (i = 0; i < pres->dimension; i++) 
      phys_end[i] = phys_unit[i] + phys_begin[i] - 1;

   /*
    * if the destination is NULL, then allocate it.  This done before
    * the update buffer intentionally, because if the update_buffer
    * fails, then that means that there was no data to update the
    * buffer with.  In that case, you are usually dealing with an
    * output file, and the user intended to use the get data as a
    * snazzy memory allocator. 
    */
   if (data == NULL)
   {
      data = kcalloc((unsigned)num_points, 
		     (unsigned)kdata_size(pres->datatype));
      
      if (data == NULL)
         return (NULL);
   }

   /*
    * get the buffer out of the transport if that is necessary.  If
    * this fails, then we return the data that was allocated above.
    * See the comment on the allocation block above for an
    * explanation.
    */
   if (!_update_buffer(object, pres, phys_begin, phys_unit, pres->dimension))
   {
      _kdms_preset_data(data, pres->datatype,
                        num_points, pres->pad_real, pres->pad_imag);
      return (data);
   }
   
   /*
    * now we check to see which stages must be executed.  This is
    * done here, because the information is necessary to determine
    * how to set up each stages destination buffer.
    */
   transpose     = __need_to_transpose(pres, phys_unit, pres_unit);
   cmplx_convert = __need_to_complex_convert(pres);
   precast       = __need_to_precast(pres, TRUE);
   postcast      = __need_to_postcast(pres, TRUE);
   scale         = __need_to_scale(pres);
   normalize     = pres->scaling == KNORMALIZE;
   interpolate   = pres->interp == KZERO_ORDER;
   outofspace    = __outside_of_data_space(pres, begin, end, phys_begin, 
					   phys_end);

   /*	
    * first check.  If there is nothing to be done after the clip
    * operation (except perhaps scale or normalize), then set the
    * destination buffer to be data, otherwise allocate space for tmp1
    * and use that. 
    */
   if ((precast || postcast || transpose || cmplx_convert) && !outofspace)
   {
      tmp1 = __adjust_size(&tmp1, &tmp1_size, num_points,
                           pres->segment->datatype);
      dst_buf = tmp1;
   }
   else
      dst_buf = data;

   /*
    * preset the values of the data to the pad value.  The only reason
    * that this seems inefficient is that it *is*.  The positioning of
    * this chunk of code is unfortunate, but it is necessary to preset
    * the destination of the clip call, if it is necessary. 
    */
   if (__need_to_preset(pres, begin, end))
      _kdms_preset_data(dst_buf, 
			outofspace ? pres->datatype : pres->segment->datatype,
                        num_points, pres->pad_real, pres->pad_imag);

   /*
    * if the requested data is totally outside of the dataspace then
    * don't bother going through the stages below.  The positioning of
    * is code chunk is unfortunate, but it is necessary to be after
    * the preset data and before the clip.  The preset must be after
    * the determination of the dst_buf for the clip operator. 
    */
   if (outofspace)
      return (data);

   /*
    * in a get pipeline, the first thing that should happen is that
    * the data is clipped out of the physical buffer.  This is done
    * right away so that the remaining pipeline stages can be freed
    * from the unnecessary complexity of coping with a subspace of the
    * buffer. 
    */

   if (interpolate) 
   {
      int dst_clipunit[KDMS_MAX_DIM];
      int src_buf_size[KDMS_MAX_DIM];
      int dst_buf_size[KDMS_MAX_DIM];
      int dst_begin[KDMS_MAX_DIM];
      int src_clipunit[KDMS_MAX_DIM];
      int src_begin[KDMS_MAX_DIM];
      int pres_begin[KDMS_MAX_DIM];
      int pres_size[KDMS_MAX_DIM];
      
      /*
       * later computations require the begin marker and sizein the
       * presentation data space, but in the physical layer index
       * ordering, so first reorder the input begin marker into the
       * physical index order. 
       */
      for (i = 0; i < pres->dimension; i++)
      {
	 pres_begin[i] = _kdms_get_order(pres->segment->order[i], pres->order,
					 begin, pres->dimension, 1);
	 pres_size[i] = _kdms_get_order(pres->segment->order[i], pres->order, 
					pres->size, pres->dimension, 1);
      }
      
      for(i = 0; i < pres->dimension; i++) 
      {
	 dst_buf_size[i] = _kdms_get_order(pres->segment->order[i], 
					   pres->order, pres_unit, 
					   pres->dimension, 1);
	 dst_begin[i] = kmax(0, -pres_begin[i]);
	 dst_clipunit[i] = kmin(pres_size[i] - pres_begin[i], 
				dst_buf_size[i] - dst_begin[i]);
      }
      
      for (i = 0; i < pres->dimension; i++)
      {
	 src_buf_size[i] = pres->segment->end[i] - pres->segment->begin[i] + 1;
	 src_clipunit[i] = (src_buf_size[i] - kmax(0, -phys_begin[i])) - 
	    kmax(0, pres->segment->end[i] - phys_end[i] );
	 src_clipunit[i] = kmax(1, kmin(phys_unit[i], src_clipunit[i]));
	 src_begin[i] = kmax(0, phys_begin[i] - pres->segment->begin[i]);
      }

      if (!_clipinterp(pres->segment->data, dst_buf, src_buf_size, 
		       dst_buf_size, pres->segment->size, pres_size,
		       pres->dimension, pres->segment->datatype, src_begin, 
		       dst_begin, src_clipunit, dst_clipunit, begin))
	 return (NULL);
   }
   else
   {
      int src_buf_size[KDMS_MAX_DIM];
      int *dst_buf_size = phys_unit;
      int src_begin[KDMS_MAX_DIM];
      int dst_begin[KDMS_MAX_DIM];
      int clipunit[KDMS_MAX_DIM];
      
      for (i = 0; i < pres->dimension; i++)
      {
	 src_buf_size[i] = pres->segment->end[i] - pres->segment->begin[i] + 1;

 	 src_begin[i] = kmax(0, phys_begin[i] - pres->segment->begin[i]); 
 	 dst_begin[i] = kmax(0, - phys_begin[i]); 

	 /* trim the unit to fit into the destination buffer */
	 clipunit[i] = (dst_buf_size[i] - dst_begin[i]) - 
	    kmax(0, phys_end[i] - pres->segment->end[i]);
         clipunit[i] = kmax(1, clipunit[i]);
      }

      /*
       * preclear the data because the bit operations OR data in.  If there
       * is garbage in the buffer, then it will come out in the resulting
       * data.
       */
      if (pres->segment->datatype == KBIT)
      {
	 int t = num_points /= phys_unit[0];
	 t *= (end[0] - begin[0] + 8) >> 3;
	 kmemset(dst_buf, 0, t);
      }

      if (!_clip(pres->segment->data, dst_buf, src_buf_size, dst_buf_size,
                 pres->dimension, pres->segment->datatype, src_begin, 
		 dst_begin,
                 clipunit))
	 return (NULL);
   }

   src_buf = dst_buf;

   /*
    * the next stage is complex conversion
    */
   if (cmplx_convert)
   {
      switch (pres->segment->datatype)
      {
	 case KDCOMPLEX: 
	    type = KDOUBLE; 
	    break;
	 case KCOMPLEX:
	    type = KFLOAT;
	    break;
	 case KDOUBLE:
	    type = KDCOMPLEX;
	    break;
	 case KFLOAT:
	    type = KCOMPLEX;
	    break;
	 default:
	    type = KCOMPLEX;
	    break;
      }
	    
      if (precast || postcast || transpose)
      {
         tmp2 = __adjust_size(&tmp2, &tmp2_size, num_points, type);

         dst_buf = tmp2;
      }
      else
         dst_buf = data;

       if (!_complex_convert(src_buf, dst_buf, pres_unit, pres->dimension, 
			     pres->complex_convert,pres->segment->datatype,
			     type, pres->pad_real, pres->pad_imag))
	  return (NULL);

      src_buf = dst_buf;
   }
   else
      type = pres->segment->datatype;
      
   /*
    * the third stage is a possible cast operation.  This only occurs
    * if the destination data type has a smaller range or less
    * precision than the source data type.  We do this first because
    * scaling and normalization are less prone to loss of precision
    * and overflow if they operate on the "higher order" data type. 
    */
   if (precast)
   {
      int _phys_unit[KDMS_MAX_DIM];
      
      if (transpose)
      {
         tmp3 = __adjust_size(&tmp3, &tmp3_size, num_points, pres->datatype);

         dst_buf = tmp3;
      }
      else
	 dst_buf = data;

      for (i = 0; i < pres->dimension; i++)
	 _phys_unit[i] = _kdms_get_order(pres->segment->order[i], pres->order,
					 pres_unit, pres->dimension, 1);

      if (!_cast(src_buf, dst_buf, _phys_unit, pres->dimension, 
		 type, pres->datatype))
         return (NULL);

      src_buf = dst_buf;

      type = pres->datatype;
   }

   /*
    * technically, the tranposition can occur almost anywhere in the
    * get pipeline.  For sake of simplicity, this is done right at the
    * beginning.  
    */
   if (transpose && pres->datatype != KBIT)
   {
      int _phys_unit[KDMS_MAX_DIM];
      
      if (postcast)
      {
         tmp3 = __adjust_size(&tmp3, &tmp3_size, num_points,
                              pres->segment->datatype);
         dst_buf = tmp3;
      }
      else
         dst_buf = data;

      for (i = 0; i < pres->dimension; i++)
	 _phys_unit[i] = _kdms_get_order(pres->segment->order[i], pres->order,
					 pres_unit, pres->dimension, 1);

      if (!_transpose(src_buf, dst_buf, _phys_unit, pres_unit, pres->dimension,
                      type, pres->segment->order, pres->order))
         return (NULL);

      src_buf = dst_buf;
   }

   /*
    * scaling and normalizing can take place in the same buffer, so
    * that's what we'll do.
    */
   if (scale)
      if (!_scale(src_buf, pres_unit, pres->dimension,
                  precast ? pres->datatype : type,
                  pres->scale_factor, pres->scale_offset_real,
                  pres->scale_offset_imag))
         return (NULL);

   if (normalize)
      if (!_normalize(src_buf, pres_unit, pres->dimension,
                      precast ? pres->datatype : type,
                      pres->norm_min, pres->norm_max))
         return (NULL);

   /*
    * if the source data type has less precision or less range than
    * the destinatin, then the cast is practically the last thing
    * done.  This is because the scaling or normalization operations
    * above will be more accurate if performed on the higher order
    * data type. 
    */
   if (postcast) 
   {
      if (!_cast(src_buf, data, pres_unit, pres->dimension, 
                 type, pres->datatype))
         return (NULL);
   }
   
   return (data);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_put_data
|       Purpose: this function puts data into a data object.
|		Here's a rundown of how the temporary buffer sharing
|		works in this routine.
|		0000
|		|||+---- cmplx conv
|		||+----- pre
|		|+------ trans
|		+------- post
|
|		0 - segment buffer
|		1 - tmp1
|		2 - tmp2
|		3 - tmp3
|		4 - tmp4
|		X - data
|
|			cmplx	pre	trans	post	clip    
|		0000					0X      
|		0001	01				1X      
|		0010		01			1X      
|		0011	01	12			2X      
|		0100			01		1X      
|		0101	01		12		2X      
|		0110		01	12		2X      
|		0111	01	12	23		3X      
|		1000				01	1X      
|		1001	01			12	2X      
|		1010					--      
|		1011					--      
|		1100			01	12	2X      
|		1101	01		12	23	3X      
|		1110					--      
|		1111					--      
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Apr 28, 1994 08:10
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_put_data(
                 kobject object,
                 kpresentation * pres,
                 int *begin,
                 int *end,
                 kaddr data)
{
   int i;
   int num_points;
   kaddr src_buf;
   kaddr dst_buf;

   /*
    * this is the presentation unit size in the presentation index order
    * and spacial size.
    */
   int pres_unit[KDMS_MAX_DIM];

   /*
    * These are the physical layer begin, end, and unit sizes
    * converted to the destination spacial-size and index order. 
    */
   int phys_begin[KDMS_MAX_DIM];
   int phys_end[KDMS_MAX_DIM];
   int phys_unit[KDMS_MAX_DIM];

   /*
    * variables which are used to indicate whether an operation should
    * occur. 
    */
   int transpose;
   int interpolate;
   int scale;
   int normalize;
   int cmplx_convert;
   int precast;
   int postcast;
   int type;

   /*
    * ...............................................................
    */

   /*
    * compute the size of, and total number of points in the resulting
    * space, given the presentation begin and end markers. 
    */
   for (i = 0, num_points = 1; i < pres->dimension; i++)
   {
      pres_unit[i] = end[i] - begin[i] + 1;
      num_points *= pres_unit[i];
   }

   /*
    * compute physical layer tuples. phys_unit is the physical version
    * of pres_unit.  phys_begin is the physical version begin, and
    * phys_end is the physical version of phys_end.
    */
   _kdms_pres2phys(pres->segment, pres, begin, phys_begin, 0);
   _kdms_pres2phys(pres->segment, pres, pres_unit, phys_unit, 1);

   for (i = 0; i < pres->dimension; i++) 
      phys_end[i] = phys_unit[i] + phys_begin[i] - 1;

   /*
    * if the requested data is totally outside of the dataspace then
    * don't bother going through the stages below.  The positioning of
    * is code chunk is unfortunate, but it is necessary to be after
    * the preset data and before the clip.  The preset must be after
    * the determination of the dst_buf for the clip operator. 
    */
   if (__outside_of_data_space(pres, begin, end, phys_begin, phys_end))
      return (FALSE);

   /*
    * flush the buffer if necessary and make sure that sufficient space
    * is available in the buffer for this operation.
    */
   if (!_update_transport(object, pres, begin, pres_unit))
      return (FALSE);

   /*
    * now we check to see which stages must be executed.  This is
    * done here, because the information is necessary to determine
    * how to set up each stages destination buffer.
    */
   transpose     = __need_to_transpose(pres, phys_unit, pres_unit);
   cmplx_convert = __need_to_complex_convert(pres);
   precast       = __need_to_precast(pres, FALSE);
   postcast      = __need_to_postcast(pres, FALSE);
   scale         = __need_to_scale(pres);
   normalize     = pres->scaling == KNORMALIZE;
   interpolate   = pres->interp == KZERO_ORDER;

   /*
    * initialize the src_buf to the input data set.
    */
   src_buf = data;
   
   /*
    * the first stage is complex conversion
    */
   if (cmplx_convert)
   {
      switch (pres->datatype)
      {
	 case KDCOMPLEX: 
	    type = KDOUBLE; 
	    break;
	 case KCOMPLEX:
	    type = KFLOAT;
	    break;
	 case KDOUBLE:
	    type = KDCOMPLEX;
	    break;
	 case KFLOAT:
	    type = KCOMPLEX;
	    break;
	 default:
	    type = KCOMPLEX;
	    break;
      }

      tmp1 = __adjust_size(&tmp1, &tmp1_size, num_points, type);
      dst_buf = tmp1;

       if (!_complex_convert(src_buf, dst_buf, pres_unit, pres->dimension, 
			     pres->complex_convert, pres->datatype,
			     type, pres->pad_real, pres->pad_imag))
	  return (FALSE);

      src_buf = dst_buf;
   }
   else
      type = pres->datatype;
      
   /*
    * the third stage is a possible cast operation.  This only occurs
    * if the destination data type has a smaller range or less
    * precision than the source data type.  We do this first because
    * scaling and normalization are less prone to loss of precision
    * and overflow if they operate on the "higher order" data type. 
    */
   if (precast)
   {
      if (cmplx_convert)
      {
         tmp2 = __adjust_size(&tmp2, &tmp2_size, num_points, 
			      pres->segment->datatype);
         dst_buf = tmp2;
      }
      else
      {
         tmp1 = __adjust_size(&tmp1, &tmp1_size, num_points, 
			      pres->segment->datatype);
         dst_buf = tmp1;
      }

      if (!_cast(src_buf, dst_buf, pres_unit, pres->dimension, 
		 type, pres->segment->datatype))
         return (FALSE);

      src_buf = dst_buf;

      type = pres->segment->datatype;
   }

   /*
    * technically, the tranposition can occur almost anywhere in the
    * get pipeline.  For sake of simplicity, this is done right at the
    * beginning.  
    */
   if (transpose && pres->datatype != KBIT)
   {
      int _phys_unit[KDMS_MAX_DIM];
      
      if (cmplx_convert)
      {
	 if (precast)
	 {
	    tmp3 = __adjust_size(&tmp3, &tmp3_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp3;
	 }
	 else
	 {
	    tmp2 = __adjust_size(&tmp2, &tmp2_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp2;
	 }
      }
      else
      {
	 if (precast)
	 {
	    tmp2 = __adjust_size(&tmp2, &tmp2_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp2;
	 }
	 else
	 {
	    tmp1 = __adjust_size(&tmp1, &tmp1_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp1;
	 }
      }
      
      for (i = 0; i < pres->dimension; i++)
	 _phys_unit[i] = _kdms_get_order(pres->segment->order[i], pres->order,
					 pres_unit, pres->dimension, 1);

      if (!_transpose(src_buf, dst_buf, pres_unit, _phys_unit, pres->dimension,
                      type, pres->order, pres->segment->order))
         return (FALSE);

      src_buf = dst_buf;
   }

   /*
    * scaling and normalizing can take place in the same buffer, so
    * that's what we'll do.
    */
   if (scale)
      if (!_scale(src_buf, pres_unit, pres->dimension,
                  precast ? pres->segment->datatype : type,
                  pres->scale_factor, pres->scale_offset_real,
                  pres->scale_offset_imag))
         return (FALSE);

   if (normalize)
      if (!_normalize(src_buf, pres_unit, pres->dimension,
                      precast ? pres->segment->datatype : type,
                      pres->norm_min, pres->norm_max))
         return (FALSE);

   /*
    * if the source data type has less precision or less range than
    * the destinatin, then the cast is practically the last thing
    * done.  This is because the scaling or normalization operations
    * above will be more accurate if performed on the higher order
    * data type. 
    */
   if (postcast) 
   {
      if (cmplx_convert)
      {
	 if (transpose)
	 {
	    tmp3 = __adjust_size(&tmp3, &tmp3_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp3;
	 }
	 else
	 {
	    tmp2 = __adjust_size(&tmp2, &tmp2_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp2;
	 }
      }
      else
      {
	 if (transpose)
	 {
	    tmp2 = __adjust_size(&tmp2, &tmp2_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp2;
	 }
	 else
	 {
	    tmp1 = __adjust_size(&tmp1, &tmp1_size, num_points, 
				 pres->segment->datatype);
	    dst_buf = tmp1;
	 }
      }
      
      if (!_cast(src_buf, dst_buf, pres_unit, pres->dimension, 
                 type, pres->segment->datatype))
         return (FALSE);

      src_buf = dst_buf;
   }
   
   /*
    * in a get pipeline, the first thing that should happen is that
    * the data is clipped out of the physical buffer.  This is done
    * right away so that the remaining pipeline stages can be freed
    * from the unnecessary complexity of coping with a subspace of the
    * buffer. 
    */

   if (interpolate) 
   {
      int dst_clipunit[KDMS_MAX_DIM];
      int src_buf_size[KDMS_MAX_DIM];
      int dst_buf_size[KDMS_MAX_DIM];
      int dst_begin[KDMS_MAX_DIM];
      int src_clipunit[KDMS_MAX_DIM];
      int src_begin[KDMS_MAX_DIM];
      int pres_begin[KDMS_MAX_DIM];
      int pres_size[KDMS_MAX_DIM];
      
      /*
       * later computations require the begin marker and sizein the
       * presentation data space, but in the physical layer index
       * ordering, so first reorder the input begin marker into the
       * physical index order. 
       */
      for (i = 0; i < pres->dimension; i++)
      {
	 pres_begin[i] = _kdms_get_order(pres->segment->order[i], pres->order,
					 begin, pres->dimension, 1);
	 pres_size[i] = _kdms_get_order(pres->segment->order[i], pres->order, 
					pres->size, pres->dimension, 1);
      }
      
      for(i = 0; i < pres->dimension; i++) 
      {
	 src_buf_size[i] = _kdms_get_order(pres->segment->order[i], 
					   pres->order, pres_unit, 
					   pres->dimension, 1);
	 src_begin[i] = kmax(0, -pres_begin[i]);
	 src_clipunit[i] = kmin(pres_size[i] - pres_begin[i], 
				src_buf_size[i] - src_begin[i]);
      }
      
      for (i = 0; i < pres->dimension; i++)
      {
	 dst_buf_size[i] = pres->segment->end[i] - pres->segment->begin[i] + 1;
	 dst_clipunit[i] = (dst_buf_size[i] - kmax(0, -phys_begin[i])) - 
	    kmax(0, pres->segment->end[i] - phys_end[i] );
	 dst_clipunit[i] = kmax(1, kmin(phys_unit[i], dst_clipunit[i]));
	 dst_begin[i] = kmax(0, phys_begin[i] - pres->segment->begin[i]);
      }

      if (!_clipinterp(src_buf, pres->segment->data, src_buf_size, 
		       dst_buf_size, pres_size, pres->segment->size, 
		       pres->dimension, pres->segment->datatype, src_begin, 
		       dst_begin, src_clipunit, dst_clipunit, phys_begin))
	 return (FALSE);
   }
   else
   {
      int *src_buf_size = phys_unit;
      int dst_buf_size[KDMS_MAX_DIM];
      int src_begin[KDMS_MAX_DIM];
      int dst_begin[KDMS_MAX_DIM];
      int clipunit[KDMS_MAX_DIM];
      
      for (i = 0; i < pres->dimension; i++)
      {
	 dst_buf_size[i] = pres->segment->end[i] - pres->segment->begin[i] + 1;

	 dst_begin[i] = kmax(0, phys_begin[i] - pres->segment->begin[i]);
	 src_begin[i] = kmax(0, - phys_begin[i]);

	 /* trim the unit to fit into the destination buffer */
	 clipunit[i] = kmin(phys_end[i],pres->segment->end[i]) - 
	    kmax(phys_begin[i],0) + 1;
      }
      
      if (!_clip(src_buf, pres->segment->data, src_buf_size, dst_buf_size,
                 pres->dimension, pres->segment->datatype, src_begin, 
		 dst_begin, clipunit))
	 return (FALSE);
   }

   return (TRUE);
}

/************************************************************
*
*  Routine Name: kdms_get_data - get data from data object
*
*       Purpose: 
*		 kdms_get_data is used to obtain data that is stored
*		 in a data object.  The data that is retrieved is
*		 designated by two "corner-markers".  These are arrays
*		 which contain N integer values, where N is the
*		 dimensionality of the segment (the dimensionality of
*		 a segment can be determined with the KDMS_DIMENSION
*		 attribute).  All values in begin argument must be
*		 less than or equal to their corresponding value in
*		 the end argument.  In a two dimensional case, the
*		 begin marker is the upper left corner and the end
*		 marker is the lower right corner of a rectangle that
*		 is obtained with this function call.  The coordinate
*		 origin is the upper, left, front corner of the data
*		 set extended to N-space.  The corner markers are
*		 specified in the index order that the data set is
*		 presented in.
*
*         Input: object - the object from which the data will be obtained.
*		 segment - the segment from which the data will be obtained.
*		 begin  - the begin marker of the region of data to
*		          be retrieved.
*		 end    - the end marker of the region of data to be
*			  retrieved.
*                data      - a pointer to the region of
*                            memory that will serve as a
*                            destination for the data. If
*                            this value is NULL, then
*                            sufficient space for this
*                            operation will be allocated
*                            for this operation.  The data
*                            type kaddr is used because
*                            it indicates a generic data
*                            pointer.
*
*        Output: If "data" is not initially NULL, then the memory
*		 that it points to will be filled with the data
*		 that is being requested.
*       Returns: If "data" is not initially NULL, then the data
*                space pointed to by "data" will be returned on
*                success.  If the "data" argument is NULL, then
*                a new pointer to the requested data will be
*                returned.  Unsuccessful calls to this routine
*                are indicated by a return value of NULL.
*
*  Restrictions: This routine assumes that if the argument
*                "data" is not NULL, then it contains the
*                appropriate amount of memory with the
*                appropriate dimensionality for the
*                requested primitive.
*
*    Written By: Jeremy Worley
*          Date: Oct 05, 1993 19:54
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kaddr
kdms_get_data(
   kobject object,
   char *segment,
   int *begin,
   int *end,
   kaddr data)
{
   kdms_segment_defin *segd = NULL;
   int i;
   kpresentation *pres;
   kaddr tmp;
   int offset_begin[KDMS_MAX_DIM], offset_end[KDMS_MAX_DIM];
   
   if (!_kdms_compiled)
      _kdms_init();

   /* -- if there is a defined segment get_data routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->get_data)
      {
	 kaddr return_data;
	 
	 segd->lock = TRUE;
	 return_data = segd->get_data(object, segment, begin, end, data);
	 segd->lock = FALSE;
	 return return_data;
      }

   /* -- otherwise, do a normal get_data -- */   
   if (!_kdms_get_segment_info(object, (int)kstring_to_token(segment),
			       &pres) || !pres)
      return (NULL);

   /*
    * it may be necessary to update the physical layer.  if so, do it now.
    */
   if (pres->segment->update == TRUE)
      _kdms_update_segment(object, pres);

   /*
    * check for zero size, dimension and unset size.
    */
   if (pres->dimension <= 0)
   {
      kerror("kdataman", "kdms_get_data", 
	     "The dimension of the %s segment has not been initialized."
	     "Unable to get data.\n", ktoken_to_string(pres->id));
      _kdms_set_error(KDMS_EATTR_UNSET);
      return (NULL);
   }

   if (pres->datatype <= 0) 
   {
      kerror("kdataman", "kdms_get_data", 
	     "The data type of the %s segment has not been initialized."
	     "Unable to get data.\n", ktoken_to_string(pres->id));
      _kdms_set_error(KDMS_EATTR_UNSET);
      return (NULL);
   }

   for (i = 0; i < pres->dimension; i++)
      if (pres->size[i] <= 0) 
      {
	 kerror("kdataman", "kdms_get_data", 
	     "The size of the %s segment has not been initialized or is not "
	     "legal.  Unable to get data.\n", ktoken_to_string(pres->id));
	 _kdms_set_error(KDMS_EATTR_UNSET);
	 return (NULL);
      }
	 
   if (!_kdms_legal_order(pres->segment->order, pres->segment->dimension))
   {
      kerror("kdataman", "kdms_get_data", 
	     "The physical layer index order of the %s segment has not been "
	     "initialized or is not legal.  Unable to get data.\n",
	     ktoken_to_string(pres->id));
      _kdms_set_error(KDMS_EATTR_UNSET);
      return (NULL);
   }

   if (!_kdms_legal_order(pres->order, pres->dimension))
   {
      kerror("kdataman", "kdms_get_data", 
	     "The presentation index order of the %s segment has not been "
	     "initialized or is not legal.  Unable to get data.\n",
	     ktoken_to_string(pres->id));
      _kdms_set_error(KDMS_EATTR_UNSET);
      return (NULL);
   }
   
   /*
    * kludgey, but it'll work.
    */
   for (i = 0; i < pres->dimension; i++)
   {
      offset_begin[i] = begin[i] + pres->offset[i];
      offset_end[i] = end[i] + pres->offset[i];
   }
   begin = &(offset_begin[0]);
   end = &(offset_end[0]);
   
   /*
    * if the interpolate type is none, then fail if the data is outside
    * of the data space.  Issue no error message, as people sometimes
    * use this functionality to simplify their code.
    */
   if (pres->interp == KNONE &&
       (!_kdms_position_in_space(begin, pres->size, pres->dimension) ||
        !_kdms_position_in_space(end, pres->size, pres->dimension)))
      return (NULL);

   /*
    * now that all that nonsense is over, call the thing that really
    * does the work. 
    */
   if ((tmp = _kdms_get_data(object, pres, begin, end, data)) == NULL)
      return (NULL);

   /*
    * for now, but don't forget to move to the end.
    */
   if (pres->segment->num_callbacks != 0)
      _kdms_process_data_callbacks(object, pres, begin, end,
                                   KDMS_CALLBACK_ACCESS);

   /*
    * return the data
    */
   return (tmp);
}

/************************************************************
*
*  Routine Name: kdms_put_data - put data into object
*
*       Purpose: 
*		 kdms_put_data is used to store data that 
*		 into a data object.  The data that is stored is
*		 designated by two "corner-markers".  These are arrays
*		 which contain N integer values, where N is the
*		 dimensionality of the segment (the dimensionality of
*		 a segment can be determined with the KDMS_DIMENSION
*		 attribute).  All values in begin argument must be
*		 less than or equal to their corresponding value in
*		 the end argument.  In a two dimensional case, the
*		 begin marker is the upper left corner and the end
*		 marker is the lower right corner of a rectangle that
*		 is obtained with this function call.  The coordinate
*		 origin is the upper, left, front corner of the data
*		 set extended to N-space.  The corner markers are
*		 specified in the index order that the data set is
*		 presented in.
*
*         Input: object - the object from which the data will be stored.
*		 segment - the segment from which the data will be stored.
*		 begin  - the begin marker of the region of data to
*		          be retrieved.
*		 end    - the end marker of the region of data to be
*			  retrieved.
*                data      - a pointer to the region of
*                            memory that will serve as a
*
*         Input: 
*
*        Output: 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Oct 05, 1993 19:54
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int
kdms_put_data(
   kobject object,
		char *segment,
		int *begin,
		int *end,
		kaddr data)
{
   kdms_segment_defin *segd = NULL;
   int i;
   kpresentation *pres;
   int offset_begin[KDMS_MAX_DIM], offset_end[KDMS_MAX_DIM];
   
   if (!_kdms_compiled)
      _kdms_init();

   /* -- if there is a defined segment put_data routine, then use that -- */
   if ((segd = kdms_locate_segment_defin(segment)) != NULL)
      if (segd->put_data)
      {
	 int status;
	 
	 segd->lock = TRUE;
	 status = segd->put_data(object, segment, begin, end, data);
	 segd->lock = FALSE;
	 return status;
      }

   /* -- otherwise, do a normal put_data -- */   
   if (!_kdms_get_segment_info(object, (int)kstring_to_token(segment),
			       &pres) || !pres)
      return (FALSE);

   /*
    * it may be necessary to update the physical layer.  if so, do it now.
    */
   if (pres->segment->update == TRUE)
      _kdms_update_segment(object, pres);

   /*
    * check for zero size, dimension and unset size.
    */
   if (pres->dimension <= 0)
   {
      kerror("kdataman", "kdms_put_data", 
	     "The dimension of the %s segment has not been initialized."
	     "Unable to put data.\n", ktoken_to_string(pres->id));
      _kdms_set_error(KDMS_EATTR_UNSET);
      return (FALSE);
   }

   if (pres->datatype <= 0) 
   {
      kerror("kdataman", "kdms_put_data", 
	     "The data type of the %s segment has not been initialized."
	     "Unable to put data.\n", ktoken_to_string(pres->id));
      _kdms_set_error(KDMS_EATTR_UNSET);
      return (FALSE);
   }

   for (i = 0; i < pres->dimension; i++)
      if (pres->size[i] <= 0) 
      {
	 kerror("kdataman", "kdms_put_data", 
		"The size of the %s segment has not been initialized or is "
		"not legal.  Unable to put data.\n", 
		ktoken_to_string(pres->id));
	 _kdms_set_error(KDMS_EATTR_UNSET);
	 return (FALSE);
      }
	 
   /*
    * copy the permanent data to a temporary data set if its
    * read only.  The pres->phys->temporary test is all powerful!
    */
   if (object->phys->protected && !pres->segment->temporary)
      _kdms_make_temporary(object, pres);

   /*
    * kludgey, but it'll work.
    */
   for (i = 0; i < pres->dimension; i++)
   {
      offset_begin[i] = begin[i] + pres->offset[i];
      offset_end[i] = end[i] + pres->offset[i];
   }
   begin = &(offset_begin[0]);
   end = &(offset_end[0]);
   
   /*
    * if the interpolate type is none, then fail if the data is outside
    * of the data space.  Issue no error message, as people sometimes
    * use this functionality to simplify their code.
    */
   if (pres->interp == KNONE &&
       !(_kdms_position_in_space(begin, pres->size, pres->dimension) ||
         !_kdms_position_in_space(end, pres->size, pres->dimension)))
      return (FALSE);

   /*
    * the save callback must occur *before* the change of data.
    */
   if (pres->segment->num_callbacks != 0)
      _kdms_process_data_callbacks(object, pres, begin, end,
                                   KDMS_CALLBACK_SAVE);

   /*
    * now call the thing that does all the work.
    */
   if (_kdms_put_data(object, pres, begin, end, data) == FALSE)
      return (FALSE);

   /*
    * for now, but don't forget to put at the end.
    */
   if (pres->segment->num_callbacks != 0)
      _kdms_process_data_callbacks(object, pres, begin, end,
                                   KDMS_CALLBACK_CHANGE);

   /*
    * if we got to this point, then all is well.
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_free_pipeline_memory
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Nov 08, 1994 08:50
| Modifications:
|
------------------------------------------------------------*/

void
_kdms_free_pipeline_memory(void)
{
   if (tmp1_size > 0)
      kfree(tmp1);

   if (tmp2_size > 0)
      kfree(tmp2);

   if (tmp3_size > 0)
      kfree(tmp3);
   
   return;
}
