 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros XVIMAGE Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             xvimage_check()
   >>>>             xvimage_input()
   >>>>             xvimage_output()
   >>>>             _init()
   >>>>             xvimage_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KXVIMAGE_DEF)
DataServiceInformation xvimage_format[] =
{NULL};

#else

#include "kdatafmt/xvimage.h"


static int xvimage_check PROTO((int));
static int xvimage_input PROTO((kobject, int, int));
static int xvimage_output PROTO((kobject, int, int));
static int xvimage_destroy PROTO((kobject));
static int xvimage_order PROTO((char *, int *));
static kaddr xvimage_read_data PROTO((kobject, int, kaddr *, int *, int *));
static int xvimage_write_data PROTO((kobject, int, kaddr, int *, int *));
static int xvimage_architecture PROTO((kobject));

static int read_compressed(int, char *, int, xvimage *);

DataServiceInformation xvimage_format[] =
{
   {
      "Old Khoros 1.0 Visualization Image File Format (xvimage)",
      "xvimage",
      xvimage_check,
      xvimage_input,
      xvimage_output,
      xvimage_destroy,
      xvimage_read_data,
      xvimage_write_data,
      xvimage_order,
      xvimage_architecture,
      NULL,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *              xvimage    -  Old Khoros 1.0 Visualization Image File Format
 */
typedef struct
{
   xvimage *image;
   int fid;
}
ResourceStruct;


#define KXVIMAGE_MAP_ENABLE "xvimageMapEnable"
#define KXVIMAGE_VALUE_SUBROW_SIZE "xvimageSubrowSize"


/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an xvimage image
|
|       Purpose: This function is used to create an xvimage image.  Which
|                means that we use the current dimension values in order
|                to create a proper xvimage.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_define_attributes
|
|       Purpose: This function is called to predefine
|                all application specific attributes that
|                are specifically needed by xvimage glue.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 04, 1993 11:26
| Modifications:
|
------------------------------------------------------------*/

static int 
xvimage_define_attributes(void)
{
   static int defed = FALSE;

   if (defed)
      return (TRUE);

   /*
    * there are a few attributes that must be defined when using the
    * xvimage file format, so define them now.
    */
   
   if (!kdms_query_attribute_definition(KDMS_OBJECT, KDMS_COMMENT))
      kdms_define_attribute(KDMS_OBJECT, KDMS_COMMENT, 1, 1, KSTRING,
                            TRUE, TRUE,
                            NULL);

   if (!kdms_query_attribute_definition(KDMS_OBJECT, _INTERNAL_COLORSPACE))
      kdms_define_attribute(KDMS_OBJECT, _INTERNAL_COLORSPACE, 1, 1, KINT,
                            TRUE, TRUE,
                            KCOLOR_COLORSPACE_DEFAULT);

   kdms_define_attribute(KDMS_OBJECT, KXVIMAGE_VALUE_SUBROW_SIZE, 1, 1, KINT,
                         TRUE, TRUE,
                         0);

   kdms_define_attribute(KDMS_OBJECT, KXVIMAGE_MAP_ENABLE, 1, 1, KINT,
                         TRUE, TRUE,
                         KOPTIONAL);

   defed = TRUE;

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_info(object)
|
|       Purpose: This routine is used primarily to query
|                the nature of a segment in the viff.
|                the main piece of information is offset.
|
|         Input: object - object.
|
|        Output: offset - the offset of the data segment for
|                         the object in question.
|
|       Returns: TRUE (1) when that segment does exist in this
|                file format, FALSE (0) when the segment does
|                not exist in this file format.
|
|    Written By: Jeremy Worley
|          Date: Jan 15, 1993 14:25
| Modifications:
|
------------------------------------------------------------*/

static int
xvimage_info(kobject object, char *segment, int *offset)
{
   ResourceStruct *resources = (ResourceStruct *) _kdms_glue_get_resources(object);
   kpresentation *presentation;
   int dim[KDMS_MAX_DIM];
   int mapsize,
      locsize,
      siz;
   int i;

   /*
    * determine the map segment size
    */
   if (!_kdms_get_segment_info(object, kstring_to_token(KDMS_SEGMENT_MAP),
            &presentation) || resources->image->map_scheme == VFF_MS_NONE ||
       resources->image->map_col_size == 0)
      mapsize = 0;
   else
   {
      if (presentation->segment->locked)
      {
         for (i = 0; i < presentation->dimension; i++)
            dim[i] = presentation->size[i];
         for (i = presentation->dimension; i < 5; i++)
            dim[i] = 1;
         siz = kdata_size(presentation->segment->datatype);
      }
      else
      {
         for (i = 0; i < presentation->dimension; i++)
            dim[i] = presentation->size[i];
         for (i = presentation->dimension; i < 5; i++)
            dim[i] = 1;
         siz = kdata_size(presentation->datatype);
      }
      mapsize = dim[0] * dim[1] * dim[2] * dim[3] * dim[4] * siz;
   }

   /*
    * determine the location segment size
    */
   if (!_kdms_get_segment_info(object, kstring_to_token(KDMS_SEGMENT_LOCATION),
                      &presentation) || resources->image->location_dim == 0)
      locsize = 0;
   else
   {
      if (presentation->segment->locked)
      {
         for (i = 0; i < presentation->dimension; i++)
            dim[i] = presentation->size[i];
         for (i = presentation->dimension; i < 5; i++)
            dim[i] = 1;
         siz = kdata_size(presentation->segment->datatype);
      }
      else
      {
         for (i = 0; i < presentation->dimension; i++)
            dim[i] = presentation->size[i];
         for (i = presentation->dimension; i < 5; i++)
            dim[i] = 1;
         siz = kdata_size(presentation->datatype);
      }
      locsize = dim[0] * dim[1] * dim[2] * dim[3] * dim[4] * siz;
   }

   if (kstrcmp(segment, KDMS_SEGMENT_MAP) == 0)
   {
      if (offset != NULL)
         *offset = 1024;
      return (TRUE);
   }
   if (kstrcmp(segment, KDMS_SEGMENT_LOCATION) == 0)
   {
      if (offset != NULL)
      {
         *offset = 1024 + mapsize;;
      }
      return (TRUE);
   }
   if (kstrcmp(segment, KDMS_SEGMENT_VALUE) == 0)
   {
      if (offset != NULL)
         *offset = 1024 + mapsize + locsize;
      return (TRUE);
   }
   else
   {
      *offset = -1;
      _kdms_set_error(KDMS_EINTERNAL);
   }
   
   return (FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_check - Checks to see if the file is
|                                an xvimage
|
|       Purpose: This function is used to check the first few bytes of
|                the stream to see if it is an xvimage.
|
|         Input: object - the xvimage object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
xvimage_check(int fid)
{
   unsigned char value;

   if ((kread_ubyte(fid, &value, 1) <= 0) ||
       ((int)value != (int)XV_FILE_MAGIC_NUM))
      return (FALSE);           /* not an error */

   else if ((kread_ubyte(fid, &value, 1) <= 0) ||
            ((int)value != (int)XV_FILE_TYPE_XVIFF))
      return (FALSE);           /* not an error */

   else if ((kread_ubyte(fid, &value, 1) <= 0) ||
            ((int)value != (int)XV_IMAGE_REL_NUM))
      return (FALSE);           /* not an error */

   else if ((kread_ubyte(fid, &value, 1) <= 0) ||
            ((int)value != (int)XV_IMAGE_VER_NUM))
      return (FALSE);           /* not an error */

   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvimage_input - Reda a xvimage image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is xvimage image.  The data was checked
|                by the xvimage_check() routine in which the xvimage
|                identifier indicated that the data is a valid xvimage.
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|         Input: object - the xvimage object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/
/*ARGSUSED2*/
static int
xvimage_input(kobject object, int fid, int flags)
{
   int type;
   int offset;
   int dsize;
   int dcount;
   int msize;
   int mcount;
   int lsize;
   int lcount;
   int ltmp[3];
   int begin[KDMS_MAX_DIM];
   int end[KDMS_MAX_DIM];
   int size[KDMS_MAX_DIM];
   int order[KDMS_MAX_DIM];
   int rorder[KDMS_MAX_DIM];
   ResourceStruct *resources;
   xvimage *image;

   ltmp[0] = ltmp[1] = ltmp[2] = -1;

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   resources->fid = fid;

   /*
    * Read the image from the object file and go ahead and set the xvimage
    * image within the resource structure.
    */
   /* kflock(fid, KLOCK_SH); */

   image = resources->image = xvimage_readheader(fid);

   if (image == NULL || !(xvimage_size(image, &dsize, &dcount, &msize,
                                       &mcount, &lsize, &lcount)))
   {
      /* kflock(fid, KLOCK_UN); */
      kfree(resources->image);
      _kdms_set_error(KDMS_EFMT_READFAIL);
      return (FALSE);
   }
   /* kflock(fid, KLOCK_UN); */

   /*
    * Initialize the data segments according to the data found in the xvimage
    * structure.
    * 
    * The xvimage structure only contains value (imagedata), maps, and location
    * data.
    */

   offset = ktell(fid);

   /*
    * check the map data
    */
   if (msize > 0)
   {
      int mw;
      int mh;
      int md;
      int mt;
      int me;

      xvimage2khoros_maptype(image->map_storage_type, &type);

      mw = image->map_row_size;
      mh = image->map_col_size;
      me = (image->map_scheme == VFF_MS_ONEPERBAND ||
            image->map_scheme == VFF_MS_CYCLE) ? image->num_data_bands
         : 1;
      md = mt = 1;

      size[0] = mh;
      size[1] = mw;
      size[2] = me;
      size[3] = md;
      size[4] = mt;

      order[0] = KMAP_HEIGHT;
      order[1] = KMAP_WIDTH;
      order[2] = KMAP_ELEMENTS;
      order[3] = KMAP_DEPTH;
      order[4] = KMAP_TIME;

      if (image->data_encode_scheme != VFF_DES_RAW)
      {
	 image->maps = kmalloc((unsigned)msize);
	 
	 read_compressed(fid, image->maps, msize, image);
	 begin[0] = begin[1] = begin[2] = begin[3] = begin[4] = 0;
	 end[0] = size[0] - 1; 
	 end[1] = size[1] - 1;
	 end[2] = size[2] - 1;
	 end[3] = size[3] - 1;
	 end[4] = size[4] - 1;
	 _kdms_set_segment(object, KDMS_SEGMENT_MAP, image->maps, type,
			   order, size, begin, end, 5);
      }
      else
	 _kdms_set_segment(object, KDMS_SEGMENT_MAP, NULL, type,
			   order, size, NULL, NULL, 5);

      /* need to make the map index order "correct" on loading */
      rorder[0] = KMAP_WIDTH;
      rorder[1] = KMAP_HEIGHT;
      rorder[2] = KMAP_DEPTH;
      rorder[3] = KMAP_TIME;
      rorder[4] = KMAP_ELEMENTS;
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_INDEX_ORDER, rorder);

      offset += msize;
   }

   /*
    * check the location data
    */
   if (lsize > 0)
   {
      size[0] = image->row_size;
      size[1] = image->col_size;
      size[2] = image->location_dim;
      size[3] = size[4] = 1;

      order[0] = KWIDTH;
      order[1] = KHEIGHT;
      order[2] = KDIMENSION;
      order[3] = KDEPTH;
      order[4] = KNONE;

      if (image->data_encode_scheme != VFF_DES_RAW)
      {
	 image->location = kmalloc((unsigned)lsize);
	 
	 read_compressed(fid, (char *)image->location, lsize, image);
	 begin[0] = begin[1] = begin[2] = begin[3] = begin[4] = 0;
	 end[0] = size[0] - 1; 
	 end[1] = size[1] - 1;
	 end[2] = size[2] - 1;
	 end[3] = size[3] - 1;
	 _kdms_set_segment(object, KDMS_SEGMENT_LOCATION, image->location, 
			   type, order, size, begin, end, 4);
      }
      else
	 _kdms_set_segment(object, KDMS_SEGMENT_LOCATION, NULL, KFLOAT,
			   order, size, NULL, NULL, 4);

      rorder[0] = KWIDTH;
      rorder[1] = KHEIGHT;
      rorder[2] = KDEPTH;
      rorder[3] = KDIMENSION;
      kdms_set_attribute(object, KDMS_SEGMENT_LOCATION, KDMS_INDEX_ORDER, rorder);

      offset += lsize;
   }

   /*
    * check the value data
    */
   if (dsize > 0)
   {
      xvimage2khoros_datatype(image->data_storage_type, &type);
      size[0] = image->row_size;
      size[1] = image->col_size;

      /*
       * This seems kinda strange, but its here to get around some
       * ambiguities in the Xvimage definition.
       */
      if (image->color_space_model == VFF_CM_NONE)
      {
         size[2] = image->num_data_bands;
         size[3] = size[4] = 1;
      }
      else if (image->num_data_bands % 3 == 0)
      {
         size[2] = 3;
         size[3] = 1;
         size[4] = image->num_data_bands / 3;
      }
      else
      {
         size[2] = size[3] = 1;
         size[4] = image->num_data_bands;
      }

      order[0] = KWIDTH;
      order[1] = KHEIGHT;
      order[2] = KELEMENTS;
      order[3] = KDEPTH;
      order[4] = KTIME;

      if (image->data_encode_scheme != VFF_DES_RAW)
      {
	 image->imagedata = kmalloc((unsigned)dsize);
	 
	 read_compressed(fid, image->imagedata, dsize, image);
	 begin[0] = begin[1] = begin[2] = begin[3] = begin[4] = 0;
	 end[0] = size[0] - 1; 
	 end[1] = size[1] - 1;
	 end[2] = size[2] - 1;
	 end[3] = size[3] - 1;
	 end[4] = size[4] - 1;
	 _kdms_set_segment(object, KDMS_SEGMENT_VALUE, image->imagedata, type,
			   order, size, begin, end, 5);
      }
      else
	 _kdms_set_segment(object, KDMS_SEGMENT_VALUE, NULL, type,
			   order, size, NULL, NULL, 5);
      
      rorder[0] = KWIDTH;
      rorder[1] = KHEIGHT;
      rorder[2] = KDEPTH;
      rorder[3] = KTIME;
      rorder[4] = KELEMENTS;
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_INDEX_ORDER, rorder);

      offset += dsize;
   }
   /*
    * predefine some attributes
    */
   xvimage_define_attributes();

   /*
    * initialize the global dimension structure to the size of of the data
    */
   kdms_set_attribute(object, NULL, KDMS_COMMENT, image->comment);

   kdms_set_attribute(object, NULL, _INTERNAL_COLORSPACE,
                      xvimage2khoros_csm(image->color_space_model));

   kdms_set_attribute(object, NULL, KXVIMAGE_MAP_ENABLE,
                (image->map_enable == VFF_MAP_FORCE) ? KFORCED : KOPTIONAL);

   kdms_set_attribute(object, NULL, KXVIMAGE_VALUE_SUBROW_SIZE,
                      (int)image->subrow_size);

   kdms_create_attribute(object, NULL, "xvimageMapScheme", 1, 1, KULONG,
                         TRUE, TRUE);
   kdms_set_attribute(object, NULL, "xvimageMapScheme", image->map_scheme);

   kdms_create_attribute(object, NULL, "ispare1", 1, 1, KULONG,
                         TRUE, TRUE);
   kdms_set_attribute(object, NULL, "ispare1", image->ispare1);

   kdms_create_attribute(object, NULL, "ispare2", 1, 1, KULONG,
                         TRUE, TRUE);
   kdms_set_attribute(object, NULL, "ispare2", image->ispare2);

   kdms_create_attribute(object, NULL, "fspare1", 1, 1, KFLOAT,
                         TRUE, TRUE);
   kdms_set_attribute(object, NULL, "fspare1", image->fspare1);

   kdms_create_attribute(object, NULL, "fspare2", 1, 1, KFLOAT,
                         TRUE, TRUE);
   kdms_set_attribute(object, NULL, "fspare2", image->fspare2);

   if (image->startx >= 0 || image->starty >= 0) 
   {
      kdms_create_attribute(object, NULL, KPDS_SUBOBJECT_POSITION, 5, 1,
			    KINT, TRUE, TRUE);
      kdms_set_attribute(object, NULL, KPDS_SUBOBJECT_POSITION,
			 (int)image->startx, (int)image->starty, 0, 0, 0);
   }

   /* see if there is already location data */
   if (kdms_query_segment(object, KDMS_SEGMENT_LOCATION))
   {
      /* 
       *  if there is already location data, then pixsizx and pixsizy
       *  correspond to the height and width image point size 
       */
      kdms_create_attribute(object, NULL, KPDS_POINT_SIZE, 5, 1,
			    KDOUBLE, TRUE, TRUE);

      kdms_set_attribute(object, NULL, KPDS_POINT_SIZE, 
			 (double)image->pixsizx, (double)image->pixsizy, 
			 1.0, 1.0, 1.0);
   }
   else
   {
      /* 
       *  if pixsizx and pixsizy are not 1.0 then we can use it to define
       *  corners for uniform location data.
       */
      if ((image->pixsizx != 1.0) || (image->pixsizy != 1.0))
      {
	 /* this is code from kaps_create_location -- if the glue was
	    in the kappserv library, I'd change all this to a 
	    kpds_create_location */
	 int loc_size[4] = {1, 1, 1, 2};
	 int ord[2] = {KWIDTH, KDIMENSION};
	 int siz[2] = {2,3};
	 int dim;

	 int cbeg[2] = {0,0};
	 int cend[2] = {1,2};
	 double dt[6] = {0., 1., 0., 1., 0., 1.};
	 
#if 0
	 int cbeg[2] = {0,0};
	 int cend[2] = {0,2};
	 double b[3] = {0, 0, 0};
	 double e[3] = {1, 1, 1};
#endif

	 loc_size[0] = (image->row_size == 0) ? 1 : image->row_size;
	 loc_size[1] = (image->col_size == 0) ? 1 : image->col_size;

	 /* -- create uniform location data by hand :( -- */
	 if (!kdms_create_segment(object, "uniform"))
	    return FALSE;

	 /* -- if kpds_init happends to have been called, 
	    then the segment is alread initialized .. scope
	    it out by seeing if the dimension is already = 2 -- */
	 if (!kdms_get_attribute(object,"uniform",KDMS_DIMENSION,&dim))
	    return FALSE;
	 
	 if (dim != 2)
	    if (!kdms_set_attributes(object, "uniform",
				     KDMS_DIMENSION,   2,
				     KDMS_SIZE,        siz,
				     KDMS_DATA_TYPE,   KDOUBLE,
				     KDMS_INDEX_ORDER, ord, NULL))
	       return FALSE;

	 dt[1] = loc_size[0]*image->pixsizx;
	 dt[3] = loc_size[1]*image->pixsizy;

	 /* -- put default uniform begin point -- */
	 kdms_put_data(object, "uniform", cbeg, cend, dt);
#if 0
	 kdms_put_data(object, "uniform", cbeg, cend, b);

	 /* -- put default uniform end points -- */
	 cbeg[0] = 1; cend[0] = 1;

	 e[0] = loc_size[0]*image->pixsizx;
	 e[1] = loc_size[1]*image->pixsizy;
	 
	 kdms_put_data(object, "uniform", cbeg, cend, e);
#endif

	 /* -- create the physical uniform location size attribute -- */
	 if (!kdms_query_attribute(object, "uniform", "uniformSize",
				   NULL, NULL, NULL, NULL))
	    if (!kdms_create_attribute(object, "uniform", "uniformSize", 1,
				       4, KINT, TRUE, TRUE))
	       return FALSE;
	 
	 if (!kdms_set_attribute(object, "uniform", "uniformSize", loc_size))
	    return FALSE;

	 kdms_create_attribute(object, NULL, "locationGrid", 1, 1, 
			       KINT, TRUE, TRUE);
	 kdms_set_attribute(object, NULL, "locationGrid", KUNIFORM);
      }
   }
   
      
   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvimage_output - Closes the xvimage image
|
|       Purpose: This function is used to close the xvimage image.  Which
|                means if the image was modified we then re-write image
|                back out.
|
|         Input: object - the xvimage object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED2*/
static int
xvimage_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;

   xvimage *image;
   int type;
   int startx;
   int starty;
   double dtmp0;
   double dtmp1;
   double wbeg;
   double hbeg;
   double wend;
   double hend;
   int itmp;
   char *tmpcomment = NULL;
   int *size;
   int *order;
   unsigned long ispare1 = 0;
   unsigned long ispare2 = 0;
   float fspare1 = 0.0;
   float fspare2 = 0.0;


   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   resources->fid = fid;

   /* kflock(fid, KLOCK_EX); */

   if (kdms_query_attribute(object, NULL, KDMS_COMMENT, NULL, NULL, NULL, NULL))
      kdms_get_attribute(object, NULL, KDMS_COMMENT, &tmpcomment);
   else
      tmpcomment = NULL;

   if (resources->image == NULL)
   {
      resources->image = xvimage_create(0, 0, VFF_TYP_1_BYTE, 0, 0,
                                    (tmpcomment == NULL ? " " : tmpcomment),
                                        0, 0, VFF_MS_NONE, VFF_MAPTYP_1_BYTE,
                                        VFF_LOC_IMPLICIT, 0);
      if (!resources->image)
      {
         _kdms_traceback("xvimage_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }
   }
   image = resources->image;

   /*
    * Initialize the image's data arrays according to the data found in the
    * data segment->
    * 
    * The xvimage structure only contains value (imagedata), maps, and location
    * data.
    */

   /*
    * check the value data
    */
   if (kdms_get_attributes(object, KDMS_SEGMENT_VALUE, 
			    KDMS_PHYSICAL_DATA_TYPE, &type,
			    KDMS_PHYSICAL_SIZE, &size,
			    KDMS_PHYSICAL_INDEX_ORDER, &order,
			    NULL) == TRUE)
   {
      khoros2xvimage_datatype(type, &image->data_storage_type);
      image->imagedata = NULL;  /* set the pointer to null for files */

      image->row_size = _kdms_get_order(KWIDTH, order, size, 5, 1);
      image->col_size = _kdms_get_order(KHEIGHT, order, size, 5, 1);
      image->num_data_bands = _kdms_get_order(KELEMENTS, order, size, 5, 1);
      image->num_of_images = _kdms_get_order(KTIME, order, size, 5, 1) *
         _kdms_get_order(KDEPTH, order, size, 5, 1);
   }

   /*
    * check the map data
    */
   if (kdms_get_attributes(object, KDMS_SEGMENT_MAP, 
			    KDMS_PHYSICAL_DATA_TYPE, &type,
			    KDMS_PHYSICAL_SIZE, &size,
			    KDMS_PHYSICAL_INDEX_ORDER, &order,
			    NULL) == TRUE)
   {
      khoros2xvimage_datatype(type, &image->map_storage_type);
      image->maps = NULL;
      image->map_row_size = _kdms_get_order(KMAP_WIDTH, order, size, 5, 1);
      image->map_col_size = _kdms_get_order(KMAP_HEIGHT, order, size, 5, 1);
      image->map_scheme = (size[2] > 1) ? VFF_MS_ONEPERBAND : VFF_MS_SHARED;
   }
   else
   {
      image->map_scheme = VFF_MS_NONE;
   }

   /*
    * check the location data
    */
   if (kdms_get_attributes(object, KDMS_SEGMENT_LOCATION, 
			    KDMS_PHYSICAL_DATA_TYPE, &type,
			    KDMS_PHYSICAL_SIZE, &size,
			    KDMS_PHYSICAL_INDEX_ORDER, &order,
			    NULL) == TRUE)
   {
      image->location = (float *)NULL;
      image->location_dim = _kdms_get_order(KDIMENSION, order, size, 4, 1);
   }

   /*
    * define some important attributes
    */
   xvimage_define_attributes();

   /*
    * get the relevant header information
    */
   tmpcomment = &(image->comment[0]);
   kdms_get_attribute(object, NULL, KDMS_COMMENT, &tmpcomment);

   kdms_get_attribute(object, NULL, _INTERNAL_COLORSPACE , &itmp);
   image->color_space_model = khoros2xvimage_csm(itmp);

   kdms_get_attribute(object, NULL, KXVIMAGE_VALUE_SUBROW_SIZE, &itmp);
   image->subrow_size = (unsigned long)itmp;

   if (kdms_query_attribute(object, NULL, "xvimageMapScheme", NULL, NULL,
			    NULL, NULL))
      kdms_get_attribute(object, NULL, "xvimageMapScheme", &image->map_scheme);
   else 
      image->map_scheme = VFF_MS_NONE;

   kdms_get_attribute(object,NULL,KPDS_SUBOBJECT_POSITION,&startx, &starty,
		      NULL, NULL, NULL);
   if (startx != 0 && starty != 0)
   {
      image->starty = (int)starty;
      image->startx = (int)startx;
   }
   else
   {
      image->startx = -1;
      image->starty = -1;
   }


   if (kdms_get_attribute(object, NULL, "ispare1", &ispare1))
      image->ispare1 = ispare1;

   if (kdms_get_attribute(object, NULL, "ispare2", &ispare2))
      image->ispare2 = ispare2;

   /* luxury exception for spectrum */
   if (kdms_get_attribute(object, NULL, "colorMapContents", &ispare1))
      image->ispare1 = ispare1;

   /* luxury exception for spectrum */
   if (kdms_get_attribute(object, NULL, "colorMapColnum", &ispare2))
      image->ispare2 = ispare2;
   
   if (kdms_get_attribute(object, NULL, "fspare1", &fspare1))
      image->fspare1 = fspare1;

   if (kdms_get_attribute(object, NULL, "fspare2", &fspare2))
      image->fspare2 = fspare2;


   /* if we have uniform location data, derive pixsizes from that */
   if (kdms_query_segment(object, "uniform"))
   {
      double wsize = image->row_size;
      double hsize = image->col_size;
      int *loc_size;
      double *b = NULL;
      double *e = NULL;
      int beg[2] = {0,0};
      int end[2] = {0,2};
      
      /* special case if there is no value data */
      if (image->row_size == 0 || image->col_size == 0)
      {
	 kdms_get_attribute(object, NULL, "locationSize", &loc_size);
	 if (loc_size != NULL)
	 {
	    wsize = loc_size[0];
	    hsize = loc_size[1];
	 }

	 /* if still 0, then make them 1 */
	 if (wsize == 0.0)
	    wsize = 1;
	 if (hsize == 0.0)
	    wsize = 1;
      }

      /* -- get uniform begin point -- */
      b = kdms_get_data(object, "uniform", beg, end, NULL);

      /* -- get uniform end point -- */
      beg[0] = 1; end[0] = 1;
      e = kdms_get_data(object, "uniform", beg, end, NULL);
      
      wbeg = b[0];
      wend = e[0];
      hbeg = b[1];
      hend = e[1];
      
      kfree(b);
      kfree(e);
      
      image->pixsizx = (float) (wend-wbeg)/wsize;
      image->pixsizy = (float) (hend-hbeg)/hsize;
   }
   else if (kdms_get_attribute(object, NULL, KPDS_POINT_SIZE, 
			       &dtmp0, &dtmp1, NULL, NULL, NULL))
   {
      image->pixsizy = dtmp0;
      image->pixsizx = dtmp1;
   }
   else
   {
      image->pixsizy = 1.0;
      image->pixsizx = 1.0;
   }


   /*
    * Write the xvimage structure out to the specified filename
    */
   klseek(resources->fid, KSEEK_SET, 0);
   xvimage_writeheader(fid, image);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_destroy - Frees an xvimage image
|
|       Purpose: This function is used to create an xvimage image.  Which
|                means that we use the current dimension values in order
|                to create a proper xvimage.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
xvimage_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * block and wait until I can have the descriptor to myself
    */
   kclose(resources->fid);

   /*
    * free the header structure
    */
   xvimage_free(resources->image);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}


static int
xvimage_order(char *seg, int *ord)
{
   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_MAP))
   {
      ord[0] = KMAP_HEIGHT;
      ord[1] = KMAP_WIDTH;
      ord[2] = KMAP_ELEMENTS;
      ord[3] = KMAP_DEPTH;
      ord[4] = KMAP_TIME;
   }
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_read_data
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 07:04
| Modifications:
|
------------------------------------------------------------*/

static kaddr
xvimage_read_data(kobject object, int seg, kaddr * data, int *begin, int *end)
{
   ResourceStruct *resources;
   kpresentation *pres;
   int offset;
   kfile *tmp;
   

   if (!_kdms_get_segment_info(object, seg, &pres) ||
       !pres)
      return (FALSE);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (!xvimage_info(object, ktoken_to_string(seg), &offset))
      return (NULL);

   tmp = kfdopen(resources->fid, "r");
   
   *data = _kdms_glue_read_segment_data(tmp,
                                kfile_getmachtype(resources->fid),
                                          kfread_generic,
                                          kfseek,
                                          offset,
                                          pres->segment->datatype,
                                          pres->segment->dimension,
                                          pres->segment->size,
					  NULL,
                                          data,
                                          begin,
					  end);

   /*
    * xvimage stores bit data as LSB, so we gotta twiddle it before
    * returning.
    */
   if (pres->segment->datatype == KBIT)
      kdata_cast(*data, KBIT_LSB, KBIT, ((end[0] - begin[0] + 1) * 
		 (end[1] - begin[1] + 1) * (end[2] - begin[2] + 1) * 
		 (end[3] - begin[3] + 1) * (end[4] - begin[4] + 1)), 
		 KNONE, 1.0, 1.0, *data);

   return (*data);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_write_data
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 07:09
| Modifications:
|
------------------------------------------------------------*/

static int
xvimage_write_data(kobject object, int seg, kaddr data, int *begin, int *end)
{
   ResourceStruct *resources;
   kpresentation *pres;
   int offset;
   int num;
   unsigned char *tdata = NULL;
   int status;
   kfile *tmp;
   
   if (!_kdms_get_segment_info(object, seg, &pres) || !pres)
      return (FALSE);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);
   
   if (!xvimage_info(object, ktoken_to_string(seg), &offset))
      return (FALSE);

   /*
    * xvimage stores bit data as LSB, so we gotta twiddle it before
    * returning.
    */
   if (pres->segment->datatype == KBIT)
   {
      num = ((end[0] - begin[0] + 7) >> 3) * 
	 (end[1] - begin[1] + 1) * (end[2] - begin[2] + 1) * 
	 (end[3] - begin[3] + 1) * (end[4] - begin[4] + 1);
      
      tdata = (unsigned char *)kmalloc((unsigned)num);
      
      kdata_cast(data, KBIT_LSB, KBIT, num << 3, KNONE, 1.0, 1.0, tdata);

      data = tdata;
   }

   tmp = kfdopen(resources->fid, "w");
   
   status = _kdms_glue_write_segment_data(
      tmp,
      kfile_getmachtype(resources->fid),
      kfwrite_generic,
      kfseek,
      offset,
      pres->segment->datatype,
      pres->segment->dimension,
      pres->segment->size,
      NULL,
      data,
      begin,
      end);

   if (pres->segment->datatype == KBIT)
      kfree(tdata);
   
   return(status);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvimage_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 08, 1993 11:35
| Modifications:
|
------------------------------------------------------------*/

static int
xvimage_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
      return (kfile_getmachtype(resources->fid));

}

/*-----------------------------------------------------------
|
|  Routine Name: read_compressed
|       Purpose: read data if it is compressed.
|         Input: fd - the file descriptor in which to read from
|		 data - a character pointer where the bytes will be 
|			read and stored to
|		 size - an integer specifying the number of bytes to 
|			to be read.
|		 image - pointer to xvimage structure.
|        Output:
|       Returns: TRUE on success, FALSE on failure.
|    Written By: Scott Wilson
|          Date: Apr 22, 1989
| Modifications: Sep 19, 1994 11:06 JW - upgraded to K 2.0
|
------------------------------------------------------------*/

static char *uncompr_cmd[] =
	{"This_should_never_happen",      /* DES-RAW */
         "uncompress",                    /* DES_COMPRESS */
         "unrle",                         /* DES_RLE */
         "untransform",                   /* DES_TRANSFORM */
         "unccitt",                       /* DES_CCITT */
         "unadpcm",                       /* DES_ADPCM */
         "ungencomp"};                    /* DES_GENERIC */

static int
read_compressed(int fd, char *data, int size, xvimage *image)
{
   char *tmpfilename;
   char command[255];
   char *buf;
   kfile *file;
   int fdtmp;
   int len,index,n;
   
   /* 
    * See if the size happends to be zero - if so, just return.
    */
   if (size == 0)
      return(FALSE);

   /* 
    * Grab the first 8 bytes from the file descriptor and see if
    * the data should be unencoded or just copied 
    */
   if (kread(fd, command, 8) != 8)
   {
      kerror("kdataman", "read_compressed", "couldn't read first 8 bytes!\n");
      return (FALSE);
   }
   
   command[8] = 0;
   len = atoi(command);

   if (len == 0)
   {
      if (kread(fd, data, size) != size)
      {
	 kerror("kdataman", "read_compressed", "Incorrect byte count!\n");
	 return (FALSE);
      }
      return (TRUE);
   }
   else
   {
      /* 
       * Read the data 
       */
      buf = kmalloc((unsigned)len);
      if (buf == NULL)
      {
	 kerror("kdataman","read_compr", "Not enough space for buffer!\n");
	 return (FALSE);
      }
      
      if (kread(fd, buf, len) != len)
      {
	 kerror("kdataman", "read_compr", 
		"Incorrect byte count on data block!\n");
	 kfree(buf);
	 return (FALSE);
      }
      
      /* 
       * Spew it thru the decompress command 
       */
      tmpfilename = ktempnam(NULL, "xvimageXXXXXX");
      index = image->data_encode_scheme;
      ksprintf(command,"%s > %s",uncompr_cmd[index],tmpfilename);
      if ((file = kpopen(command, "w")) == NULL)
      {
	 kerror("kdataman", "read_compressed", "Unable to open process\n");
	 kfree(buf);
	 return (FALSE);
      }
      if (kfwrite(buf,1,len,file) != len)
      {
	 kerror("kdataman","read_compressed", "Unable to write source data\n");
	 kpclose(file);
	 kfree(buf);
	 return (FALSE);
      }
      kpclose(file);
      kfree(buf);
      
      fdtmp = kopen(tmpfilename, KOPEN_RDONLY, 0444);
      n = kread(fdtmp, data, size);
      if (n != size)
      {
	 kerror("kdataman", "read_compressed", 
		 "Incorrect decoded byte count!\n");
	 kclose(fdtmp);
	 kunlink(tmpfilename);
	 return (FALSE);
      }
      kclose(fdtmp);
      kunlink(tmpfilename);
      return (TRUE);
   }
}

#endif /* KXVIMAGE_DEF */

/* don`t add after the endif */
