 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros AVS Data Service Routines
   >>>>
   >>>>   Static:
   >>>>                     _init()
   >>>>  Private:
   >>>>                     avs_check()
   >>>>                     avs_input()
   >>>>                     avs_output()
   >>>>                     avs_destroy()
   >>>>                     avs_order()
   >>>>                     avs_architecture()
   >>>>   Public:
   >>>>             None - no public routines should ever exist as 
   >>>>                    these are internal routines only accessible 
   >>>>                    via the DataServiceInformation *services[] 
   >>>>                    structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KAVS_DEF)
DataServiceInformation avs_format[] =
{NULL};
#else

#include "kdatafmt/avs.h"
#include "kappserv/kcolor.h"    /* need KCOLOR_HAS_ALPHA */

static int avs_check PROTO((int));
static int avs_input PROTO((kobject, int, int));
static int avs_output PROTO((kobject, int, int));
static int avs_destroy PROTO((kobject));
static int avs_order PROTO((char *, int *));
static int avs_architecture PROTO((kobject));

DataServiceInformation avs_format[] =
{
   {
      "Advanced Visualization Software Image format (AVS)",
      "avs",
      avs_check,
      avs_input,
      avs_output,
      avs_destroy,
      NULL,
      NULL,
      avs_order,
      avs_architecture,
      NULL,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data service :
 *
 *      avs  -  Advanced Visualization Software Format Utilities (AVS)
 */
typedef struct
{
   kfile *file;
   avs *image;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: _init - Initialize an resource structure
|
|       Purpose: This function is used to initialize a resource 
|                structure.
|
|         Input: object - object to set resource structure on
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;

   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: avs_check - Check if a file is an AVS file
|
|       Purpose: This function is used to check the supposed size
|                of the AVS image according to the first few bytes 
|                of the file and compares it to the overall size of
|                the file to see if it really is an AVS file.
|
|         Input: object - the avs object to be checked
|
|        Output: none
|
|       Returns: TRUE if the given file is an AVS file,
|                FALSE otherwise
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
avs_check(int fid)
{
   int size;
   int width;
   int height;

   /*
    *  Data is always stored in Sun format so we need to set the
    *  descriptor machine type to be of type KMACH_SPARC.  This
    *  will allow us to read the 4 byte width and height fields
    *  properly.
    */
   kfile_setmachtype(fid, KMACH_SPARC);

   if (kread_int(fid, &width, 1) != 1 || kread_int(fid, &height, 1) != 1)
      return (FALSE);

   /*   
    *  Verify that the image has the number of bytes that
    *  are actually indicated by the header.
    */
   size = klseek(fid, 0, SEEK_END);

   if ((size - 8) != width * height * 4)
      return (FALSE);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: avs_input - Read an AVS image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is an AVS image.  The data was checked
|                by the avs_check() routine in which the AVS
|                identifier indicated that the data is a valid AVS
|                image.
|
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|                This routine will convert from the AVS index order
|                into the more standard index order of width by
|                height by elements.  It will also shift the 
|                data ordering from RGBalpha to alphaRGB.
|                
|         Input: object - the avs object to be initialized
|
|        Output: Returns: TRUE on success, FALSE otherwise
|
|    Written By: Mark Young and Steve Kubica
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
avs_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   avs *image;

   int type = KUBYTE,
      i;
   int size[KDMS_MAX_DIM],
      order[KDMS_MAX_DIM];
   int begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM];
   int total,
      total2,
      total3;
   unsigned char *reordered_data = NULL;

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * Read the image from the object file and go ahead and set the AVS image
    * within the resource structure.
    */
   image = resources->image = avs_fdread(fid);
   if (image == NULL)
   {
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /*
    * Initialize the data segments according to the data found in the 
    * AVS structure.
    * 
    * The AVS structure only contains value (image->data)
    */

   /* set has alpha to true and colorspace model to rgb */
   kdms_create_attribute(object, NULL, KCOLOR_HAS_ALPHA, 
			 1, 1, KINT, TRUE, TRUE);
   kdms_set_attribute(object, NULL, KCOLOR_HAS_ALPHA, TRUE);
   kdms_create_attribute(object, NULL, _INTERNAL_COLORSPACE, 
			 1, 1, KINT, TRUE, TRUE);
   kdms_set_attribute(object, NULL, _INTERNAL_COLORSPACE, KRGB);

   /* If the width and height are resonable, then go for it! */
   if (image->width != 0 && image->height != 0)
   {
      size[0] = image->width;
      size[1] = image->height;
      size[2] = size[3] = 1;
      size[4] = 4;              /* alpha R G B */

      order[0] = KWIDTH;
      order[1] = KHEIGHT;
      order[2] = KDEPTH;
      order[3] = KTIME;
      order[4] = KELEMENTS;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      /* Precompute array offsets for faster indexing */
      total = image->width * image->height;
      total2 = total * 2;
      total3 = total * 3;

      /* 
       *  This next loop serves two purposes.  The first purpose is to
       *  reorder the data from [alpha R G B] to [R G B alpha]. 
       *  The second purpose is to change the index order from 
       *  elements first to elements last (might as well do this
       *  here since we are looping through every data point anyway).
       */
      reordered_data = (unsigned char *)kmalloc(total * 4 *
                                                sizeof(unsigned char));
      for (i = 0; i < total; i++)
      {
         reordered_data[i] = image->data[1 + i * 4];    /* R */
         reordered_data[i + total] = image->data[2 + i * 4];    /* G */
         reordered_data[i + total2] = image->data[3 + i * 4];   /* B */

         /* note that AVS uses this for opacity, we use it for alpha */
         reordered_data[i + total3] = 255 - image->data[i * 4];         /* alpha */
      }

      /* Put the reordered data into the object's value segment */
      _kdms_set_segment(object, KDMS_SEGMENT_VALUE, reordered_data, type,
                        order, size, begin, end, 5);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_COUPLING,KUNCOUPLED);

      /* This would ideally be done with straight kdms calls..  - SK */
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: avs_output - Closes an AVS image
|
|       Purpose: This function is used to close the AVS image.  
|                The data will be reordered by this function
|                from our default index order to the proper 
|                AVS index order.  It will also shift the 
|                data ordering from RGBalpha to alphaRGB.
|
|         Input: object - the avs object to be closed
|
|        Output: Returns: TRUE on success, FALSE otherwise
|
|    Written By: Mark Young and Steve Kubica
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
avs_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;

   avs *image;
   int type;
   int *size;
   int begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM];
   int i,
      total,
      total2,
      total3;
   unsigned char *data,
     *reordered_data = NULL;

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources->image == NULL)
   {
      resources->image = avs_create();
      if (!resources->image)
      {
         _kdms_traceback("avs_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }
   }
   image = resources->image;

   /*
    * Initialize the image's data arrays according to the data found in the
    * data segment.  
    *
    * The avs structure contains only value data.  (image->value)
    */

   /*
    * If there is value data awaiting output, then go and write it out.
    */
   if (kdms_query_segment(object, KDMS_SEGMENT_VALUE))
   {
      /* Make the object syncable so we can access the physical layer */
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_COUPLING, KCOUPLED);


      if (!kdms_sync(object, KDMS_SEGMENT_VALUE, KPHYS2PRES))
      {
         _kdms_traceback("avs_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }

      /* Get the size and data type */
      if (!kdms_get_attributes(object, KDMS_SEGMENT_VALUE,
                               KDMS_SIZE, &size,
                               KDMS_DATA_TYPE, &type,
                               NULL))
      {
         _kdms_traceback("avs_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }

      /* The width and height are in the order we explicitly specified */
      image->width = (int)size[0];
      image->height = (int)size[1];

      /* Set up the begin and end arrays for retrieving the data */
      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      /* Retrieve the data in the expected index order */
      if ((data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end, NULL))
	 == NULL)
      {
         _kdms_traceback("avs_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }

      /* Precompute array offsets for faster indexing */
      total = image->width * image->height;
      total2 = total * 2;
      total3 = total * 3;

      /* 
       *  This next loop serves two purposes.  The first purpose is to
       *  reorder the data from [R G B alpha] to [alpha R G B]. 
       *  The second purpose is to change the index order from 
       *  elements last to elements first (might as well do this
       *  here since we are looping through every data point anyway).
       */
      reordered_data = (unsigned char *)kcalloc((unsigned)(total * 4),
                                                sizeof(unsigned char));
      for (i = 0; i < total; i++)
      {
	 if (size[2] == 4)
	 {
	    /* note that AVS uses this for opacity, we use it for alpha */
	    reordered_data[i * 4] = 255 - data[i + total3];        /* alpha */
	    
	    reordered_data[1 + i * 4] = data[i];   /* R */
	    reordered_data[2 + i * 4] = data[i + total];   /* G */
	    reordered_data[3 + i * 4] = data[i + total2];  /* B */
	 }
	 else
	 {
	    /* note that AVS uses this for opacity, we use it for alpha */
	    reordered_data[i * 4] = 255;        /* alpha */
	    
	    reordered_data[1 + i * 4] = data[i];   /* R */
	    reordered_data[2 + i * 4] = data[i];   /* G */
	    reordered_data[3 + i * 4] = data[i];  /* B */
	 }
	 
      }

      /* free the image and temporary data */
      kfree(data);
      kfree(image->data);

      /* Assign the reordered image data */
      image->data = reordered_data;
   }

   /*
    * Write the avs structure out to the specified filename
    */
   return(avs_fdwrite(fid, image));
}


/*-----------------------------------------------------------
|
|  Routine Name: avs_destroy - Frees an avs image
|
|       Purpose: This function is used to destroy an AVS image. 
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: TRUE on success, FALSE otherwise
|
|    Written By: Mark Young
|          Date: Jul 13, 1992 15:50
| Modifications:
|
------------------------------------------------------------*/

static int
avs_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * free the avs structure
    */
   avs_free(resources->image);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: avs_order - return the AVS index order
|
|       Purpose: This function returns the index order in which
|                the avs_input returns the value data and in 
|                which the avs_output expects the value data.
|
|         Input: seg   - the data segment to get the order for
|                
|        Output: order - array with the expected index order
|                        for the requested segment
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 12, 1993 13:12
| Modifications:
|
------------------------------------------------------------*/

static int
avs_order(char *seg, int *ord)
{
   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_VALUE))
   {
      ord[0] = KWIDTH;
      ord[1] = KHEIGHT;
      ord[2] = KDEPTH;
      ord[3] = KTIME;
      ord[4] = KELEMENTS;
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: avs_architecture - Return the architecture of an AVS image
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.  AVS files
|                should always appear to in a KMACH_SPARC 
|                format.
|
|         Input: object - the data object to return the architecture for
|
|        Output: none
|
|       Returns: The architecture type.
|
|    Written By: Jeremy Worley and Steve Kubica
|          Date: Nov 08, 1993 11:35
| Modifications:
|
------------------------------------------------------------*/

static int
avs_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_SPARC);
   else
      return (kfile_getmachtype(kfileno(resources->file)));
}

#endif /* KAVS_DEF */
/* don`t add after the endif */
