/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for #object-name#
   >>>> 
   >>>>	Private: 
   >>>>                 kapu_multi_input_loc()
   >>>>                 kapu_multi_input_time()
   >>>>	Static: 
   >>>>	Public: 
   >>>>                 lkcall_free()
   >>>>                 kapu_mask_ops()
   >>>>                 kapu_upgrade_objects()
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/****************************************************************
*
*  Routine Name: lkcall_free() - go through a list of objects
*                and free them using the necesssary routines,
*                typically kfree() or kpds_close_object().
*
*       Purpose: Given an list of type *klist, step through
*                each entry and free memory resources
*                according the the following scheme: 1) if the
*                client_data for the list entry is "KMALLOC",
*                then call kfree() on the list identifier. If the client_data
*                for the list object is "KOBJECT", then call kpds_close_object()
*                on the list identifier.
*
*                By immediately adding items that allocate dynamic memory
*                resources to the klist, a single call to lkcall_free()
*                will return all of those resources in a systematic
*                and reliable manner. This routines is called, for example,
*                by the KCALL() macro, which relieves the programmer of the
*                need to check the return value of each kpds, kdms, or other
*                Data Services call and to clean up if a failure occurs.
*
*		 Items can be added to or removed from the object list by
*		 using the routines klist_add() and klist_delete() from the
*		 kutils(3) library.
*
*         Input: klist *objlist - klist of things to be free'd
*
*        Output: None.
*
*       Returns: TRUE if successful, FALSE if a failure occurs.
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson
*          Date: Nov 15, 1994
*      Verified:
*  Side Effects:
* Modifications:
****************************************************************/
int lkcall_free(klist *p)
  {
    char *lib="kapputils";
    char *rtn="lkcall_free";
 
    if (kstrcmp(p->client_data,"KOBJECT")==0)
      {
        if (!kpds_close_object((kobject)(p->identifier)))
          kerror(lib,rtn,"Unable to close object");
      }
    else if (kstrcmp(p->client_data,"KMALLOC")==0)
      kfree(p->identifier);
    else
        (void)kinfo(KSYSLIB,"lkcall_free: Unrecognized item in list: %s\n",
                             p->client_data);
    return(TRUE);
  }


/************************************************************
*
*  Routine Name: kapu_mask_ops - perform AND, OR, or XOR on data object masks
*
*       Purpose: This function combines the masks of the source
*		 objects according to the operation flag supplied.
*		 Valid operations are:
*		 !  KAPU_MASK_AND
*		 !  KAPU_MASK_OR
*		 !  KAPU_MASK_XOR
*
*		 If no source object has a mask, then kapu_mask_ops
*		 returns without modifying dest_obj.
*
*		 If only one source object has a mask, then its mask
*		 is copied to the destination object.
*
*		 If more than one source object has a mask, then the
*		 specified operation is performed on the source mask
*		 data, and stored in dest_obj.  dest_obj mask data,
*		 if it exists, is not included in the operation.
*
*		 Invalid objects (KOBJECT_INVALID) may be passed in.
*
*         Input: operation - defines operation to perform
*                num_src_objs - number of source objects provided in kvalist
*                kvalist - list of source objects (kobject)
*
*        Output: dest_obj - destination object where resulting mask is stored
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Jun  8, 1994
*      Verified:
*  Side Effects: Data may be written to mask of dest_object (mask may 
*		 be created if it does not already exist).
* Modifications:
*
*************************************************************/

int kapu_mask_ops(
   kobject  dest_obj,
   int	    operation,
   int	    num_src_objs,
   kvalist)
{
        kva_list        valist;
        kobject         *objlist;
	int		*mask_exists; 
	int		i, j, k, mask_count=0, error=FALSE;
	int		dw, dh, dd, dt, de;
	int		rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, num_rgns, num_pts;
	unsigned char	*data=NULL, *result=NULL;
	char		*prog="kapu_mask_ops";
	char		*lib="kdatamanip";

	/*
	 * Destination object must be valid.
	 */
	if (dest_obj == KOBJECT_INVALID)
	{
	   kerror(lib,prog,"destination object must be a valid object");
	   return(FALSE);
	}

        if ((objlist = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for objlist");
	   return(FALSE);
	}

        if ((mask_exists = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for mask_exists array");
	   kfree(objlist);
	   return(FALSE);
	}

        kva_start(valist, num_src_objs);

        for (i=0; i<num_src_objs; i++)
	{
	   objlist[i] = kva_arg(valist, kobject);
	   if (objlist[i] == KOBJECT_INVALID)
	      mask_exists[i] = FALSE;
	   else 
	   {
	      if ((mask_exists[i] = kpds_query_mask(objlist[i])) == TRUE)
	         mask_count ++;
	   }
	}

        kva_end(valist);

	/* 
	 *  If none of the objects contain a mask, do nothing and return.
	 */
	if (mask_count == 0)
	{
	   kfree(objlist);
	   kfree(mask_exists);
	   return(TRUE);
	}
		
	/* 
	 *  Get and set attributes of destination object.
	 */
	if (!kpds_query_mask(dest_obj))
	{
	   if (!kpds_create_mask(dest_obj))
	   {
	      kerror(lib,prog,"Unable to create destination object mask.");
	      kfree(objlist);
	      kfree(mask_exists);
	      return(FALSE);
	   }
	}
	if (!kpds_get_attribute(dest_obj, KPDS_MASK_SIZE, &dw,&dh,&dd,&dt,&de))
	{
	   kerror(lib,prog,"Unable to get destination object attributes.");
           kfree(objlist);
           kfree(mask_exists);
           return(FALSE);
	}

	/* 
	 *  If only one object contains a mask, simply copy it to the 
	 *  destination object and return.
	 */
	if (mask_count == 1)
	{
	   for (i=0; i<num_src_objs; i++)
	   {
	      if (mask_exists[i])
	      {
		 if ((objlist[i] = kpds_reference_object(objlist[i])) 
			== KOBJECT_INVALID)
		 {
	   	    kerror(lib,prog,"Unable to reference object %d", i);
	   	    error = TRUE;
		 }
	         if (!kpds_set_attribute(objlist[i], KPDS_MASK_SIZE, 
				dw,dh,dd,dt,de))
	         {
	            kerror(lib,prog,"Unable to set mask size of object %d.",i);
                    kfree(objlist);
                    kfree(mask_exists);
                    return(FALSE);
	         }
	         if (!kpds_copy_mask((objlist[i]), dest_obj, TRUE))
		    error = TRUE;

		 kpds_close_object(objlist[i]);
	         kfree(objlist);
	         kfree(mask_exists);
		 if (error)
		    return(FALSE);
		 else
	            return(TRUE);
	      }
	   }
	}

	if ((dest_obj = kpds_reference_object(dest_obj)) == KOBJECT_INVALID)
	{
	   kerror(lib,prog,"Unable to reference destination object");
	   error = TRUE;
	}

	if (!kpds_set_attribute(dest_obj, KPDS_MASK_DATA_TYPE, KUBYTE))
	{
	   kerror(lib,prog,"Unable to set destination object mask data type.");
	   error = TRUE;
	}
	if (!kpds_get_attributes(dest_obj, 
			KPDS_MASK_OPTIMAL_REGION_SIZE, 
			&rgn_w, &rgn_h, &rgn_d, &rgn_t, &rgn_e, &num_rgns, 
			NULL))
	{
	   kerror(lib,prog,"Unable to get destination object attributes.");
	   error = TRUE;
	}
	if (!kpds_set_attributes(dest_obj, 
		KPDS_MASK_REGION_SIZE, rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, 
		KPDS_MASK_POSITION, 0,0,0,0,0,
		KPDS_MASK_OFFSET, 0,0,0,0,0,
		NULL))
	{
	   kerror(lib,prog,"Unable to set destination object mask attributes.");
	   error = TRUE;
	}

	if (error)
	{
	   kpds_close_object(dest_obj);
	   kfree(objlist);
	   kfree(mask_exists);
	   return(FALSE);
	}

	/* 
	 *  If more than one mask exists, we will need set attributes and
	 *  get and put data.  Therefore, reference all objects that contain 
	 *  masks so that their attributes are not permanently affected.  
	 */
        for (i=0; i<num_src_objs; i++)
	{
	   if (mask_exists[i])
	   {
	      if ((objlist[i] = kpds_reference_object(objlist[i])) 
			== KOBJECT_INVALID)
	      {
	         kerror(lib,prog,"Unable to reference object %d.",i);
		 error = TRUE;
	      }
	      if (!kpds_set_attributes(objlist[i], 
			KPDS_MASK_SIZE, dw, dh, dd, dt, de,
			KPDS_MASK_REGION_SIZE, rgn_w,rgn_h,rgn_d,rgn_t,rgn_e, 
			KPDS_MASK_POSITION, 0,0,0,0,0,
			KPDS_MASK_OFFSET, 0,0,0,0,0,
			KPDS_MASK_DATA_TYPE, KUBYTE,
			NULL))
	      {
	         kerror(lib,prog,"Unable to set attributes for object %d.",i);
		 error = TRUE;
	      }

	      if (error)
	      {
	         kpds_close_object(dest_obj);
	         for (j=0; j<i; j++)
		 {
		    if (mask_exists[j])
		       kpds_close_object(objlist[j]);
	         }
		 kfree(mask_exists);
		 kfree(objlist);
	         return(FALSE);
	      }
	   }
	}
	
	num_pts = rgn_w * rgn_h * rgn_d * rgn_t * rgn_e;

	/*
	 *  Initialize a temporary results array, initial values are
	 *  based on operation being performed (AND - initialize to 1, 
	 *  OR - initialize to 0, XOR - initialize to 0).
	 */

        if ((result=(unsigned char *)kmalloc(num_pts * sizeof(unsigned char))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for result array.");
	   kpds_close_object(dest_obj);
	   for (j=0; j<i; j++)
	   {
	      if (mask_exists[j])
	         kpds_close_object(objlist[j]);
	   }
	   kfree(mask_exists);
	   kfree(objlist);
	   return(FALSE);
	}
	if (operation == KAPU_MASK_AND)
	{
	   for (i=0; i<num_pts; i++)
	      result[i] = 1;
	}
	else
	{
	   for (i=0; i<num_pts; i++)
	      result[i] = 0;
	}
	
	for (i=0; i<num_rgns; i++)
	{
	   for (j=0; j<num_src_objs; j++)
	   {
	      if (mask_exists[j])
	      {
	         if ((data = kpds_get_data(objlist[j], KPDS_MASK_REGION, data))
			   == NULL)
		 {
	            kerror(lib,prog,"Unable to get region from object %d.",j);
		    error = TRUE;
		    break;
	 	 }
	         if (error)  break;
		 if (operation == KAPU_MASK_AND)
		 {
		    for (k=0; k<num_pts; k++)
		       result[k] = result[k] & data[k]; 
		 }
		 else if (operation == KAPU_MASK_OR)
		 {
		    for (k=0; k<num_pts; k++)
		       result[k] = result[k] | data[k]; 
		 }
		 else if (operation == KAPU_MASK_XOR)
		 {
		    for (k=0; k<num_pts; k++)
		       result[k] = result[k] ^ data[k]; 
		 }
		 else 
		 {
	            kerror(lib,prog,"Unrecognized operation (%d).",operation);
		    error = TRUE;
		    break;
		 }
	      }
	      if (error)  break;
	   }

	   if (!kpds_put_data(dest_obj, KPDS_MASK_REGION, result))
	   {
	      kerror(lib,prog,"Unable to get region from object %d.",j);
	      error = TRUE;
	      break;
	   }
	   if (error)  break;
	}
	
	/* clean up */
	kpds_close_object(dest_obj);
	for (j=0; j<i; j++)
	{
	   if (mask_exists[j])
	      kpds_close_object(objlist[j]);
	}
	kfree(objlist);
	kfree(mask_exists);
	kfree(result);
	kfree(data);
	if (error)
	   return(FALSE);
	else
	   return(TRUE);
	
}


/*-----------------------------------------------------------
|
|  Routine Name: kapu_multi_input_loc - check object list for location data
|	
|       Purpose: This function checks whether location data exists in
|		 the source objects.  Currently, the function returns 
|		 TRUE if the first object has location, and FALSE if it 
|		 does not.
|
|		 The grid_type parameter is set to the grid type of the 
|		 first input object if it has location.
|
|		 The function will print a warning (KSYSLIB) if any
|		 of the other source objects have location, and it is
|		 being lost.
|
|		 kapu_multi_input_loc is under construction.  In the future, 
|		 as we start supporting location more fully, it will do more 
|		 than just check the first object and set the grid type to 
|		 that object's type.  For example, 
|		 the grid_type parameter will be set to the lowest common
|		 grid type of the input objects, there will be the option
|		 to set up location on all of the objects so that they are
|		 presented the same, and more information on which source 
|		 objects have location will be returned.  This will happen
|		 later. 
|
|         Input: num_src_objs - number of source objects provided in kvalist
|                kvalist - list of source objects (kobject)
|
|        Output: grid_type - processing grid type: set to KNONE, KUNIFORM, 
|			     KRECTILINEAR, or KCURVILINEAR
|
|       Returns: TRUE (1) if location exists in first source, FALSE (0) 
|		 otherwise
|
|  Restrictions: 
|    Written By: Donna Koechner
|          Date: Jan  3, 1995
|      Verified:
|  Side Effects: None
| Modifications:
|
------------------------------------------------------------*/

int kapu_multi_input_loc(
   int	    *grid_type,
   int	    num_src_objs,
   kvalist)
{
        kva_list        valist;
        kobject         *objlist;
	int		*grid; 
	int		*loc_exists; 
	int		i, loc_count=0;
	char		*prog="kapu_multi_input_loc";
	char		*lib="kdatamanip";

        if ((objlist = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for objlist");
	   return(FALSE);
	}

        if ((grid = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for grid array");
	   kfree(objlist);
	   return(FALSE);
	}

        if ((loc_exists = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for loc_exists array");
	   kfree(objlist);
	   kfree(grid);
	   return(FALSE);
	}

        kva_start(valist, num_src_objs);

        for (i=0; i<num_src_objs; i++)
	{
	   objlist[i] = kva_arg(valist, kobject);
	   grid[i] = KNONE;
	   if (objlist[i] != KOBJECT_INVALID)
	   {
	      if ((loc_exists[i] = kpds_query_location(objlist[i])) == TRUE)
	      {
	         loc_count ++;
		 if (!kpds_get_attribute(objlist[i], 
				KPDS_LOCATION_GRID, &grid[i]))
		 {
                    kerror(lib,prog,
			   "Unable to get location grid type for object %d",i);
                    kfree(objlist); kfree(grid); kfree(loc_exists);
                    return(FALSE);
		 }
	      }
	   }
	}
        kva_end(valist);

        kfree(objlist);

        *grid_type = KNONE;

	if (loc_count > 0)
	{
	   if (loc_exists[0] && (loc_count == 1))
	      *grid_type = grid[0];
	   else
	   {
	      kinfo(KSYSLIB, "Only Location data from the first object will "
			      "be propagated. \nLocation that exists in other "
			      "source objects will be lost. \n"
		     	      "(%d out of %d have location)", 
				loc_count, num_src_objs);
	      *grid_type = grid[0];
	   }
	}
	else
	{
           kfree(loc_exists);
           kfree(grid);
	   return(FALSE);
	}
	
        kfree(loc_exists);
        kfree(grid);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: kapu_multi_input_time - check object list for time data
|	
|       Purpose: This function checks whether time data exists in
|		 the source objects.  Currently, the function returns 
|		 TRUE if the first object has time, and FALSE if it 
|		 does not.
|
|		 The function will print a warning (KSYSLIB) if any
|		 of the other source objects have time, and it is
|		 will potentially be lost.
|
|		 kapu_multi_input_time is under construction.  In the future, 
|		 as we start supporting time more fully, it will do more 
|		 than just check the first object.  For example a grid type 
|		 will be added when data services supports it.
|		 The grid_type parameter will be set to the lowest common
|		 grid type of the input objects, there will be the option
|		 to set up time on all of the objects so that they are
|		 presented the same, and more information on which source 
|		 objects have time will be returned.  This will happen
|		 later. 
|
|         Input: num_src_objs - number of source objects provided in kvalist
|                kvalist - list of source objects (kobject)
|
|       Returns: TRUE (1) if time exists in first source, FALSE (0) otherwise
|  Restrictions: 
|    Written By: Donna Koechner
|          Date: Jan  3, 1995
|      Verified:
|  Side Effects: None
| Modifications:
|
------------------------------------------------------------*/

int kapu_multi_input_time(
   int	    num_src_objs,
   kvalist)
{
        kva_list        valist;
        kobject         *objlist;
	int		*time_exists; 
	int		i, count=0;
	int		result;
	char		*prog="kapu_multi_input_time";
	char		*lib="kdatamanip";

        if ((objlist = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for objlist");
	   return(FALSE);
	}

        if ((time_exists = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for time_exists array");
	   kfree(objlist);
	   return(FALSE);
	}

        kva_start(valist, num_src_objs);

        for (i=0; i<num_src_objs; i++)
	{
	   objlist[i] = kva_arg(valist, kobject);
	   if (objlist[i] != KOBJECT_INVALID)
	   {
	      if ((time_exists[i] = kpds_query_time(objlist[i])) == TRUE)
	         count ++;
	   }
	}
        kva_end(valist);

        kfree(objlist);

	result = FALSE;
	if (count > 0)
	{
	   if (time_exists[0] && (count == 1))
	      result = TRUE;
	   else
	   {
	      kinfo(KSYSLIB, "Only Time data from the first object will "
			      "be propagated. \nTime that exists in other "
			      "source objects will be lost. \n"
		     	      "(%d out of %d have time)", count, num_src_objs);
	      result = time_exists[0];
	   }
	}
	
        kfree(time_exists);
	return(result);
}


/************************************************************
*
*  Routine Name: kapu_upgrade_objects - check objects for value, map, and mask data, and set up for polymorphic processing
*	
*       Purpose: This function checks the source objects for map, 
*		 value, and mask data, and it sets the processing 
*		 flags and type parameters passed in to indicate how the
*		 objects would best be processed as polymorphic objects.  
*		 If the upgrade flags are TRUE, this function will also 
*		 set up the objects for the recommended processing. 
*
*		 .IP "Upgrade Parameters:" 5
*		 The upgrade parameters are applied in this order - mapping
*		 first, then type, then size.
*		 .RS
*		 .IP "upgrade_mapping"
*		 If the upgrade_mapping parameter is true, and the 
*		 recommended processing is not process_map, then any 
*		 object that has map data will be modified so that the
*		 KPDS_MAPPING_MODE is KMAPPED.
*		 .IP "upgrade_type" 
*		 If the upgrade_type flag is TRUE, then all segments 
*		 that exist in the data objects will be upgraded to 
*		 the highest common data type for that segment.  For
*		 example, if the value data in source 1 is unsigned 
*		 byte, and the data type in source 2 it is float, source 
*		 1's value data will be upgraded to float.  If mapping 
*		 has been upgraded, the type upgrading will not be applied 
*		 to map data.
*		 .IP "upgrade_size"
*		 If the upgrade_size flag is TRUE, then all segments that
*		 exist in the data objects will be upgraded to the highest
*		 size of each dimension, for each segment.  Again, if
*		 mapping has already been upgraded, size upgrading will 
*		 not be applied to map data.
*		 .RE
*
*		 .IP "Subobject Position Flag" 5
*		 If the use_subobj_pos flag is TRUE, when size is calculated
*		 the subobject position attribute for that object will be 
*		 added before its maximum size is determined.  The subobject
*		 position is not applied to the first source object, 
*		 because the standard way of using subobject position for
*		 multi-input polymorphic processing is to use the first
*		 object as the base object.  If you want the subobject
*		 position to be applied to all objects, simply pass in
*		 NULL as the first object and add 1 to the num_src_objs
*		 parameter.
*		 
*		 If invalid objects are passed in, this function will not
*		 fail, but will pass over those objects.  Invalid objects
*		 will not be taken into consideration.
*
*         Input: 
*		 upgrade_mapping - if TRUE, upgrade mapping mode is necessary
*		 upgrade_type - if TRUE, upgrade map, value, mask data types
*		 upgrade_size - if TRUE, upgrade map, value, mask data size
*		 use_subobj_pos - if TRUE, use subobject position in size 
*				  calculation
*		 num_src_objs - number of source objects provided in kvalist
*                kvalist - list of source objects (kobject)
*
*        Output: 
*		 process_map - set to TRUE when map processing is recommended
*		 process_value - set to TRUE when value processing recommended
*		 process_mask - set to TRUE when mask processing recommended
*		 map_type - recommended processing type for map data
*		 value_type - recommended processing type for value data
*		 mask_type - recommended processing type for mask data
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Feb 10, 1995
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

int kapu_upgrade_objects(
   int	    *process_map,
   int	    *process_value,
   int	    *process_mask,
   int	    *map_type,
   int	    *value_type,
   int	    *mask_type,
   int	    upgrade_mapping,
   int	    upgrade_type,
   int	    upgrade_size,
   int	    use_subobj_pos,
   int	    num_src_objs,
   kvalist)
{
        klist  		*objlist=NULL;
	char		*rtn="kapu_upgrade_objects", *lib="kapputils";
	int		*has_value, *has_mask, *has_map;
	int		*mw, *mh, *md, *mt, *me;
	int		*vw, *vh, *vd, *vt, *ve;
	int		*woff=0, *hoff=0, *doff=0, *toff=0, *eoff=0;
	int		map_w=0, map_h=0, map_d=0, map_t=0, map_e=0;
	int		value_w=0, value_h=0, value_d=0, value_t=0, value_e=0;
        kobject         *src_list;
        kva_list	valist;
	int		valid_objects, i;
	int		map_count=0, mask_count=0, value_count=0;
	int		maptype=KBIT,masktype=KBIT,valuetype=KBIT,type=KBIT;

        /*----------------------------------------------------------------
        | Allocate arrays
        ----------------------------------------------------------------*/
        if ((src_list = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for src_list");
	   return(FALSE);
	}
        objlist = klist_add(objlist,src_list,"KMALLOC");

        if ((has_map = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for has_map array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,has_map,"KMALLOC");

        if ((has_mask = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for has_mask array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,has_mask,"KMALLOC");

        if ((has_value = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for has_value array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,has_value,"KMALLOC");

        if ((mw = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for mw array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,mw,"KMALLOC");

        if ((mh = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for mh array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,mh,"KMALLOC");

        if ((md = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for md array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,md,"KMALLOC");

        if ((mt = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for mt array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,mt,"KMALLOC");

        if ((me = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for me array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,me,"KMALLOC");

        if ((vw = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for vw array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,vw,"KMALLOC");

        if ((vh = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for vh array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,vh,"KMALLOC");

        if ((vd = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for vd array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,vd,"KMALLOC");

        if ((vt = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for vt array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,vt,"KMALLOC");

        if ((ve = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,rtn,"Unable to allocate memory for ve array");
	   (void)klist_free(objlist, (kfunc_void)lkcall_free);
	   return(FALSE);
	}
        objlist = klist_add(objlist,ve,"KMALLOC");

	if (use_subobj_pos)
	{
           if ((woff = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	   {
	      kerror(lib,rtn,"Unable to allocate memory for woff array");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
           objlist = klist_add(objlist,woff,"KMALLOC");

           if ((hoff = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	   {
	      kerror(lib,rtn,"Unable to allocate memory for hoff array");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
           objlist = klist_add(objlist,hoff,"KMALLOC");

           if ((doff = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	   {
	      kerror(lib,rtn,"Unable to allocate memory for doff array");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
           objlist = klist_add(objlist,doff,"KMALLOC");

           if ((toff = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	   {
	      kerror(lib,rtn,"Unable to allocate memory for toff array");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
           objlist = klist_add(objlist,toff,"KMALLOC");

           if ((eoff = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	   {
	      kerror(lib,rtn,"Unable to allocate memory for eoff array");
	      (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
           objlist = klist_add(objlist,eoff,"KMALLOC");
	}

        /*----------------------------------------------------------------
        | Get map, mask, and value information about each source object.
        ----------------------------------------------------------------*/
        kva_start(valist, num_src_objs);

	valid_objects = num_src_objs;
        for (i=0; i<num_src_objs; i++)
	{
	   src_list[i] = kva_arg(valist, kobject);
	   if (src_list[i] == KOBJECT_INVALID)
	   {
	      has_map[i] = FALSE;
	      has_mask[i] = FALSE;
	      has_value[i] = FALSE;
	      valid_objects --;
	   }
	   else 
	   {
	      if ((has_map[i] = kpds_query_map(src_list[i])) == TRUE)
	      {
		 KCALL(kpds_get_attribute(src_list[i],
				KPDS_MAP_DATA_TYPE, &type));
		 maptype = kdatatype_cast_output(maptype, type);
		 KCALL(kpds_get_attribute(src_list[i],
			KPDS_MAP_SIZE, &mw[i],&mh[i],&md[i],&mt[i],&me[i]));
		 map_w = kmax(map_w, mw[i]);
		 map_h = kmax(map_h, mh[i]);
		 map_d = kmax(map_d, md[i]);
		 map_t = kmax(map_t, mt[i]);
		 map_e = kmax(map_e, me[i]);
	         map_count ++;
	      }
	      if ((has_mask[i] = kpds_query_mask(src_list[i])) == TRUE)
	      {
		 KCALL(kpds_get_attribute(src_list[i],
				KPDS_MASK_DATA_TYPE, &type));
		 masktype = kdatatype_cast_output(masktype, type);
	         mask_count ++;
	      }
	      if ((has_value[i] = kpds_query_value(src_list[i])) == TRUE)
	      {
		 KCALL(kpds_get_attribute(src_list[i],
				KPDS_VALUE_DATA_TYPE, &type));
		 valuetype = kdatatype_cast_output(valuetype, type);
		 KCALL(kpds_get_attribute(src_list[i],
			KPDS_VALUE_SIZE, &vw[i],&vh[i],&vd[i],&vt[i],&ve[i]));
		 if (use_subobj_pos)
		 {
		    KCALL(kpds_get_attribute(src_list[i], 
				KPDS_SUBOBJECT_POSITION, 
				&woff[i],&hoff[i],&doff[i],&toff[i],&eoff[i]));
		    if (i != 0)
		    {
		       vw[i] += kabs(woff[i]);
		       vh[i] += kabs(hoff[i]);
		       vd[i] += kabs(doff[i]);
		       vt[i] += kabs(toff[i]);
		       ve[i] += kabs(eoff[i]);
		    }
		 }
		 value_w = kmax(value_w, vw[i]);
		 value_h = kmax(value_h, vh[i]);
		 value_d = kmax(value_d, vd[i]);
		 value_t = kmax(value_t, vt[i]);
		 value_e = kmax(value_e, ve[i]);
	         value_count ++;
	      }
	   }
	}

        kva_end(valist);

	if (map_count == 0)
	   *process_map = FALSE;
	if (mask_count == 0)
	   *process_mask = FALSE;
	if (value_count == 0)
	{
	   *process_value = FALSE;
	   *process_mask = FALSE;
	}

        /*----------------------------------------------------------------
	| If both map and value exist, then any object that has map must
	| also have value.  
        ----------------------------------------------------------------*/
	if ((valid_objects == 1) && (map_count > 0))
	{
	   *process_map = TRUE;
           *process_value = FALSE;
	   if (mask_count > 0)
             *process_mask = TRUE;
	}
	else if ((value_count > 0) && (map_count > 0))
	{
	   for (i=0; i<num_src_objs; i++)
	   {
	      if (has_map[i])
	      {
	         if (!has_value[i])
		 {
		    kerror(lib, rtn, "Object %d must have value data.",i);
	   	    (void)klist_free(objlist, (kfunc_void)lkcall_free);
		 }
	      }
	   }
	   *process_map = FALSE;
	   *process_value = TRUE;
	   if (mask_count > 0)
	      *process_mask = TRUE;
	}
        else if ((value_count > 0) && (map_count == 0))
	{
	   *process_map = FALSE;
	   *process_value = TRUE;
	   if (mask_count > 0)
	      *process_mask = TRUE;
	}
        else if ((value_count == 0) && (map_count > 0))
	{
	   *process_map = TRUE;
	   *process_value = FALSE;
	   *process_mask = FALSE;
	}

        /*----------------------------------------------------------------
        | If the flag is true, set the objects up for 
	| processing.  
	| First modify type, then mapping (may need to redo size indices), 
	| then size
        ----------------------------------------------------------------*/

	if (upgrade_mapping && (map_count > 0) && (value_count > 0))
	{
           kva_start(valist, num_src_objs);

           for (i=0; i<num_src_objs; i++)
	   {
	      src_list[i] = kva_arg(valist, kobject);
	      if (src_list[i] != KOBJECT_INVALID)
	      {
                 /*-------------------------------------------------------
		 | Set mapping mode on all objects with maps, set map 
		 | count to zero.  determine greater of map/value type.
                 -------------------------------------------------------*/
	         if (has_map[i])
		 {
		    KCALL(kpds_set_attribute(src_list[i],
					KPDS_MAPPING_MODE, KMAPPED));
		    map_count --;
		    has_map[i] = FALSE;
		    KCALL(kpds_get_attribute(src_list[i], KPDS_VALUE_SIZE, 
				&vw[i], &vh[i], &vd[i], &vt[i], &ve[i]));
		    if (use_subobj_pos)
		    {
		       if (i != 0)
		       {
		          vw[i] += kabs(woff[i]);
		          vh[i] += kabs(hoff[i]);
		          vd[i] += kabs(doff[i]);
		          vt[i] += kabs(toff[i]);
		          ve[i] += kabs(eoff[i]);
		       }
		    }
		    value_w = kmax(value_w, vw[i]);
		    value_h = kmax(value_h, vh[i]);
		    value_d = kmax(value_d, vd[i]);
		    value_t = kmax(value_t, vt[i]);
		    value_e = kmax(value_e, ve[i]);
		 }
	   	 valuetype = kdatatype_cast_output(valuetype, maptype);
	      }
	   }
           kva_end(valist);
	}

	if (upgrade_type || upgrade_size)
	{
           kva_start(valist, num_src_objs);

           for (i=0; i<num_src_objs; i++)
	   {
	      src_list[i] = kva_arg(valist, kobject);
	      if (src_list[i] != KOBJECT_INVALID)
	      {
		 if (upgrade_type)
		 {
                    /*----------------------------------------------------
		    | If count of segment > 0, set type on all objects 
                    ----------------------------------------------------*/
		    if (map_count > 0)
		    {
	               if (has_map[i])
		       {
		          KCALL(kpds_set_attribute(src_list[i],
					KPDS_MAP_DATA_TYPE, maptype));
		       }
		    }
		    if (value_count > 0)
		    {
	               if (has_value[i])
		       {
		          KCALL(kpds_set_attribute(src_list[i],
					KPDS_VALUE_DATA_TYPE, valuetype));
		       }
		    }
		    if (mask_count > 0)
		    {
	               if (has_mask[i])
		       {
		          KCALL(kpds_set_attribute(src_list[i],
					KPDS_MASK_DATA_TYPE, masktype));
		       }
		    }
		 }

		 if (upgrade_size)
		 {
                    /*----------------------------------------------------
		    | If upgrade_size, match size of value and or map 
		    | segments if their count is greater than 1.  don't 
		    | operate on mask 
                    ----------------------------------------------------*/
		    if (map_count > 0)
		    {
		       if (has_map[i])
		       {
		          KCALL(kpds_set_attribute(src_list[i], KPDS_MAP_SIZE, 
				map_w, map_h, map_d, map_t, map_e));
		       }
		    }
		    if (value_count > 0)
		    {
		       if (has_value[i])
		       {
		          KCALL(kpds_set_attribute(src_list[i],KPDS_VALUE_SIZE, 
				value_w, value_h, value_d, value_t, value_e));
		       }
		    }
		 }
	      }
	   }
           kva_end(valist);
	}

	*map_type = maptype;
	*value_type = valuetype;
	*mask_type = masktype;

	(void)klist_free(objlist, (kfunc_void)lkcall_free);
	return(TRUE);
}
