 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kcpfromval
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkcpfromval
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkcpfromval - copy data from value into specified segment
* 
*       Purpose: This routine copies the five dimensional value segment
*                from the source_object to segment specified by
*                segment_name, of the destination_object. The data
*                will be copied into the lowest order dimension of the
*                destination segment, then the next order will be
*                used, and the next, and so forth, until all the
*                value data is copied.  If the dimensionality of the
*                destination is less than five, then the last dimension
*                of the destination segment will have all the higher
*                order dimensions rolled into it.
*
* 		 If segment_name is one of the polymorphic segments
*		 (KDMS_SEGMENT_VALUE, KDMS_SEGMENT_MASK,
*		 KDMS_SEGMENT_MAP, * KDMS_SEGMENT_TIME or
*		 KDMS_SEGMENT_LOCATION) than the * segment
*		 dimensionalities and index orders specified by the
*		 polymorphic model will be used. If the segment is not
*		 one of the polymorphic segments, then the
*		 dimensionality and index order of the destination
*		 segment must be specified.
*
*                Only the size and data type attribute will be changed, all
*                others will not be transferred to the destination segment.
*                If the segment does not exist in the destination_object
*                and is not a polymorphic segment the dimension and
*                index order will be set.  If it is a polymorphic segment
*                the default polymorphic dimension and index order will be
*                used.
*
*         Input: source_object - object to copy the value segment from
*                segment_name  - segment name to copy to
*                dimension     - dimension of segment (not yet supported)
*                index_order   - index order of segment (not yet supported)
*
*        Output: destination_object - object to put the segment into
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Results guaranteed only if dimension of destination
*		 segment is 5.  Other dimensions are not tested and 
*		 may produce poor results.
*    Written By: John M. Salas & Donna Koechner
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int 
lkcpfromval(
   kobject source_object,
   char    *segment_name,
   int     dimension,
   int     *index_order,
   kobject destination_object)
/* -library_def_end */

/* -library_code */
{
   klist *objlist=NULL;
   char  *lib="ksegops", *rtn="lkcpfromval";
   int   i;
   int   dim;
   int   ndim;
   int   size;
   int   data_type;
   int  *size_vect;
   int  *nsize_vect;
   int  *begin;
   int  *end;
   int  *nbegin;
   int  *nend;
   int  *copy_size;
   int   value_data = FALSE;
   int   mask_data = FALSE;
   int   map_data = FALSE;
   int   time_data = FALSE;
   int   location_data = FALSE;
   kaddr data;

   /*
    * See if the segment exists in the source object.
    */
   if (!kdms_query_segment(source_object, KDMS_SEGMENT_VALUE))
   {
      kerror("ksegops", "lkcpfromval", "Segment %s does not exist \
in source object.", segment_name);
      return (FALSE);
   }

   /*
    * Determine if this is a polymorphic segment.
    */
   if (kstrcmp(segment_name, KDMS_SEGMENT_VALUE) == 0)
   {
      value_data = TRUE;
   }
   else if (kstrcmp(segment_name, KDMS_SEGMENT_MASK) == 0)
   {
      mask_data = TRUE;
   }
   else if (kstrcmp(segment_name, KDMS_SEGMENT_MAP) == 0)
   {
      map_data = TRUE;
   }
   else if (kstrcmp(segment_name, KDMS_SEGMENT_TIME) == 0)
   {
      time_data = TRUE;
   }
   else if (kstrcmp(segment_name, KDMS_SEGMENT_LOCATION) == 0)
   {
      location_data = TRUE;
   }

   /*
    * Get the dimension, size and data type of the value segment.
    */
   KCALL(kdms_get_attributes(source_object, KDMS_SEGMENT_VALUE,
		       KDMS_SIZE,      &size_vect,
		       KDMS_DATA_TYPE, &data_type,
		       KDMS_DIMENSION, &dim,
		       NULL));

   /*
    * If the segment does not exist create it.  If it is a polymorphic
    * segment use kpds_create_XXX() so that the dimension and index order
    * are set correctly.  If it is not we need those piece of information
    * to create the segment using the given information.
    */
   if (!kdms_query_segment(destination_object, segment_name))
   {
      if (value_data)
      {
	 KCALL(kpds_create_value(destination_object));
	 KCALL(kdms_get_attribute(destination_object, segment_name,
			    KDMS_DIMENSION, &ndim));
      }
      else if (mask_data)
      {
	 KCALL(kpds_create_mask(destination_object));
	 KCALL(kdms_get_attribute(destination_object, segment_name,
			    KDMS_DIMENSION, &ndim));
      }
      else if (map_data)
      {
	 KCALL(kpds_create_map(destination_object));
	 KCALL(kdms_get_attribute(destination_object, segment_name,
			    KDMS_DIMENSION, &ndim));
      }
      else if (time_data)
      {
	 KCALL(kpds_create_time(destination_object));
	 KCALL(kdms_get_attribute(destination_object, segment_name,
			    KDMS_DIMENSION, &ndim));
      }
      else if (location_data)
      {
	 KCALL(kpds_create_location(destination_object));
	 KCALL(kdms_get_attribute(destination_object, segment_name,
			    KDMS_DIMENSION, &ndim));
      }
      else
      {
	 KCALL(kdms_create_segment(destination_object, segment_name));
	 KCALL(kdms_set_attributes(destination_object, segment_name,
			     KDMS_DIMENSION,   dimension,
			     KDMS_INDEX_ORDER, index_order,
			     NULL));
	 ndim = dimension;
      }
   }

   /*
    * If the segment does exist, get the dimension and size.
    */
   else
   {
      KCALL(kdms_get_attributes(destination_object, segment_name,
			  KDMS_DIMENSION, &ndim,
			  NULL));
   }

   copy_size = (int *)kmalloc(dim * sizeof(int));
   objlist = klist_add(objlist,copy_size,"KMALLOC");
   nsize_vect = (int *)kmalloc(kmax(ndim, 5) * sizeof(int));
   objlist = klist_add(objlist,nsize_vect,"KMALLOC");

   if (!copy_size || !nsize_vect)
   {
      kerror("ksegops", "lkcpfromval", 
	     "Could not allocate memory for temporary variable.");
      return (FALSE);
   }

   /*
    * Determine the dimension of the segment.  
    * If it is greater than 5 then return (JMAS).  
    */
   if (ndim > 5)
   {
      kerror("ksegops", "lkcpfromval", 
	     "Data Services does not yet support dimensions greater than "
	     "five.  Sorry for the inconvenience.");
      return (FALSE);
   }


   /*
    * Adjust the size vector.  If the dimension of the destination segment
    * is < the value segment roll the higher order dimensions into the
    * highest dimension of the destination segment.  But, first copy the
    * the size vector because it is the value segments actual size data
    * and we do not want to change it.
    */
   for (i = 0; i < dim; i++)
   {
      copy_size[i] = size_vect[i];
   }

   if (ndim < dim)
   {
      for (i = 0; i < ndim; i++)
      {
	 nsize_vect[i] = copy_size[i];
      }

      for (i = ndim; i < 5; i++)
      {
	 nsize_vect[i] = 1;
      }

      for (i = ndim, size = copy_size[ndim - 1]; i < dim; i++)
      {
	 size *= copy_size[i];
      }
      nsize_vect[ndim - 1] = size;
   }
   /*
    * If the dimension of the destination segment is > the value segment
    * copy the lower order dimensions and set the higher ones to 1.
    */
   else if (ndim > dim)
   {
      for (i = 0; i < dim; i++)
      {
	 nsize_vect[i] = copy_size[i];
      }

      for (i = dim; i < ndim; i++)
      {
	 nsize_vect[i] = 1;
      }
   }
   else
   {
      for (i = 0; i < dim; i++)
      {
	 nsize_vect[i] = copy_size[i];
      }
   }

   /*
    * Set the data type and size of the segment.
    */
   KCALL(kdms_set_attribute(destination_object, segment_name,
		       KDMS_SIZE, nsize_vect));
   KCALL(kdms_set_attribute(destination_object, segment_name,
		       KDMS_DATA_TYPE, data_type));

   /*
    * Copy the data from the value to the destination segment.  If the
    * dimensions are the same just do a kdms_copy_data().
    */
   if (ndim == dim &&
       !kdms_copy_segment_data(source_object, KDMS_SEGMENT_VALUE,
			       destination_object, segment_name))
   {
      kerror("ksegops", "lkcpfromval", "Failed to copy value segment \
data into %s segment.", segment_name);
      return (FALSE);
   }
   /*
    * If not, copy the data with kdms_get_data() and then put it with
    * kdms_put_data().  First figure at the begin and end region first.
    */
   else if (ndim != dim)
   {
      begin = (int *)kmalloc(dim * sizeof(int *));
      objlist = klist_add(objlist,begin,"KMALLOC");
      end = (int *)kmalloc(dim * sizeof(int *));
      objlist = klist_add(objlist,end,"KMALLOC");
      nbegin = (int *)kmalloc(kmax(ndim, 5) * sizeof(int *));
      objlist = klist_add(objlist,nbegin,"KMALLOC");
      nend = (int *)kmalloc(kmax(ndim, 5) * sizeof(int *));
      objlist = klist_add(objlist,nend,"KMALLOC");

      if (!begin || !end || !nbegin || !nend)
      {
	 kerror("ksegops", "lkcpfromval", "Could not allocate memory \
for temporary variable.");
	 return (FALSE);
      }

      for (i = 0; i < dim; i++)
      {
	 begin[i] = 0;
	 end[i] = size_vect[i] - 1;
      }

      for (i = 0; i < kmax(ndim, 5); i++)
      {
	 nbegin[i] = 0;
	 nend[i] = nsize_vect[i] - 1;
      }

      data = kdms_get_data(source_object, KDMS_SEGMENT_VALUE,
			   begin, end, NULL);
      kdms_put_data(destination_object, segment_name,
		    nbegin, nend, data);
   }

   (void)klist_free(objlist, (kfunc_void)lkcall_free);
   return (TRUE);
}
/* -library_code_end */
