 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kstats
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkstats
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkstats - calculate statistics of data object
* 
*       Purpose: Library Routine for kstats.  For the input src_obj, 
*		 lkstats calculates the set of statistics defined by the 
*                flags set in the active_stats variable.  The results are
*		 stored in the stats_obj data object.  The active_stats
*		 flag can be built by or'ing the statistics flags defined 
*		 in include/kdatamanip/kstats.h.  The aux_info variable
*		 defines additional, non-statistical information, such
*		 as object data dimensions and position of the minimum
*		 and maximum values, that may also be recorded in 
*		 stats_obj.  The aux_info flag can be built by or'ing the 
*		 auxiliary information flags defined 
*		 in include/kdatamanip/kstats.h.
*
*		 Data with corresponding mask or gate values of zero (gating 
*		 defined by gate_obj) are not included in any statistics 
*		 calculations.  If the gating object is different sized 
*		 than the source object, it will be resized to match the 
*		 source object, padded with 1.
*
*		 If the input data object contains a map, the data will
*		 be mapped before the statistics are calcualted.  All
*		 output information will reflect this mapping. 
*
*                The following statistics are available:
*                !
*                !  mean = 1/N*sum(x(i)) i=1..N
*                !  var  = 1/(N-1)*sum( (x(i) - mean)**2 ) i=1..N 
*                !  stddev = sqrt(var) i=1..N 
*                !  rms  = sqrt(1/N*sum(x(i)**2)) i=1..N 
*                !  skew = 1/N*sum( ((x(i) - mean)/stddev) **3 ) i=1..N 
*                !  kurtosis=(1/N*sum( ((x(i) - mean)/stddev) **4)) - 3 i=1..N 
*                !  minimum value
*                !  maximum value
*                !  minimum value coordinates
*                !  maximum value coordinates
*                !  sum of all points
*                !  sum of all positive points
*                !  sum of all negative points
*                !  number of contributing points
*                !  number of positive contributing points
*                !  number of negative contributing points
*                !  number of zero-valued contributing points
*                !  dimensions of data
*
*		 Statistics are stored in the stats_obj as double float.
*		 Each statistic is stored as an element of a N-D vector
*		 defined along the value data "elements" dimension of the 
*		 stats_obj, where N is the number of statistics defined 
*		 by the active_stats variable.
*
*                The order in which the elements of the statistics vector 
*		 are stored in the statistics data object is given in the
*                object's comment attribute.
*
*		 When M sets of statistics are calcualted for M multiple 
*		 regions of the input data object (defined by the flags
*		 set in the stats_region variable), the statistics vectors 
*		 are stored along the width dimension (width=M, elements=N).  
*		 
*		 The stats_region variable defines how the data object is 
*                sliced along the dimensions for calculation of statistics 
*                over sub-regions. For example, if the width and height 
*                flags are set in the "stats_region" variable, the statistics
*		 for each width-height plane will be calculated separately.
*                For an image object this can be viewed as calculating the 
*                statistics across the various bands of the image.  The 
*		 stats_region flag can be built by or'ing the region 
*                dimensionality flags defined in include/kdatamanip/kstats.h.
*
*                While computing variance, the division factor is set to
*                (N-1) in order to obtain a more accurate estimate.
*                For a more detailed explanation refer to any standard
*                statistics text.
*
*         Input: src_obj - source data object from which stats are calculated.
*		 gate_obj - gating data object
*		 stats_obj - data object in which stat results are stored
*                active_stats - the bits of this flag indicate whether a 
*			particular statistic is to be calculated.
*                aux_info - the bits of this flag indicate whether a 
*			particular piece of auxiliary information is 
*			to be recorded in the stats_obj.
*                stats_region - the bits of this flag specify which dimensions
*			make up each region from which statistics are to 
*			be computed.
*		 ascii_flag - if this flag is TRUE, output statistics
*			in formatted ASCII to ascii_file.
*
*        Output: stats_obj - destination object for statistics results
*        	 ascii_file - output file for printing ASCII data.
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Ashish Malhotra & Donna Koechner
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkstats (
   kobject src_obj,
   kobject gate_obj,
   kobject stats_obj,
   unsigned long  active_stats,
   unsigned long  aux_info,
   unsigned long  stats_region,
   kfile          *ascii_file,
   int		  ascii_flag)
/* -library_def_end */

/* -library_code */
{
     double mean, var, std_dev, rms, integ, pos_integ, neg_integ;
     double max_val, min_val, skew, kurtosis;
     int    maxw, maxh, maxd, maxt, maxe;
     int    num_pts, pos_pts, neg_pts, zpts;
     int    minw, minh, mind, mint, mine;
     int    w_orig=0, h_orig=0, d_orig=0, t_orig=0, e_orig=0;
     int    first_point;
     unsigned long temp;

     int    w1, d1, h1, t1, e1, type1, num_subrgns=1, num_stats=0;
     long   num_statrgns=1,num_elems=1;
     int    cnt1, cnt2;
     int    i, j, k, i_end;
     int    reg_w, reg_h, reg_d, reg_t, reg_e;
     char   *comment = NULL;
     char   *str_ptr = NULL;
     char   str_array[KLENGTH] ;
     char   *lib = "kdatamanip";
     char   *prog = "lkstats";
     double *data = NULL;
     double *gate_data = NULL;
     double *mask_data = NULL;
     double *stat_data = NULL;
     double tmp_double=0;

     /* If no information is requested, print a warning and return...
      */

     if (active_stats == 0 && aux_info == 0)
     {
        kinfo(KSTANDARD, "lkstats:  active_stats and aux_info flags indicate that no statistics are requested, therefore no action taken.");
        return(TRUE);
     }

     /* Reference the input object and map the value data if a map exists.  
      * Check the datatype, and return if the data is complex.  Set the 
      * presentation type to double. Obtain its dimension.
      */

     if ( (src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID) 
     {
        kerror(lib, prog, "Unable to reference source data object.");
        return(FALSE);
     }

     if (kpds_query_map(src_obj)) 
     {
        if (!kpds_set_attribute(src_obj, KPDS_MAPPING_MODE, KMAPPED) )
	{
           kerror(lib, prog, "Unable to map source object data.");
	   (void) kpds_close_object(src_obj);
           return(FALSE);
	}
     }

     kpds_get_attribute(src_obj, KPDS_VALUE_DATA_TYPE, &type1);
     if ((type1 == KCOMPLEX) || (type1 == KDCOMPLEX))
     {
        kerror(lib, prog, "Cannot compute complex statistics at this time.\n"
		"If you wish to process the real and imagary components\n"
		"independently, separate components using kcmplx2real.");
	(void) kpds_close_object(src_obj);
        return(FALSE);
     }
     type1 = KDOUBLE;
     if (! kpds_set_attribute(src_obj, KPDS_VALUE_DATA_TYPE, type1))
     {
        kerror(lib, prog, "Unable to set source object value data type.");
        (void) kpds_close_object(src_obj);
        return(FALSE);
     }
     
     if (!kpds_get_attribute(src_obj, KPDS_VALUE_SIZE, &w1,&h1,&d1,&t1,&e1))
     {
        kerror(lib, prog, "Unable to get source object value data size.");
        (void) kpds_close_object(src_obj);
        return(FALSE);
     }

    /* The stats_region variable contains a set of dimension flags 
     * which define the regions over which the statistics will be 
     * calculated independently.  For example, if the width and
     * height flags are set, a separate set of statistics will
     * be calculated for each width-height plane in the object.
     *
     * To speed up processing, we want to get an optimal sized chunk 
     * of data (use the optimal region size attribute).  Since kstats 
     * needs to place restrictions on the "optimal" region definition,
     * we first need to check which dimensions must be processed separately, 
     * and temporarily set the size of those dimensions to 1 before 
     * getting the optimal region size.  After getting this size, reset 
     * the dimensions of the input data.  The query for optional region
     * size will also return num_subrgns, which will be the number of
     * sub-regions per statistics region.
     *
     * num_elems will be the number of elements in each sub-region,
     * and num_statrgns contains the number of statistics regions to be 
     * fetched from the object.
     */

     num_statrgns *= (stats_region & KSTATS_WIDTH)   ?  1 : w1;
     reg_w =         (stats_region & KSTATS_WIDTH)   ? w1 :  1;
     
     num_statrgns *= (stats_region & KSTATS_HEIGHT)  ?  1 : h1;
     reg_h =         (stats_region & KSTATS_HEIGHT)  ? h1 :  1;

     num_statrgns *= (stats_region & KSTATS_DEPTH)   ?  1 : d1;
     reg_d =         (stats_region & KSTATS_DEPTH)   ? d1 :  1;

     num_statrgns *= (stats_region & KSTATS_TIME)    ?  1 : t1;
     reg_t =         (stats_region & KSTATS_TIME)    ? t1 :  1;

     num_statrgns *= (stats_region & KSTATS_ELEMENT) ?  1 : e1;
     reg_e =         (stats_region & KSTATS_ELEMENT) ? e1 :  1;

     kpds_set_attribute(src_obj, KPDS_VALUE_SIZE,
			reg_w, reg_h, reg_d, reg_t, reg_e);

/*
kpds_print_attribute(src_obj, KPDS_VALUE_SIZE, kstdout);
kinfo(KVERBOSE, "\t<---- KPDS_VALUE_SIZE of src_obj");
*/

kinfo(KVERBOSE, "%d stat regions: w=%d, h=%d, d=%d, t=%d, e=%d\n",
	num_statrgns, reg_w, reg_h, reg_d, reg_t, reg_e);

     kpds_get_attribute(src_obj, KPDS_VALUE_OPTIMAL_REGION_SIZE,
			&reg_w, &reg_h, &reg_d, &reg_t, &reg_e, &num_subrgns);

kinfo(KVERBOSE, "%d optimal rgns/stat rgn: w=%d, h=%d, d=%d, t=%d, e=%d\n",
	num_subrgns, reg_w, reg_h, reg_d, reg_t, reg_e);

     num_elems = reg_w * reg_h * reg_d * reg_t * reg_e;

     kpds_set_attribute(src_obj, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);

     kpds_set_attribute(src_obj, KPDS_VALUE_REGION_SIZE,
			reg_w, reg_h, reg_d, reg_t, reg_e);

     if (kpds_query_mask(src_obj)) 
     {
        if (!kpds_set_attributes(src_obj, KPDS_MASK_REGION_SIZE,
				 reg_w, reg_h, reg_d, reg_t, reg_e, 
				 KPDS_MASK_DATA_TYPE, KDOUBLE, 
				 NULL))
        {
           kerror(lib, prog, "Unable to set input mask data attributes.");
           (void) kpds_close_object(src_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           return(FALSE);
        }
     }

     /* If a gating object exists, reference it, map the value data if a map
      * exists, and set the datatype attribute to double.  Set the padding 
      * value to 1, and then set the size to equal that of the source object.  
      */

     if (gate_obj)
     {
        if ( (gate_obj = kpds_reference_object(gate_obj)) == KOBJECT_INVALID) 
        {
           kerror(lib, prog, "Unable to duplicate gating data object.");
	   (void) kpds_close_object(src_obj);
           return(FALSE);
        }
        if (kpds_query_mask(gate_obj))
        {
           kerror(lib, prog, "Gating object cannot contain mask data.");
           (void) kpds_close_object(src_obj);
           (void) kpds_close_object(gate_obj);
           return(FALSE);
        }
        if (kpds_query_map(gate_obj)) 
        {
           if (!kpds_set_attribute(gate_obj, KPDS_MAPPING_MODE, KMAPPED) )
	   {
              kerror(lib, prog, "Unable to map gating data.");
	      (void) kpds_close_object(src_obj);
	      (void) kpds_close_object(gate_obj);
              return(FALSE);
	   }
        }
        if (!kpds_set_attributes(gate_obj, 
			   KPDS_VALUE_DATA_TYPE, KDOUBLE,
			   KPDS_VALUE_PAD_VALUE, 1.0, 1.0,
			   KPDS_VALUE_SIZE, w1,h1,d1,t1,e1,
        		   KPDS_VALUE_REGION_SIZE, 
			      reg_w, reg_h, reg_d, reg_t, reg_e,
			   NULL))
        {
           kerror(lib, prog, "Unable to set attribues on gating object.");
           (void) kpds_close_object(src_obj);
           (void) kpds_close_object(gate_obj);
           return(FALSE);
        }
     }

     /* Determine the number of statistics to be calculated 
      */
     num_stats = 0;
     temp = active_stats;
     i_end = 8*sizeof(unsigned long);
     for (i=0; i<i_end; i++)
     {
        num_stats += (temp & 1) ? 1: 0;
        temp = temp >> 1;
     }
     temp = aux_info;
     for (i=0; i<i_end; i++)
     {
        num_stats += (temp & 1) ? 1: 0;
        temp = temp >> 1;
     }

     /* Set up the statistics object - create value segment if it does
      * not already exist, remove other polymorhpic segments if they
      * do exist.  Set the value width size to correspond with the 
      * number of statistics regions analyzed, and the value elements
      * size to the number of statistics and auxiliary parameters 
      * requested.  
      */
     if (kpds_query_location(stats_obj)) kpds_destroy_location(stats_obj);
     if (kpds_query_map(stats_obj))      kpds_destroy_map(stats_obj);
     if (kpds_query_mask(stats_obj))     kpds_destroy_mask(stats_obj);
     if (kpds_query_time(stats_obj))     kpds_destroy_time(stats_obj);
     if (!kpds_query_value(stats_obj))   kpds_create_value(stats_obj);
     if (!kpds_set_attributes(stats_obj,
		KPDS_VALUE_SIZE, num_statrgns, 1, 1, 1, num_stats,
		KPDS_VALUE_DATA_TYPE, KDOUBLE, 
		NULL))
     {
        kerror(lib, prog, "Unable to set attributes on statistics object.");
        (void) kpds_close_object(src_obj);
        (void) kpds_close_object(gate_obj);
        return(FALSE);
     }


     /* If ascii output, print the object size now */
     if (ascii_flag)
     {
        if ((aux_info & KSTATS_WSIZE) ||
            (aux_info & KSTATS_HSIZE) ||
            (aux_info & KSTATS_DSIZE) ||
            (aux_info & KSTATS_TSIZE) ||
            (aux_info & KSTATS_ESIZE))
	    kfprintf(ascii_file,"  Object Dimension:  ");
             
         if (aux_info & KSTATS_WSIZE)
         {
            if (ascii_flag)
               kfprintf(ascii_file,"w=%d  ", w1);
         }
         if (aux_info & KSTATS_HSIZE)
         {
            if (ascii_flag)
               kfprintf(ascii_file,"h=%d  ", h1);
         }
         if (aux_info & KSTATS_DSIZE)
         {
            if (ascii_flag)
               kfprintf(ascii_file,"d=%d  ", d1);
         }
         if (aux_info & KSTATS_TSIZE)
         {
            if (ascii_flag)
               kfprintf(ascii_file,"t=%d  ", t1);
         }
         if (aux_info & KSTATS_ESIZE)
         {
            if (ascii_flag)
               kfprintf(ascii_file,"e=%d", e1);
         }
         if ((aux_info & KSTATS_WSIZE) ||
             (aux_info & KSTATS_HSIZE) ||
             (aux_info & KSTATS_DSIZE) ||
             (aux_info & KSTATS_TSIZE) ||
             (aux_info & KSTATS_ESIZE))
	     kfprintf(ascii_file,"\n");
      }

     /* Obtain each primitive of data and do the processing 
      * on it.  So far, all of the calculated statistics require only 
      * one pass on the data, and can therefore be "streamed".
      */

     for (i=0; i< num_statrgns; i++)
     {

         /* Initialise all variables to appropriate values */
         mean = var = std_dev = rms = integ = pos_integ = neg_integ = 0.0;
         max_val = min_val = skew = kurtosis = 0.0;
         num_pts = pos_pts = neg_pts = zpts = 0;
         maxw = maxh = maxd = maxt = maxe = 0;
         minw = minh = mind = mint = mine = 0;
         cnt1 = cnt2 = 0;
         first_point = TRUE;

         /* Compute statistics over all sub-regions within the statistics 
          * region 
	  */
         for (k=0; k<num_subrgns; k++) 
         {
            data = (double *) kpds_get_data(src_obj, KPDS_VALUE_REGION, data);

            if (kpds_query_mask(src_obj)) 
            {
              mask_data = (double *) kpds_get_data(src_obj, KPDS_MASK_REGION, 
						   mask_data);
            }
	    if (gate_obj)
	    {
              gate_data = (double *) kpds_get_data(gate_obj, KPDS_VALUE_REGION,
						   gate_data);
            }

	    /* Get the region origin if ascii output is requested */
            if (k==0 && ascii_flag) 
            {
               kpds_get_attribute(src_obj, KPDS_VALUE_POSITION,
				  &w_orig, &h_orig, &d_orig, &t_orig, &e_orig);
            }

            for (j=0; j<num_elems; j++)
            {
                if ((!gate_data || gate_data[j]) && 
		    (!mask_data || mask_data[j]) )
                {

                   /* The mean/variance/skew need to be calculated 
                    * when any higher order statistic is enabled
                    */

                   if (  (active_stats & KSTATS_MEAN) 
                      || (active_stats & KSTATS_SUM)
                      || (active_stats & KSTATS_VAR)
                      || (active_stats & KSTATS_SKEW)
                      || (active_stats & KSTATS_STDDEV) 
                      || (active_stats & KSTATS_KURT) )
                   {
                     mean += data[j];
                   }
              
                   if (  (active_stats & KSTATS_KURT)
                      || (active_stats & KSTATS_VAR)
                      || (active_stats & KSTATS_RMS) 
                      || (active_stats & KSTATS_SKEW)
                      || (active_stats & KSTATS_STDDEV) )
                   {
                      tmp_double = data[j]*data[j];
                      var += tmp_double;
                   }
   
                   if ( (active_stats & KSTATS_KURT)
                      || (active_stats & KSTATS_SKEW) )
                   {
		      tmp_double *= data[j];
                      skew += tmp_double;
                   }
                   if (active_stats & KSTATS_KURT)
                   {
		      tmp_double *= data[j];
                      kurtosis += tmp_double;
                   }


                   if (  (active_stats & KSTATS_PTS)
		      || (active_stats & KSTATS_MEAN)
                      || (active_stats & KSTATS_VAR)
                      || (active_stats & KSTATS_RMS) 
                      || (active_stats & KSTATS_SKEW)
                      || (active_stats & KSTATS_STDDEV)
                      || (active_stats & KSTATS_KURT) )
                      num_pts += 1;
                   if (active_stats & KSTATS_POSPTS) 
                      pos_pts += (data[j]>0.0) ? 1 : 0;
                   if (active_stats & KSTATS_NEGPTS)
                      neg_pts += (data[j]<0.0) ? 1 : 0;
                   if (active_stats & KSTATS_ZEROPTS)
                      zpts += (data[j]==0.0) ? 1 : 0;
            

                   if (active_stats & KSTATS_POSSUM) 
                      pos_integ += (data[j]>0.0) ? data[j] : 0;
                   if (active_stats & KSTATS_NEGSUM)
                      neg_integ += (data[j]<0.0) ? data[j] : 0;


		   /* Check for maximum and minimum values, and query the 
		    * postion attribute to get the positions of these values.
                    */

                   if ( (active_stats & KSTATS_MAX) ||
		        (aux_info & KSTATS_WMAX) ||
                        (aux_info & KSTATS_HMAX) ||
			(aux_info & KSTATS_DMAX) ||
			(aux_info & KSTATS_TMAX) ||
			(aux_info & KSTATS_EMAX) )
                   {
                      if (data[j]>max_val || (first_point == TRUE) ) 
                      {
 		         max_val = data[j];
		         kpds_get_attribute(src_obj, KPDS_VALUE_POSITION,
					   &maxw, &maxh, &maxd, &maxt, &maxe);
			 if (aux_info & KSTATS_WMAX)
			    maxw += j%reg_w;
			 if (aux_info & KSTATS_HMAX)
			    maxh += (j/reg_w)%reg_h;
			 if (aux_info & KSTATS_DMAX)
			    maxd += (j/(reg_w*reg_h))%reg_d;
			 if (aux_info & KSTATS_TMAX)
			    maxt += (j/(reg_w*reg_h*reg_d))%reg_t;
			 if (aux_info & KSTATS_EMAX)
			    maxe += (j/(reg_w*reg_h*reg_d*reg_t))%reg_e;
                      }
                   }

                   if ( (active_stats & KSTATS_MIN) ||
		        (aux_info & KSTATS_WMIN) ||
                        (aux_info & KSTATS_HMIN) ||
			(aux_info & KSTATS_DMIN) ||
			(aux_info & KSTATS_TMIN) ||
			(aux_info & KSTATS_EMIN) )
                   {
                      if (data[j]<min_val || (first_point == TRUE) )
                      {
                         min_val = data[j];
		         kpds_get_attribute(src_obj, KPDS_VALUE_POSITION,
					   &minw, &minh, &mind, &mint, &mine);
			 if (aux_info & KSTATS_WMIN)
			    minw += j%reg_w;
			 if (aux_info & KSTATS_HMIN)
			    minh += (j/reg_w)%reg_h;
			 if (aux_info & KSTATS_DMIN)
			    mind += (j/(reg_w*reg_h))%reg_d;
			 if (aux_info & KSTATS_TMIN)
			    mint += (j/(reg_w*reg_h*reg_d))%reg_t;
			 if (aux_info & KSTATS_EMIN)
			    mine += (j/(reg_w*reg_h*reg_d*reg_t))%reg_e;
                      }
                   }
                   first_point = FALSE;
               }    
	    }
         }
         integ = mean;
	 if ( num_pts != 0 )
	 {
            mean /= num_pts;
            var  /= num_pts;
            skew /= num_pts;
            kurtosis /= num_pts;
	 }
	 else
	 {
            mean = 0.0;
            var  = 0.0;
            skew = 0.0;
            kurtosis = 0.0;
	 }
         rms   = ksqrt(var);

         /* Ref: Numerical recipes in C Pg.456/7
          * Eqn: 13.1.2,3,5,6
          * All the 3 below are algebraic expansions of those formulae
          */
         if ( (active_stats & KSTATS_KURT)
                || (active_stats & KSTATS_VAR)
                || (active_stats & KSTATS_RMS)
                || (active_stats & KSTATS_SKEW)
                || (active_stats & KSTATS_STDDEV) )
         {
            kurtosis += 6*kpow(mean,2.0)*var -4*mean*skew - 3*kpow(mean,4.0);
            skew +=2*kpow(mean,3.0) - 3*mean*var;
            var  -=kpow(mean,2.0);
            var  *=((num_pts==1)?1.0:(float)(num_pts)/(float)(num_pts-1));
	    std_dev = ksqrt(var);

	    if (var != 0.0)
            {
               skew /= kpow(var,1.5);

               kurtosis /= kpow(var,2.0);
               kurtosis -= 3.0;
            }
	    else
            skew = kurtosis = 0.0;
         }

 
	 /* .... Store and Print Statistics .... */
         /* Store statistics in data object.  If this is the first 
          * statistics region analyzed, also print out how the statistics
          * are stored in the object's comment attribute.
          * If the ascii_flag is set, print the statistics information
          * to the ascii_file.
          */
         
         stat_data = (double *) kpds_get_data(stats_obj, KPDS_VALUE_VECTOR, 
						stat_data);
         if (ascii_flag && (num_statrgns > 1))
         {
            kfprintf(ascii_file,"\n****  Statistics for Region # %d  ****\n",i);
            kfprintf(ascii_file,
		     "  Region origin:  w=%d  h=%d  d=%d  t=%d  e=%d\n",
		     w_orig, h_orig, d_orig, t_orig, e_orig);
         }

	 temp = 0;
         if (i==0)
         {
            kpds_get_attribute(src_obj, KPDS_NAME, &str_ptr);
            comment = kstring_3cat("Statistics for File:  ",
				    str_ptr, "\n\n", NULL);
         }

         if (active_stats & KSTATS_MEAN)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Mean\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = mean;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Mean:     %g\n", mean);
         }
       
         if (active_stats & KSTATS_VAR)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Variance\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = var;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Variance: %g\n", var);
         }

         if (active_stats & KSTATS_STDDEV)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Std Dev\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = ksqrt(var);
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Std Dev:  %g\n", ksqrt(var));
         }

         if (active_stats & KSTATS_RMS)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: RMS\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = rms;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  RMS:      %g\n", rms);
         }

         if (active_stats & KSTATS_SKEW)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Skewness\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = skew;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Skewness: %g\n", skew);
         }

         if (active_stats & KSTATS_KURT)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Kurtosis\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = kurtosis;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Kurtosis: %g\n", kurtosis);
         }

         if (active_stats & KSTATS_MIN)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Minimum\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = min_val;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Minimum:  %g\n", min_val);
         }

         if (active_stats & KSTATS_MAX)    
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Maximum\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = max_val;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Maximum:  %g\n", max_val);
         }

         if (ascii_flag && (aux_info & (KSTATS_MINPOS)))
            kfprintf(ascii_file,"  First minima at location:  ");

         if (aux_info & KSTATS_WMIN)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Width Coordinate of First Minima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) minw;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"w=%d  ",minw);
         }

         if (aux_info & KSTATS_HMIN)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Height Coordinate of First Minima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) minh;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"h=%d  ",minh);
         }

         if (aux_info & KSTATS_DMIN)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Depth Coordinate of First Minima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) mind;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"d=%d  ",mind);
         }

         if (aux_info & KSTATS_TMIN)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Time Coordinate of First Minima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) mint;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"t=%d  ",mint);
         }

         if (aux_info & KSTATS_EMIN)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Elements Coordinate of First Minima\n",
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) mine;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"e=%d",mine);
         }

         if (ascii_flag && (aux_info & (KSTATS_MINPOS)))
            kfprintf(ascii_file,"\n");

         if (ascii_flag && (aux_info & (KSTATS_MAXPOS)))
            kfprintf(ascii_file,"  First maxima at location:  ");

         if (aux_info & KSTATS_WMAX)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Width Coordinate of First Maxima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) maxw;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"w=%d  ",maxw);
         }

         if (aux_info & KSTATS_HMAX)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Height Coordinate of First Maxima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) maxh;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"h=%d  ",maxh);
         }

         if (aux_info & KSTATS_DMAX)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Depth Coordinate of First Maxima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) maxd;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"d=%d  ",maxd);
         }

         if (aux_info & KSTATS_TMAX)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Time Coordinate of First Maxima\n",temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) maxt;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"t=%d  ",maxt);
         }

         if (aux_info & KSTATS_EMAX)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Elements Coordinate of First Maxima\n",
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) maxe;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"e=%d",maxe);
         }

         if (ascii_flag && (aux_info & (KSTATS_MAXPOS)))
            kfprintf(ascii_file,"\n");

         if (active_stats & KSTATS_SUM)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Total Integral\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = integ;
            temp ++;
            if (ascii_flag)
               kfprintf(ascii_file,"  Total Integral:    %g\n",integ);
         }

         if (active_stats & KSTATS_POSSUM)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Positive Integral\n", 
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = pos_integ;
            temp ++;
            if (ascii_flag)
            {
               kfprintf(ascii_file,"  Positive Integral: %g\n",
			pos_integ);
            }
         }

         if (active_stats & KSTATS_NEGSUM)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Negative Integral\n", 
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = neg_integ;
            temp ++;
            if (ascii_flag)
            {
               kfprintf(ascii_file,"  Negative Integral: %g\n",
			neg_integ);
            }
         }

         if (active_stats & KSTATS_PTS)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Total Contributing Points\n", 
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) num_pts;
            temp ++;
            if (ascii_flag)
	    {
               kfprintf(ascii_file,"  Total Contributing Points:    %d\n",
			num_pts);
	    }
         }
          
         if (active_stats & KSTATS_POSPTS)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Contributing Positive Points\n",
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) pos_pts;
            temp ++;
            if (ascii_flag)
            {
               kfprintf(ascii_file,"  Contributing Positive Points: %d\n",
			pos_pts);
            }
         }

         if (active_stats & KSTATS_NEGPTS)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Contributing Negative Points\n",
			temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) neg_pts;
            temp ++;
            if (ascii_flag)
            {
               kfprintf(ascii_file,"  Contributing Negative Points: %d\n",
			neg_pts);
            }
         }

         if (active_stats & KSTATS_ZEROPTS)
         {
            if(i==0)
            {
	       ksprintf(str_array, 
			"Element %d: Contributing Zero-Valued Points\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) zpts;
            temp ++;
            if (ascii_flag)
            {
               kfprintf(ascii_file,"  Contributing Zero-Valued Pts: %d\n",
			zpts);
            }
         }

         if (aux_info & KSTATS_WSIZE)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Object Width\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) w1;
            temp ++;
         }

         if (aux_info & KSTATS_HSIZE)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Object Height\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) h1;
            temp ++;
         }

         if (aux_info & KSTATS_DSIZE)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Object Depth\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) d1;
            temp ++;
         }

         if (aux_info & KSTATS_TSIZE)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Object Time Size\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) t1;
            temp ++;
         }

         if (aux_info & KSTATS_ESIZE)
         {
            if(i==0)
            {
	       ksprintf(str_array, "Element %d: Object Elements Size\n", temp);
               str_ptr = kstring_cat(comment, str_array, NULL);
               kfree(comment);
               comment = str_ptr;
            }
            stat_data[temp] = (double) e1;
            temp ++;
         }

         if (i==0)
         {
	    kpds_set_attribute(stats_obj, KPDS_COMMENT, comment);
	    kfree(comment);
	 }
	 kpds_put_data(stats_obj, KPDS_VALUE_VECTOR, stat_data);

     }
     kfree(data);
     kfree(stat_data);
     kfree(gate_data);
     kfree(mask_data);
     kpds_close_object(src_obj);
     if (gate_obj) kpds_close_object(gate_obj);
     return(TRUE);
}
/* -library_code_end */
