 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kpad
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkpad
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkpad - perform the padding of a data object
* 
*       Purpose: Library Routine for kpad.  lkpad performs the 
*                padding of a data object.  The offset parameters
*		 define where the origin of the source object will 
*		 lie with respect to the output object.  Specifying 
*		 an offset greater than zero can be used to generate 
*                a border around the original data.  Negative offset 
*		 values result in truncation of data lying in the 
*		 region (0-w_off, 0-h_off, 0-d_off, 0-t_off, 0-e_off)
*		 For example, given an offset of (-1,-1,0,0,0), the 
*		 object will effectively be shifted by -1 in the width 
*		 and height dimensions, and that data truncated.
*		 If the offset plus the original data size is greater
*		 than the specified destination size, truncation of
*		 the data will occur.
*                
*		 The imag and real input parameters are the pad 
*		 values used.  If the input data object type is 
*                not complex, the imaginary pad value will be ignored.  
*                For data objects that are not of type double, pad 
*                values will be cast so that the output object is the
*                same type as the input object.  
*
*                Mask data corresponding to padded data will be padded
*                non-zero if the valid flag is TRUE, and padded with
*		 zeros if the valid flag is FALSE.  Mask data corresponding 
*		 to the source object will be transferred unaltered.
*
*                If curvilinear or rectilinear time or location data 
*		 exist, it is unclear how to alter the dimensionality, 
*		 and lkpad will fail.  If uniform location data exist
*		 lkpad will not fail, but it will not alter the location
*		 data in any way.  yet.
*
*                lkpad does not modify map data - it directly pads the 
*		 value data.  The data is not mapped first.
*
*         Input: src_obj - Source data object.
*                w_size  - Size of width dimension after padding.
*                h_size  - Size of height dimension after padding.
*                d_size  - Size of depth dimension after padding.
*                t_size  - Size of time dimension after padding.
*                e_size  - Size of elements dimension after padding.
*                w_off   - Offset of src_obj in width dimension.
*                h_off   - Offset of src_obj in height dimension.
*                d_off   - Offset of src_obj in depth dimension.
*                t_off   - Offset of src_obj in time dimension.
*                e_off   - Offset of src_obj in elements dimension.
*                real    - Real part of pad value.
*                imag    - Imaginary part of pad value.
*                valid   - if TRUE, padded data is considered valid
*        Output: dst_obj - Destination object (padded) data object.
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Donna Koechner & Mark Young
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkpad (
   kobject src_obj,
   int     w_size,
   int     h_size,
   int     d_size,
   int     t_size,
   int     e_size,
   int     w_off,
   int     h_off,
   int     d_off,
   int     t_off,
   int     e_off,
   double  real,
   double  imag,
   int	   valid,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	char	*lib="kdatamanip", *prog="lkpad";
	int	grid, has_value, has_mask, mtype=KUBYTE, created_mask=FALSE;
	double	mask_pad;

	/* make sure parameters that were passed in are valid */
	if ((w_size<1) || (h_size<1) || (d_size<1) || (t_size<1) || (e_size<1))
	{
	   kerror(lib, prog,
		 "destination size must be greater than zero (%d,%d,%d,%d,%d)",
		  w_size, h_size, d_size, t_size, e_size);
	   return(FALSE);
	}

	/* make sure the objects are valid */
	if (src_obj == KOBJECT_INVALID)
	{
           kerror(lib, prog, "Source object is not a valid data object.");
	   return(FALSE);
	}
	if (dst_obj==KOBJECT_INVALID)
	{
           kerror(lib, prog, "Destination object is not a valid data object.");
	   return(FALSE);
	}

        /*
         *  Check if the source object has location or time data.  We
         *  currently cannot operate on objects with rectilinear or
         *  curvilinear location data or time data, so if either
         *  exist, return.
         */
        if (kpds_query_location(src_obj))
        {
           if (!kpds_get_attribute(src_obj, KPDS_LOCATION_GRID, &grid))
           {
              kerror(lib, prog, "Unable to retrieve location grid attribute.");
              return(FALSE);
           }
           if (grid != KUNIFORM && (grid == KRECTILINEAR ||
                                    grid == KCURVILINEAR))
           {
              kerror(lib, prog, "source object can not contain explicit "
                     "rectilinear or curvilinear location data.");
	      return(FALSE);
           }
        }
        if (kpds_query_time(src_obj))
        {
           kerror(lib, prog, "Source object cannot contain time data.");
           return(FALSE);
        }

	/* bail out if source does not have value or mask data */
	has_value = kpds_query_value(src_obj);
	has_mask  = kpds_query_mask(src_obj);
	if ((!has_mask) && (!has_value))
        {
           kerror(lib, prog, "Source object must contain value or mask data.");
           return(FALSE);
        }

	/* reference the source object to avoid side effects */
	if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, prog, "Unable to reference source object.");
           return(FALSE);
        }

	/* if valid flag is FALSE, create mask segment if it doesn't exist */
	if ((!valid) && (!has_mask))
	{
	   if (!kpds_create_mask(src_obj))
           {
              kerror(lib, prog, "Unable to create mask for source object.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	   if (!kpds_set_attribute(src_obj,KPDS_MASK_DATA_TYPE,mtype)) 
           {
              kerror(lib, prog, "Unable to set source object mask attributes.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	   if (!kpds_initialize_mask(src_obj, 1.0))
           {
              kerror(lib, prog, "Unable to initialize source object mask.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	   has_mask = TRUE;
	   /* set this flag so that we can destroy the mask we created on
	      the source before returning - to remove side effects */
	   created_mask = TRUE;
	}

	/* if source value exists, set pad, size, and position attributes */
	if (has_value)
	{
	   if (!kpds_set_attributes(src_obj,
		KPDS_VALUE_SIZE, w_size, h_size, d_size, t_size, e_size,
		KPDS_VALUE_OFFSET, -w_off, -h_off, -d_off, -t_off, -e_off,
		KPDS_VALUE_INTERPOLATE, KPAD,
		KPDS_VALUE_PAD_VALUE, real, imag, NULL))
           {
              kerror(lib,prog,"Unable to set source object value attributes.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	}

	/* if source mask exists, set pad value, size, and offset attributes */
	if (valid)	 mask_pad = 1;
	else		 mask_pad = 0;

	if (has_mask)
	{
	   if (!kpds_set_attributes(src_obj,
		KPDS_MASK_SIZE, w_size, h_size, d_size, t_size, e_size,
		KPDS_MASK_OFFSET, -w_off, -h_off, -d_off, -t_off, -e_off,
		KPDS_MASK_INTERPOLATE, KPAD,
		KPDS_MASK_PAD_VALUE, mask_pad, NULL))
           {
              kerror(lib,prog,"Unable to set source object mask attributes.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	}

	/* if source value exists, copy it to destination */
	if (has_value)
	{
	   if (!kpds_copy_value(src_obj, dst_obj, TRUE))
           {
              kerror(lib,prog,"Unable to copy source value to destination.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	}

	/* if source mask exists, copy it to destination */
	if (has_mask)
	{
	   if (!kpds_copy_mask(src_obj, dst_obj, TRUE))
           {
              kerror(lib,prog,"Unable to copy source mask to destination.");
	      kpds_close_object(src_obj);
              return(FALSE);
           }
	   if (created_mask)
	   {
	      if (!kpds_destroy_mask(src_obj))
              {
                 kerror(lib,prog,"Unable to destroy source mask.");
	         kpds_close_object(src_obj);
                 return(FALSE);
              }
	   }
	}

	kpds_close_object(src_obj);

	return(TRUE);
}
/* -library_code_end */
