 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for korient
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkorient
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkorient - reorient object to new dimensions
* 
*       Purpose: The following is the library routine that performs
*	 	 data reorientation with respect to the 5 polymorphic
*		 dimensions, width, height, depth, time, and elements.
*                The routine takes an input data object and a mapping,
*		 defined by the src2dst_map array, which specifies how 
*		 each dimension in the old object will be mapped to the 
*		 new object.  More than one dimension from the input 
*		 object can be mapped to the same dimension in the 
*		 output object.  
*
*		 The src2dst_map array is a set of five integers that 
*		 can have one of the following values: 
*		 !	KORIENT_WIDTH
*		 !	KORIENT_HEIGHT
*		 !	KORIENT_DEPTH
*		 !	KORIENT_TIME
*		 !	KORIENT_ELEMENTS
*		 
*		 src2dst_map[0] 
*		 !	defines where the WIDTH dimension of the source 
*		 !	object will be mapped to in the destination object.
*		 src2dst_map[1] 
*		 !	defines where the HEIGHT dimension of the source 
*		 !	object will be mapped to in the destination object.
*		 src2dst_map[2] 
*		 !	defines where the DEPTH dimension of the source 
*		 !	object will be mapped to in the destination object.
*		 src2dst_map[3] 
*		 !	defines where the TIME dimension of the source 
*		 !	object will be mapped to in the destination object.
*		 src2dst_map[4] 
*		 !	defines where the ELEMENTS dimension of the source 
*		 !	object will be mapped to in the destination object.
*		 
*		 
*		 If the source object has map data, the map data cannot
*		 have depth, time, or elements dimensions of greater than
*		 one.  The map width and height dimensions can be any size.
*		 
*		 If the source object has rectilinear or curvilinear 
*		 location or time data, lkorient will fail.  If the 
*		 source object has uniform location data, lkorient will
*		 continue, but the location data will not be modified
*		 to reflect the reorientation.
*
*		 If the object has mask data, it will be reoriented along
*		 with the value data.
*
*		 lkorient will fail if the source object does not contain
*		 value data.
*
*         Input: src_obj     - the input object to be processed
*                src2dst_map - dimension mapping from src_obj to dst_obj
*        Output: dst_obj     - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Donna Koechner, Mark Young & Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkorient (
   kobject src_obj,
   int     *src2dst_map,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	char 	*prog="lkorient", *lib="kdatamanip";
	int	mw, mh, md, mt, me, grid, has_mask=FALSE;
	int 	line_size, num_pts;
        int 	*indx_order, *out_dims, *inp_dims, *out_reg_size;
	int	*out_index_order;
        kaddr 	data = NULL, mdata=NULL;
        int  	i,j,k;

	if (!kpds_query_value(src_obj))
	{
	   kerror(lib, prog, "source object must contain value data");
	   return(FALSE);
        }
	
	if (kpds_query_map(src_obj))
	{
	   if (!kpds_get_attribute(src_obj,KPDS_MAP_SIZE,&mw,&mh,&md,&mt,&me))
	   {
	      kerror(lib, prog, "Unable to get src_obj value size");
	      return(FALSE);
           }
	   if ((md > 1) || (mt > 1) || (me > 1))
	   {
	      kerror(lib, prog, 
		"Unable operate on object with map depth, time, or elements "
		"dimension greater than 1 (md=%d, mt=%d, me=%d)", md, mt, me);
	      return(FALSE);
	   }
	}

        /* Object may not have rectilinear or curvilinear location data */
        if (kpds_query_location(src_obj))
        {
           if (!kpds_get_attribute(src_obj, KPDS_LOCATION_GRID, &grid))
           {
              kerror(lib, prog, "Unable to retrieve location grid attribute.");
              return(FALSE);
           }
           if (grid != KUNIFORM && (grid == KRECTILINEAR ||
                                    grid == KCURVILINEAR))
           {
              kerror(lib, prog,
                     "Source object cannot contain explicit "
                     "rectilinear or curvilinear location data.\n"
		     "Use the kaxis program for reorienting data along "
		     "the width, height, and depth dimensions");
              return(FALSE);
           }
        }

        /* Object may not have time data */
        if (kpds_query_time(src_obj))
        {
           kerror(lib, prog,
                  "Source object cannot contain explicit time data. ");
           return(FALSE);
        }

	if (kpds_query_mask(src_obj))
	   has_mask = TRUE;

	/* reference the source object */
	if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
	{
	   kerror(lib, prog, "failed to reference input object");
	   return(FALSE);
        }

        indx_order      = (int *) kmalloc(5 * sizeof(int));
        out_dims        = (int *) kmalloc(5 * sizeof(int));
        inp_dims        = (int *) kmalloc(5 * sizeof(int));
        out_reg_size    = (int *) kmalloc(5 * sizeof(int));
        out_index_order = (int *) kmalloc(5 * sizeof(int));

	if (!kpds_get_attribute(src_obj, KPDS_VALUE_SIZE, &inp_dims[0],
			&inp_dims[1],&inp_dims[2],&inp_dims[3],&inp_dims[4]))
	{
	   kerror(lib, prog, "Unable to get src_obj value size");
	   kpds_close_object(src_obj);
	   return(FALSE);
        }

        /* num_pts contains the total no. of points in the object
         * (w*h*d*t*e)
         */
        num_pts = 1;
        for (i=0; i<5; i++)
        {
            out_dims[i] = 1;
            out_reg_size[i] = 1;
            num_pts *= inp_dims[i];
        }

        /* Determines the order in which the input object should be 
         * indexed. It depends on the way the input object dimensions
         * are mapped to the output object dims. Each entry is 
         * compared to w,h,d,t,e in that order(Outermost loop). If it
         * matches it goes into the index order array.
         */ 
        for (k= KORIENT_WIDTH,i=0; k < (KORIENT_WIDTH + 5); k++)
            for (j=0; j<5; j++)
            {
                if (src2dst_map[j] == k)
                {
                   switch (j)
                   {
                       case KORIENT_WIDTH:
                            indx_order[i++] = KWIDTH;
                            break;

                       case KORIENT_HEIGHT:
                            indx_order[i++] = KHEIGHT;
                            break;

                       case KORIENT_DEPTH:
                            indx_order[i++] = KDEPTH;
                            break;

                       case KORIENT_TIME:
                            indx_order[i++] = KTIME;
                            break;

                       case KORIENT_ELEMENTS:
                            indx_order[i++] = KELEMENTS;
                            break;

                   }    
                   /*indx_order[i++] = j + KORIENT_WIDTH ;*/        
                   out_dims[k - KORIENT_WIDTH] *= inp_dims[j];
                }   
            }

  	if (!kdms_set_attribute(src_obj, KDMS_SEGMENT_VALUE,
			KDMS_INDEX_ORDER,indx_order))
	{
	   kerror(lib, prog, "Unable to set src_obj value index order");
	   kpds_close_object(src_obj);
	   return(FALSE);
        }
	
	if (has_mask)
	{
  	   if (!kdms_set_attribute(src_obj, KDMS_SEGMENT_MASK,
			KDMS_INDEX_ORDER,indx_order))
	   {
	      kerror(lib, prog, "Unable to set src_obj mask index order");
	      kpds_close_object(src_obj);
	      return(FALSE);
           }
	}


        if (!kpds_set_attribute(dst_obj, KPDS_VALUE_SIZE, out_dims[0],
                           out_dims[1],out_dims[2],out_dims[3],out_dims[4]))
	{
	   kerror(lib, prog, "Unable to set dst_obj value size");
	   kpds_close_object(src_obj);
	   return(FALSE);
        }

	out_index_order[0] = KWIDTH;
	out_index_order[1] = KHEIGHT;
	out_index_order[2] = KDEPTH;
	out_index_order[3] = KTIME;
	out_index_order[4] = KELEMENTS;
	if (!kdms_set_attribute(dst_obj, KDMS_SEGMENT_VALUE, KDMS_INDEX_ORDER, 
                           out_index_order))
	{
	   kerror(lib, prog, "Unable to set dst_obj index order");
	   kpds_close_object(src_obj);
	   return(FALSE);
        }
	if (has_mask)
	{
	   if (!kdms_set_attribute(dst_obj, KDMS_SEGMENT_MASK, 
			KDMS_INDEX_ORDER, out_index_order))
	   {
	      kerror(lib, prog, "Unable to set dst_obj index order");
	      kpds_close_object(src_obj);
	      return(FALSE);
           }
	}


        if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
	{
	   kerror(lib, prog, "Unable to create reference to dst_obj");
	   kpds_close_object(src_obj);
	   return(FALSE);
        }

        /* The size of the line from the input is determined by the
         * new index order of the input. This line is put to a region
         * in the output. The size of the output region is determined 
         * by the dimension to which indx_order[0] maps.
         */

        switch (indx_order[0])
        {
            case KWIDTH:
                 line_size = inp_dims[0];
                 out_reg_size[src2dst_map[0]] = line_size;
                 break;

            case KHEIGHT:
                 line_size = inp_dims[1];
                 out_reg_size[src2dst_map[1]] = line_size;
                 break;

            case KDEPTH:
                 line_size = inp_dims[2];
                 out_reg_size[src2dst_map[2]] = line_size;
                 break;

            case KTIME:
                 line_size = inp_dims[3];
                 out_reg_size[src2dst_map[3]] = line_size;
                 break;

            case KELEMENTS:
                 line_size = inp_dims[4];
                 out_reg_size[src2dst_map[4]] = line_size;
                 break;

        }

        if (!kpds_set_attribute(dst_obj, KPDS_VALUE_REGION_SIZE, 
                           out_reg_size[0],out_reg_size[1],out_reg_size[2],
                           out_reg_size[3],out_reg_size[4]))
	{
	   kerror(lib, prog, "Unable to set dst_obj value size");
	   kpds_close_object(src_obj); kpds_close_object(dst_obj);
	   return(FALSE);
        }
        
	if (has_mask)
        {
	   if (!kpds_set_attribute(dst_obj, KPDS_MASK_REGION_SIZE, 
                           out_reg_size[0],out_reg_size[1],out_reg_size[2],
                           out_reg_size[3],out_reg_size[4]))
	   {
	      kerror(lib, prog, "Unable to set dst_obj mask size");
	      kpds_close_object(src_obj); kpds_close_object(dst_obj);
	      return(FALSE);
           }
	}

        (void) kfree(indx_order);
        (void) kfree(out_dims);
        (void) kfree(inp_dims);
        (void) kfree(out_reg_size);
        (void) kfree(out_index_order);

        /* It is possible that more than one dimension of the input 
         * object maps to the same dimension of the output object.
         * To take this case into account, a line get on the input
         * and a region put on the output is done.
         */

        for (i=0; i< (num_pts/line_size); i++)
        {
           if ((data = kpds_get_data(src_obj, KPDS_VALUE_LINE, data)) == NULL)
	   {
	      kerror(lib, prog, "Unable to get src_obj data, line %d", i);
	      kpds_close_object(src_obj); kpds_close_object(dst_obj);
	      kfree(data);
	      if (has_mask) kfree(mdata);
	      return(FALSE);
           }

           if (!kpds_put_data(dst_obj, KPDS_VALUE_REGION, data))
           {
	      kerror(lib,prog,"Failed to put region in dst_obj, region %d",i);
              kfree(data); 
	      if (has_mask) kfree(mdata);
	      kpds_close_object(src_obj); kpds_close_object(dst_obj);
              return(FALSE);
           }

	   if (has_mask)
	   {
              if ((mdata = kpds_get_data(src_obj, KPDS_MASK_LINE, mdata)) 
			== NULL)
	      {
	         kerror(lib, prog, 
			"Unable to get src_obj mask data, line %d", i);
	         kpds_close_object(src_obj); kpds_close_object(dst_obj);
		 kfree(data); kfree(mdata);
	         return(FALSE);
              }

              if (!kpds_put_data(dst_obj, KPDS_MASK_REGION, mdata))
              {
	         kerror(lib,prog,
			"Failed to put mask region in dst_obj, region %d",i);
                 kfree(data); kfree(mdata);
	         kpds_close_object(src_obj); kpds_close_object(dst_obj);
                 return(FALSE);
              }
           }
        }
        kfree(data);  
	if (has_mask) kfree(mdata);
        kpds_close_object(src_obj);
        kpds_close_object(dst_obj);
	return(TRUE);
}
/* -library_code_end */
