 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kfft
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkfft
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include <klibm/kcfft.h>
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkfft - perform multidimensional FFT on data object
* 
*       Purpose: Performs a multidimensional FFT on the input object.
*                The FFT can be requested on up to 5 dimensions.
*                The direction of the FFT can be specified.
*                The same routine performs both the forward
*                and the inverse FFT. The result of the FFT operation
*                can be scaled by either a factor of 1 (no scaling),
*                1/N or 1/sqrt(N) where N is the size of the FFT.
*                Centering on the FFT/IFFT is also available as an option.
*                lkfft performs the FFT operation on all the data
*                primitives in the object. For example, if the FFT
*                is requested along the width for a 5-D object then
*                it is taken along all width primitives in the object.
*                The size of the dimension for which FFT is requested
*                must be a power of two and greater than 1.
*                The output data object is always complex. It can
*                either be single precision complex or double precision complex
*                depending on the value of the parameter datatype.
*         Input: in_obj : Input complex object
*                center : flag to request centered freq domain
*                w : flag to request procesing along width
*                h : flag to request procesing along height
*                d : flag to request procesing along depth
*                t : flag to request procesing along time
*                e : flag to request procesing along elements
*                scale_factor : scaling factor on the output
*                direction : direction along which FFT is taken
*                datatype : datatype of the output object
*
*        Output: cmplx_out : Output object containing real and imag components
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Scott Wilson, Ashish Malhotra
*          Date: Apr 14, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkfft (
	   kobject in_obj,
           int     center,
           int     w,
           int     h,
           int     d,
           int     t,
           int     e,
           int     scale_factor,
           int     direction,
           int     datatype,
           kobject cmplx_out)
/* -library_def_end */

/* -library_code */
{
   int wi,hi,di,ti,ei,type; /* Size and type of input object */
   int wu,hu,du,tu,eu; /* Size of processing unit (a vector) */
   int wp,hp,dp,tp,ep; /* Data point location in storage array */
   int wor,hor,dor,tor,eor; /* Optimal region size */
   int unit_len,num_units,numo,numr;
   int num, count, loop_cnt;
   double  scale_mult,size,mult;
   int error_flag;
   int grid;
   double *real_data=NULL,*imag_data=NULL;
   kdcomplex *cmplx_data=NULL;
   kobject   cmplx_ref;
   int dim_list[5],num_dims;
   int i,x,index; /* For mapping calculations */
   int efactor,tfactor,dfactor,hfactor,wfactor;

   num_dims=0;
   if (w) { dim_list[num_dims] = KWIDTH; num_dims++; }
   if (h) { dim_list[num_dims] = KHEIGHT; num_dims++; }
   if (d) { dim_list[num_dims] = KDEPTH; num_dims++; }
   if (t) { dim_list[num_dims] = KTIME; num_dims++; }
   if (e) { dim_list[num_dims] = KELEMENTS; num_dims++; }
   
   if ( (in_obj = kpds_reference_object(in_obj)) == NULL)
     {
        kerror("kdatamanip","lkfft",
               "Failed to reference input object.");
        return(FALSE);
     }

   if (!kpds_query_value(in_obj))
     {
        kerror("kdatamanip","lkfft",
               "No value data in source object to operate on.");
        (void) kpds_close_object(in_obj);
        return(FALSE);
     }

   if (kpds_query_location(in_obj))
   {
      if (!kpds_get_attribute(in_obj, KPDS_LOCATION_GRID, &grid))
      {
	 kerror("kdatamanip", "lkfft", 
		"Unable to retrieve location grid attribute.");
	 return(FALSE);
      }
      if (grid != KUNIFORM && (grid == KRECTILINEAR || 
			       grid == KCURVILINEAR))
      {
	 kerror("kdatamanip","lkfft",
               "FFT does not support explicit rectilinear or curvilinear "
               "data in the source object.");
	 return(FALSE);
      }
   }
   
   if (kpds_query_mask(in_obj) || kpds_query_time(in_obj))
     {
        kerror("kdatamanip","lkfft",
               "FFT does not support mask and/or time %s",
               "data in the source object.");
        (void) kpds_close_object(in_obj);
        return(FALSE);
     }

   kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&wi,&hi,&di,&ti,&ei);
   kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE,KDCOMPLEX);

  /* IEEE FFT is good only up to 32768 point vectors, so limit things to
      that size */
  if (wi > 32768 || hi > 32768 || di > 32768 || ti > 32768 || ei > 32768)
    {
        kerror("kdatamanip","lkfft",
               "FFT currently limited to 32768 points along any dimension.");
        (void) kpds_close_object(in_obj);
        return(FALSE);
    }

  /*
   * Check the size of the dimensions for which FFT is requested.
   * If they are not a power of two error out. Find the correct scale
   * factor for multiplying the destination object.
   */
   error_flag = FALSE;
   scale_mult = 1.0;
   size = 1.0; 
   for (loop_cnt = 0; loop_cnt < num_dims; loop_cnt++)
   {
     if (w)
       {
         if (kpowtwo(wi) == FALSE) error_flag = TRUE;
         if (scale_factor == 2) scale_mult *= wi;
         else if (scale_factor == 3) scale_mult *= ksqrt((double)wi); 
         size *= wi;
       }
     if (h)
       {
         if (kpowtwo(hi) == FALSE) error_flag = TRUE;
         if (scale_factor == 2) scale_mult *= hi;
         else if (scale_factor == 3) scale_mult *= ksqrt((double)hi);
         size *= hi;
       }
     if (d)
       {
         if (kpowtwo(di) == FALSE) error_flag = TRUE;
         if (scale_factor == 2) scale_mult *= di;
         else if (scale_factor == 3) scale_mult *= ksqrt((double)di);
         size *= di;
       }
     if (t)
       {
         if (kpowtwo(ti) == FALSE) error_flag = TRUE;
         if (scale_factor == 2) scale_mult *= ti;
         else if (scale_factor == 3) scale_mult *= ksqrt((double)ti);
         size *= ti;
       }
     if (e)
       {
         if (kpowtwo(ei) == FALSE) error_flag = TRUE;
         if (scale_factor == 2) scale_mult *= ei;
         else if (scale_factor == 3) scale_mult *= ksqrt((double)ei);
         size *= ei;
       }
     if (error_flag)
       {
         (void) kpds_close_object(in_obj);
         kerror("kdatamanip","lkfft",
                "Size of dimension requested for FFT is not an integral %s",
                "power of 2 or equal to 1.");
         return(FALSE); 
       }
   }

  /*
   * The IEEE FFT routine used for lkfft scales the output data by 1/N 
   * when doing an inverse FFT. The following step is done to 
   * compensate for it.
   */
   if (direction == 1) scale_mult /= size;

  /* 
   * Set the datatype of the destination object as single precision complex
   * (KCOMPLEX) or double complex(KDCOMPLEX) depending on the flag
   * passed in. Set the size of the dest obj to be the same as the 
   * src object. Create value segment if not there already.
   */
   type = ((datatype == 1) ? KDCOMPLEX : KCOMPLEX);
   if (!kpds_query_value(cmplx_out)) kpds_create_value(cmplx_out);
   kpds_set_attributes(cmplx_out, 
                       KPDS_VALUE_DATA_TYPE,type,
                       KPDS_VALUE_SIZE,wi,hi,di,ti,ei,NULL);

  /* 
   * Reference the destination object.
   */
   if ( (cmplx_out = kpds_reference_object(cmplx_out)) == NULL)
     {
       (void) kpds_close_object(in_obj);
       kerror("kdatamanip","lkfft", "Failed to reference output object");
       return(FALSE);
     }
   kpds_set_attributes(cmplx_out,
                       KPDS_VALUE_INTERPOLATE,KNONE,
                       KPDS_VALUE_DATA_TYPE,KDCOMPLEX,
                       NULL);

  /* Make another reference of the output object for intermediate data. */
   if ( (cmplx_ref = kpds_reference_object(cmplx_out)) == NULL)
     {
       (void) kpds_close_object(in_obj);
       (void) kpds_close_object(cmplx_out);
       kerror("kdatamanip","lkfft", "Failed to reference output object");
       return(FALSE);
     }
   kpds_copy_object_attr(cmplx_out,cmplx_ref);

   /* Copy data into the output object. We'll then use it as input.
   If Nyquist modulation is needed, do that along with the copy. */
   kpds_get_attribute(in_obj,KPDS_VALUE_OPTIMAL_REGION_SIZE,
                              &wor,&hor,&dor,&tor,&eor,&numr);
   kpds_set_attribute(cmplx_out,KPDS_VALUE_REGION_SIZE,wor,hor,dor,tor,eor);
   kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wor,hor,dor,tor,eor);
   numo = wor*hor*dor*tor*eor;
   index=0;
   efactor = wi*hi*di*ti;
   tfactor = wi*hi*di;
   dfactor = wi*hi;
   hfactor = wi;
   wfactor = 1;
   for (num=0; num<numr; num++)
     {
       cmplx_data = (kdcomplex *)kpds_get_data(in_obj,
                                      KPDS_VALUE_REGION, (kaddr)cmplx_data);
       if (direction == 0 && center) /* Nyquist modulation */
         {
           for (i=0; i<numo; i++)
             {
               x = index;            /* Offset into linear storage area */
               ep = x/efactor;     /* Inverse mapping function */
               x=x-ep*efactor;
               tp = x/tfactor;
               x=x-tp*tfactor;
               dp = x/dfactor;
               x=x-dp*dfactor;
               hp = x/hfactor;
               x=x-hp*hfactor;
               wp = x/wfactor;
               if ((0x1 & (wp+hp+dp+tp+ep)) == 0) mult=1.0;
               else mult=-1.0;
               cmplx_data[i].r *= mult; 
               cmplx_data[i].i *= mult; 
               index++;
             }
         }
       kpds_put_data(cmplx_out,KPDS_VALUE_REGION,(kaddr)cmplx_data);
     }
   (void)kpds_close_object(in_obj);
   kfree(cmplx_data);
    
   /* Perform FFT processing along the dimensions in dim_list[] */
   for (num = 0; num < num_dims; num++)
     {
       /* Set up the dimensions of the processing vector for this dimension */
       wu=hu=du=tu=eu=1;
       if (dim_list[num] == KWIDTH)         wu=wi;
       else if (dim_list[num] == KHEIGHT)   hu=hi;
       else if (dim_list[num] == KDEPTH)    du=di;
       else if (dim_list[num] == KTIME)     tu=ti;
       else if (dim_list[num] == KELEMENTS) eu=ei;

       /* Set units sizes for input object */
       kpds_set_attributes(cmplx_out,
                           KPDS_VALUE_REGION_SIZE,wu,hu,du,tu,eu,
                           KPDS_VALUE_POSITION,0,0,0,0,0,NULL);

       /* The game with cmplx_ref is that we will later read the
          data from cmplx_ref, crunch it, and write it to cmplx_out.
          Since reading from cmplx_ref moves the data pointer, we
          use cmplx_out to write to because it still has the unmodified
          data origin unmolested by the read. */
       kpds_copy_object_attr(cmplx_out,cmplx_ref);

       /* Compute number of units for processing along the axis selected */
       unit_len = wu*hu*du*tu*eu;
       num_units = (wi*hi*di*ti*ei)/unit_len;

/*
       kinfo(KVERBOSE,"lkfft: num_dims:%d num:%d\n",num_dims,num);
       kinfo(KVERBOSE,"lkfft: wi:%d hi:%d di:%d ti:%d ei:%d\n",wi,hi,di,ti,ei);
       kinfo(KVERBOSE,"lkfft: unit_len:%d\n",unit_len);
       kinfo(KVERBOSE,"lkfft: wu:%d hu:%d du:%d tu:%d eu:%d\n",wu,hu,du,tu,eu);
       kinfo(KVERBOSE,"lkfft: num_units:%d\n",num_units);
*/
       for (count=0; count < num_units; count++)
       {
             /* Subsequent axes, read from complex_ref temporary object */
             cmplx_data = (kdcomplex *)kpds_get_data(cmplx_ref,
                                       KPDS_VALUE_REGION, (kaddr)cmplx_data);
             kdcomplex_to_arrays(cmplx_data,unit_len,&real_data,&imag_data);

          /* Call f2c'd and hand hacked IEEE F77 FFT */
          fft842_(&direction,&unit_len,real_data,imag_data);

          if (num == (num_dims-1))
          {
             /* Do whatever scaling has been set up. */
             if (scale_mult != 1.0)
             {
                for (loop_cnt = 0; loop_cnt < unit_len; loop_cnt++)
                {
                   real_data[loop_cnt] /= scale_mult;
                   imag_data[loop_cnt] /= scale_mult;
                }
             }
          }
         
          /* Combine real_data & imag_data into one array */
          kcombine_to_complex(&cmplx_data,unit_len,real_data,imag_data);
          kpds_put_data(cmplx_out,KPDS_VALUE_REGION,(kaddr)cmplx_data);
       }
       if (cmplx_data) kfree(cmplx_data);
       if (real_data)  kfree(real_data);
       if (imag_data)  kfree(imag_data); 
   }

   /* Do Nyquist modulation on output if needed */
   kpds_get_attribute(cmplx_out,KPDS_VALUE_OPTIMAL_REGION_SIZE,
                                &wor,&hor,&dor,&tor,&eor,&numr);
   kpds_set_attribute(cmplx_out,KPDS_VALUE_REGION_SIZE,wor,hor,dor,tor,eor);
   kpds_set_attribute(cmplx_out,KPDS_VALUE_POSITION,0,0,0,0,0);
   kpds_copy_object_attr(cmplx_out,cmplx_ref);
   numo = wor*hor*dor*tor*eor;
   index=0;
   for (num=0; num<numr; num++)
     {
       cmplx_data = (kdcomplex *)kpds_get_data(cmplx_out,
                                      KPDS_VALUE_REGION, (kaddr)cmplx_data);
       if (direction == 1 && center) /* Nyquist modulation */
         {
           for (i=0; i<numo; i++)
             {
               x = index;            /* Offset into linear storage area */
               ep = x/efactor;     /* Inverse mapping function */
               x=x-ep*efactor;
               tp = x/tfactor;
               x=x-tp*tfactor;
               dp = x/dfactor;
               x=x-dp*dfactor;
               hp = x/hfactor;
               x=x-hp*hfactor;
               wp = x/wfactor;
               if ((0x1 & (wp+hp+dp+tp+ep)) == 0) mult=1.0;
               else mult=-1.0;
               cmplx_data[i].r *= mult;
               cmplx_data[i].i *= mult;
               index++;
             }
         }
       kpds_put_data(cmplx_ref,KPDS_VALUE_REGION,(kaddr)cmplx_data);
     }
   kfree(cmplx_data);

   (void) kpds_close_object(cmplx_out);
   (void) kpds_close_object(cmplx_ref);

   return(TRUE);
}
/* -library_code_end */
