 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kcompare
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkcompare
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/*
 * Operation structure declaration
 */
struct _operation
{
        char    *name;
	int     routine;
};

/*
 * Function declaration list
 */
static struct _operation operations[] =
{
#define KEQ     0
    { "eq",      KEQ },
#define KNE     1
    { "ne",      KNE  },
#define KGT     2
    { "gt",      KGT },
#define KGE     3
    { "ge",      KGE },
#define KLT     4
    { "lt",      KLT },
#define KLE     5
    { "le",      KLE },
};
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkcompare - perform comparison operation on data object(s)
* 
*       Purpose: The following is the library routine that performs
*		 a comparison operation on two data objects.
*		 The routine takes two input data object (or a single data
*		 object and a constant), a function name, and an output
*		 data object in which to store the result of the
*		 function.
*
*		 The supported functions are:
*                .TS 
*                center tab(:) ;
*                l l .
*		 eq  :  o = i1 == (i2 or val)
*		 ne  :  o = i1 != (i2 or val)
*		 gt  :  o = i1 >  (i2 or val)
*		 ge  :  o = i1 >= (i2 or val)
*		 lt  :  o = i1 <  (i2 or val)
*		 le  :  o = i1 <= (i2 or val)
*                .TE
*
*		 Therefore if lkcompare() were called with two input data
*		 objects:
*
*		 !    lkcompare(i1, i2, 0.0, 1.0, 0.0, "eq", o);
*
*		 then the following would be performed:
*
*		 !    o = (i1 == i2) ? 1.0 : 0.0
*
*		 If lkcompare were called with a single data object and
*		 a constant:
*
*		 !    lkcompare(i1, NULL, 10.0, 1.0, 0.0, "eq", o);
*
*		 then the following would be performed:
*
*		 !    o = (i1 == 10.0) ? 1.0 : 0.0
*
*         Input: i1   - the input object to be processed
*		 i2   - the second input object (if NULL then uses the
*			 constant "val")
*		 val  - the constant to be instead of the second data object
*		 function - the function to be applied
*		 tval - the value to be assigned if comparison is TRUE
*		 fval - the value to be assigned if comparison is FALSE
*		 tol  - the plus/minus tolerence in which to do the comparison
*			(not functional yet)
*        Output: o    - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions: 
*    Written By: Donna Koechner & Mark Young
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkcompare (
   kobject i1,
   kobject i2,
   double  val,
   char    *function,
   double  tval,
   double  fval,
   double  tol,
   kobject o)
/* -library_def_end */

/* -library_code */
{
	double	*idata1 = NULL;
        double  *idata2 = NULL;
        double  *odata  = NULL;
	int	routine, j, indx, num;
        int     num_lines,line_size;
        int	type1, type2, w1, w2, h1, h2,
		d1, d2, t1, t2, e1, e2;


	if ((i1 = kpds_reference_object(i1)) == KOBJECT_INVALID)
	{
	   kerror("kdatamanip", "lkcompare",
		  "failed to reference input object");
	   return(FALSE);
	}

        if (i2 != NULL && ((i2 = kpds_reference_object(i2)) == KOBJECT_INVALID))
        {
	   (void) kpds_close_object(i1);
           kerror("kdatamanip", "lkcompare",
		  "failed to reference input object 2");
           return(FALSE);
        }

        for (num = 0, indx = -1; num < knumber(operations); num++)
        {
           if (kstrcmp(function, operations[num].name) == 0)
           {
              indx = num;
              break;
           }
        }

        if (indx == -1)
        {
	   (void) kpds_close_object(i1);
	   (void) kpds_close_object(i2);
           kerror("kdatamanip", "lkcompare", "Sorry, but the function \
'%s' is not in the list of callable operations", function);
           return(FALSE);
        }
        else if (operations[indx].routine == ERROR)
        {
	   (void) kpds_close_object(i1);
	   (void) kpds_close_object(i2);
           kerror("kdatamanip", "lkcompare", "Sorry, but i am unable to \
run the operation '%s' on data services objects at this time.", function);
           return(FALSE);
        }
        else
           routine = (int) operations[indx].routine;


	/*
	 *  Decide the data type to be used in processing and set it for
	 *  i1, i2, and o.
	 */ 
	kpds_get_attribute(i1, KPDS_VALUE_DATA_TYPE, &type1);

        if (i2 != NULL)
        {
	   kpds_get_attribute(i2, KPDS_VALUE_DATA_TYPE, &type2);
           type1 = kdatatype_cast_process(type1, type2, KANYTYPE);
        }

        kpds_set_attribute(o, KPDS_VALUE_DATA_TYPE, type1);

	/*
	 *  Get the maximum size of the value segment for i1 & i2 and make
	 *  sure that i1, i2, o are the same.
	 */
	kpds_get_attribute(i1, KPDS_VALUE_SIZE, &w1, &h1, &d1, &t1, &e1);
        if (i2 != NULL)
        {
	   kpds_get_attribute(i2, KPDS_VALUE_SIZE, &w2, &h2, &d2, &t2, &e2);

	   w1 = kmax(w1, w2); h1 = kmax(h1, h2); d1 = kmax(d1, d2);
	   t1 = kmax(t1, t2); e1 = kmax(e1, e2);

	   kpds_set_attribute(i2, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);

	   kpds_set_attribute(i1, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);
        }

        kpds_set_attribute(o, KPDS_VALUE_SIZE, w1, h1, d1, t1, e1);

        if ((o = kpds_reference_object(o)) == KOBJECT_INVALID)
        {
           (void) kpds_close_object(i1);
           if (i2) (void) kpds_close_object(i2);
           kerror("kdatamanip","lkcompare",
                  "Failed to reference output object.");
           return(FALSE);
        }
 
	type1 = KDOUBLE;
	kpds_set_attribute(i1, KPDS_VALUE_DATA_TYPE, type1);
        if (i2 != NULL)
        {
	   kpds_set_attribute(i2, KPDS_VALUE_DATA_TYPE, type1);
        }
	kpds_set_attribute(o, KPDS_VALUE_DATA_TYPE, type1);

	/*
	 *  Do the processing by simply requesting the lines of data from
	 *  i1 & i2 and then evaluate the symbol list and then store the
	 *  result into the output. 
	 */

        kpds_get_attribute(i1, KPDS_VALUE_LINE_INFO, &line_size, &num_lines);

        odata  = (double *) kmalloc(line_size * sizeof(double));

	for (num = 0; num < num_lines; num++)
	{
	   idata1 = (double *) kpds_get_data(i1, KPDS_VALUE_LINE,
			(kaddr) idata1);
      
           if (i2 != NULL)
           {
	      idata2 = (double *) kpds_get_data(i2, KPDS_VALUE_LINE,
                	(kaddr) idata2);
           }

	   for (j = 0; j < line_size; j++)
	   {
	      switch (routine)
	      {
		  case KEQ:
		       odata[j] = (idata1[j] == ((!i2) ? val : idata2[j]) ?
					tval : fval);
		       break;

		  case KNE:
		       odata[j] = (idata1[j] != ((!i2) ? val : idata2[j]) ?
					tval : fval);
                       break;

		  case KGT:
		       odata[j] = (idata1[j] > ((!i2) ? val : idata2[j]) ?
					tval : fval);
                       break;

		  case KGE:
		       odata[j] = (idata1[j] >= ((!i2) ? val : idata2[j]) ?
					tval : fval);
                       break;

		  case KLT:
		       odata[j] = (idata1[j] < ((!i2) ? val : idata2[j]) ?
					tval : fval);
                       break;

		  case KLE:
		       odata[j] = (idata1[j] <= ((!i2) ? val : idata2[j]) ?
					tval : fval);
                       break;
	      }
	   }

	   if (!kpds_put_data(o, KPDS_VALUE_LINE, (kaddr) odata))
	   {
	      (void) kpds_close_object(i1);
	      if (i2) (void) kpds_close_object(i2);
              (void) kpds_close_object(o);

              kfree(idata1);
              kfree(idata2);
              kfree(odata);

	      kerror("kdatamanip", "lkcompare",
		     "failed to put line in output object");
	      return(FALSE);
	   }
	}
	(void) kpds_close_object(i1);
	if (i2) (void) kpds_close_object(i2);
        (void) kpds_close_object(o);
        kfree(idata1);
        kfree(idata2);
        kfree(odata);

	return(TRUE);
}
/* -library_code_end */
