 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kcmplx
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkcmplx
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/*
 * Function structure declaration
 */
struct _function
{
   kstring      name;
   kdcomplex  (*routine)(kdcomplex);
};

/*
 * Function declaration list
 */
static struct _function functions[] =
{
    { "conj",    kdcconj },
    { "p2r",     kdcp2r  },
    { "r2p",     kdcr2p  },
};
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkcmplx - perform a unary complex function on data object
* 
*       Purpose: The following is the library routine that performs
*		 a unary complex function on a data object.  The routine takes
*		 an input data object, a function name, and an output
*		 data object in which to store the result of the
*		 function.
*
*		 The supported functions are:
*                .TS 
*                center tab(:) ;
*                l l .
*		 conj :  Output is Complex Conjugate of the input
*		 p2r  :  Output is Polar Coordinates to Rectangular
*		      :  Coordinates of the input
*		 r2p  :  Output is Rectangular Coordinates to Polar
*		      :  Coordinates of the input
*		 .TE
*
*		 Therefore if lkcmplx() were called with:
*
*		 !    lkcmplx(i, "p2r", o);
*
*		 then the following would be performed:
*
*		 !    o = polar_to_rect(i);
*
*         Input: i        - the input object to be processed
*		 function - the function to be applied
*        Output: o        - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions: 
*    Written By: Donna Koechner, Ashish Malhotra, Mark Young, Jeremy Worley
*          Date: Apr 26, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkcmplx(
   kobject  i,
   kstring  function,
   kobject  o)
/* -library_def_end */

/* -library_code */
{
   struct _function *func     = NULL;
   kdcomplex        *dc_data  = NULL;
   kaddr             data     = NULL;
   kdcomplex        *c_data;
   int               k;
   int               j;
   int               w, h, d, t, e;
   int               datatype;
   int               num_regions;
   int               region_size;
   kstring           prim;
   kstring           size;
   kstring           type;
   kstring           rsize;
   kstring           orsize;

   
   for (j = 0; j < knumber(functions) && func == NULL; j++)
      func = kstrcmp(function, functions[j].name) == 0 ? &functions[j] : func;
      
   kinfo(KVERBOSE, "Function value = %d",func);

   if (func == NULL)
   {
      kerror("kdatamanip", "lkcmplx", "Sorry, but the function '%s' is not in "
	     "the list of callable functions", function);
      return FALSE;
   }

   if (func->routine == NULL)
   {
      kerror("kdatamanip", "lkcmplx", "Sorry, but i am unable to run the "
	     "function '%s' on data services objects at this time.", function);
      return FALSE;
   }

   /*
    * reference the input object to prevent side effects
    */
   if ((i = kpds_reference_object(i)) == KOBJECT_INVALID)
   {
      kerror("kdatamanip", "lkcmplx", "failed to reference input object");
      return FALSE;
   }

   /*
    * determine if we're dealing with mapped data or value data.
    * depending on what kind of beast we're dealing with, set the appropriate
    * primitives and attributes to bang on.
    */
   if (kpds_query_map(i))
   {
      prim   = KPDS_MAP_REGION;
      size   = KPDS_MAP_SIZE;
      orsize = KPDS_MAP_OPTIMAL_REGION_SIZE;
      rsize  = KPDS_MAP_REGION_SIZE;
      type   = KPDS_MAP_DATA_TYPE;
   }
   else
   {
      prim   = KPDS_VALUE_REGION;
      size   = KPDS_VALUE_SIZE;
      orsize = KPDS_VALUE_OPTIMAL_REGION_SIZE;
      rsize  = KPDS_VALUE_REGION_SIZE;
      type   = KPDS_VALUE_DATA_TYPE;
   }
   
   kpds_get_attributes(i, type, &datatype, size, &w, &h, &d, &t, &e, NULL);

   if ((datatype != KCOMPLEX) && (datatype != KDCOMPLEX))
   {
      kerror("kcmplx", "lkcmplx",
	     "Cannot operate on non-complex data types.");
      return FALSE;
   }

   /*
    * set the size of the output value segment according to the input.  This
    * is different that what may have occurred in the calling routine because
    * the mapping mode of the input object is set to KMAPPED which means
    * that potentially the size is different.
    */
   kpds_set_attribute(o, size, w, h, d, t, e);
   
   /* 
    * output of complex calculation is real if the func variable is greater
    * than two.
    */
   kpds_set_attribute(o, type, datatype);

   /*
    * now that the output data object has its data type and size set
    * to what it should be, reference the object to prevent further
    * changes to the physical layer. 
    */
   if ((o = kpds_reference_object(o)) == KOBJECT_INVALID)
   {
      kerror("kdatamanip", "lkcmplx", "failed to reference output object");
      return FALSE;
   }

   /*	
    *  Get the size of the line primitive for the input
    */
   kpds_get_attribute(i, orsize, &w, &h, &d, &t, &e, &num_regions);

   region_size = w * h * d * t * e;
   
   /*
    * Set the (presentation) type to be KDCOMPLEX on the input, and either
    * KDOUBLE or KDCOMPLEX on output.  Also, set the region size on the
    * inputs and outputs to the appropriate sizes.
    */ 
   kpds_set_attributes(i, type, KDCOMPLEX, rsize, w, h, d, t, e, NULL);

   kpds_set_attributes(o, type, KDCOMPLEX, rsize, w, h, d, t, e, NULL);

   /*
    * get a sample of the output data set so that when processing below,
    * the data set is already allocated.  Also, set double and complex
    * pointers to the data set for tightness of code.
    */
   data   = kpds_get_data(o, prim, NULL);
   c_data = (kdcomplex *)data;
   
   /*
    *  Do the processing by simply requesting the regions of data from
    *  it and then evaluate the expression and then storing the result
    *  into the output. 
    */
   for (k = 0; k < num_regions; k++)
   {
      dc_data = (kdcomplex *)kpds_get_data(i, prim, dc_data);

      for (j = 0; j < region_size; j++)
	 c_data[j] = func->routine(dc_data[j]);
      
      if (!kpds_put_data(o, prim, data))
      {
	 kerror("kdatamanip", "lkcmplx",
		"failed to put line in output object");
	 kfree(dc_data);
	 kfree(data);
	 kpds_close_object(i);
	 kpds_close_object(o);
	 return FALSE;
      }
   }

   kfree(dc_data);
   kfree(data);
   kpds_close_object(i);
   kpds_close_object(o);

   return TRUE;
}
/* -library_code_end */
