 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kblend
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkblend
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkblend - blend two data objects using the specified ratio
* 
*       Purpose: lkblend blends two objects in the specified ratio.
*                The blending ratio of the objects can be specifed as
*                a constant or an input object can be used for supplying
*                the ratios. The constant is used only if the ratio 
*                object is NULL. An optional input object can be used 
*                to specify regions in which the operation is not to be 
*                performed. The value segment of the gating object
*                is used to determine if the blending on a point is to
*                be gated. 
*
*         Input: src1_obj  - first source object to be blended
*                src2_obj  - second source object to be blended
*                gate_obj  - optional operation gating object
*                ratio_obj - optional object specifying blending ratios
*                ratio     - value of the blending ratio 
*
*        Output: dest_obj  - destination object to be written into
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkblend(kobject src1_obj,
            kobject src2_obj,
            kobject gate_obj,
            kobject ratio_obj,
            double  ratio,
            kobject dest_obj)
/* -library_def_end */

/* -library_code */
{
        int  src1_map_exists, src1_val_exists, src1_mask_exists;
        int  src1_time,src1_loc,src2_time,src2_loc;
        int  src2_map_exists, src2_val_exists, src2_mask_exists;
        int  gate_map_exists=FALSE, gate_val_exists=FALSE, gate_mask_exists;
        int  ratio_map_exists=FALSE, ratio_val_exists=FALSE, ratio_mask_exists;
        char  *prim,*info,*datasize,*datatype,*padtype,*interptype;
        int  w1,h1,d1,t1,e1,w2,d2,h2,t2,e2;
        int  type1, type2;
        int  error_flag = FALSE;
        int  ignore;
        kdcomplex cmplx_ratio1,cmplx_ratio2;
        int  num_lines,line_size,j;
        int  num;
	int  grid;

        double *idata1 = NULL,*idata2 = NULL;
        kdcomplex *idata1_cmplx = NULL;
        kdcomplex *idata2_cmplx = NULL;

        double *ratio_data= NULL;
        double *gate_data = NULL;
        unsigned char *mask1 = NULL; 
        unsigned char *mask2 = NULL;

        src1_val_exists = kpds_query_value(src1_obj);
        src1_map_exists = kpds_query_map(src1_obj);
        src1_mask_exists = kpds_query_mask(src1_obj);
        src1_time = kpds_query_time(src1_obj);
        src1_loc = kpds_query_location(src1_obj);

        src2_val_exists = kpds_query_value(src2_obj);
        src2_map_exists = kpds_query_map(src2_obj);
        src2_mask_exists = kpds_query_mask(src2_obj);
        src2_time = kpds_query_time(src2_obj);
        src2_loc = kpds_query_location(src2_obj);

        if (gate_obj)
        {
           gate_val_exists = kpds_query_value(gate_obj);
           gate_map_exists = kpds_query_map(gate_obj);
           gate_mask_exists = kpds_query_map(gate_obj);
        }
           
        if (ratio_obj)
        {
           ratio_val_exists = kpds_query_value(ratio_obj);
           ratio_map_exists = kpds_query_map(ratio_obj);
           ratio_mask_exists = kpds_query_map(ratio_obj);
        }

	if (src1_loc)
	{
	   if (!kpds_get_attribute(src1_obj, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror("kdatamanip", "lkblend", 
		     "Unable to retrieve location grid attribute.");
	      return(FALSE);
	   }
	   if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				    grid == KCURVILINEAR))
	   {
	      kerror("kdatamanip", "lkblend", 
		     "First source object cannot contain explicit "
		     "rectilinear or curvilinear location data.");
	      return(FALSE);
	   }
	}

	if (src2_loc)
	{
	   if (!kpds_get_attribute(src1_obj, KPDS_LOCATION_GRID, &grid))
	   {
	      kerror("kdatamanip", "lkblend",
		     "Unable to retrieve location grid attribute.");
	      return(FALSE);
	   }
	   if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				    grid == KCURVILINEAR))
	   {
	      kerror("kdatamanip", "lkblend",
		     "Second source object cannot contain explicit "
		     "rectilinear or curvilinear location data.");
	      return(FALSE);
	   }
	}
	
        if (src1_time || src2_time)
        {
           kerror("kdatamanip", "lkblend",
                  "Source object(s) cannot contain time data.");
           return FALSE;
        }

        if (!ratio_val_exists && ratio_obj)
        {
           kerror("kdatamanip", "lkblend",
                  "No value segment in ratio object.");
           return (FALSE);
        }

        if (gate_obj && (!gate_map_exists && !gate_val_exists))
        { 
           kerror("kdatamanip", "lkblend",
                 "Gating object missing value/map data or contains mask data.");
           return (FALSE);
        }

        if (!src1_map_exists && !src1_val_exists)
        {
           kerror("kdatamanip", "lkblend",
                  " No Maps or Value exists for first source object");
           return (FALSE);
        }

        if (!src2_map_exists && !src2_val_exists)
        {
           kerror("kdatamanip", "lkblend",
                  " No Maps or Value exists for second source object");
           return (FALSE);
        }

        if ( (!src2_map_exists && src2_val_exists && 
              !src1_val_exists && src1_map_exists)||
             (src2_map_exists && !src2_val_exists && 
              src1_val_exists && !src1_map_exists)||
             (src2_map_exists && !src2_val_exists && 
              src1_val_exists && src1_map_exists) ||
             (src1_map_exists && !src1_val_exists && 
              src2_map_exists && src2_val_exists) )
        {
           kerror("kdatamanip", "lkblend",
                  "Source objects have illegal combination of segments.");
           return(FALSE);
        }

        if ( (src1_obj = kpds_reference_object(src1_obj)) == KOBJECT_INVALID)
        {
           kerror("kdatamanip","lkblend",
                  "Failed to reference source object 1.");
           return(FALSE);
        }
         
        if ( (src2_obj = kpds_reference_object(src2_obj)) == KOBJECT_INVALID)
        {
           (void) kpds_close_object(src1_obj);

           kerror("kdatamanip","lkblend",
                  "Failed to reference source object 2.");
           return(FALSE);
        }

        if (gate_obj)
        {
           if ( (gate_obj = kpds_reference_object(gate_obj)) == KOBJECT_INVALID)
           {
              (void) kpds_close_object(src1_obj);
              (void) kpds_close_object(src2_obj);
              kerror("kdatamanip","lkblend",
                     "Failed to reference gating object.");
              return(FALSE);
           }
        }

        if (ratio_obj)
        {
           if ((ratio_obj=kpds_reference_object(ratio_obj)) == KOBJECT_INVALID)
           {
              (void) kpds_close_object(src1_obj);
              (void) kpds_close_object(src2_obj);
              if (gate_obj) (void) kpds_close_object(gate_obj);
              kerror("kdatamanip","lkblend",
                     "Failed to reference gating object.");
              return(FALSE);
           }
        }

        if ((src1_val_exists && src2_val_exists &&
           (src1_map_exists || src2_map_exists)))  
        {
           src1_map_exists = src2_map_exists = FALSE;
           kpds_set_attribute(src1_obj, KPDS_MAPPING_MODE, KMAPPED);
           kpds_set_attribute(src2_obj, KPDS_MAPPING_MODE, KMAPPED);
           if (gate_obj)
              kpds_set_attribute(gate_obj, KPDS_MAPPING_MODE, KMAPPED);
           if (ratio_obj)
              kpds_set_attribute(ratio_obj, KPDS_MAPPING_MODE, KMAPPED);
        }

        if (src1_val_exists || src2_val_exists)
        {
           if (!kpds_query_value(dest_obj))
              kpds_create_value(dest_obj); 
        }

        if (src1_mask_exists || src2_mask_exists)
        {
           if (!kpds_query_mask(dest_obj))
              kpds_create_value(dest_obj);
        }

        if (src1_map_exists && !src1_val_exists && 
            src2_map_exists && !src2_val_exists)
        {
           if ((gate_obj && !gate_map_exists) ||
              (ratio_obj && !ratio_map_exists))
           {
              kerror("kdatamanip","lkblend",
                     "Gate/Ratio object needs map data.");
              error_flag = TRUE;
           }

           prim = KPDS_MAP_LINE;
           info = KPDS_MAP_LINE_INFO;
           datasize   = KPDS_MAP_SIZE;
           datatype   = KPDS_MAP_DATA_TYPE;
           padtype    = KPDS_MAP_PAD_VALUE;
           interptype = KPDS_MAP_INTERPOLATE;
        }
        else
        {
           prim = KPDS_VALUE_LINE;
           info = KPDS_VALUE_LINE_INFO;
           datasize   = KPDS_VALUE_SIZE;
           datatype   = KPDS_VALUE_DATA_TYPE;
           padtype    = KPDS_VALUE_PAD_VALUE;
           interptype = KPDS_VALUE_INTERPOLATE;
        }

        kpds_set_attributes(src1_obj,
                KPDS_MAP_POSITION,   0,0,0,0,0,
                KPDS_VALUE_POSITION, 0,0,0,0,0,
                interptype, KPAD,
                padtype,0.0,0.0, NULL);

        kpds_set_attributes(src2_obj,
                KPDS_MAP_POSITION,   0,0,0,0,0,
                KPDS_VALUE_POSITION, 0,0,0,0,0,
                interptype, KPAD,
                padtype,0.0,0.0, NULL);

        if (gate_obj)
           kpds_set_attributes(gate_obj,
                   KPDS_MAP_POSITION,   0,0,0,0,0,
                   KPDS_VALUE_POSITION, 0,0,0,0,0,
                   interptype, KPAD,
                   padtype,0.0,0.0, NULL);

        if (ratio_obj)
           kpds_set_attributes(ratio_obj,
                   KPDS_MAP_POSITION,   0,0,0,0,0,
                   KPDS_VALUE_POSITION, 0,0,0,0,0,
                   interptype, KPAD,
                   padtype,0.0,0.0, NULL);
        
        kpds_get_attribute(src1_obj, datasize,&w1,&h1,&d1, &t1, &e1);
        kpds_get_attribute(src2_obj,  datasize,&w2,&h2,&d2,&t2,&e2);
        w1 = kmax(w1, w2); h1 = kmax(h1, h2); d1 = kmax(d1, d2);
        t1 = kmax(t1, t2); e1 = kmax(e1, e2);

        kpds_set_attribute(src1_obj, datasize, w1, h1, d1, t1, e1);
        kpds_set_attribute(src2_obj, datasize, w1, h1, d1, t1, e1);

        kpds_set_attribute(dest_obj, datasize, w1, h1, d1, t1, e1);
        if (gate_obj)
           kpds_set_attribute(gate_obj, datasize, w1, h1, d1, t1, e1);
        if (ratio_obj)
           kpds_set_attribute(ratio_obj, datasize, w1, h1, d1, t1, e1);

        kpds_get_attribute(src1_obj, datatype, &type1);
        kpds_get_attribute(src2_obj, datatype, &type2);

        type1 = kdatatype_cast_output(type1, type2);
        kpds_set_attribute(dest_obj, datatype, type1); 

        if ((dest_obj = kpds_reference_object(dest_obj)) == NULL)
        {
           error_flag = TRUE;
           kerror("kdatamanip","lkblend",
                  "Failed to reference destination object.");
        }

        if (error_flag)
        {
           (void) kpds_close_object(src1_obj);
           (void) kpds_close_object(src2_obj);
           if (gate_obj) (void) kpds_close_object(gate_obj);
           if (ratio_obj) (void) kpds_close_object(ratio_obj);
           return (FALSE);
        }

        type1 = kdatatype_cast_process(type1,type2,KDOUBLE | KDCOMPLEX);

        if (gate_obj)
           kpds_set_attribute(gate_obj, datatype, KDOUBLE);
        if (ratio_obj)
           kpds_set_attribute(ratio_obj, datatype, KDOUBLE);

        kpds_set_attribute(dest_obj, datatype, type1);
        kpds_set_attribute(src1_obj, datatype, type1);
        kpds_set_attribute(src2_obj, datatype, type1);
                            
        kpds_get_attribute(src1_obj, info, &line_size, &num_lines);

        for (num=0; num < num_lines; num++)
        {
           if (type1 == KDOUBLE)
           {
              idata1 =(double *) kpds_get_data(src1_obj, prim, (kaddr) idata1);
              idata2 =(double *) kpds_get_data(src2_obj, prim, (kaddr) idata2);
           }
           else
           if (type1 == KDCOMPLEX)
           {
              idata1_cmplx = (kdcomplex *)kpds_get_data(src1_obj, prim, 
                             (kaddr) idata1_cmplx);
              idata2_cmplx = (kdcomplex *)kpds_get_data(src2_obj, prim, 
                             (kaddr) idata2_cmplx);
           }

           if (gate_obj)
               gate_data = (double *) kpds_get_data(gate_obj, prim, 
                           (kaddr)gate_data); 

           if (ratio_obj)
               ratio_data = (double *) kpds_get_data(ratio_obj, prim, 
                            (kaddr)ratio_data);

           if (src1_mask_exists && (prim == KPDS_VALUE_LINE_INFO))
               mask1 = (unsigned char *) kpds_get_data(src1_obj,
                             KPDS_MASK_LINE,(kaddr)mask1);

           if (src2_mask_exists && (prim == KPDS_VALUE_LINE_INFO))
               mask2 = (unsigned char *) kpds_get_data(src1_obj,
                             KPDS_MASK_LINE,(kaddr)mask2);

           if (!gate_obj && !src1_mask_exists && !src2_mask_exists)
           {
              for (j=0; j<line_size; j++)
              {
                 ratio = (ratio_obj) ? ratio_data[j] : ratio; 

                 if (ratio > 1.0 || ratio < 0.0)
                 {
                    kerror("kdatamanip","kblend",
                           "Ratio object can only contain values between %s",
                           "0 and 1.");
                    error_flag = TRUE;
                    break;
                 }
                 if (type1 == KDOUBLE)
                    idata1[j] = ratio*idata1[j] + (1.0 - ratio)*idata2[j];
                 else
                 if (type1 == KDCOMPLEX)
                 {
                    cmplx_ratio1 = kdccomp(ratio,0.0);
                    cmplx_ratio2 = kdccomp((1.0-ratio),0.0);
                    idata1_cmplx[j] = kdcadd(
                                      kdcmult(cmplx_ratio1,idata1_cmplx[j]),
                                      kdcmult(cmplx_ratio2,idata2_cmplx[j]));
                 }
              }
           }
           else
           {
              for (j=0; j<line_size; j++)
              {
                 ratio = (ratio_obj) ? ratio_data[j] : ratio;
 
                 if (ratio > 1.0 || ratio < 0.0)
                 {
                    kerror("kdatamanip","kblend",
                           "Ratio object can only contain values between %s",
                           "0 and 1.");
                    error_flag = TRUE;
                    break;
                 }
                 ignore = FALSE;
                 
                 if (mask1) 
                    ignore = ignore || !(mask1[j]);
                 if (mask2)
                    ignore = ignore || !(mask2[j]);
                 if (gate_data)
                    if (gate_data[j] == 0.0)
                       ignore = TRUE;
                 
                 if (mask1)
                    if (mask1[j] == 0)
                       idata1[j] = idata2[j];

                 if (!ignore)
                 {
                    if (type1 == KDOUBLE)
                       idata1[j] = ratio*idata1[j] + (1.0 - ratio)*idata2[j];
                    else
                    if (type1 == KDCOMPLEX)
                    {
                       cmplx_ratio1 = kdccomp(ratio,0.0);
                       cmplx_ratio2 = kdccomp((1.0-ratio),0.0);
                       idata1_cmplx[j] = kdcadd(
                                         kdcmult(cmplx_ratio1,idata1_cmplx[j]),
                                         kdcmult(cmplx_ratio2,idata2_cmplx[j]));
                    }   
                 }
              } 
           }
           if (type1 == KDOUBLE)
           {
              if (kpds_put_data(dest_obj,prim,(kaddr)idata1) == FALSE)
                 error_flag = TRUE;
           }
           else
           if (type1 == KDCOMPLEX)
           {
              if (kpds_put_data(dest_obj,prim,(kaddr)idata1_cmplx) == FALSE)
                 error_flag = TRUE;
           }
           if (error_flag) break;
        }

        (void) kpds_close_object(src1_obj);
        (void) kpds_close_object(src2_obj);
        (void) kpds_close_object(dest_obj);

        if (gate_obj) (void) kpds_close_object(gate_obj);
        if (ratio_obj) (void) kpds_close_object(ratio_obj);

        if (idata1) kfree(idata1);
        if (idata2) kfree(idata2);
        if (idata1_cmplx) kfree(idata1_cmplx);
        if (idata2_cmplx) kfree(idata2_cmplx);
        if (gate_data) kfree(gate_data);
        if (ratio_data) kfree(ratio_data);
        if (mask1) kfree(mask1);
        if (mask2) kfree(mask2);

        if (error_flag) 
           return FALSE;
	else
           return TRUE;
}
/* -library_code_end */
