 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kbessel
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkbessel
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkbessel - compute bessel function of data object
* 
*       Purpose: Library Routine for kbessel.
*                The following is the library routine that performs
*                the bessel function on a data object. The routine
*                takes an input data object, user-specified function
*                kind, user-specified function order, and an output
*                data object in which to store the result of the 
*                function.
*                
*                Therefore if lkbessel() were called with:
*
*                !       lkbessel(i, 1, 0, o)
*
*                !       o = J0(i)
*
*                and if it were called with:
*
*                !      lkbessel(i, 2, 1, o)
*
*                !       o = Y1(i)
*
*         Input: i        - the input object to be processed
*                fn_type  - bessel fn. of the 1st/2nd kind
*                order    - order of the bessel fn.
*
*        Output: o        - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkbessel (
   kobject i,
   int     fn_type,
   int     order,
   kobject o)
/* -library_def_end */

/* -library_code */
{
        double *idata = NULL;
        int   type,error = FALSE;
        int   num_lines,line_size;
        int   num, j;

        if ((i = kpds_reference_object(i)) == KOBJECT_INVALID)
        {
           kerror("kdatamanip", "lkbessel",
		  "failed to reference input object");
           return(FALSE);
        }

        if ((o = kpds_reference_object(o)) == KOBJECT_INVALID)
        {
           (void) kpds_close_object(i);
           kerror("kdatamanip", "lkbessel",
                  "failed to reference input object");
           return(FALSE);
        }

        /*
         *  Decide the data type to be used in processing and set it for
         *  both i and o.
         */ 
        kpds_get_attribute(i, KPDS_VALUE_DATA_TYPE, &type);

        kpds_set_attribute(i,  KPDS_VALUE_DATA_TYPE, KDOUBLE);
        kpds_set_attribute(o, KPDS_VALUE_DATA_TYPE, KDOUBLE);

        /*
         *  Get the size of the value segment for i
         */

        kpds_get_attribute(i, KPDS_VALUE_LINE_INFO, &line_size, &num_lines);

        /*
         *  Do the processing by simply requesting the lines of data from
         *  i and then evaluate the expression and then storing the
         *  result into the output.
         */

         for (num = 0; num < num_lines; num++)
         {
             idata = (double *) kpds_get_data(i, KPDS_VALUE_LINE,
                                (kaddr) idata);

             for (j = 0; j < line_size; j++)
             {
                 if (fn_type==1)  /* Bessel fn. of the first kind */
                 {
                    if (order==0)
                    {
                       idata[j] = kj0(idata[j]);
                    }
                    else if (order==1)
                    {
                       idata[j] = kj1(idata[j]);
                    }
                    else 
                    {
                       idata[j] = kjn(order,idata[j]);
                    }
                 }
                 else if (fn_type==2)  /* Bessel fn. of the second kind */
                 {
                    if (order==0)
                    {
                       idata[j] = (idata[j] > 0.0) ? ky0(idata[j]) :
                                             KMAXFLOAT;
                    }
                    else if (order==1)
                    {
                       idata[j] = (idata[j] > 0.0) ? ky1(idata[j]) :
                                             KMAXFLOAT;
                    }
                    else 
                    {
                       idata[j] = (idata[j] > 0.0) ? kyn(order,idata[j]) :
                                             KMAXFLOAT;
                    }
                 }
             }

             if (!kpds_put_data(o, KPDS_VALUE_LINE, idata))
             {
               error = TRUE;
               kerror("kdatamanip", "lkbessel",
		      "Failed to put line in output object");
               break;
             }
         }
         kfree(idata);
         (void) kpds_close_object(i);
         (void) kpds_close_object(o);

         if (error)
            return(FALSE);
         else
            return(TRUE);
}
/* -library_code_end */
