#!/bin/sh

#
# Khoros: $Id$
#
# $Log$
#

# Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
# All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.

#************************************************************
#
#  Routine Name: kbugreport - Report a Bug to KRI via Email
#
#       Purpose: This program allows a user to report a bug
#		 in the Khoros system.  It provides a standard
#		 Bug Report form as a template, and executes
#		 their favorite editor (specified by $EDITOR)
#		 to allow them to edit it.
#    Written By: Steven Jorgensen
#          Date: Aug 28, 1994
# Modifications: 
#
#************************************************************/

progname=kbugreport
subject="\"$progname Bug Report\""
email="khoros-bugs@khoros.unm.edu"

# This section begins a useful library of routines that are general enough to
# be used in other sh scripts.
# ------------------------------------------------------------------------
# set up some other standard commands.
# eventually we should do something like the below code for echo,
# check for each of them on the path.  Oh for perl!
 
  cp="cp"
  mv="mv -f"
  rm="rm -f"
 awk="awk"
 sed="sed"
grep="grep"
TMPDIR=${TMPDIR:-/tmp}

# clean up if they interrupt the script.
#----------------------------------------------------------
cleanup()
{
        echo ""
        echo "Tidying up all temporary files for $progname"
        $rm $TMPDIR/$progname*.$$
        echo "bye"
        echo ""
        exit 0
}
 
trap cleanup 2

# build a $echoN variable and $echoEnd combination that will give
# will allow the echon procedure to work well.
FindEchoN()
{
        # build a list of potential echos, starting with sh built-in
        echoEnd=""
        elist="echo"
        for dir in `echo $PATH | $sed -e 's/:/ /g'`; do
                [ -x "$dir/echo" ] && elist="$elist $dir/echo"
        done
 
        # work through list, see if one of them groks -n
        for echo in $elist; do
                result=`$echo -n hello`
                if [ "$result" != "-n hello" ]; then
                        echoN="$echo -n"
                        break
                fi
        done
        if [ "$echoN" = "" ]; then
                echoN=echo
                echoEnd="\c"
                result=`$echoN "testing"$echoEnd`
                if [ "$result" != "testing" ]; then
                        unset echoEnd
                fi
        fi
}

# quitting
quit()
{
        echo "Ok bye.."
        cleanup
}
 
# echo without without return
echon()
{
        $echoN "$* $echoEnd"
}
 
# clear screen function
clr()
{
        clear
}
 
# prompt the user, to see if we should go on
YesNoPrompt()
{
        if [ "$#" -eq 0 ]; then
                echon "Do you wish to continue? (y/n) [y]:"
        else
                echon "$1 (y/n) [y]:"
        fi
        read resp
        if [ "$resp" = "n" -o "$resp" = "N" ]; then
                return 1
        fi
        return 0
}

# email the program to send email
FindMailer()
{
        mailer="/usr/ucb/mail"
        if [ -x $mailer ]; then
                return
        fi
 
        dlist=`echo $PATH | $sed -e 's/:/ /g'`
 
        for dir in $dlist; do
                if [ -x "$dir/mailx" ]; then
                        mailer="$dir/mailx"
                        return
                fi
        done
 
        for dir in $dlist; do
                if [ -x "$dir/mail" ]; then
                        mailer="$dir/mail"
                        return
                fi
        done
        mailer="none"
}
 
# find the date
FindDate()
{
        date_prog="/bin/date"
        if [ -x $date_prog ]; then
                date=`$date_prog`
                return
        fi
 
        dlist=`echo $PATH | $sed -e 's/:/ /g'`
 
        date_prog=""
        for dir in $dlist; do
                if [ -x "$dir/date" ]; then
                        date_prog="$dir/date"
                        break
                fi
        done
        if [ "$date_prog" = "" ]; then
                date="none"
        else
                date=`$date_prog`
        fi
}

# email the program to send email
FindEditor()
{
	if [ "$EDITOR" != "" ]; then
		return
	fi

        dlist=`echo $PATH | $sed -e 's/:/ /g'`

        for dir in $dlist; do
                if [ -x "$dir/vi" ]; then
                        EDITOR="$dir/vi"
                        return
                fi
        done
 
        for dir in $dlist; do
                if [ -x "$dir/emacs" ]; then
                        EDITOR="$dir/emacs"
                        return
                fi
        done
        for dir in $dlist; do
                if [ -x "$dir/pico" ]; then
                        EDITOR="$dir/pico"
                        return
                fi
        done
        EDITOR="none"
}
 
# This section begins the routines that are specific to this shell script
# ------------------------------------------------------------------------
FindFiles()
{
	if [ "$KHOROS_TOOLBOX" = "" ]; then
		cat <<"		EndError"
	The environment variable KHOROS_TOOLBOX has not been set.
	Please use the 'kconfigure' program to correctly set up
	your environment to use Khoros, and rerun kregister.

		EndError
		cleanup
	fi

	for tbfile in `echo $KHOROS_TOOLBOX | $sed -e 's/:/ /g'`; do
		[ -f "$tbfile" ] && tblist="$tblist $tbfile"
	done
	for tbfile in $tblist; do
		topdir=`$grep BOOTSTRAP $tbfile | $awk -F: '{ print $2 }'`
		if [ "$topdir" != "" ]; then
			break
		fi
	done

	if [ "$topdir" = "" ]; then
		echo ""
		echo "Cannot find the top directory for the BOOTSTRAP Toolbox"
		echo "in the file(s) listed in your KHOROS_TOOLBOX"
		echo "variable.  Please set this variable correctly, or"
		echo "use 'kconfigure' to correctly set up your environment"
		cleanup
	fi

	topdir=$topdir/repos

	if [ ! -d $topdir ]; then
		echo ""
		echo "The directory '$topdir' that"
		echo "was found for kregister's support files is not a"
		echo "directory.  This implies that either the BOOTSTRAP"
		echo "entry in your Toolbox file is incorrect, or kregister"
		echo "is not installed in BOOTSTRAP.  Please fix the problem"
		echo "and rerun 'kregister'."
		cleanup
	fi
	bug_file=$topdir/BugReport
	if [ ! -r $bug_file ]; then
		echo ""
		echo "ERROR:  Unable to read one or all of the template files"
		echo "for the registration program.  This implies that the"
		echo "files are either gone or unreadable."
		cleanup
	fi
}

PrintIntro()
{
	clr
	echo "This program provides a general template bug report, and then executes"
	echo "your favorite editor to allow you to fill out the bug report."
	echo ""
	YesNoPrompt "Continue with kbugreport?"

	if [ "$?" -eq 1 ]; then
		quit
	fi
}

ReportBug()
{
	clr
	FindEditor
	$cp $bug_file $TMPDIR/kbugreport.$$
	val=1
	until [ $val -eq 0 ]; do
		if [ "$EDITOR" != "none" ]; then
			$EDITOR $TMPDIR/kbugreport.$$
		else
			echo "Cannot find an editor to call, Please edit the file 'kbugreport_info' in"
			echo "the current directory and email it to 'khoros-bugs@khoros.unm.edu'"
			$cp $TMPDIR/kbugreport.$$ ./kbugreport_info
			cleanup
		fi

		clr
		YesNoPrompt "Are you done editing the Bug Report file?"
		val=$?
	done
}

MailIt()
{
	FindMailer
	echo ""
	echo "About to mail the information to '$email'"
	YesNoPrompt "Ready to send it? "
	if [ "$?" -eq 0 ] ; then
		if [ "$mailer" != "none" ]; then
			$mailer $email <$TMPDIR/kbugreport.$$ >/dev/null 2>&1
		else
			$cp $TMPDIR/kbugreport.$$ ./kbugreport_info
			echo ""
			echo "I cannot find a mailer program that I understand, please"
			echo "send the file 'kbugreport_info' in the current directory"
			echo "to 'khoros-bugs@khoros.unm.edu'."
		fi
	else
		$cp $TMPDIR/kbugreport.$$ ./kbugreport_info
		echo ""
		echo "Ok exiting kbugreport, but I have left a file in the current"
		echo "directory called kbugreport_info with the information you typed"
		echo "into the bug report.  When you have finalized the bugreport"
		echo "please send the file to 'khoros-bugs@khoros.unm.edu'"
	fi
}

FindEchoN
FindFiles
PrintIntro
ReportBug
MailIt
echo ""
echo "Thank you for reporting this bug and supporting Khoros."
$rm $TMPDIR/kbugreport*.$$
