/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>   Public:
   >>>>             kfullpath()
   >>>>             kexpandpath()
   >>>>             ktbpath()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	



/************************************************************
*
*  Routine Name: kfullpath - Expand a environment variables in a path
*
*       Purpose: This function returns the full path to the
*		 user supplied file.  "kfullpath" expands
*		 all environment variables in a path, and
*		 returns a full path back the user.
*		 If the file specified with a ~username, then
*		 the path will be expanded.
*
*         Input: filename    - the file to be expanded
*		 global_dir  - the global directory that will be used as
*			       a prefix to filename.  If it is NULL,
*			       then it will ignore this parameter.
*        Output: return_file - if the return file is not NULL then
*			       this will be where it place the fullpath
*			       to the expanded file.  Otherwise, it will
*			       kmalloc the space required.
*                RETURNS: pointer to return_file if it is not NULL, or
*			  a pointer to a kmalloc'ed string containing
*			  the result.
*
*  Restrictions: 
*    Written By: Mark Young  
*          Date: Jul 03, 1992 13:57
*      Verified:
*  Side Effects: if return_file is NULL, it kmalloc's the return string
* Modifications: Converted from vfullpath in Khoros 1.0 (SJ)
*
*************************************************************/

char *kfullpath(
   char *filename,
   char *global_dir,
   char *return_file)
{
	int i;
	char file[KLENGTH], directory[KLENGTH], identifier[KLENGTH],
	     *dir, *buffer;


	/*
	 *  If the filename is null then error and return NULL
	 */
	if (!filename)
	{
	   errno = KNULL_PARAMETER;
	   kinfo(KSYSLIB, "Error!  NULL input file encountered.");
	   return(NULL);
	}

	/*
	 *  If the filename is  empty then error and return NULL
	 */
	if (kstrcmp(ktransport_identifier(filename, identifier), "@") == 0)
	{
	   if (return_file == NULL)
	      return(kstrdup(filename));
	   else
	   {
	      kstrcpy(return_file, filename);
	      return(return_file);
	   }
	}
	ktransport_token(filename, file);
	buffer = _cleanup_string(file);
	if (kstrlen(buffer) == 0)
	{
	   errno = KINVALID_PARAMETER;
	   kinfo(KSYSLIB, "Error!  Empty input file encountered.");
	   return(NULL);
	}

	/*
	 *  if the filename is a alias representation (with colons in it) &
	 *  if we haven't already created the khoros_alias list, do so now.
	 */
	if (kstrchr(buffer,':'))
	{
	   if ((dir = _kalias_replace(buffer)) == NULL)
	   {
	      errno = KINVALID_ALIAS;
	      kinfo(KSYSLIB, "Error!  Bogus alias encountered.");
	      return(NULL);
	   }
	}
	else
	{
	   if (!global_dir)
	      dir = NULL;
	   else if (kstrlen(global_dir) > 0)
	   {
	      kstrcpy(directory, global_dir);
	      dir = _cleanup_string(directory);
	   }
	   else
	      dir = NULL;

	   /*
	    *  Now check to see if the filename or global_dir contains
	    *  in which case we will need to expand the string first.
	    */
	   if ((buffer = _expand_variable(buffer, NULL)) == NULL)
	      return(NULL);
 
 	   if (dir != NULL)
	   {
	      dir = _expand_variable(dir, NULL);
	      if (!dir) return(NULL);
	   }

	   /*
	    *  Expand the username if the string begins with a '~'.
	    */
	   if (buffer[0] == '~' || buffer[0] == '/' || buffer[0] == '.')
	   {
	      if ((buffer = _expand_tilda(buffer)) == NULL)
	         return(NULL);

	      dir = NULL;
	   }
	   else if (dir != NULL)
	   {
	      dir = _expand_tilda(dir);
	   }

	   /*
	    *  Return the fullpath back to the user.
	    */
	   if (dir != NULL)
	   {
	      i = kstrlen(dir);
	      if (i > 0 && dir[i -1] != '/')
	         kstrcat(dir, "/");

	      kstrcat(dir, buffer);
	   }
	   else
	      dir = buffer;
	}

	if (kstrlen(identifier) == 0)
	   kstrcpy(identifier, dir);
	else
	{
	   kstrcat(identifier, "=");
	   kstrcat(identifier, dir);
	}

	/*NOT A MEMORY LEAK*/
	if (return_file == NULL)
	   return(kstrdup(identifier));
	else
	{
	   kstrcpy(return_file, identifier);
	   return(return_file);
	}
}


/************************************************************
*
*  Routine Name: kexpandpath - Expand a path to it's "true" path
*
*       Purpose: This routine takes an input pathname, and expands
*		 environment variables, khoros variables,  ~'s , and
*		 logical paths to return the true path from / to the
*		 filename specified in the input.  It does this by
*		 calling kfullpath to expand the first three expansions,
*		 and then changing the current working directory to the
*		 one containing the specified file and calling getcwd.
*
*         Input: filename    - the file to be expanded
*		 global_dir  - a directory that can be used as a prefix
*			       to filename.  This parameter can be NULL.
*
*        Output: return_file - a string to put the expanded path into.  If
*			       it is NULL, the result will be kmalloc'ed
*                RETURNS: return_file if it is not NULL, the kmalloc'ed output
*			  string if return_file is NULL, or NULL on error
*
*  Restrictions: kexpandpath will fail if the directory specified does not
*		 already exist.
*    Written By: Mark Young & Tom Sauer & Charlie Gage
*          Date: Jul 03, 1992 13:56
*      Verified:
*  Side Effects: if return_file is NULL, the output string is kmalloc'ed
* Modifications: Converted from vexpandpath in khoros 1.0 (SJ)
*
*************************************************************/

char *kexpandpath(
   char *filename,
   char *global_dir,
   char *return_file)
{
	struct stat buf;
	char *basename, *dirname;
	char fullpath[KLENGTH], dir[KLENGTH], save[KLENGTH];
	char message[KLENGTH];


	/*
	 *  Get the fullpath to the file so that we may expand it
	 */
	if (global_dir != NULL || filename != NULL)
	{
	   if (filename == NULL)
	   {
	      if (kfullpath(global_dir, NULL, fullpath) == NULL)
	         return(NULL);
	   }
	   else
	   {
	      if (kfullpath(filename, global_dir, fullpath) == NULL)
	         return(NULL);
	   }
	}
	else
	   kstrcpy(fullpath, "./");

	/*
	 *  Stat the fullpath to see if it is a directory or file.
	 */
	basename = NULL;
	if (stat(fullpath, &buf) == -1)
	{
	   dirname  = kdirname(fullpath,NULL);
	   basename = kbasename(fullpath,NULL);
	   kstrcpy(fullpath, dirname);
	   kfree(dirname);
	   if (stat(fullpath, &buf) == -1)
	   {
	      kinfo(KSYSLIB, "Error! Unable to expand path: '%s/%s'",
		     fullpath, basename);
	      kfree(basename);
	      return(NULL);
	   }
	}

	if (!S_ISDIR(buf.st_mode))
	{
	   /*
	    *  The fullpath is a file not a directory so get the
	    *  path up to the file
	    */
	   dirname  = kdirname(fullpath,NULL);
	   basename = kbasename(fullpath,NULL);
	   kstrcpy(fullpath, dirname);
	   kfree(dirname);
	}

	/*
	 *  Now get the fullpath to the expanded directory.
	 */
	if (getcwd(save, sizeof(save)) == NULL)
	{
	   errno = KINVALID_DIR;
	   kinfo(KSYSLIB, "Error! Unable to get current directory");
	   kfree(basename);
	   return(NULL);
	}

	if (chdir(fullpath) == -1)
	{
	   errno = KINVALID_DIR;
	   kinfo(KSYSLIB, "Error! Unable to change working directory");
	   kfree(basename);
	   return(NULL);
	}

	if (getcwd(dir, sizeof(dir)) == NULL)
	{
	   kinfo(KSYSLIB, "Error! Unable to get new directory");
	   kfree(basename);
	   return(NULL);
	}

	if (chdir(save) == -1)
	{
	   errno = KINVALID_DIR;
	   kinfo(KSYSLIB, "Unable to restore working directory after \
expanding the path:\n\t'%s'", fullpath);
	   kfree(basename);
	   return(NULL);
	}

	/*
	 *  If basename is not NULL then we need to cancatenate the basename
	 *  back onto the expanded director.
	 */
	if (basename != NULL)
	{
	   kstrcat(dir, "/");
	   kstrcat(dir, basename);
	   kfree(basename);
	}

	if (return_file != NULL)
	   kstrcpy(return_file, dir);
	else
	   return_file = kstrdup(dir);

	return(return_file);
}


/************************************************************
*
*  Routine Name: ktbpath - Expand a environment variables local to a toolbox
*
*       Purpose: This function returns the full path to the
*		 user supplied file.  "ktbpath", like "kfullpath",
*		 expands all environment variables in a path, and
*		 returns a full path back the user.
*		 If the file specified with a ~username, then
*		 the path will be expanded.  The difference
*		 between "ktbpath" and "kfullpath" is that
*		 "ktbpath" tries to expand the relative to the
*		 list of toolboxes.  So for example if the path
*		 handed in is:
*
*		 !      /usr/data/test/tb/repos/Aliases
*
*		 So if the path is inside the toolbox TBTEST which
*		 has the path "/usr/data/test/tb", then the resulting
*		 path will be
*		 handed back as:
*
*		 !     $TBPATH/repos/Aliases
*
*         Input: filename    - the file to be expanded
*        Output: return_file - if the return file is not NULL then
*			       this will be where it place the path
*			       to the expanded file.  Otherwise, it will
*			       kmalloc the space required.
*       Returns: pointer to return_file if it is not NULL, or
*		 a pointer to a kmalloc'ed string containing
*		 the result.
*
*  Restrictions: 
*    Written By: Mark Young  
*          Date: Sep 28, 1993
*      Verified:
*  Side Effects: if return_file is NULL, it kmalloc's the return string
* Modifications: 
*
*************************************************************/

char *ktbpath(
   char *filename,
   char *return_file)
{
	int  i, num, len;
	char **toolboxes, temp1[KLENGTH], temp2[KLENGTH], tbname[KLENGTH];


	/*
	 *  If the filename is null then error and return NULL
	 */
	if (!filename)
	{
	   errno = KNULL_PARAMETER;
	   kinfo(KSYSLIB, "Error!  NULL input file encountered.");
	   return(NULL);
	}

	kexpandpath(filename, NULL, temp1);
	toolboxes = ktoolbox_file_list2array(&num);
	for (i = 0; i < num; i++)
	{
	   ksprintf(tbname, "$%s", toolboxes[i]);
	   kexpandpath(tbname, NULL, temp2); len = kstrlen(temp2);
	   if (kstrncmp(temp1, temp2, len) == 0)
	   {
	      ksprintf(temp2, "%s%s", tbname, &temp1[len]);
	      karray_free(toolboxes, num, NULL);

	      if (return_file == NULL)
	         return(kstrdup(temp2));
	      else
	      {
	         kstrcpy(return_file, temp2);
	         return(return_file);
	      }
	   }
	}
	karray_free(toolboxes, num, NULL);
	return(NULL);
}
