/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Link List Utility Routines
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>                 klist_insert()
   >>>>                 klist_add()
   >>>>                 klist_copy()
   >>>>                 klist_delete()
   >>>>                 klist_free()
   >>>>                 klist_locate()
   >>>>                 klist_locate_clientdata()
   >>>>                 klist_head()
   >>>>                 klist_tail()
   >>>>                 klist_size()
   >>>>                 klist_sort()
   >>>>                 klist_merge()
   >>>>                 klist_split()
   >>>>                 klist_to_array()
   >>>>
   >>>>			klist_filelist()
   >>>>			klist_dirlist()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: klist_insert - insert an entry into the linked list
*       Purpose: Inserts an entry into the linked list.  This is done by
*		 adding the entry to the end of the linked list
*		 (if the list currently exists).  The new list
*		 is then passed back to the calling routine.
*		 The routine first scans the list to make sure the
*		 identifier is not already on the list, if so then we
*		 don't change original list.
*
*		 The position field is used to indicate where in the
*		 list an entry should be inserted:
*
*			KLIST_HEAD - insert at the head of the list
*			KLIST_PREV - insert previous to the current position
*			KLIST_NEXT - insert next from th current position
*			KLIST_TAIL - insert at the tail of the list
*	  Input: list        - The current list in which we will be adding the
*			       entry to
*		 identifier  - The entry identifier to be added to the linked
*			       list
*		 client_data - client data to be associated with the identifier
*		 position    - where in the list to add the entry
*		 duplicate_entries - whether to be allowed multiple occurences
*				     of an identifier on the linked list.
*       Returns: The head of the modified linked list.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_insert(
   klist *list,
   kaddr identifier,
   kaddr client_data,
   int   position,
   int   duplicate_entries)
{
	klist *entry, *head;


	/*
	 *  If duplicate entries aren't allowed then make sure that the
	 *  identifier doesn't currently exist anywhere in the list
	 */ 
	head = klist_head(list);
	if (duplicate_entries == FALSE && klist_locate(head,identifier) != NULL)
	   return(list);

	entry = (klist *) kcalloc(1, sizeof(klist));
	entry->identifier = identifier;
	entry->client_data = client_data;

	/*
	 *  Initialize the list if it is currently NULL
	 */
	if (list == NULL)
	{
	   entry->head = TRUE;
	   list = entry;
	   list->next  =
	   list->prev  = NULL;
	   return(list);
	}

	/*
	 *  Change the linked list to be the head or tail if the position
	 *  indicates that the programmer wants the entry inserted at
	 *  the head or tail of the list.
	 */
	if (position == KLIST_HEAD)
	   list = klist_head(list);
	else if (position == KLIST_TAIL)
	   list = klist_tail(list);

	/*
	 *  Insert the entry depending of whether we are to add the
	 *  entry before or after the list's current position.
	 */
	if (position == KLIST_HEAD || position == KLIST_PREV)
	{
	   /*
	    *  Relinked the list previous to be the new entry and the
	    *  entry's previous to be the list's previous.  The entry's
	    *  next is then set to the list.
	    */
	   if (list->prev != NULL)
	      list->prev->next = entry;
	   entry->prev = list->prev;
	   list->prev  = entry;
	   entry->next = list;

	   /*
	    *  If the list was the head then we are now the head of
	    *  list.
	    */
	   entry->head = list->head; list->head = FALSE;
	}
	else if (position == KLIST_TAIL || position == KLIST_NEXT)
	{
	   /*
	    *  Relinked the list next to be the new entry and the
	    *  entry's next to be the list's next.  The entry's
	    *  previous is then set to the list.
	    */
	   if (list->next != NULL)
	      list->next->prev = entry;
	   entry->next = list->next;
	   list->next  = entry;
	   entry->prev = list;
	}
	return(klist_head(entry));
}


/************************************************************
*
*  Routine Name: klist_add - add an entry into the linked list
*       Purpose: Adds an entry to the linked list.  This is done by
*		 adding the entry to the end of the linked list
*		 (if the list currently exists).  If the list is
*		 currently NULL then the new item is returned as the
*		 head of the list.  If the list is not NULL then
*		 the original list is passed back to the calling
*		 routine.
*
*		 The routine first scans the list to make sure the
*		 identifier is not already on the list, if so then we
*		 don't change original list.  
*	  Input: list       - The current list in which we will be adding the
*			      entry to (if NULL then return the newly
*			      malloc'ed head).
*		 identifier - The entry identifier to be added to the linked
*			      list
*		 client_data - client data to be associated with the identifier
*       Returns: The modified linked list.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_add (
   klist *list,
   kaddr identifier,
   kaddr client_data)
{
	/*
	 *  Call klist_insert() to do the actually work
	 */
	return(klist_insert(list, identifier, client_data, KLIST_TAIL, FALSE));
}


/************************************************************
*
*  Routine Name: klist_copy - copy a linked list into a new linked list
*       Purpose: Copies a linked list.  This is done by
*		 adding the items from one list to a new list.
*	  Input: list - The current linked list to be copied
*       Returns: The new linked list or NULL if we fail.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_copy(
   klist *list)
{
	klist *newlist = NULL;


	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (list == NULL)
	   return(NULL);

	/*
	 *  Race thru the list copying each entry and adding it to the
	 *  new list.
	 */
	do
	{
	   newlist = klist_add(newlist, list->identifier, list->client_data);
	   list = list->next;

	} while (list != NULL && (int) list->head == FALSE);
	return(newlist);
}


/************************************************************
*
*  Routine Name: klist_delete - delete an entry from the linked list
*       Purpose: Delete an entry from the linked list.  This is done by
*		 deleting the entry from the linked list.  The new
*		 list is then passed back to the calling routine.
*	  Input: list       - The current linked list in which the entry will be
*			      deleted.
*		 identifier - the identifier which identifies the entry to
*			      be deleted.
*       Returns: The modified linked list.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_delete(
   klist *list,
   kaddr identifier)
{
	klist *entry;


	/*
	 *  Make sure that the entry exists
	 */ 
	if ((entry = klist_locate(list, identifier)) == NULL)
	   return(list);

	/*
	 *  If the next entry exists then set it to point to our
	 *  previous entry.
	 */
	if (entry->next != NULL)
	{
	   entry->next->prev = entry->prev;

	   /*
	    *  If our entry is the head of the list then make the next
	    *  entry the new head
	    */
	   if ((int) entry->head == TRUE)
	      entry->next->head = TRUE;
	}

	/*
	 *  If the previous entry exists then set it to point to our
	 *  next entry.
	 */
	if (entry->prev != NULL)
	   entry->prev->next = entry->next;

	/*
	 *  Advance the list if the list and entry are one and the
	 *  same.
	 */
	if (entry == list && entry->next != NULL)
	   list = entry->next;
	else if (entry == list)
	   list = entry->prev;

	kfree(entry);
	return(list);
}

/************************************************************
*
*  Routine Name: klist_free - free the entire linked list
*       Purpose: Destroys the linked list by walking thru the list
*                and freeing each entry.  One of the parameters is
*                an element routine pointer that will allow the user
*                to specify a routine to be used to destroy each
*                element of the array.  This routine must have a
*                return type of void.  It must also take a single argument
*                as a parameter; the pointer to the list element to reclaim,
*                which is of the data type klist.
*                If the routine parameter is NULL, the routine uses kfree.
*         Input: list    - the linked list to be destroyed
*                routine - the routine to destroy each entry
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/
void klist_free(
   klist *list,
   kfunc_void routine)
{
	klist *temp;


	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (!list)
	   return;

	/*
	 *  Break the previous entry in case we are destroying the remainder
	 *  of the linked list.
	 */
	if (list->prev != NULL)
	   list->prev->next = NULL;

	list = klist_tail(list);
	while (list != NULL)
	{
	   temp = list->prev;
	   if (routine)
	      routine(list);

	   kfree(list);
	   list = temp;
	}
	return;
}


/************************************************************
*
*  Routine Name: klist_locate - locate an entry in the linked list
*
*       Purpose: Tries to locate an entry on the linked list.  This 
*		 is done by racing thru the linked list until the
*		 desired identifier is found or the end of the
*		 list is encountered.  If the identifier exists
*		 then the klist structure is returned.  If the
*		 identifier doesn't exist then NULL is returned.
*	  Input: list       - The current linked list in which the entry will be
*			      located.
*		 identifier - the identifier which identifies the entry to
*			      be located.
*       Returns: The item's klist entry or NULL if the
*		 identifier is not found.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_locate(
   klist *list,
   kaddr identifier)
{
	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (list == NULL)
	   return(NULL);

	do
	{
	   if (((unsigned *) list->identifier) == ((unsigned *) identifier))
	      return(list);

	   list = list->next;
	} while (list != NULL && (int) list->head == FALSE);
	return(NULL);
}


/************************************************************
*
*  Routine Name: klist_locate_clientdata - locate an entry in the linked list
*					   according to it's client data
*       Purpose: Tries to locate an entry on the linked list according to
*		 it's client data, rather than it's identifier.  This 
*		 is done by racing thru the linked list until the
*		 desired identifier is found or the end of the
*		 list is encountered.  If the identifier exists
*		 then the klist structure is returned.  If the
*		 identifier doesn't exist then NULL is returned.
*	  Input: list       -  The current linked list in which the entry will
*			       be located.
*		 client_data - the client_data which identifies the entry to
*			       be located.
*       Returns: The item's klist entry or NULL if the
*		 identifier is not found.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_locate_clientdata(
   klist *list,
   kaddr client_data)
{
	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (list == NULL)
	   return(NULL);

	do
	{
	   if (((unsigned) list->client_data) == ((unsigned) client_data))
	      return(list);

	   list = list->next;
	} while (list != NULL && (int) list->head == FALSE);
	return(NULL);
}


/************************************************************
*
*  Routine Name: klist_head - locate the head of the linked list
*       Purpose: Tries to locate the head of the linked list.  This
*		 is done by taking to current position and racing
*		 up the previous links until the head is found.
*		 The head klist structure is returned, NULL if the
*		 list is currently empty.
*	  Input: list  -  The current linked list to find the head for.
*       Returns: The item's head entry or NULL if the
*		 current list does not exist.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_head(
   klist *list)
{
	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (list == NULL)
	   return(NULL);

	while (list->prev != NULL && (int) list->head == FALSE)
	   list = list->prev;

	return(list);
}


/************************************************************
*
*  Routine Name: klist_tail - locate the tail of the linked list
*       Purpose: Tries to locate the tail of the linked list, or last
*		 entry in the linked list.  This is done by taking to
*		 current position and racing down the next links
*		 until the tail is found.  The last klist structure
*		 is returned, NULL if the list is currently empty.
*	  Input: list - The current linked list to find the tail for.
*       Returns: The item's tail entry or NULL if the
*		 current list does not exist.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

klist *klist_tail(
   klist *list)
{
	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (list == NULL)
	   return(NULL);

	while (list->next != NULL && (int) list->next->head == FALSE)
	   list = list->next;

	return(list);
}


/************************************************************
*
*  Routine Name: klist_sort - sort the linked list 
*       Purpose: Tries to sort the linked list into ascending order.
*		 This is done by simply racing thru the list swapping
*		 each set of entries depending if the entry's identifier.
*		 If the compare routine is not NULL then the routine
*		 is called with the two identifier and the routine is
*		 expected to return whether the first identifier is less
*		 than, equal to, or greater than the second.  This is
*		 done by returning -1, 0, 1 respectively.
*	  Input: list    - The current linked list to be sorted.
*		 compare - An integer function which performs the comparision
*			   of whether the first identifier is less than (-1),
*			   equal to (0), or greater than (1) the second
*			   identifier.
*		 duplicate_entries - toggle where TRUE allows duplicate entries
*				     and FALSE removes duplicate entries.
*       Returns: The sorted linked list
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*   Declaration: klist *klist_sort(
*                !   klist *list,
*                !   kfunc_int compare,
*                !   int   duplicate_entries)
*************************************************************/

/*ARGSUSED*/
klist *klist_sort(
   klist *list,
   kfunc_int compare,
   int   duplicate_entries)
{
	klist *temp1, *temp2;
	int   i, j, sorted, num, value;


	/*
	 *  Passing in a NULL compare operator should be legal, but for
	 *  now just go ahead and say we are already sorted.
	 */
	if (!compare)
	   return(list);

	num = klist_size(list);
        for (i = 0; i < num; i++)
        {
           sorted = TRUE;
	   temp1 = klist_head(list);
	   temp2 = klist_next(temp1);
           for (j = 1; j < num - i; j++)
           {
	      value = compare(temp1->identifier, temp2->identifier);
              if (value > 0)
              {
		 if (temp1->prev != NULL)
		    temp1->prev->next = temp2;
		 if (temp2->next != NULL)
		    temp2->next->prev = temp1;

		 temp2->prev = temp1->prev;
		 temp1->next = temp2->next;
		 temp1->prev = temp2;
		 temp2->next = temp1;

		 temp2->head = temp1->head;
		 temp1->head = FALSE;
		 temp2 = klist_next(temp1);
                 sorted = FALSE;
              }
              else if (compare == 0 && duplicate_entries == FALSE)
              /*EMPTY*/
              {
		 /* what to do ... what to do */
	         temp1 = klist_next(temp1);
	         temp2 = klist_next(temp2);
              }
	      else
	      {
	         temp1 = klist_next(temp1);
	         temp2 = klist_next(temp2);
	      }
           }
           if (sorted == TRUE) break;
	}
	return(klist_head(list));
}


/************************************************************
*
*  Routine Name: klist_merge - merge two linked list into a single linked list
*       Purpose: Merge two linked list into a single linked by tacking the
*		 second list onto the end of the first.
*	  Input: list1 - The first linked list
*		 list2 - The second linked list
*       Returns: The newly merge linked list, which is really
*		 the head of the first list, or NULL upon failure.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*   Declaration: klist *klist_merge(
*                !   klist *list1,
*                !   klist *list2)
*************************************************************/

/*ARGSUSED*/
klist *klist_merge(
   klist *list1,
   klist *list2)
{
	klist *list = (list1 ? list1 : list2);


	list1 = klist_tail(list1);
	list2 = klist_head(list2);
	if (list1) list1->next = list2;
	if (list2) list2->prev = list1;
	if (list1 && list2) list2->head = FALSE;
	return(list);
}


/************************************************************
*
*  Routine Name: klist_split - split a single linked list into two linked lists
*       Purpose: Split a single linked list into two linked lists.  Given
*		 the entry in which we will break into the head of the
*		 new second list.  Since this is the head of the second
*		 list and the head of the first list is really the
*		 klist_head(entry), before the klist_split() routine is
*		 called, it is not necessary to pass back the heads of
*		 the two lists.  But to make life easier if list1 or
*		 list2 are not NULL the respective heads will be initialized
*		 there.
*         Input: entry - the linked list entry in which we will be performing
*			 the split
*	 Output: list1 - If not NULL, then the head of the first linked list
*			 is passed back.
*		 list2 - If not NULL, then the head of the second linked list
*			 is passed back.
*       Returns: TRUE if we were able to split the linked list or
*		 FALSE upon failure.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*   Declaration: int klist_split(
*                !   klist  *entry,
*                !   klist **list1,
*                !   klist **list2)
*************************************************************/
/*ARGSUSED*/
int klist_split(
   klist *entry,
   klist **list1,
   klist **list2)
{
	kinfo(KFORCE, "klist_split:  Not available as of yet, sorry!");
	return(FALSE);
}


/************************************************************
*
*  Routine Name: klist_size - compute the size or number of entries in the list
*       Purpose: Given a list will indicate how many entries are
*		 on the list.  This is done by counting the number
*		 of entries until the tail entry is reached.
*	  Input: list - The current linked list to count the number entries for.
*       Returns: The number of entries or zero if a NULL list is passed in.
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

int klist_size(
   klist *list)
{
	int num = 0;


	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if (list == NULL)
	   return(0);

	do
	{
	   num++;
	   list = list->next;
	} while (list != NULL && (int) list->head == FALSE);

	return(num);
}


/************************************************************
*
*  Routine Name: klist_to_array - convert the linked list into an array
*       Purpose: Creates an array from a linked list.  This is
*		 done by malloc'ing an array of pointers and
*		 then loading the assigning the identifiers into
*		 each array.
*	  Input: list - The current linked list in which the array
*			of identifiers will be created.
*        Output: num  - The number of entries in the array.
*       Returns: The newly malloc'ed array or NULL upon failure
*    Written By: Mark Young
*          Date: Sep 09, 1992 10:05
*************************************************************/

char **klist_to_array(
   klist *list,
   int   *num)
{
	int  i;
	char **array;


	/*
	 *  Sanity check to make sure that the linked list is not NULL
	 */
	if ((*num = klist_size(list)) == 0)
	   return(NULL);

	/*
	 *  Malloc the array to hold the identifiers
	 */
	if ((array = (char **) kmalloc(*num * sizeof(char *))) == NULL)
	{
	   return(NULL);
	}

	/*
	 *  Copy each list entry identifier into the array
	 */
	for (i = 0; i < *num; i++, list = list->next)
	   array[i] = list->identifier;

	return(array);
}


/************************************************************
*
*  Routine Name: klist_filelist - create a linked list of strings from a file
*       Purpose: This module is used to create a linked list from the
*		 supplied file.  It reads the lines out of the
*		 specified file, and returns them in a linked list of
*		 strings.  The programmer also has the option of
*		 specifying if they want the resulting string sorted or not.
*         Input: filename    - base name of the files to be listed.
*		 global_dir  - the global directory is used as a prefix
*			       to the filename
*		 sort_flag   - a flag indicating whether we are to sort
*			       the link list before returning
*       Returns: The head of the linked list to the names in the file upon
*		 successfull completion of this routine, otherwise NULL is
*		 returned.
*    Written By: Mark Young  
*          Date: Nov 28, 1992 14:36
*************************************************************/

klist *klist_filelist(
   char *filename,
   char *global_dir,
   int  sort_flag)
{
	klist   *list;
	int	num;
	char	**array;


	/*
	 *  Call karray_filelist() to do the actually work.  Then simply convert
	 *  the return array to a link list and free the old array pointer.
	 */
	if ((array = karray_filelist(filename, global_dir,
				     sort_flag, &num)) == NULL)
	{
	   return(NULL);
	}

	/*
	 *  Convert the array to a linked list and then free the pointer
	 *  to the array of strings.
	 */
	list = karray_to_list(array, num); kfree(array);
	return(list);
}


/************************************************************
*
*  Routine Name: klist_dirlist - create a linked list of file names
*       Purpose: This module is used to create a linked list of file
*		 names according to a user supplied basename and an
*		 initial global directory.  The list mode is used
*		 to indicate what we are going to list in the
*		 directory.  The possible defines are listed in
*		 $KHOROS/include/khoros/kdefines.h.  The following
*		 symbols are the current list mode:
*
*		 !	KPATH		- prepend the path to each file
*		 !	KFILE		- list plain text files
*		 !	KDIR		- list directories
*		 !	KDOT		- list dot files
*		 !	KLINK		- list symbolic files
*		 !	KSOCK		- list socket files
*		 !	KREAD		- file is readable by caller
*		 !	KWRITE		- file is writable by caller
*		 !	KEXEC		- file is executable by caller
*
*		The selections are or'ed together in order to choose
*		the set of attributes that are desired.  (e.g KFILE | KDIR)
*		will list only files and directories that match the basename.
*         Input: basename    - base name of the files to be listed.
*		 global_dir  - the global directory is used as a prefix to
*			       basename
*		 filter	     - the filter is used in matching the directory
*			       entries.  The syntax used is the same as the
*			       "kparse" utilities.  If filter is NULL then
*			       all entries are accepted.
*		 list_mode   - a flag indicating what we are to list
*			       (ie. files, directories, dot files, symbolic
*				links, etc.)
*		 format      - whether the files should be prepended with
*			       type of file. (ie. "@" for sym links, "/" for
*			       directories)
*       Returns: A link list of strings to filenames that match the basename
*    Written By: Mark Young
*          Date: Nov 28, 1992 14:39
*************************************************************/

klist *klist_dirlist(
   char *basename,
   char *global_dir,
   char *filter,
   int  list_mode,
   int  format)
{
	klist   *list;
	int	num;
	char	**array;


	/*
	 *  Call karray_dirlist() to do the actually work.  Then simply convert
	 *  the return array to a link list and free the old array pointer.
	 */
	if ((array = karray_dirlist(basename, global_dir, filter, list_mode,
			format, &num)) == NULL)
	{
	   return(NULL);
	}

	/*
	 *  Convert the array to a linked list and then free the pointer
	 *  to the array of strings.
	 */
	list = karray_to_list(array, num); kfree(array);
	return(list);
}
