/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Conversion Routines
   >>>>
   >>>>      Private:
   >>>>               convert_order()
   >>>>               convert_signed()
   >>>>               convert_unsigned()
   >>>>
   >>>> Semi-Private:
   >>>>               convert_float()
   >>>>               convert_double()
   >>>>               convert_complex()
   >>>>               convert_dcomplex()
   >>>>               convert_short()
   >>>>               convert_ushort()
   >>>>               convert_int()
   >>>>               convert_uint()
   >>>>               convert_long()
   >>>>               convert_ulong()
   >>>>               convert_byte()
   >>>>               convert_ubyte()
   >>>>
   >>>>       Public: 
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "kconvert.h" 


/*---------------------------------------------------------------------
|
|  Routine Name: convert_order()
|
|       Purpose: converts a string of MSB or LSB words of any length to
|                another order.  i.e. this can be used to convert MSB
|                words to LSB or LSB to MSB, etc.
|
|         Input: data           - input data cast to unsigned char
|		 src_mach       - source machine type
|		 dest_mach      - destination machine type 
|                num_elem       - number of nbyte elements in the string
|                bytes_per_elem - number of bytes per element.
|
|        Output: data           - reordered data.
|
|        Status: Private Routine
|
|  Restrictions: none
|
|    Written By: Mark Young, Jeremy Worley 
|
|          Date: Sat Sep 21 10:26:06 MDT 1991
|
| Modifications:
|
----------------------------------------------------------------------*/

static int convert_order(
   register unsigned char *data,
   unsigned int           src_order,
   unsigned int           dest_order,
   unsigned int           num_elem,
   unsigned int           bytes_per_elem)
{
   register unsigned int i,j,indx1,indx2,num_swaps;
   unsigned char tmp;

   if (src_order != dest_order){
      num_swaps = bytes_per_elem >> 1;
      if(num_swaps==0)return(1);

      for(i=0,indx1=0;i<num_elem;i++){
         indx2 = indx1 + (bytes_per_elem-1);
         for(j=0;j<num_swaps;j++){
	    tmp = data[indx1];
            data[indx1++] = data[indx2];
            data[indx2--] = tmp;
         }
         indx1 += num_swaps;
      }
   } /* end switch */
   return(1);
}

/*-------------------------------------------------------------------
|
|  Routine Name: convert_signed()
|        
|       Purpose: Converts a vector of signed integer information 
|                of any wordsize originating on one architecture 
|                to another architecture. 
|
|         Input: inptr       - address of the input data cast to 
|                              (char *)
|                outptr      - address of the output data cast to 
|                              (char *) or NULL, which indicates that 
|                              memory should should be allocated.
|                srctype     - unsigned int specifying source 
|                              architecture.
|                desttype    - another unsigned int specifying the 
|                              destination (usually current) architecture.
|                inwordsize  - size of one signed integer word in the
|                              input
|                outwordsize - size of one signed integer word in the
|                              output
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer 
|                if outptr == NULL
|
|        Status: Private
|
|  Restrictions: 
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

static unsigned char *convert_signed(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  inwordsize,
   unsigned int  outwordsize,
   unsigned int  num)
{
   unsigned char *tmpin, *tmpout, *output;
   unsigned int i;
   long cnt,cur,sign;
   long srcorder,destorder;
 
   srcorder = kmach_order(srctype);
   destorder = kmach_order(desttype);

   /*
    * if the outptr is NULL, then that means that this routine must
    * allocate space that is returned at exit.
    */

   if(outptr == NULL){
      if((output = (unsigned char *)malloc(num*outwordsize))==NULL)
         return(NULL);
   }else
      output = outptr;

   tmpout = output;
   tmpin = inptr;

   /*
    * sign extend nonsense here.
    */

    if(inwordsize > outwordsize){
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)srcorder,
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
      for(i=0;i<num;i++){
	cnt = outwordsize;
	sign = (*tmpin) & 0x80;
        tmpin += inwordsize - outwordsize;
        while(cnt--) *tmpout++ = *tmpin++;
	*(tmpout-outwordsize) = (*(tmpout-outwordsize) & 0x7f) | sign;
      }
      i = convert_order((unsigned char *)(output),
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)destorder,
                     (unsigned int)num,
                     (unsigned int)outwordsize);
      tmpin = inptr;
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)srcorder,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
    }else if(inwordsize < outwordsize){
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)srcorder,
                     (unsigned int)KORDER_LITTLE_ENDIAN,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
      tmpin += inwordsize*num - 1;
      tmpout += outwordsize*num - 1;
      for(i=0;i<num;i++){
	cur = outwordsize - inwordsize;
        cnt = outwordsize;
	sign = *tmpin & 0x80;
	while(cnt--) *tmpout-- = (cur-- > 0) ? ((sign)?0xff:0x0) : *tmpin--;
      }
      i = convert_order((unsigned char *)(output),
                     (unsigned int)KORDER_LITTLE_ENDIAN,
                     (unsigned int)destorder,
                     (unsigned int)num,
                     (unsigned int)outwordsize);
      tmpin = inptr;
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)KORDER_LITTLE_ENDIAN,
                     (unsigned int)srcorder,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
    }else{
      if (tmpin != tmpout)
	 kmemcpy(tmpout, tmpin, num*outwordsize);
      i = convert_order((unsigned char *)(output),
                     (unsigned int)srcorder,
                     (unsigned int)destorder,
                     (unsigned int)num,
                     (unsigned int)outwordsize);
   }
   return(output);
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_unsigned()
|        
|       Purpose: Converts a vector of unsigned integer information 
|                of any wordsize originating on one architecture 
|                to another architecture. 
|
|         Input: inptr       - address of the input data cast to 
|                              (char *)
|                outptr      - address of the output data cast to 
|                              (char *) or NULL, which indicates that 
|                              memory should should be allocated.
|                srctype     - unsigned int specifying source 
|                              architecture.
|                desttype    - another unsigned int specifying the 
|                              destination (usually current) architecture.
|                inwordsize  - size of one signed integer word in the
|                              input
|                outwordsize - size of one signed integer word in the
|                              output
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer 
|                if outptr == NULL
|
|        Status: Private
|
|  Restrictions: 
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

static unsigned char *convert_unsigned(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  inwordsize,
   unsigned int  outwordsize,
   unsigned int  num)
{
   unsigned char *tmpin, *tmpout, *output;
   unsigned int i;
   long int cnt,cur;
   long srcorder,destorder;
 
   srcorder = kmach_order(srctype);
   destorder = kmach_order(desttype);

   /*
    * if the outptr is NULL, then that means that this routine must
    * allocate space that is returned at exit.
    */

   if(outptr == NULL){
      if((output = (unsigned char *)malloc(num*outwordsize))==NULL)
         return(NULL);
   }else
      output = outptr;

   tmpout = output;
   tmpin = inptr;

   /*
    * sign extend nonsense here.
    */

    if(inwordsize > outwordsize){
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)srcorder,
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
      for(i=0;i<num;i++){
	cnt = outwordsize;
        tmpin += inwordsize - outwordsize;
        while(cnt--) *tmpout++ = *tmpin++;
      }
      i = convert_order((unsigned char *)(output),
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)destorder,
                     (unsigned int)num,
                     (unsigned int)outwordsize);
      tmpin = inptr;
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)srcorder,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
    }else if(inwordsize < outwordsize){
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)srcorder,
                     (unsigned int)KORDER_LITTLE_ENDIAN,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
      tmpin += inwordsize*num - 1;
      tmpout += outwordsize*num - 1;
      for(i=0;i<num;i++){
	cur = outwordsize - inwordsize;
        cnt = outwordsize;
	while(cnt--) *tmpout-- = (cur-- > 0) ? 0x0 : *tmpin--;
      }
      i = convert_order((unsigned char *)(output),
                     (unsigned int)KORDER_LITTLE_ENDIAN,
                     (unsigned int)destorder,
                     (unsigned int)num,
                     (unsigned int)outwordsize);
      tmpin = inptr;
      i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)KORDER_LITTLE_ENDIAN,
                     (unsigned int)srcorder,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
    }else{
      if (tmpin != tmpout)
	 kmemcpy(tmpout, tmpin, num*outwordsize);
      i = convert_order((unsigned char *)(output),
                     (unsigned int)srcorder,
                     (unsigned int)destorder,
                     (unsigned int)num,
                     (unsigned int)outwordsize);
   }
   return(output);
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_float()
|        
|       Purpose: Converts a vector of float information originating
|                on one architecture to another architecture. 
|
|         Input: inptr  - address of the input data cast to 
|                        (char *)
|                outptr - address of the output data cast to 
|                         (char *) or NULL, which indicates that 
|                         memory should should be allocated.
|                src    - unsigned int specifying source architecture.
|                dest   - another unsigned int specifying the 
|                         destination (usually current) architecture.
|                num    - unsigned int specifying number of float 
|                         elements in *data (NOT the number of bytes 
|                         in the array)
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer 
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions: only works on single precision floats.
|
|    Written By: Jeremy Worley
|
|          Date: Sat Feb 22 11:39:06 MST 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_float(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
   unsigned char *tmpin, *tmpout, *output, c;
   unsigned int i, inwordsize, outwordsize;
   unsigned long sign, man1, man2, tmp1;
   long int expo;
   long srcorder,destorder,srcformat,destformat;
 
   srcorder = kmach_order(srctype);
   destorder = kmach_order(desttype);
   srcformat = kmach_format(srctype);
   destformat = kmach_format(desttype);

   /*
    * determine word size
    */

   inwordsize = kmach_sizeof((int)srctype,KFLOAT);
   outwordsize = kmach_sizeof((int)desttype,KFLOAT);

   /*
    * if the outptr is NULL, then that means that this routine must
    * allocate space that is returned at exit.
    */

   if(outptr == NULL){
      if((output = (unsigned char *)malloc(num*outwordsize))==NULL)
         return(NULL);
   }else
      output = outptr;

   tmpout = output;
   tmpin = inptr;
  
   /*
    * convert the input to "big endian" byte ordering.
    */
   i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)srcorder,
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)num,
                     (unsigned int)inwordsize);
 
   switch(srcformat){
     case KFORMAT_IEEE: 
     case KFORMAT_IEEE_ALPHA: 
          switch(destformat){
            case KFORMAT_VAX:
                 for(i=0;i<num;i++){ 
                     load_ieee_float(tmpin,sign,expo,man1,man2); 
                     unload_vax_float(tmpout,sign,expo,man1,man2); 
                     diddle_vax_float(tmpout);
                     tmpin += inwordsize;
                     tmpout += outwordsize;
                 }
                 break;
            case KFORMAT_CRAY:
                 tmpin += inwordsize*(num - 1); 
                 tmpout += outwordsize*(num - 1); 
                 for(i=0;i<num;i++){ 
                     load_ieee_float(tmpin,sign,expo,man1,man2); 
                     unload_cray_float(tmpout,sign,expo,man1,man2); 
                     tmpin -= inwordsize;
                     tmpout -= outwordsize;
                 }
                 break;
          }
          break;
     case KFORMAT_VAX: 
          switch(destformat){
            case KFORMAT_IEEE: 
            case KFORMAT_IEEE_ALPHA: 
                 for(i=0;i<num;i++){ 
                     diddle_vax_float(tmpin);
                     load_vax_float(tmpin,sign,expo,man1,man2); 
                     unload_ieee_float(tmpout,sign,expo,man1,man2);
                     tmpin += inwordsize; 
                     tmpout += outwordsize;
                 }
                 break;
            case KFORMAT_CRAY:
                 tmpin += inwordsize*(num - 1); 
                 tmpout += outwordsize*(num - 1); 
                 for(i=0;i<num;i++){ 
                     diddle_vax_float(tmpin);
                     load_vax_float(tmpin,sign,expo,man1,man2); 
                     unload_cray_float(tmpout,sign,expo,man1,man2); 
                     tmpin -= inwordsize;
                     tmpout -= outwordsize;
                 }
                 break;
          }
          break;
     case KFORMAT_CRAY: 
          switch(destformat){
            case KFORMAT_IEEE: 
            case KFORMAT_IEEE_ALPHA: 
                 for(i=0;i<num;i++){ 
                     load_cray_float(tmpin,sign,expo,man1,man2);
                     unload_ieee_float(tmpout,sign,expo,man1,man2); 
                     tmpin += inwordsize; 
                     tmpout += outwordsize;
                 }
                 break;
            case KFORMAT_VAX: 
                 for(i=0;i<num;i++){ 
                     load_cray_float(tmpin,sign,expo,man1,man2); 
                     unload_vax_float(tmpout,sign,expo,man1,man2); 
                     diddle_vax_float(tmpout);
                     tmpin += inwordsize; 
                     tmpout += outwordsize;
                 }
                 break;
          }
          break;
   } /* end big switch */


  /*
   * convert the byte ordering back
   */

  i = convert_order((unsigned char *)(output),
                    (unsigned int)KORDER_BIG_ENDIAN,
                    (unsigned int)destorder,
                    (unsigned int)num,
                    (unsigned int)outwordsize);

  /* if(tmpin != tmpout)free(tmpin); */

  return(output);

}

/*-------------------------------------------------------------------
|
|  Routine Name: convert_double()
|        
|       Purpose: Converts a vector of double information originating
|                on one architecture to another architecture. 
|
|         Input: inptr  - address of the input data cast to 
|                         (char *)
|                outptr - address of the output data cast to 
|                         (char *) or NULL, which indicates that 
|                         memory should should be allocated.
|                src    - unsigned int specifying source architecture.
|                dest   - another unsigned int specifying the 
|                         destination (usually current) architecture.
|                num    - unsigned int specifying number of float 
|                         elements in *data (NOT the number of bytes 
|                         in the array)
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer 
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions: no comment
|
|    Written By: Jeremy Worley
|
|          Date: Sat Feb 22 11:39:06 MST 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_double(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
   unsigned char *tmpin, *tmpout, *output, c;
   unsigned int i, inwordsize, outwordsize;
   unsigned long sign, man1, man2, tmp1;
   long int expo;
   long srcorder,destorder,srcformat,destformat;
 
   srcorder = kmach_order(srctype);
   destorder = kmach_order(desttype);
   srcformat = kmach_format(srctype);
   destformat = kmach_format(desttype);

   /*
    * determine word size
    */

   inwordsize = kmach_sizeof((int)srctype,KDOUBLE);
   outwordsize = kmach_sizeof((int)desttype,KDOUBLE);

   /*
    * if the outptr is NULL, then that means that this routine must
    * allocate space that is returned at exit.
    */

   if(outptr == NULL){
      if((output = (unsigned char *)malloc(num*outwordsize))==NULL)
         return(NULL);
   }else
      output = outptr;

   tmpout = output;
   tmpin = inptr;

   /*
    * convert the input to "big endian" byte ordering.
    */
   i = convert_order((unsigned char *)(tmpin),
                     (unsigned int)srcorder,
                     (unsigned int)KORDER_BIG_ENDIAN,
                     (unsigned int)num,
                     (unsigned int)inwordsize);

   switch(srcformat){
     case KFORMAT_IEEE: 
     case KFORMAT_IEEE_ALPHA: 
          switch(destformat){
            case KFORMAT_VAX:
                 for(i=0;i<num;i++){ 
                     load_ieee_double(tmpin,sign,expo,man1,man2); 
                     unload_vax_double(tmpout,sign,expo,man1,man2); 
                     diddle_vax_double(tmpout);
                     tmpin += inwordsize;
                     tmpout += outwordsize;
                 }
                 break;
            case KFORMAT_CRAY:
                 tmpin += inwordsize*(num - 1); 
                 tmpout += outwordsize*(num - 1); 
                 for(i=0;i<num;i++){ 
                     load_ieee_double (tmpin,sign,expo,man1,man2); 
                     unload_cray_double(tmpout,sign,expo,man1,man2); 
                     tmpin -= inwordsize;
                     tmpout -= outwordsize;
                 }
                 break;
          }
          break;
     case KFORMAT_VAX: 
          switch(destformat){
            case KFORMAT_IEEE: 
            case KFORMAT_IEEE_ALPHA: 
                 for(i=0;i<num;i++){ 
                     diddle_vax_double(tmpin);
                     load_vax_double(tmpin,sign,expo,man1,man2); 
                     unload_ieee_double(tmpout,sign,expo,man1,man2);
                     tmpin += inwordsize; 
                     tmpout += outwordsize;
                 }
                 break;
            case KFORMAT_CRAY:
                 tmpin += inwordsize*(num - 1); 
                 tmpout += outwordsize*(num - 1); 
                 for(i=0;i<num;i++){ 
                     diddle_vax_double(tmpin);
                     load_vax_double(tmpin,sign,expo,man1,man2); 
                     unload_cray_double(tmpout,sign,expo,man1,man2); 
                     tmpin -= inwordsize;
                     tmpout -= outwordsize;
                 }
                 break;
          }
          break;
     case KFORMAT_CRAY: 
          switch(destformat){
            case KFORMAT_IEEE: 
            case KFORMAT_IEEE_ALPHA: 
                 for(i=0;i<num;i++){ 
                     load_cray_double(tmpin,sign,expo,man1,man2); 
                     unload_ieee_double(tmpout,sign,expo,man1,man2); 
                     tmpin += inwordsize; 
                     tmpout += outwordsize;
                 }
                 break;
            case KFORMAT_VAX: 
                 for(i=0;i<num;i++){ 
                     load_cray_double(tmpin,sign,expo,man1,man2); 
                     unload_vax_double(tmpout,sign,expo,man1,man2); 
                     diddle_vax_double(tmpout);
                     tmpin += inwordsize; 
                     tmpout += outwordsize;
                 }
                 break;
          }
          break;
   } /* end big switch */

  /*
   * convert the byte ordering back
   */

  i = convert_order((unsigned char *)(output),
                    (unsigned int)KORDER_BIG_ENDIAN,
                    (unsigned int)destorder,
                    (unsigned int)num,
                    (unsigned int)outwordsize);

  /* if(tmpin != tmpout)free(tmpin); */

  return(output);
 }

/*-------------------------------------------------------------------
|
|  Routine Name: convert_complex()
|        
|       Purpose: Converts a vector of single precision complex information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|                srctype  - unsigned int specifying source 
|                           architecture.
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|        Output: 
|       Returns: NULL if any error occured. Otherwise, 
|                it returns outptr if outptr != NULL or a new 
|                pointer if outptr == NULL
|
|      Comments: This is essentially a binding for convert_float that
|                doubles the number of elements to convert...pretyy
|                sly eh?
|
|        Status: Semi-Private
|
|  Restrictions: Really causes problems if you pass integers in.
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_complex(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  return(convert_float(inptr,outptr,srctype,desttype,num<<1));
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_dcomplex()
|        
|       Purpose: Converts a vector of double precision complex information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|                srctype  - unsigned int specifying source 
|                           architecture.
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new if 
|                outptr == NULL
|
|      Comments: This is a binding for convert_double that essentially
|                doubles the number of words to read.
|
|        Status: Semi-Private
|
|  Restrictions: -
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_dcomplex(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  return(convert_double(inptr,outptr,srctype,desttype,num<<1));
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_short()
|        
|       Purpose: Converts a vector of signed short information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr      - address of the input data cast to 
|                             (char *)
|
|                outptr     - address of the output data cast to 
|                             (char *) or NULL, which indicates that 
|                             memory should should be allocated.
|
|                srctype    - unsigned int specifying source 
|                             architecture.
|
|                desttype   - another unsigned int specifying the 
|                             destination (usually current) architecture.
|
|        Output: 
|       Returns: NULL if any error occured. Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions: 
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_short(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KSHORT);
  outsize = kmach_sizeof((int)desttype,KSHORT);
  return(convert_signed(inptr,outptr,srctype,desttype,insize,outsize,num));
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_ushort()
|        
|       Purpose: Converts a vector of unsigned short information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|        Output: 
|       Returns: NULL if any error occured. Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer 
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions: 
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_ushort(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KUSHORT);
  outsize = kmach_sizeof((int)desttype,KUSHORT);
  return(convert_unsigned(inptr,outptr,srctype,desttype,insize,outsize,num));
}

/*-------------------------------------------------------------------
|
|  Routine Name: convert_int()
|        
|       Purpose: Converts a vector of signed int information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|        Output: 
|       Returns: NULL if any error occured. Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions: 
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_int(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KINT);
  outsize = kmach_sizeof((int)desttype,KINT);
  return(convert_signed(inptr,outptr,srctype,desttype,insize,outsize,num));
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_uint()
|        
|       Purpose: Converts a vector of unsigned int information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    -  address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|
|        Output: 
|       Returns: NULL if any error occured. Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions:
| 
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_uint(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KUINT);
  outsize = kmach_sizeof((int)desttype,KUINT);
  return(convert_unsigned(inptr,outptr,srctype,desttype,insize,outsize,num));
}

/*--------------------------------------------------------------------
|
|  Routine Name: convert_long()
|        
|       Purpose: Converts a vector of signed long information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|        Output: 
|       Returns: NULL if any error occured. Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions:
|
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
--------------------------------------------------------------------*/

unsigned char *convert_long(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KLONG);
  outsize = kmach_sizeof((int)desttype,KLONG);
  return(convert_signed(inptr,outptr,srctype,desttype,insize,outsize,num));
}

/*-------------------------------------------------------------------
|
|  Routine Name: convert_ulong()
|        
|       Purpose: Converts a vector of unsigned long information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions:
| 
|    Written By: Jeremy Worley
|
|          Date: Thu Jun 25 11:34:12 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_ulong(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KULONG);
  outsize = kmach_sizeof((int)desttype,KULONG);
  return(convert_unsigned(inptr,outptr,srctype,desttype,insize,outsize,num));
}

/*-------------------------------------------------------------------
|
|  Routine Name: convert_byte()
|        
|       Purpose: Converts a vector of signed byte information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions:
| 
|    Written By: Jeremy Worley
|
|          Date: Fri Jun 26 16:35:36 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_byte(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KBYTE);
  outsize = kmach_sizeof((int)desttype,KBYTE);
  return(convert_signed(inptr,outptr,srctype,desttype,insize,outsize,num));
}


/*-------------------------------------------------------------------
|
|  Routine Name: convert_ubyte()
|        
|       Purpose: Converts a vector of unsigned byte information 
|                originating on one architecture to another 
|                architecture. 
|
|         Input: inptr    - address of the input data cast to 
|                           (char *)
|
|                outptr   - address of the output data cast to 
|                           (char *) or NULL, which indicates that 
|                           memory should should be allocated.
|
|                srctype  - unsigned int specifying source 
|                           architecture.
|
|                desttype - another unsigned int specifying the 
|                           destination (usually current) architecture.
|
|
|        Output: 
|       Returns: NULL if any error occured.  Otherwise, it 
|                returns outptr if outptr != NULL or a new pointer
|                if outptr == NULL
|
|        Status: Semi-Private
|
|  Restrictions:
| 
|    Written By: Jeremy Worley
|
|          Date: Fri Jun 26 16:36:33 MDT 1992
|
| Modifications:
|
----------------------------------------------------------------------*/

unsigned char *convert_ubyte(
   unsigned char *inptr,
   unsigned char *outptr,
   unsigned int  srctype,
   unsigned int  desttype,
   unsigned int  num)
{
  unsigned int insize, outsize;

  insize = kmach_sizeof((int)srctype,KUBYTE);
  outsize = kmach_sizeof((int)desttype,KUBYTE);
  return(convert_unsigned(inptr,outptr,srctype,desttype,insize,outsize,num));
}
