/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Directory Utilities
   >>>>
   >>>>   Public:
   >>>>             kdirname()
   >>>>             kbasename()
   >>>>		    kremove_dir()
   >>>>		    kmake_dir()
   >>>>
   >>>>		    krmdir()
   >>>>		    kmkdir()
   >>>>		    kchdir()
   >>>>		    kgetcwd()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


/************************************************************
*
*  Routine Name: kdirname - find directory component of a given pathname
*       Purpose: This routine searches an input pathname for the final
*		 occurance of a '/', and returns the string existing
*		 before the '/'.  If a '/' is not found, it looks for
*		 a '~', as in ~username.  If neither a '/' or a '~' are
*		 found, it will assume the pathname does not have a
*		 directory component, and returns the string ".".
*         Input: pathname - A valid pathname
*        Output: return_path - A string that will hold the resulting directory
*			       portion of the string.  If this parameter is
*			       NULL, it will automatically malloc space for
*			       the result.
*       Returns: directory component of pathname, NULL on error
*    Written By: Mark Young
*          Date: Jul 03, 1992 13:56
*  Side Effects: This routine kmalloc's the return string, and will remove
*		 whitespace (as defined by isspace()) from the end of the input
*		 string
* Modifications: Converted from vdirname in khoros 1.0 (SJ)
************************************************************/

char *kdirname(
   char *pathname,
   char *return_path)
{
	int   num;
	char  *file, temp[KLENGTH], dirname[KLENGTH];


	/*
	 *  If the pathname is null then error and return NULL
	 */
	if (pathname == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kdirname", "Parameter pathname is NULL");
	   return(NULL);
	}

	kstrcpy(temp, pathname);
	pathname = _cleanup_string(temp);
	if (kstrlen(pathname) == 0)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kdirname", "Parameter pathname is empty");
	   return(NULL);
	}


	/*
	 *  basically want to get the component or the prefix to the filename.
	 *  This should be the last occurence of a '/'.  If no '/' is found
	 *  then we check to see if the string begins with a ~username.  If
	 *  neither of these cases occur then it must be a filename in which
	 *  case we will return "." for the current directory.
	 */
	if ((file = kstrrchr(pathname,'/')) != NULL)
	{
	   /*
	    *  Now that we have found the last occurence of '/' we
	    *  need to make sure that it is not the root of the file
	    *  system ie) /usr, or that the string is not ~/.  Otherwise
	    *  we copy everything in front of the "file" part of the string.
	    */
	   if (file == pathname)
	      kstrcpy(dirname, "/");
	   else if ((int) file[-1] == '~')
	   {
	      num = file-pathname+1;
	      kstrncpy(dirname, pathname, num);
	      dirname[num] = '\0';
	   }
	   else
	   {
	      num = file-pathname;
	      kstrncpy(dirname, pathname, num);
	      dirname[num] = '\0';
	   }
	}
	else if ((int) pathname[0] == '~')
	   kstrcpy(dirname, pathname);
	else
	   kstrcpy(dirname, ".");

	if (return_path == NULL)
	   return(kstrdup(dirname));
	else
	   return(kstrcpy(return_path, dirname));
}

/************************************************************
*
*  Routine Name: kbasename - return the filename component of a pathname
*       Purpose: This routine searches an input pathname for the final
*		 occurance of a '/', and returns the string existing
*		 after it.  If a '/' is not found, it looks for a '~',
*		 as in ~username.  If neither a '/' or a '~' is found and
*		 the input string is not empty, then it will return
*		 the original string.
*         Input: pathname - A valid pathname
*        Output: return_base - if return_base is not NULL, then this will be
*			       where the result is placed.  Otherwise, it will
*			       malloc the space required.
*       Returns: filename component of pathname, NULL on error
*    Written By: Mark Young & Tom Sauer
*          Date: Sep 15, 1993
*  Side Effects: This routine kmalloc's the return string, and will remove
*		 whitespace (as defined by isspace() from the end of the string)
* Modifications: Converted from vbasename in khoros 1.0 (SJ)
************************************************************/

char *kbasename(
   char *pathname,
   char *return_base)
{
	char *file, temp[KLENGTH];


	/*
	 *  If the pathname is null then error and return NULL
	 */
	if (pathname == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kbasename", "Parameter pathname is NULL");
	   return(NULL);
	}

	/*
	 *  If the pathname is  empty then error and return NULL
	 */
	kstrcpy(temp, pathname);
	pathname = _cleanup_string(temp);
	if (kstrlen(pathname) == 0)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kbasename", "Parameter pathname is empty");
	   return(NULL);
	}

	if ((file = kstrrchr(pathname,'/')) == NULL)
	{
	   if (pathname[0] == '~')
	      return(NULL);
	   else
	      file = pathname;
	}
	else 
	   file = (file + 1);

	if (return_base == NULL)
	   return(kstrdup(file));
	else
	   return(kstrcpy(return_base, file));
}

/************************************************************
*
*  Routine Name: kremove_dir - remove a directory and it's contents
*       Purpose: This routine removes a specified path.  If it is
*		 a directory, it recursively calls itself on all
*		 pathnames under it.
*         Input: path - the pathname to remove
*       Returns: 0 on success and -1 on failure
*  Restrictions: Hard links to other directory trees are not recognized by
*		 this routine, so if a hard link exists to a directory,
*		 it will follow it as a normal directory.
*    Written By: Steven Jorgensen
*          Date: Jul 15, 1992 12:52
* Modifications: Converted from vrmdir in Khoros 1.0 (SJ)
*		 Updated to Khoros style return values (TRUE ==> success)
*************************************************************/
int kremove_dir(
   char *path)
{
	int		rtl = TRUE,cnt, i;
	char		*npath,**npaths, var[KLENGTH];
	struct stat	buf;

	if (path == NULL || (npath = kfullpath(path,NULL,NULL)) == NULL)
		return FALSE;

#ifndef SYSV
	if (lstat(npath,&buf) == -1) 
#else
	if (stat(npath,&buf) == -1) 
#endif
	{
		kfree(npath);
		return FALSE;
	}

#if defined(S_ISLNK)
	if (S_ISREG(buf.st_mode) || S_ISLNK(buf.st_mode))
#else
	if (S_ISREG(buf.st_mode))
#endif
	{
		rtl = (kunlink(npath) == 0);
		kfree(npath);
		return rtl;
	}

	npaths = karray_dirlist(NULL, npath, NULL, KDIR | KPATH | KDOT, 0,&cnt);
	for (i = 0; i < cnt; i++)
	{
		kbasename(npaths[i], var);
		if (var != NULL && (!kstrcmp(var,"..") || !kstrcmp(var,".")))
			continue;
		else if ((rtl = kremove_dir(npaths[i])) == FALSE)
			break;
	}
	karray_free(npaths, cnt, NULL);

	if (!rtl)
	{
		kfree(npath);
		return rtl;
	}

	npaths = karray_dirlist(NULL, npath, NULL, KDOT | KSOCK | KPATH |
			    KLINK | KFILE, 0, &cnt);
	for (i = 0 ; i < cnt; i++)
	{
		kbasename(npaths[i], var);
		if (var != NULL && (!kstrcmp(var,"..") || !kstrcmp(var,".")))
			continue;
		else if ((rtl = (kunlink(npaths[i]) == 0)) == FALSE)
			break;
	}
	karray_free(npaths, cnt, NULL);

	if (rtl)
		rtl = (krmdir(npath) == 0);

	kfree(npath);
	return rtl;
}

/************************************************************
*
*  Routine Name: kmake_dir - make a directory and all parent directories
*			     if necessary
*       Purpose: This routine calls kmkdir as many times as needed
*		 to build a new directory.  This means that if making
*		 a directory, and the parent directories do not exist,
*		 it will first make all the parent directories that
*		 are needed, and then create the directory requested by
*		 the calling routine.
*         Input: path - directory to create
*		 mode - octal permission mode of the new directory
*       Returns: TRUE (1) on success, FALSE (0) on failure
*    Written By: Steven Jorgensen
*          Date: Jan 14, 1993 15:09
* Modifications: changed to Khoros style return values (neilb)
*************************************************************/
int kmake_dir(
   char *path,
   int   mode)
{
	int		call = TRUE;
	char		parent[KLENGTH];


	if (kdirname(path, parent) == NULL)
		call = FALSE;

	while (kmkdir(path, mode) == -1 && errno == ENOENT)
	{
		if (!kmake_dir(parent, mode) && call == FALSE)
			return FALSE;
	}
	return TRUE;
}

/************************************************************
*
*  Routine Name: krmdir - remove a directory
*       Purpose: This routine is a replacement for the system routine
*		 rmdir.  It has the same actions, error status, and
*		 return values of rmdir, but it will expand ~'s, environment
*		 variables, and Khoros variables.
*         Input: path - directory name to delete
*       Returns: 0 on success, -1 otherwise
*    Written By: Steven Jorgensen
*          Date: Jul 15, 1992 13:08
*************************************************************/

int krmdir(
   char *path)
{
	int rtl;
	char *npath;

	if (path == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "krmdir","path parameter is NULL");
	   return(-1);
	}

	if ((npath = kfullpath(path, NULL, NULL)) == NULL)
	{
	   errno = KINTERNAL;
	   kerror("kutils", "krmdir", "path could not be expanded");
	   return(-1);
	}
	rtl = rmdir(npath);
	kfree(npath);
	return(rtl);
}

/************************************************************
*
*  Routine Name: kmkdir - library call to create a directory
*       Purpose: This routine acts the same as the system routine
*		 mkdir(), with one minor change.  It makes an
*		 internal call to kfullpath to expand environment
*		 variables, toolbox variables, and ~'s.  NOTE:
*		 errno is set by the internal call to mkdir()
*         Input: path - pathname of new directory to create
*		 mode - octal permission mode of the new directory
*       Returns: (0) on success, (-1) otherwise
*    Written By: Steven Jorgensen
*          Date: Jul 06, 1992
* Modifications: Converted from vmkdir() in khoros 1.0 (SJ)
*************************************************************/
int kmkdir(
   char *path,
   int  mode)
{
	int rtl;
	char *npath;


	if (path == NULL)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kmkdir","path parameter is NULL");
	   return(-1);
	}

	if ((npath = kfullpath(path, NULL, NULL)) == NULL)
	{
	   errno = KINTERNAL;
	   kerror("kutils", "kmkdir", "path could not be expanded");
	   return(-1);
	}

	rtl = mkdir(npath,mode);
	kfree(npath);
	return(rtl);
}

/************************************************************
*
*  Routine Name: kchdir - library call to change the current working directory
*       Purpose: This routine acts the same as the system routine
*		 chdir(), with one minor change.  It makes an
*		 internal call to kfullpath to expand environment
*		 variables, toolbox variables, and ~'s.  NOTE:
*		 errno is set by the internal call to chdir()
*         Input: path - pathname of new directory to create
*       Returns: (0) on success, (-1) otherwise
*    Written By: Neil Bowers & Mark Young
*          Date: Sep 15, 1993
*************************************************************/

int kchdir(
   char *path)
{
	int  status;
	char *temp, pwd[KLENGTH];


	if (!path)
	{
	   errno = KNULL_PARAMETER;
	   kerror("kutils", "kchdir","path parameter is NULL");
	   return(-1);
	}

	if ((temp = kfullpath(path, NULL, NULL)) == NULL)
	{
	   errno = KINTERNAL;
	   kerror("kutils", "kchdir", "path could not be expanded");
	   return(-1);
	}

	/*
	 *  Change to the desired directory by calling chdir().
	 */
	status = chdir(temp);
	kfree(temp);

	/*
	 *  If we successfully changed directories then update our PWD
	 *  environment variable.
	 */
	if (status == 0)
	{
	   kgetcwd(pwd, KLENGTH);
	   temp = kstring_cat("PWD=", pwd, NULL);
	   kputenv(temp); kfree(temp);
	}
	return(status);
}

/************************************************************
*
*  Routine Name: kgetcwd - library call to get the current working directory
*       Purpose: This routine acts the same as the system routine
*		 getcwd(), with one minor change.  It will malloc
*		 the return path irregardless of the length size
*		 passed in.
*         Input: length - the length of the path string
*        Output: path   - the buffer that holds the return string
*       Returns: path on success, NULL otherwise
*    Written By: Neil Bowers & Mark Young
*          Date: Sep 15, 1993
*************************************************************/

char *kgetcwd(
   char *path,
   int  length)
{
	char *temp;

	do
	{
	    if ((temp = getcwd(path, length)) == NULL)
	    {
	       if (errno == EACCES)
		  return(NULL);

	       length += KLENGTH;
	    }
        } while (!temp);
	return(temp);
}
