/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         UNIX Pipe Transport Routines
   >>>>
   >>>>  Private:
   >>>>      		pipe_tempnam
   >>>>			pipe_open
   >>>>			pipe_close
   >>>>			pipe_read
   >>>>			pipe_write
   >>>>			pipe_access
   >>>>			pipe_unlink
   >>>>			pipe_lock
   >>>>			pipe_descriptors
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


#if !defined(KPIPE_DEF)
TransportInformation pipe_transport[] = {NULL};
#else

static int pipe_tempnam	    PROTO((char *, char *, char *));
static int pipe_open	    PROTO((char *, int, int, kfile *));
static int pipe_close	    PROTO((kfile *));
static int pipe_read	    PROTO((kfile *, char *, int));
static int pipe_write	    PROTO((kfile *, char *, int));
static int pipe_unlink	    PROTO((char *, char *));
static int pipe_lock	    PROTO((kfile *, int));
static int pipe_descriptors PROTO((kfile *, int *, int *));

TransportInformation pipe_transport[] =
{
    {
	"Standard Unix Pipe",
	"pipe",
	FALSE,
	pipe_tempnam,
	pipe_open,
	pipe_close,
	pipe_read,
	pipe_write,
	NULL,
	NULL,
	pipe_unlink,
	pipe_lock,
	pipe_descriptors,
    }
};

/*
 *  Internal resource structures for "standard pipe" transport
 */
typedef struct
{
	int	fid[2];
} ResourceStruct;


/*-------------------------------------------------------------
|  
|  Routine Name: pipe_tempnam
|  
|       Purpose: This function initializes a tempnam for a "pipe"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"pipe=[in,out]"
|		  	"pipe=-"
|		  	"pipe=#"
|  
|	         This routine creates a pipe and returns the template
|	         in the following syntax:
|  
|		  	"pipe=[in,out]"
|  
|         Input:  identifier - the transport identifier
|		  template   - the filename template to be modeled
|        Output:  result     - the newly created transport name
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_tempnam(
   char *identifier,
   char *template,
   char *result)
{
	int	fid[2];


	/*
	 *  Create a pipe for the transport
	 */
	if (pipe(fid) == -1)
	{
	   kinfo(KSYSLIB, "pipe_tempnam: Unable to create temporary pipe");
	   return(-1);
	}
	else
	{
	   if (identifier == NULL)
	      (void) sprintf(result,"pipe=[%d,%d]", fid[0], fid[1]);
	   else
	      (void) sprintf(result,"%s=[%d,%d]", identifier, fid[0], fid[1]);
	}
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_open
|  
|       Purpose: This function opens a "pipe" connection.  It is the
|	         internal driver to open a pipe, which is called by
|	         the user when they call kopen().  kopen() calls when
|	         it gets an open request with a filename of the following
|	         syntax:
|  
|		  	"pipe=[in,out]"
|		  	"pipe=-"   (stdin or stdout)
|		  	"pipe=#"   (stderr)
|  
|         Input: 
|        Output:  returns 0 or -1 if an error occurs
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_open(
   char  *path,
   int   flags,
   int   mode,
   kfile *file)
{
	int	   fid_access, fid[2];
	ResourceStruct *resources;


	/*
	 *  Assign the string to the appropriate pipe.
	 */
	fid[0] = fid[1] = -1;
	if (sscanf(path," [ %d , %d ]", &fid[0], &fid[1]) != 2)
	{
	   if (kstrcmp(path,"-") == 0)
	   {
	      fid_access = flags & KOPEN_ACCMODE;
	      if (fid_access == KOPEN_RDONLY)
	         fid[0] = fileno(stdin);
	      else if (fid_access == KOPEN_WRONLY)
	         fid[1] = fileno(stdout);
	      else
	      {
                 kinfo(KSYSLIB, "pipe_open: Invalid open access for file '-'. \
The '-' can only be used for exclusively reading or writing.");
	         return(-1);
	      }
	   }
	   else if (kstrcmp(path,"#") == 0)
	   {
	      fid[1] = fileno(stderr);
	   }
	}

	/*
	 *  See if the user wants the KOPEN_NONBLOCK set
	 */
	if (flags & KOPEN_NONBLOCK)
	{
	   fcntl(fid[0], F_SETFL, KOPEN_NONBLOCK);
	   fcntl(fid[1], F_SETFL, KOPEN_NONBLOCK);
	}

	if ((resources = (ResourceStruct *) kcalloc(1,
			sizeof(ResourceStruct))) == NULL)
	{
	   kinfo(KSYSLIB, "pipe_open: Not enough memory to kcalloc (%d) bytes \
for the khoros pipe structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}

	/*
	 *  Check to see if the transport is opened for reading.  If so then
	 *  initialize fid[0], otherwise set it to -1.
	 */
	if (kfile_isread(file))
	   resources->fid[0] = fid[0];
	else
	   resources->fid[0] = -1;

	/*
	 *  Check to see if the transport is opened for writing.  If so then
	 *  initialize fid[1], otherwise set it to -1.
	 */
	if (kfile_iswrite(file))
	   resources->fid[1] = fid[1];
	else
	   resources->fid[1] = -1;

	file->resources = (kaddr) resources;
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_close
|  
|       Purpose: This function closes a "pipe".  It is the internal driver
|	         to close a data pipe, which is called by the user
|	         when they call kclose().  kclose() calls the "pipe" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the initial kopen().
|  
|         Input:  file - the kfile structure.
|        Output:  returns whether we were able to close the pipe
|   Called From:  internal routine called from kclose()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_close(
   kfile *file)
{
	int	   status = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;

	/*
	 *  Simply call close() since "kfile" is just a regular file
	 *  descriptor.
	 */
	if (resources->fid[1] != -1)
	   status |= close(resources->fid[1]);

	if (resources->fid[0] != -1)
	   status |= close(resources->fid[0]);

	kinfo(KDEBUG,"in close: status %d", status);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_read
|  
|       Purpose: This function reads a "pipe".  It is the internal driver
|	         to read data from a pipe, which is called by the user
|	         when they call kread().  kread() calls the "pipe" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the kopen().
|  
|         Input:  file - the kfile structure.
|		  ptr  - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes read from the file
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int	   count;
	int	   numread = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call read() since "kfile" is just a regular file
	 *  descriptor.
	 */
	do
	{
	   count = kmin(nbytes - numread, DefaultPageSize);
	   kinfo(KDEBUG,"in read: desire %d bytes", count);
	   count = read(resources->fid[0], (char *) (ptr + numread), count);
	   numread += count;
	   kinfo(KDEBUG,"in read: got %d bytes", count);
	}
	while (numread < nbytes && count > 0);

	kinfo(KDEBUG,"read %d out of %d bytes", numread, nbytes);
	return(numread);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_write
|  
|       Purpose: This function writes to a "pipe".  It is the internal driver
|	         to write data from the supplied data array to a pipe, which
|	         is called by the user when they call kwrite().  kwrite() calls
|	         the "pipe" internal drivers by the nature of the transport,
|	         which is dictated by the kopen().
|  
|         Input:  id   - the kfile structure.
|		  ptr  - the pointer to write the data from.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int	   count;
	int	   numwrite = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Safety check..
	 */
	if (nbytes == 0)
	   return(0);

	/*
	 *  Simply call write() since "kfile" is just a regular file
	 *  descriptor.
	 */
	do
	{
	    count = kmin(nbytes - numwrite, DefaultPageSize);
	    kinfo(KDEBUG,"desire to write %d bytes", count);
	    count = write(resources->fid[1], (char *) (ptr + numwrite), count);
	    kinfo(KDEBUG,"wrote %d bytes '%s'", count, file->path);
	    numwrite += count;
	} while (numwrite < nbytes && count > 0);

	kinfo(KDEBUG,"wrote %d out of %d bytes '%s'\n'%s'", numwrite, nbytes, file->path, ptr);
	return(numwrite);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_unlink
|  
|       Purpose: This function initializes a tempnam for a "file"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"pipe=[in,out]"
|  
|         Input: 
|        Output:  returns -1 or 0 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kunlink()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_unlink(
   char *identifier,
   char *filename)
{
	int	fid[2], status1 = -1, status2 = -1;


	fid[0] = fid[1] = -1;
	(void) sscanf(filename," [ %d , %d ]", &fid[0], &fid[1]);

	if (fid[0] != -1)
	{
	   if ((status1 = close(fid[0])) == -1)
	      if (errno == EBADF) status1 = 0;
	}

	if (fid[1] != -1)
	{
	   if ((status2 = close(fid[1])) == -1)
	      if (errno == EBADF) status2 = 0;
	}
	return(status1 | status2);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_lock
|  
|       Purpose: This function locks a "pipe" transport.  Given
|	         a request with a filename of the following syntax:
|  
|		  	"pipe=[in,out]"
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kflock()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_lock(
   kfile *file,
   int   operation)
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*status = flock(resources->fid[0], operation);*/
	return(TRUE);
}

/*-------------------------------------------------------------
|  
|  Routine Name: pipe_descriptors
|  
|       Purpose: This function shouldn't exist, but there is occasion
|	         when the user wants to know the actual file descriptor
|	         associated with the file descriptor.
|  
|         Input: file structure
|        Output:  inum - the input file descriptor
|   Called From:  internal routine called from kdescriptor()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int pipe_descriptors(
   kfile *file,
   int   *inum,
   int   *onum)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (inum != NULL) *inum = resources->fid[0];
	if (onum != NULL) *onum = resources->fid[1];
	return(0);
}
#endif  /* KPIPE_DEF */
/* don't add after the endif */
