/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>		    kdata_manip()
   >>>>   Static:
   >>>>		    _data_calloc
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: _data_calloc
|
|       Purpose: This routine is simply a convenient way
|		 to allocate data, given a number of points
|		 and a data type.  The main motivation for
|		 this routine was that I did not want to
|		 have to test for whether it was KBIT and
|		 then allocate differently.
|
|         Input: num  - number of elements
|		 type - element data type.
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 05, 1992 12:50
| Modifications:
|
------------------------------------------------------------*/

static char *
_data_calloc(unsigned num, unsigned type)
{
   unsigned size;

   if (num == 0)
      return (NULL);

   if (type == KBIT) {
      return (kcalloc((unsigned) ((num >> 3) + (num % 8 != 0)), sizeof(char)));
   } else {
      if ((size = (unsigned) kdata_size((int) type)) == 0)
	 return (NULL);
      return (kcalloc(num, size));
   }
}


/*------------------------------------------------------------
|
|  Routine Name: kdata_manip
|
|       Purpose: This routine is used to manipulate data
|		 a variety of ways.  It can be used to
|		 interpolate data; it can be used to
|		 cast data; it can be used to slice data;
|		 it can be used to dice data; it can be
|		 used to puree data...
|
|		 This routine should be considered unfinished.
|		 In its finished form, it should use a
|		 static buffer to reduce memory usage and
|		 eliminate the time wasted in allocation
|		 of temporary space.  Also, this is not
|		 a feasible solution if the input and
|		 output vectors passed in are huge. Sigh...
|
|		 The main reason that this was not implemented
|		 in the first cut of this routine is that
|		 I'm a little confused about how to approach
|		 nth order interpolation in a reasonable
|		 way if I am interpolating on small sub-
|		 sections of the input stream.  Points on
|		 the edges of the sections could get really
|		 screwed up if I am not careful about
|		 solving the problem.
|
|         Input: idata   - the input data vector.
|		 inum    - the number of points in the input
|			   data vector.
|		 onum    - the desired number of points in
|			   the data vector.
|		 idtype  - the input data type.
|		 odtype  - the output data type.
|		 istride - the input "skip factor" or stride
|			   value.
|		 ostride - the output "skip factor" or stride
|			   value.
|		 rfact   - a magnitude factor for scaling or
|			   normalizing the input data.  If
|			   flags argument is set to
|			   KDATAOP_NORMALIZE, then this refers to
|			   the real component of the desired
|			   maximum value of the input data.
|			   If the flag field is set to
|			   KDATAOP_SCALE then this argument
|			   represents the real component of
|			   the scale factor.
|		 ifact   - a magnitude factor for scaling or
|			   normalizing the input data.  If
|			   flags argument is set to
|			   KDATAOP_NORMALIZE, then this refers to
|			   the imaginary component of the
|			   desired maximum value of the
|			   input data.  If the flag field is
|			   set to KDATAOP_SCALE then this argument
|			   represents the imaginary component
|			   of the scale factor.  This argument
|			   is only used if the input data
|			   is kcomplex or kdcomplex.  Otherwise
|			   it is ignored.
|		 rshift  - a shift value that is used only
|			   when the flags argument is set to
|			   KDATAOP_SCALE.  If used, this value
|			   specifies the real component of an
|			   amount that should be added to each
|			   data value.
|		 ishift  - a shift value that is used only
|			   when the flags argument is set to
|			   KDATAOP_SCALE and the data is either
|			   kcomplex or kdcomplex.  If used, this
|			   value specifies the imaginary
|			   component of an amount that should
|			   be added to each data value.
|		 convert - complex convert.  KREAL, KIMAGINARY,
|			   KMAGNITUDE, and KPHASE are the only
|			   legal values.
|		 order   - Order of interpolation.  This value
|			   specifies what kind of polynomial
|			   based interpolation should be used.
|			   For example, a value of 0 indicates
|			   that a 0-order interpolation (also
|			   referred to as replication) should
|			   be used.  A value of 1 implies a
|			   linear interpolation, and so forth.
|			   this value is only used if the
|			   arguments inum and onum are not
|			   equal.
|		 flags   - This argument determines whether
|			   normalization, scale and shift,
|			   or neither should be performed
|			   on the data.  Legal values are
|			   KDATAOP_SCALE, KDATAOP_NORMALIZE,
|			   and KDATAOP_NOP.
|
|        Output: odata   - The output data vector.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Oct 02, 1992 17:50
| Modifications:
|
------------------------------------------------------------*/

int 
kdata_manip(
   kaddr idata,
   int inum,
   int onum,
   int idtype,
   int odtype,
   double istride,
   double ostride,
   double rfact,
   double ifact,
   double rshift,
   double ishift,
   int order,
   int convert,
   double rpad,
   double ipad, 
   unsigned long flags,
   kaddr odata)
{
   char *tmp1, *tmp2, *tmp3;
   int alloced = 0;

   /*
    * If the flags indicate normalization or scaling/shifting then do these
    * operations on the input.
    */
   if (flags == KDATAOP_NORMALIZE) {
      if ((tmp1 = (char *) _data_calloc((unsigned) inum, (unsigned) idtype)) 
	  == NULL) {
	 kerror("kmath", "kdata_manip()",
	    "Unable to allocate temporary space for operation.");
	 return (FALSE);
      }
      if (!kdata_normalize(idata, inum, idtype, rfact, ifact, tmp1)) {
	 kfree(tmp1);
	 return (FALSE);
      }
      alloced = 1;
   } else if (flags == KDATAOP_SCALE) {
      if ((tmp1 = (char *) _data_calloc((unsigned) inum, (unsigned) idtype))
	 == NULL) {
	 kerror("kmath", "kdata_manip()",
	    "Unable to allocate temporary space for operation.");
	 return (FALSE);
      }
      if (!kdata_scale(idata, inum, idtype, rfact, ifact, rshift, ishift, tmp1)) {
	 kfree(tmp1);
	 return (FALSE);
      }
      alloced = 1;
   } else {
      tmp1 = idata;
   }

   /*
    * Extract a temporary buffer full of data and cast it.
    */
   if ((tmp2 = (char *) _data_calloc((unsigned) inum, (unsigned) odtype)) 
       == NULL) {
      kerror("kmath", "kdata_manip()",
	 "Unable to allocate temporary space for operation.");
      return (FALSE);
   }
   if (!kdata_cast(tmp1, idtype, odtype, inum, convert, istride, 1.0, tmp2)) {
      if (alloced)
	 kfree(tmp1);
      kfree(tmp2);
      return (FALSE);
   }

   /*
    * we're done with tmp1 now, so free it.
    */
   if (alloced)
      kfree(tmp1);

   /*
    * Call the interpolation routine if necessary
    */
   if (inum != onum) {
      if ((tmp3 = (char *) _data_calloc((unsigned) onum, (unsigned) odtype)) 
	  == NULL) {
	 kerror("kmath", "kdata_manip()",
	    "Unable to allocate temporary space for operation.");
	 return (FALSE);
      }
      if (!kdata_sample(tmp2, inum, onum, odtype, order, convert, rpad, ipad, 0.0, 0.0, tmp3)) {
	 if (alloced)
	    kfree(tmp1);
	 kfree(tmp2);
	 kfree(tmp3);
	 return (FALSE);
      }
      if (!kdata_cast(tmp3, odtype, odtype, onum, convert, 1.0, ostride, odata)) {
	 kfree(tmp3);
	 return (FALSE);
      }
      kfree(tmp3);
   } else {
      kdata_cast(tmp2, odtype, odtype, onum, convert, 1.0, ostride, odata);
   }

   /*
    * Clean up our mess
    */
   kfree(tmp2);

   /*
    * Looks like everything worked out after all...
    */
   return (TRUE);
}

