/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Miscellaneous Math Routines
   >>>>
   >>>>  Private:
   >>>>
   >>>>   Static:
   >>>>
   >>>>   Public:
   >>>>             kpowtwo()
   >>>>             kfact()
   >>>>             kimpulse()
   >>>>             kstep()
   >>>>             ksign()
   >>>>             ksinc()
   >>>>             krecip()
   >>>>             kneg()
   >>>>             kfraction()
   >>>>             kset()
   >>>>             kclear()
   >>>>             knot()
   >>>>		    klogn()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kpowtwo() - determine if number is an 
*                            integer power of two.
*
*       Purpose: The function kpowtwo checks to see if the integer
*		 input argument (a sequence length)
*                is an integer power of 2.
*
*         Input: length - length of sequence to be checked.
*
*        Output: 
*                RETURNS: TRUE (1) if the argument is an
*                integer power of two, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 04, 1992 12:28
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

int kpowtwo(
   int length)
{
    double size;

    size = (double)length;
	
    while(size > 1.0){
       size /= 2.0;
       if(size == 1.0)
          return(TRUE);
       else 
          if(size < 1.0)
             return(FALSE);
    }
    return(FALSE);
}

/************************************************************
*
*  Routine Name: kfact() - compute factorial of input.
*
*       Purpose: kfact() computes the factorial of the double precision input
*                argument.
*
*         Input: num - number to have factorial of computed.
*
*        Output: 
*                RETURNS: The factorial of the input argument.
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Jul 04, 1992 12:34
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

double kfact(
   double num)
{
  int i;
  double factorial;

  factorial = 1.0;
  for (i = 1;i <= (int)num;i++)factorial *= (double)i;
  return(factorial);
}

/************************************************************
*
*  Routine Name: kimpulse() - evaluate impulse function.
*
*       Purpose: The kimpulse function evaluates the impulse function at
*                the specified double precision argument x.
*
*         Input: x - value to evaluate the impulse function 
*                    at.
*
*        Output: 
*                RETURNS: 1.0 if x==0 and 0.0 otherwise
*
*  Restrictions: If the value of x falls between -KEPSILON
*                and +KEPSILON, then the value returned
*                will be 1.0.
* 
*    Written By: Jeremy Worley
*          Date: Jul 10, 1992 15:59
*      Verified:
*  Side Effects:
* Modifications: 
*
*************************************************************/

double kimpulse (
   double x)
{
    if ((x > -KEPSILON) && (x < KEPSILON)) return (1.0);
    return(0.0);
}

/************************************************************
*
*  Routine Name: kstep() - evaluate step function.
*
*       Purpose: The function kstep evaluates the step function at
*                the value of the specified double precision argument.
*
*         Input: x - input argument to step
*
*        Output: 
*                RETURNS: The double precision value of the step function
*                at x.
*
*  Restrictions: 
*    Written By: Marcelo Teran
*          Date: Jul 10, 1992 16:00
*      Verified:
*  Side Effects:
* Modifications: Converted from vstep() in Khoros 1.0 (JW)
*
*************************************************************/

double kstep (
   double x)
{
  if(x < 0.0)
     return (0.0);
  else
     return (1.0);
}

/************************************************************
*
*  Routine Name: ksign() - evaluate sign function.
*
*       Purpose: The ksign function evaluates the sign function at the value
*                of the double precision input argument.  The result is
*                -1 if the argument is negative, 0 if
*                the argument is 0, and 1 if the argument
*                is positive.
*
*         Input: x - input argument to evaluate
*
*        Output: 
*                RETURNS: -1 if the input argument is negative,
*		 0 if the input argument is 0, and 1 if the input
*		 argument is positive
*
*  Restrictions: 
*    Written By: Tom Sauer
*          Date: Jul 10, 1992 16:01
*      Verified:
*  Side Effects:
* Modifications: Converted from vsign() in Khoros 1.0 (JW)
*
*************************************************************/

double ksign (
   double x)
{
  if(x < 0.0)
    return (-1.0);
  else if (x > 0.0)
    return(1.0);
  else
    return(0.0);
}

/************************************************************
*
*  Routine Name: ksinc() - sinc function which is "sin(x)/x"
*
*       Purpose: The function ksinc evaluates the sinc function at the value of
*		 the input argument.  The result is sin(x)/x
*		 for a given x.
*
*         Input: x - input argument to the sinc function
*        Output: 
*       Returns: The double precision result of the sinc function evaluated
*		 from sin(x)/x.
*  Restrictions: 
*    Written By: Mark Young
*          Date: Apr 19, 1993 09:54
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

double ksinc(
   double x)
{
	return(x != 0.0 ? ksin(x)/x : 1.0);
}

/************************************************************
*
*  Routine Name: krecip() - reciprocal function.
*
*       Purpose: The krecip function evaluates the reciprocal value of the 
*		 double precision input 
*		 argument.  The result is 1/x for a given x.
*
*         Input: x - argument to get the reciprocal of.
*        Output: 
*       Returns: The result of the krecip function evaluated 
*		 as 1/x.		 
*
*  Restrictions: 
*    Written By: Mark Young & Donna Koechner
*          Date: Dec 1, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

double krecip(
   double x)
{
	return(x != 0.0 ? 1/x : KMAXFLOAT);
}


/************************************************************
*
*  Routine Name: kneg() - negative function 
*
*       Purpose: The kneg function evaluates the negative value of the 
*		 double precisoin input argument.  The result is -x for 
*		 a given x.
*
*         Input: x - input argument to find the negative value of.
*        Output: 
*       Returns: The double precision negative value of the input argument.
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Dec 1, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

double kneg(
   double x)
{
	return(0.0 - x);
}


/************************************************************
*
*  Routine Name: kfraction() - returns the fractional part of x 
*
*       Purpose: The kfraction function evaluates the fractional 
*		 value of the double precision input 
*		 argument.  The result is kmodf(x, *iptr) for a 
*		 given x.
*
*         Input: x - argument to the kfraction function
*        Output: 
*       Returns: The double precision fractional part of the input
*		 argument.
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Dec 1, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

double kfraction(
   double x)
{
	double iptr;
	return( kmodf(x, &iptr));
}


/************************************************************
*
*  Routine Name: kset() - bitwise set function 
*
*       Purpose: The kset function sets all bits (set to 1) of the input 
*		 argument.  
*
*         Input: x - input argument to set all bits on.
*        Output: 
*       Returns: An unsigned long with all bits set.
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Dec 1, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

unsigned long kset(
   unsigned long x)
{
	x = (~0L >> 1);
	return( x );
}

/************************************************************
*
*  Routine Name: kclear() - bitwise clear function 
*
*       Purpose: Clears all of the bits (sets to 0) of the unsigned
*		 long input argument.  
*
*         Input: x - argument to the kclear function
*        Output: 
*       Returns: The cleared unsigned long result. 
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Dec 1, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

unsigned long kclear(
   unsigned long x)
{
	x = 0;
	return(x);
}

/************************************************************
*
*  Routine Name: knot() - bitwise not (invert) function 
*
*       Purpose: The knot function inverts all bits of the input argument.  
*
*         Input: x - input argument to invert the bits on.
*        Output: 
*       Returns: The unsigned long inverted bit value of the input argument.
*
*  Restrictions: 
*    Written By: Donna Koechner
*          Date: Dec 1, 1993 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

unsigned long knot(
   unsigned long x)
{
	return( ~x );
}

/************************************************************
*
*  Routine Name: klogical_not() - logical not (invert) function 
*
*       Purpose: Changes the logic of the argument.  TRUE becomes
*		 FALSE, FALSE becomes TRUE.
*
*         Input: x - argument to the knot function
*        Output: 
*       Returns: The result of the knot function 
*
*  Restrictions: 
*    Written By: Jeremy Worley
*          Date: Oct 03, 1994 16:00
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

long klogical_not(
   long x)
{
   return( !x );
}

/************************************************************
*
*  Routine Name: klogn - base log n of argument
*       Purpose: The klogn function computes the base n log of the 
*		 first double precision input argument, 
*		 where n is the second double precision input argument.
*         Input: x - value to take base-n log of.
*		 n - base of logarithm.
*        Output:
*       Returns: the double precision base n logarithm of the double
*		 precision input argument x.
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Aug 16, 1994 09:59
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/

double 
klogn(double x, double n)
{
   return (klog(x) / klog(n));
}
