/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>           Form  Tree Manipulation Routines            <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                 kvf_add_subform()                     <<<<
   >>>>                 kvf_add_guide_to_subform()            <<<<
   >>>>                 kvf_add_selection()                   <<<<
   >>>>                 kvf_delete_subform()                  <<<<
   >>>>                 kvf_delete_guide()                    <<<<
   >>>>                 kvf_delete_selection()                <<<<
   >>>>                 kvf_update_backlinks()                <<<<
   >>>>                 kvf_transfer_selections()             <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"

static int delete_submenu PROTO((kselection *));

/*-----------------------------------------------------------------
|
|  Routine Name: kvf_add_subform
|
|       Purpose: Adds the given subform to the form tree provided.
|
|         Input: form    - pointer to the form tree
|                subform - subform branch to be added
|
|        Output: Returns TRUE on success, FALSE on failure
|
|    Written By: Danielle Argiro
|          Date: Jul 07, 1994
| Modifications: 
|
------------------------------------------------------------------*/

int kvf_add_subform(
   kform    *form,
   ksubform *new_subform)
{
        kguide   *guide;
        ksubform *subformptr;

       /*
        * the new subform branch may have been linked to others -
        * sever the link
        */
        new_subform->next = NULL;

        /*
         * reset backlinks
         */
        new_subform->back_form = form;
        if (new_subform->guidepane != NULL)
        {
            guide = new_subform->guidepane->guide_list;
            kvf_update_backlinks(new_subform->guidepane->sel_list,
                             form, new_subform, guide,
                             new_subform->guidepane);
        }
        else guide = new_subform->guide;
        while (guide != NULL)
        {
            guide->back_form = form;
            guide->pane->back_form = form;
            kvf_update_backlinks(guide->pane->sel_list, form, new_subform,
                             guide, new_subform->guidepane);
            guide = guide->next;
        }

        /*
         * form has no subforms at all: a single subform
         * goes to form->subform
         */
        if ((form->subform == NULL) && (form->master == NULL))
        {
            form->subform = new_subform;
            kvf_unlink_formstruct(new_subform->back_kformstruct);
            kfree(new_subform->back_kformstruct);
            new_subform->back_kformstruct =
                        kvf_create_struct((kaddr) new_subform,
                                           KUIS_STARTSUBFORM, KSUBFORM);
            kvf_link_formstruct(new_subform->back_kformstruct);
        }

        /*
         * form has a single subform in form->subform, but
         * we're adding another, which means we must add a master
         * and start a subform list with the two subforms
         */
        else if (form->master == NULL)
        {
            form->master = kvf_begin_control(form, NULL, NULL, KMASTER);
            form->master->subform_list = form->subform;
            form->master->subform_list->next = new_subform;
            form->subform = NULL;
            kvf_unlink_formstruct(new_subform->back_kformstruct);
            kfree(new_subform->back_kformstruct);
            new_subform->back_kformstruct =
                        kvf_create_struct((kaddr) new_subform,
                                           new_subform->type, KSUBFORM);
            kvf_link_formstruct(new_subform->back_kformstruct);
        }

        /*
         * form already has a master - simply add to end of subform list.
         */
        else
        {
            subformptr = form->master->subform_list;
            while (subformptr->next != NULL)
                subformptr = subformptr->next;
            subformptr->next = new_subform;
            kvf_unlink_formstruct(new_subform->back_kformstruct);
            kfree(new_subform->back_kformstruct);
            new_subform->back_kformstruct =
                        kvf_create_struct((kaddr) new_subform,
                                          new_subform->type, KSUBFORM);
            kvf_link_formstruct(new_subform->back_kformstruct);
        }


        return(TRUE);
}

/*-----------------------------------------------------------------
|
|  Routine Name: kvf_add_guide_to_subform
|
|       Purpose: Adds a guide branch to the subform branch provided.
|
|         Input: subform   - pointer to the subform branch
|                new_guide - guide branch to be added
|
|        Output: Returns TRUE on success, FALSE on failure
|
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications:
|
------------------------------------------------------------------*/

int kvf_add_guide_to_subform(
   ksubform *subform,
   kguide   *new_guide)
{
        kguide *guideptr;

       /*
        * the new guide branch may have been linked to others -
        * sever the link
        */
        new_guide->next = NULL;

        /* * subform has no guides at all: a single guide
         * goes to subform->guide
         */
        if ((subform->guide == NULL) && (subform->guidepane == NULL))
        {
            subform->guide = new_guide;
        }

        /*
         * subform has a single guide in subform->guide, but
         * we're adding another, which means we must add a guidepane
         * and start a guide list with the two guides
         */
        else if (subform->guidepane == NULL)
        {
            subform->guidepane = kvf_begin_control(subform->back_form,
                                                   subform, NULL,
                                                   KGUIDEPANE);
            subform->guide->back_control = subform->guidepane;
            new_guide->back_control = subform->guidepane;

            subform->guidepane->guide_list = subform->guide;
            subform->guidepane->guide_list->next = new_guide;

            subform->guide = NULL;
        }

        /*
         * subform already has a guide - simply add to end of guide list.
         */
        else
        {
            guideptr = subform->guidepane->guide_list;
            while (guideptr->next != NULL)
                guideptr = guideptr->next;
            guideptr->next = new_guide;

            new_guide->back_control = subform->guidepane;
        }

        new_guide->back_subform       = subform;
        new_guide->pane->back_subform = subform;
        new_guide->pane->back_form    = subform->back_form;
        new_guide->back_form          = subform->back_form;

        if (subform->guidepane != NULL)
            kvf_update_backlinks(subform->guidepane->sel_list,
                             subform->back_form, subform, new_guide,
                             subform->guidepane);

        kvf_update_backlinks(new_guide->pane->sel_list, subform->back_form,
                         subform, new_guide, new_guide->pane);

        return(TRUE);
}


/*-----------------------------------------------------------------
|
|  Routine Name: kvf_add_selection
|
|       Purpose: Adds a selection to the end of the selection list.
|
|         Input: sel_list - selection list to be added to
|                new_sel  - selection to be added
|        Output:
|       Returns: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications:
|
------------------------------------------------------------------*/

void kvf_add_selection(
    kselection *sel_list,
    kselection *new_sel)
{
        kselection *selection;

	if (new_sel == NULL) return;

	selection = sel_list;
        while (selection->next != NULL)
            selection = selection->next;
        selection->next = new_sel;
        new_sel->back_toggle  = NULL;
        new_sel->back_group   = selection->back_group;

}

/*-----------------------------------------------------------------
|
|  Routine Name: kvf_delete_subform
|
|	Purpose: Deletes a subform from the form tree provided. 
|
|	  Input: form - pointer to the form tree
|		 old_subform - subform branch to be deleted 		
|
|        Output: Returns TRUE on success, FALSE on failure
|
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications: 
|
------------------------------------------------------------------*/

int kvf_delete_subform(
   kform     *form,
   ksubform  *old_subform)
{
	ksubform   *subformptr;

	/*
	 *  no subforms in form tree to delete
	 */
	if ((form->subform == NULL) && (form->master == NULL)) 
	    return(FALSE);

	/*
	 * delete only subform in form tree, from form->subform 
	 */
	if (form->subform != NULL)
	{
	    kvf_destroy_subform(form->subform);
	    form->subform = NULL;
	}

	/*
	 * deleting a subform from the list of subforms on the master.
	 */
	else if (form->master->subform_list != NULL)
	{
	    /*
	     * deleting the last subform from the master.  Move any selections
             * on the master to the single subform, delete the master. 
	     */
            subformptr = form->master->subform_list;
            if (subformptr == old_subform)
            {
		if (form->master->subform_list->next == NULL)
		{
		    form->subform = form->master->subform_list;
			
		     /* if there were selections on the master, transfer them 
		        to subform - either guidepane or single pane */
		    if (form->subform->guidepane != NULL)
			kvf_transfer_selections(form->master, 
				 	        form->subform->guidepane);
		    else kvf_transfer_selections(form->master, 
					         form->subform->guide->pane);

		    form->subform->selected = TRUE;
		    form->master->subform_list = NULL;
		    kfree(form->master->control_line);
	            kvf_unlink_formstruct(form->master->back_kformstruct);
                    kfree(form->master->back_kformstruct);
	            kfree(form->master);
		    form->master = NULL;
		}

		/*
	         *  deleting the 1st (of several) subforms from the master 
	         */
		else
		{
                    form->master->subform_list = subformptr->next;
                    kvf_destroy_subform(old_subform);
		}
            }

           /*
            * delete nth subform from list on master
            */
            else
            {
                while ((subformptr->next != old_subform) &&
                       (subformptr->next != NULL))
                    subformptr = subformptr->next;

                if (subformptr->next == old_subform)
                {
                    subformptr->next = subformptr->next->next;
                    kvf_destroy_subform(old_subform);
                }
                else return(FALSE);
            }
	}
	return(TRUE);
}

/*-----------------------------------------------------------------
|
|  Routine Name: kvf_delete_guide
|
|       Purpose: Deletes a guide button & pane from the form tree
|
|         Input: guide - pointer to the guide
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 08, 1994
| Modifications:
|
------------------------------------------------------------------*/

int kvf_delete_guide(
   kguide *guide)
{
        kguide   *guideptr;
        ksubform *subform;

	subform = guide->back_subform;

        /*
         * make sure this is not the only guide/pane pair on the subform
         */
        if (subform->guide == guide)
        {
            kerror("kforms", "kvf_delete_guide",
                   "Illegal request to delete the only pane from a subform");
            return(FALSE);
        }

        /* if this one was selected, select another one */
        if (guide->selected)
        {
            guideptr = subform->guidepane->guide_list;
            while (guideptr != NULL)
            {
                if (guideptr != guide) break;
                guideptr = guideptr->next;
            }
            if (guideptr != NULL)
	    {
                kvf_set_attribute(guideptr->back_kformstruct,
                                  KVF_SELECTED, 1);
                kvf_set_attribute(guideptr->pane->back_kformstruct,
                                  KVF_SELECTED, 1);
	    }
        }

        /*
         * deleting one of 2 or more guide buttons from the
         * list of guide buttons on the guidepane.
         */
        if (subform->guidepane->guide_list != NULL)
        {
            /*
             * delete 1st guide from list on guidepane
             */
            guideptr = subform->guidepane->guide_list;
            if (guideptr == guide)
            {
                subform->guidepane->guide_list = guideptr->next;
                kvf_destroy_control(guideptr->pane);
            }

            else
            {
                /*
                 * delete nth guide from list on guidepane
                 */
                while ((guideptr->next != guide) &&
                       (guideptr->next != NULL))
                    guideptr = guideptr->next;

                if (guideptr->next == guide)
                {
                    guideptr->next = guideptr->next->next;
                    kvf_destroy_control(guide->pane);
                }
                else return(FALSE);
            }

            /*
             * if we're deleting 2nd-to-last guide button, delete
             * guidepane, and put guide in subform->guide, for single guide
             * Move any selections on the guidepane to the single pane.
             */
            if (subform->guidepane->guide_list->next == NULL)
            {
                /*
                 * if there were selections on the guidepane,
                 * transfer them to the single pane
                 */
                kvf_transfer_selections(subform->guidepane, 
				         subform->guidepane->guide_list->pane);

                subform->guide = subform->guidepane->guide_list;
                subform->guide->pane->back_guide = subform->guide;

                /*
                 *  destroy the guidepane
                 */
                kfree(subform->guidepane->control_line);
                kvf_unlink_formstruct(subform->guidepane->back_kformstruct);
                kfree(subform->guidepane->back_kformstruct);
                kfree(subform->guidepane);
                subform->guidepane = NULL;
            }
        }
        return(TRUE);
}


/*-----------------------------------------------------------------
|
|  Routine Name: kvf_delete_selection
|
|       Purpose: Deletes a selection from the form tree.
|
|         Input: selection   - selection to be deleted
|        Output:
|       Returns: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications:
|
------------------------------------------------------------------*/

int kvf_delete_selection(
   kselection *selection)
{
        kselection   *prev; 
	kform_struct *prev_ptr;

	/*
	 * if selection is part of a submenu, need to take it out 
	 */
	if (selection->back_submenu != NULL)
	{
	   /* i'm too tired, i can't take it out */
	   prev_ptr = selection->back_submenu->submenu_next;
	   if (prev_ptr->Selptr == selection)
		selection->back_submenu->submenu_next = prev_ptr->next;
	   else
	   {
	       while ((prev_ptr->next != NULL) &&
		      (prev_ptr->next->Selptr != selection)) 
                   prev_ptr = prev_ptr->next;
               if (prev_ptr->next->Selptr != selection)
               {
                    kerror("kforms", "kvf_delete_selection",
                           "Can't find selection in submenu member list to delete");
                    return(FALSE);
               }
               prev_ptr->next = prev_ptr->next->next;
	    }
	}

        /*
         * selection may be plain, part of a toggle, or part of a group;
         * find pointer to previous selection
         */
        if (selection->back_toggle != NULL)
            prev = selection->back_toggle->toggle_next;
        else if (selection->back_group != NULL)
            prev = selection->back_group->group_next;
        else prev = selection->back_control->sel_list;

        /*
         * link the selection list around the selection being deleted,
         * isolating it from the rest of the selection list
         */
        if (prev == selection)
        {
           if (selection->back_toggle != NULL)
               selection->back_toggle->toggle_next = prev->next;
           else if (selection->back_group != NULL)
               selection->back_group->group_next = prev->next;
           else selection->back_control->sel_list = prev->next;
        }
        else
        {
            while ((prev->next != selection) && (prev->next != NULL))
                prev = prev->next;
            if (prev->next != selection)
            {
                kerror("kforms", "kvf_delete_selection",
                       "Can't find selection in sel_list to delete");
                return(FALSE);
            }
            prev->next = selection->next;
        }

	/*
	 *  if the selection being deleted is a submenu, also delete
         *  all the selections that comprise the contents of the menu
	 */
	if (selection->type == KUIS_STARTSUBMENU)
	    delete_submenu(selection);

	/*
	 * deleting last member of toggle: delete toggle itself
	 */
	if ((selection->back_toggle != NULL) &&
            (selection->back_toggle->toggle_next == NULL))
	    kvf_destroy_selection(selection->back_toggle);

	/*
	 *  destroy the selection & free associated memory
	 */
	kvf_destroy_selection(selection);

	return(TRUE);
}


/*-----------------------------------------------------------------
|
|  Routine Name: delete_submenu
|
|       Purpose: Deletes the contents of a submenu button (all the
|                selections accessable from the submenu). 
|
|         Input: submenu - submenu to be deleted
|        Output:
|       Returns: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications:
|
------------------------------------------------------------------*/
static int delete_submenu(
    kselection *submenu)
{
        char         *name;
        ksubform     *subform;
        kguide       *guide;
        kform_struct *kformstruct;

        if (submenu->submenu_next == NULL)
            return(TRUE);

	kformstruct = submenu->submenu_next;
        while (kformstruct != NULL)
        {
            switch(kformstruct->flag)
            {
                 case KUIS_SUBFORMBUTTON:
                      subform = kformstruct->Subformptr;
                      name = ktoken_to_string(subform->var_token);

                      if (!(kvf_delete_subform(subform->back_form, subform)))
                          return(FALSE);

                      break;

                 case KUIS_GUIDEBUTTON:
                      guide = kformstruct->Guideptr;
                      name = ktoken_to_string(guide->var_token);
/*
                      if (!(kvf_delete_guide(guide)))
                          return(FALSE);
*/
                      break;

                 case KUIS_MASTERACTION:
                 case KUIS_SUBFORMACTION:
                 case KUIS_PANEACTION:
                 case KUIS_BLANK:
                 case KUIS_HELP:
                 case KUIS_QUIT:
                 case KUIS_ROUTINE:

                      if (!(kvf_delete_selection(kformstruct->Selptr)))
                         return(FALSE);

                      break;

            }
            kformstruct = kformstruct->next;
	}
	return(TRUE);
}



/*-----------------------------------------------------------------
|
|  Routine Name: kvf_update_backlinks
|
|       Purpose: Updates the backlinks after a guide is added to
|                a subform, or after a subform is added to a master.
|
|         Input: sel_list - pointer to the selection list
|                form     - pointer to the form tree
|                subform  - pointer to the subform branch
|                guide    - pointer to the guide
|                control  - pointer to the master, guidepane, or pane
|        Output:
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications:
|
------------------------------------------------------------------*/

void kvf_update_backlinks(
   kselection *sel_list,
   kform      *form,
   ksubform   *subform,
   kguide     *guide,
   kcontrol   *control)
{
        kselection *selection;

        selection = sel_list;
        while (selection != NULL)
        {
            if (form != NULL)
                selection->back_form = form;
            if (subform != NULL)
                selection->back_subform = subform;
            if (guide != NULL)
                selection->back_guide = guide;
            if (control != NULL)
                selection->back_control = control;

            if (selection->toggle_next != NULL)
                kvf_update_backlinks(selection->toggle_next, form,
                                     subform, guide, control);

            if (selection->group_next != NULL)
                kvf_update_backlinks(selection->group_next, form,
                                     subform, guide, control);

            selection = selection->next;
        }
}

/*-----------------------------------------------------------------
|
|  Routine Name: kvf_transfer_selections
|
|       Purpose: Transfers a selection list from the master to 
|                a guidepane or pane, when the master is deleted, or
|                transfers a selection list from the guidepane to
|                a pane when the guidepane is deleted.
|
|         Input: from_control - pointer to the master
|                to_control   - pointer to the guidepane or pane
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: July 07, 1994
| Modifications:
|
------------------------------------------------------------------*/

int kvf_transfer_selections(
    kcontrol *from_control, 
    kcontrol *to_control)
{
	int        var_token;
	Line_Info  line_info;
	kselection *selection;
	kselection *from_sel, *to_sel;

	/* 
	 * no selections to transfer 
         */
	if (from_control->sel_list == NULL)
	    return(TRUE);

	/* 
	 * change any master actions to subform or pane actions 
         */
	selection = from_control->sel_list;
	while (selection != NULL)
	{
	    if (selection->type == KUIS_MASTERACTION)
	    {
		kvf_clear_line_info(&line_info);
		if (to_control->type == KPANE)
		{
		    selection->type = KUIS_PANEACTION;
		    kvf_gen_parse(selection->line, &line_info);
		    line_info.typeflag = KUIS_PANEACTION;
		    kvf_gen_deparse(&line_info, &selection->line);
		    kvf_free_line_info_strings(&line_info);
		}
		else if (to_control->type == KGUIDEPANE)
		{
		    selection->type = KUIS_SUBFORMACTION;
		    kvf_gen_parse(selection->line, &line_info);
		    line_info.typeflag = KUIS_SUBFORMACTION;
		    kvf_gen_deparse(&line_info, &selection->line);
		    kvf_free_line_info_strings(&line_info);
		}
	    }
	    selection = selection->next;
	}

	/*
         * selections on both "from" and "to" controls: try to merge
         * "intelligently".  this means no duplicate quit, help, or 
         * license buttons, etc.  what a pain.
         */
        if ((from_control->sel_list != NULL) &&
            (to_control->sel_list != NULL))
        {
            /* 
	     * if they both have a quit button, delete the one coming from
	     * the masterform or the guidepane; the trick is to save the UIS 
	     * line of that one, and impose its UIS line on the one that will
	     * be saved on the guidepane or pane, so that it *looks* like
	     * the one is "moving" to the new location.
	     */
            if ((kvf_gui_item_exists(to_control->sel_list, KUIS_QUIT)
                && kvf_gui_item_exists(from_control->sel_list, KUIS_QUIT)))
            {
                from_sel = kvf_find_specified_selection(from_control->sel_list,
                                                     KUIS_QUIT, -1);
		to_sel = kvf_find_specified_selection(to_control->sel_list,
                                                     KUIS_QUIT, -1);
		kfree(to_sel->line);
	        to_sel->line = kstrdup(from_sel->line);
                kvf_delete_selection(from_sel);
            }

            /*
             * if they both have a help button w/ variable name "help",
             * delete one. (one "help" and one "license" are allowed).
	     * see trick documented above for quit button
             */
            var_token = kstring_to_token("help");
            if ((kvf_gui_named_item_exists(to_control->sel_list, KUIS_HELP,
                                           var_token) != NULL) &&
                (kvf_gui_named_item_exists(from_control->sel_list, KUIS_HELP,
                                           var_token) != NULL))
            {
                from_sel = kvf_find_specified_selection(from_control->sel_list,
                                                     KUIS_HELP, var_token);
		to_sel = kvf_find_specified_selection(to_control->sel_list,
                                                     KUIS_HELP, var_token);
		kfree(to_sel->line);
	        to_sel->line = kstrdup(from_sel->line);
                kvf_delete_selection(from_sel);
            }

            /*
             * if they both have a help button w/ variable name "license",
             * delete one. (one "help" and one "license" are allowed)
	     * see trick documented above for quit button
             */
            var_token = kstring_to_token("license");
            if ((kvf_gui_named_item_exists(to_control->sel_list, KUIS_HELP,
                                           var_token) != NULL) &&
                (kvf_gui_named_item_exists(from_control->sel_list, KUIS_HELP,
                                           var_token) != NULL))
            {
                from_sel = kvf_find_specified_selection(from_control->sel_list,
                                                     KUIS_HELP, var_token);
		to_sel = kvf_find_specified_selection(to_control->sel_list,
                                                     KUIS_HELP, var_token);
		kfree(to_sel->line);
	        to_sel->line = kstrdup(from_sel->line);
                kvf_delete_selection(from_sel);
            }
        }


	 
	/* 
	 * no selections on "to" control - transfer directly 
         */
	if (to_control->sel_list == NULL)
	    to_control->sel_list = from_control->sel_list;

	/* 
 	 * selections on both "from" and "to" controls - merge 
         */
	else 
	{
	    /*
	     *  link front of "from" selection to end of "to" selection list
	     */
	    selection = to_control->sel_list;
            while (selection->next != NULL)
                selection = selection->next;
            selection->next = from_control->sel_list;
	}

	/* 
	 *  make sure "from" selections' backplanes are NULL so that 
	 *  they will be re-created on the new "to" control. also be sure
         *  to update backlinks, or many interesting & bizarre bugs will occur.
	 */
	selection = from_control->sel_list;
        while (selection != NULL)
        {
	    selection->back = NULL;
	    selection->back_control = to_control;
	    selection->back_subform = to_control->back_subform;
	    if (to_control->type == KPANE)
	        selection->back_guide = to_control->back_guide;
            selection = selection->next;
        }
	return(TRUE);
}
