/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>          Utilities to Maintain Form List 
   >>>>
   >>>>  Private:
   >>>>             kvf_get_entry()
   >>>>             kvf_add_entry()
   >>>>             kvf_delete_entry()
   >>>>             kvf_change_entry()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static kform_list *kformlist_hdr = NULL;

/*-----------------------------------------------------------
|
|  Routine Name: kvf_get_entry - retrieve an entry from the form list
|
|       Purpose: Returns the form list structure associated
|                to the given form structure.
|
|         Input: treeptr - the kform or kvf_subform that is 
|                         wanted out of the form list.
|        Output: none
|       Returns: The corresponding kform_list structure on success,
|		 NULL on failure
|
|    Written By: Mark Young & Stephanie Hallet
|          Date: Jul 15, 1992 13:50
| Modifications: Converted from Khoros 1.0 (MY)
|
------------------------------------------------------------*/

kform_list *kvf_get_entry(
   kaddr treeptr)
{
	kform_list	*current;

	/*
	 *  Search through the linked form list for the desired list
	 *  structure.
	 */
	if (kformlist_hdr == NULL)
	   return(NULL);
	 
	else if (kformlist_hdr->treeptr == treeptr || treeptr == NULL)
	   return(kformlist_hdr);

	current = kformlist_hdr;
	while (current->next != NULL && current->treeptr != treeptr)
	      current = current->next;

	if (current->treeptr == treeptr)
	{
	   /*
            *  Stick the current list form structure at the top of the list,
            *  since it will probably be used again soon.
            */
           current->prev->next = current->next;
           if (current->next != NULL)
              current->next->prev = current->prev;

           current->next = kformlist_hdr;
           current->prev = NULL;

           kformlist_hdr->prev = current;
           kformlist_hdr = current;
	   return(current);
	}

	else return(NULL);
}



/*-----------------------------------------------------------
|
|  Routine Name: kvf_add_entry - add an entry to the form list
|
|       Purpose: This routine is used to create and add a new
|		 form list structure to the list of forms.  Given
|		 a form, a callback routines, and corresponding
|		 client_data a new kform_list is malloc'ed and added
|		 to the static form list structure.
|
|         Input: treeptr - the kform or kvf_subform that is
|                           wanted out of the form list.
|                type        - the type of treeptr (KFORM or KSUNFORM)
|                busy        - whether to map the busy window on callback
|                routine     - the routine to call when a form has
|			       been modified
|                client_data - the client data to be returned with
|			       callback routine.
|
|        Output: none
|       Returns: TRUE (1) on success, FALSE (0) on failure
|
|    Written By: Mark Young & Stephanie Hallet
|          Date: Jul 15, 1992 13:55
| Modifications: Converted from Khoros 1.0 (MY)
|
------------------------------------------------------------*/

int kvf_add_entry(
   kaddr treeptr,
   int   type,
   int   busy,
   kfunc_void routine,
   kaddr client_data)
{
	kform_list	*entry;

	entry = (kform_list *) kcalloc(1,sizeof(kform_list));
	if (entry == NULL)
	{
	   kerror("kforms", "kvf_add_entry", 
                   "Unable to allocate new entry in form list");
	   return(FALSE);
	}

	if (kvf_get_entry(treeptr) != NULL)
        {
	   errno = KCALL;
	   kerror("kforms", "kvf_add_entry",
		  "Form has already been added to list!");
	   return(FALSE);
        }

	entry->type   	   = type;
	entry->treeptr	   = treeptr;
	entry->busy   	   = busy;
	entry->routine	   = routine;
	entry->client_data = client_data;
	entry->next	   = kformlist_hdr;
	entry->prev	   = NULL;

	if (kformlist_hdr != NULL)
	   kformlist_hdr->prev	= entry;

	kformlist_hdr = entry;
	return(TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: kvf_delete_entry - delete an entry from the form_list
|
|       Purpose: This routine is used to delete a previously added
|		 entry from the form_list structure.  Given the
|		 form structure we search the linked list of the
|		 form_list structure removing the entry from the
|		 list.
|
|         Input: treeptr - the kform or kvf_subform that is wanted out of the
|			  form_list.
|
|        Output: RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Stephanie Hallet
|          Date: Jul 15, 1992 14:09
| Modifications: Converted from Khoros 1.0 (MY)
|
------------------------------------------------------------*/

void kvf_delete_entry(
   kaddr treeptr)
{
	kform_list	*current;

	
	if ((current = kvf_get_entry(treeptr)) != NULL)
	{
	   /*
	    *  Delete the entry from the list by adjusting the
	    *  next & previous pointers and then freeing the
	    *  kform_list structure.
	    */
	   if (current->next == NULL && current->prev == NULL)
	   {
	      kformlist_hdr = NULL;
	   }
	   else if (current->prev == NULL)
	   {
	      current->next->prev = NULL;
	      kformlist_hdr = current->next;
	   }
	   else if (current->next == NULL)
	   {
	      current->prev->next = NULL;
	   }
	   else
	   {
	      current->next->prev = current->prev;
	      current->prev->next = current->next;
	   }
	   kfree(current);
	}
}



/*-----------------------------------------------------------
|
|  Routine Name: kvf_change_entry - change the entry associated
|				    with the form.
|
|       Purpose: This routine is used to cahnge an already existing
|		 form_list structure in the list of forms.  Given
|		 the form the user can then change the callback
|		 routine and corresponding client_data.
|
|         Input: treeptr - the kform or kvf_subform that is wanted out of the
|			  form_list.
|		 busy        - whether to map the busy window on callback
|                routine     - the routine to call when a form has
|			       been modified
|                client_data - the client data to be returned with
|			       callback routine.
|
|        Output: RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young & Stephanie Hallet
|          Date: Aug 10, 1992 15:04
| Modifications: Converted from Khoros 1.0 (MY)
|
------------------------------------------------------------*/

int kvf_change_entry(
   kaddr treeptr,
   int   busy,
   kfunc_void routine,
   kaddr client_data)
{
	kform_list	*entry;

	if ((entry = kvf_get_entry(treeptr)) == NULL)
        {
	   errno = KCALL;
	   kerror("kforms", "kvf_change_entry", "Form entry doesn't exist!");
	   return(FALSE);
        }
	entry->busy   	   = busy;
	entry->routine	   = routine;
	entry->client_data = client_data;
	return(TRUE);
}
