/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>             Attribute Actions For Printing UIS        <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                 kvf_get_print_uis()                   <<<<
   >>>>                 kvf_set_print_uis()                   <<<<
   >>>>                 kvf_get_print_pane()                  <<<<
   >>>>                 kvf_set_print_pane()                  <<<<
   >>>>                 kvf_get_print_subform()               <<<<
   >>>>                 kvf_set_print_subform()               <<<<
   >>>>                                                       <<<<
   >>>>                 kvf_get_uis_location()                <<<<
   >>>>                 kvf_set_uis_location()                <<<<
   >>>>                 kvf_get_subform_file()                <<<<
   >>>>                 kvf_set_subform_file()                <<<<
   >>>>                 kvf_get_pane_file()                   <<<<
   >>>>                 kvf_set_pane_file()                   <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"

static char *get_uis_fullpath PROTO((kform_struct *, char *));

/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_print_uis
|       Purpose: Initializes the filename for kvf_print_UIS to NULL, as
|                we don't know what the name of the file should be.
|
|         Input: kformstruct - ptr to pane, subform, or master
|                attribute   - KVF_PRINT_UIS ("kvf_print_uis")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: March 22, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_print_uis(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char **uis_filename;

        /* set the pointer to be returned */
        uis_filename = (char **) calldata;

	/* initialize to NULL */
	*uis_filename = NULL;

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_print_uis
|
|       Purpose: This routine writes out the UIS file;
|                it is an action attribute routine.
|
|         Input: kformstruct - ptr to pane, subform, or master
|                attribute   - KVF_PRINT_UIS ("kvf_print_uis")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: March 22, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_print_uis(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kform        *form;
	char         **uis_filename;

        uis_filename  = (char **) calldata;
	if ((kformstruct->type == KCONTROL) ||
	    (kformstruct->type == KMASTER)  ||
	    (kformstruct->type == KGUIDEPANE)  ||
	    (kformstruct->type == KPANE))
	    form    = kformstruct->Controlptr->back_form;
	else if (kformstruct->type == KFORM)
	    form    = kformstruct->Formptr;
	else if (kformstruct->type == KSUBFORM)
	    form    = kformstruct->Subformptr->back_form;
	else if (kformstruct->type == KGUIDE)
	    form    = kformstruct->Guideptr->back_form;
	else if (kformstruct->type == KSELECTION)
	    form    = kformstruct->Selptr->back_form;
	else 
	{    
	    kerror("kforms", "kvf_set_print_uis", 
		   "kformstruct of unexpected type");
	    return(FALSE);
	}

	if (kvf_print_UIS_file(form, *uis_filename, FALSE, FALSE))
	{
	    kinfo(KSTANDARD,  "UIS written to file %s", *uis_filename);
	    return(TRUE);
	}
	else return(FALSE);
}


/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_print_pane
|       Purpose: Initializes the filename for kvf_print_pane to NULL, as
|                we don't know what the name of the file should be.
|
|         Input: kformstruct - ptr to pane, subform, or master
|                attribute   - KVF_PRINT_PANE ("kvf_print_pane")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: March 22, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_print_pane(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char **pane_filename;

        /* set the pointer to be returned */
        pane_filename = (char **) calldata;

	/* initialize to filename stored in comment of pane (if any) */
	*pane_filename = kvf_retrieve_pane_filename(kformstruct->Controlptr);


	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_print_pane
|
|       Purpose: This routine writes out the *.pane file;  
|                it is an action attribute routine.
|
|		 ALSO modifies the control comment of the pane so that it has:
|                @-p {filename}
|                @-F ....
|                @-M ....
|
|                Adding these lines will cause the main form to substitute
|                a [-p] line for the pane definition when (if) it is written
|                out.
|
|         Input: object    - backplane of the selection
|                attribute - KVF_PRINT_PANE ("kvf_print_pane")
|                calldata  - passes in the filename to print UIS into
|        Output: none
|
|          Date: Sept 22, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_print_pane(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	/* set the name of the *.pane file in the pane structure comment */
	if (!(kvf_set_pane_file(kformstruct, KVF_PANE_FILE, calldata)))
	    return(FALSE);

	/* now, actually write out the *.pane file */
	if (!(kvf_print_panefile(kformstruct->Controlptr, TRUE)))
	    return(FALSE);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_print_subform
|       Purpose: Initializes the filename for kvf_print_subform to NULL, as
|                we don't know what the name of the file should be.
|
|         Input: object    - backplane of the selection
|                attribute - KVF_PRINT_SUBFORM ("kvf_print_subform")
|                calldata  - passes in the filename to print UIS into
|        Output: none
|          Date: Sept 22, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_print_subform(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char **subform_filename;

        /* set the pointer to be returned */
        subform_filename = (char **) calldata;

	/* initialize to filename stored in comment of subform (if any) */
	*subform_filename = kvf_retrieve_subform_filename(kformstruct->Subformptr);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_print_subform
|
|       Purpose: This routine writes out the *.sub file;
|                it is an action attribute routine.
|
|		 ALSO modifies the control comment of the pane so that it has:
|                @-k {filename}
|                @-F ....
|
|                Adding these lines will cause the main form to substitute
|                a [-k] line for the subform definition when (if) it is written
|                out.
|
|         Input: object    - backplane of the selection
|                attribute - KVF_PRINT_SUBFORM ("kvf_print_subform")
|                calldata  - passes in the filename to print UIS into
|        Output: none
|
|          Date: Sept 22, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_print_subform(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        /* set the name of the *.pane file in the pane structure comment */
        if (!(kvf_set_subform_file(kformstruct, KVF_SUBFORM_FILE, calldata)))
	    return(FALSE);

	if (!(kvf_print_subformfile(kformstruct->Subformptr, FALSE))) 
	    return(FALSE);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_uis_location
|       Purpose: Gets the directory location for UIS files that might
|                be written out: *.form, *.pane, and *.subform.  This
|                directory is saved in form struct as form->uis_location.
|
|         Input: kformstruct - kformstruct associated with form ptr
|                attribute   - KVF_UIS_LOCATION ("kvf_uis_location")
|        Output: calldata    - passes back UIS directory 
|          Date: June 15, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_uis_location(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        char     **uis_location;

        /* set the pointer to be returned */
        uis_location = (char **) calldata;
        *uis_location = kstrdup(kformstruct->Formptr->uis_location);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|
|       Routine: kvf_set_uis_location
|       Purpose: Sets the directory location for UIS files that might
|                be written out: *.form, *.pane, and *.subform.  This
|                directory is stored in form struct as form->uis_location.
|
|         Input: kformstruct - kformstruct associated with form ptr
|                attribute   - KVF_UIS_LOCATION ("kvf_uis_location")
|        Output: calldata    - passes back UIS filename (NULL)
|          Date: June 15, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_uis_location(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	char  **uis_location = (char **) calldata;

	kformstruct->Formptr->uis_location = kstrdup(*uis_location);
        return(TRUE);
}




/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_pane_file
|       Purpose: Gets the name of the *.pane file associated with the pane.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_PANE_FILE ("kvf_pane_file")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_pane_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        kcontrol *pane;
        char     **pane_filename;

        /* set the pointer to be returned */
        pane_filename = (char **) calldata;

        pane = kformstruct->Controlptr;

        *pane_filename = kvf_retrieve_pane_filename(pane);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_pane_file
|
|       Purpose: Sets the name of the *.pane file associated with
|                the pane, by saving it in the comment of the pane 
|                control structure in a "secret" format, which is just
|                the -p line as it needs to be written out in the 
|                controlling *.subform or *.form file, plus the -F and -M
|                lines which need to go out to the *.pane file so it will
|                be syntactically correct.  For example:
|               
| @ -p $KHOROS_CORE/objects/xvroutine/craftsman/uis/script.pane
| @ -F 4.3 1 1 52x20+15+3.5 +1+1 'Creating a new script object' newScript 
| @   -M 1 1 52x20+10+5 +0+0 ' ' template
|                
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_PANE_FILE ("kvf_pane_file")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_pane_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kcontrol     *pane;
	kfile        *file;
	char         *fullpath;
        char         temp[KLENGTH];
	char         **pane_filename = (char **) calldata;

	fullpath = get_uis_fullpath(kformstruct, *pane_filename);
	if (fullpath == NULL)
	    return(FALSE);

	/*
	 *  open the file to make sure it is writable by user.
         *  return FALSE if it's not.  doublechecking, as get_uis_fullpath()
         *  should have returned FALSE in this case.
	 */
	if ((file = kfopen(fullpath, "w")) == NULL)
        {
            kerror("kforms", "kvf_set_pane_file",
                    "Cannot write to output UIS file %s", fullpath);
            return(FALSE);
        }

	/*
	 *  write the *.pane file info in it's "secret" format, 
         *  and store it in the comment of the pane structure 
	 */
	pane = kformstruct->Controlptr;
        ksprintf(temp, "@-p %s\n@%s\n@%s\n", fullpath,
                 pane->back_form->control_line, 
		 pane->back_subform->control_line);
        kfree(pane->control_comment);
        pane->control_comment = kstrdup(temp);

        return(TRUE);
}



/*-----------------------------------------------------------
|
|
|       Routine: kvf_get_subform_file
|       Purpose: Gets the name of the *.subform file associated with the subform.
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_SUBFORM_FILE ("kvf_subform_file")
|        Output: calldata    - passes back UIS filename (NULL)
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int kvf_get_subform_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	ksubform *subform;
	char     **subform_filename;

        /* set the pointer to be returned */
        subform_filename = (char **) calldata;

	subform = kformstruct->Subformptr;

	*subform_filename = kvf_retrieve_subform_filename(subform);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: kvf_set_subform_file
|
|       Purpose: Sets the name of the *.subform file associated with
|                the subform, by saving it in the comment of the subform
|                control structure in a "secret" format, which is just
|                the -k line as it needs to be written out in the
|                controlling *.form file, plus the -F line which needs
|                to go out to the *.subform file so it will
|                be syntactically correct.  For example:
|
| @ -k $ENVISION/objects/xvroutine/xprism/uis/print.subform
| @ -F 4.3 1 0 170x7+10+20 +35+1 ' ' form
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - KVF_SUBFORM_FILE ("kvf_subform_file")
|                calldata    - passes in the filename to print UIS into
|        Output: none
|
|          Date: June 10, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
/* ARGSUSED */
int kvf_set_subform_file(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	ksubform *subform;
	char     *fullpath;
        char     temp[KLENGTH];
        kfile    *file;
        char     **subform_filename = (char **) calldata;

        /*
         * determine fullpath for *.subform file
         */
	fullpath = get_uis_fullpath(kformstruct, *subform_filename);
        if (fullpath == NULL)
            return(FALSE);

        /*
         *  open the file to make sure it is writable by user.
         *  return FALSE if it's not.  doublechecking, as get_uis_fullpath()
         *  should have returned FALSE in this case.
         */
        if ((file = kfopen(fullpath, "w")) == NULL)
        {
            kerror("kforms", "kvf_set_subform_file",
                    "Cannot write to output UIS file %s", fullpath);
            return(FALSE);
        }

        /*
         *  write the *.subform file info in it's "secret" format,
         *  and store it in the comment of the pane structure
         */
	subform = kformstruct->Subformptr;
        ksprintf(temp, "@-k %s\n@%s\n", fullpath, 
	         subform->back_form->control_line);
        kfree(subform->control_comment);
        subform->control_comment = kstrdup(temp);

        return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: get_uis_fullpath
|
|       Purpose: Determines the full path to which a UIS file will be written 
|                out.  Path precedence is determined by the following cases:
|
|          1. a. pane_filename DOES include path, eg, "./float.pane",
|             b. KVF_UIS_LOCATION is ignored, NULL or non-NULL
|             c. current working directory is ignored.
|             => the path is used exactly as it appears.
|             => "./float.pane"
|         
|          2. a. pane_filename DOES NOT include path, eg, "float.pane",
|             b. KVF_UIS_LOCATION non-NULL, eg "$FOO/objects/xvroutine/test/uis"
|             c. current working directory is ignored.
|             => path combines KVF_UIS_LOCATION value with filename given
|             => "$FOO/objects/xvroutine/test/uis/float.pane"
|         
|          3. a. pane_filename DOES NOT include path, eg, "float.pane",
|             b. KVF_UIS_LOCATION is NULL,
|             c. current working directory is in a toolbox.
|             => the current working directory wrt toolbox is used.
|             => eg, if location is in the "objects" directory of $FOO toolbox,
|             => "$FOO/objects/float.pane"
|         
|          4. a. pane_filename DOES NOT include path, eg, "float.pane",
|             b. KVF_UIS_LOCATION is NULL,
|             c. current working directory is NOT in a toolbox,
|             => the absolute path to the current working directory is used.
|             => eg, if working directory was the user's home, path might be:
|             => "/research/vision/danielle/float.pane"
|         
|          5. IN ANY OF THE ABOVE CASES, if the user CANNOT WRITE to the
|             resolved directory (whatever it was),
|             => we obtain the value of the TMPDIR environment variable,
|             => and drop the file there.
|         
|          6. If TMPDIR is not set, we try to use /usr/tmp.
|
|          7. After all this, if we cannot write to the file, we return FAILURE.
|          
|         Input: kformstruct - kformstruct associated with calling kvf_get_*() 
|                              or kvf_set_*() routine, so we can call 
|                              kvf_get_attribute to get KVF_UIS_LOCATION
|                filename    - name of file, sent in as calldata to 
|                              kvf_set_print_pane, kvf_set_print_subform,
|                              kvf_set_pane_file, or kvf_set_subform_file.
|                              If initiated through use of internal menuform
|                              for subform or pane, this will be the text that
|                              was entered by the user.
|        Output: none
|       Returns: The full path for UIS file on success, NULL on failure
|          Date: June 15, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
static char *get_uis_fullpath(
   kform_struct *kformstruct,
   char         *filename)
{
	kform    *form;
	ksubform *subform;
	kcontrol *control;
	char     *uis_dir, *tmp_dir = NULL;
	char     path[KLENGTH];
	char     basename[KLENGTH];
	char     directory[KLENGTH];

	path[0] = '\0';
	if (kformstruct->type == KPANE)
	{
	    control = kformstruct->Controlptr;
	    form = control->back_form;
	}
	else if (kformstruct->type == KSUBFORM)
	{
	    subform = kformstruct->Subformptr;
	    form = subform->back_form;
	}
	else 
	{
	    kerror("kforms", "get_uis_fullpath", 
	           "only kformstructs of type KPANE and KSUBFORM supported");
	    return(NULL);
	}
/*
	kvf_get_attribute(form->back_kformstruct, 
			  KVF_UIS_LOCATION, &uis_dir);
*/
	uis_dir = kdirname(form->uis_location, NULL);

	/* 
	 * case 1: pane_filename DOES include path, KVF_UIS_LOCATION is ignored,
         *         path is used exactly as it appears and is returned.
	 */
	(void) kbasename(filename, basename);
	if (kstrcmp(basename, filename) != 0)
	{
	    ksprintf(path, "%s", filename);
	}
	else
	{
            /*
             * case 2:  pane_filename DOES NOT include path, KVF_UIS_LOCATION 
             *          non-NULL, path combines KVF_UIS_LOCATION with filename 
             *          and is returned.
             */
	    if (uis_dir != NULL)
	    {
		ksprintf(path, "%s/%s", uis_dir, filename);
	        kfree(uis_dir);
	    }

	    else
	    {
	        /*
	         * case 3: pane_filename DOES NOT include path, KVF_UIS_LOCATION
                 *         NULL, current working directory is in a toolbox,
                 *         path of current working directory wrt toolbox is used
	         */
        	if (!(ktbpath(filename, path)))
	
		    /*
 	             * case 4: pane_filename DOES NOT include path, 
		     *         KVF_UIS_LOCATION is NULL, current working 
                     *         directory is NOT in a toolbox, absolute path used
	             */
                     (void) kexpandpath(filename, NULL, path);
	    }
	}

	/*
	 * 5. IN ANY OF THE ABOVE CASES, if the user CANNOT WRITE to the
         *    resolved directory (whatever it was), we use the path 
         *    specified by the TMPDIR environment variable
	 */ 
	(void) kdirname(path, directory);
	if (kaccess(directory, W_OK) == -1)
	{
	    (void) kbasename(filename, basename);
	    tmp_dir = kgetenv("TMPDIR");
	    if (tmp_dir == NULL) 
	    {
	        /*
	         * 6. TMPDIR was not set; try to use /usr/tmp
	         */ 
		tmp_dir = "/usr/tmp";
	        kerror("kforms", "get_uis_fullpath",
		       "Directory %s is not accessable.  Attempting to write %s to %s instead.", directory, basename, tmp_dir);
	    }
	    else
	    {

	        kerror("kforms", "get_uis_fullpath",
		       "Directory %s is not accessable.  Attempting to write %s to %s (as specified by your TMPDIR environment variable) instead.", directory, basename, tmp_dir);
	    }
	    ksprintf(path, "%s/%s", tmp_dir, basename);
	}

	/*
	 *  access the file to make sure it is writable by user.
         *  return FALSE if it's not.
	 */
	(void) kdirname(path, directory);
	if (kaccess(directory, W_OK) == -1)
	{
	    kerror("kforms", "get_uis_fullpath",
		   "Cannot write to UIS file, '%s'", path);
            return(NULL);
	}
	
	return(kstrdup(path));
}
