/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>         Routines To Print CLUI Program Syntax         <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                kgen_clui_print_selections()           <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

static void print_arg_expls        PROTO((kfile *, kselection *, int, int));
static void print_toggle_expls     PROTO((kfile *, kselection *, int, int));
static void print_list_expls       PROTO((kfile *, kselection *, int, int));
static void print_group_expls      PROTO((kfile *, kselection *, int, int));
static void print_spaces           PROTO((kfile *, int, int));

/*-----------------------------------------------------------
|
|  Routine Name: kgen_clui_print_selections
|
|       Purpose: Prints the explanation of each selection
|                in the GUI (pane) of the software object.
|
|         Input: file - file to which to write explanation
|                       (associated with the online help file)
|	 Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: May 16, 1993
| Modifications: 
|
------------------------------------------------------------*/


void kgen_clui_print_selections(
    kfile *file)
{
	int        indx;
	kselection *selection;

	/* 
	 * print out usage for required arguments 
	 */
        kfprintf(file, "\n");
        if (kgen_req_num > 0)
        {
            indx = 0;
	    while (indx < kgen_req_num)
	    {
		selection = kgen_req_sels[indx++];

		if (selection->type == KUIS_TOGGLE)
		    print_toggle_expls(file, selection, TRUE, FALSE);

		else if ((selection->type == KUIS_LIST)        ||
		         (selection->type == KUIS_DISPLAYLIST) ||
		         (selection->type == KUIS_CYCLE))
		    print_list_expls(file, selection, TRUE, FALSE);

		else if ((selection->type == KUIS_MUTEXCL) ||
		         (selection->type == KUIS_MUTINCL) ||
		         (selection->type == KUIS_GROUP))
		    print_group_expls(file, selection, TRUE, FALSE);

		else print_arg_expls(file, selection, TRUE, FALSE);
	    }
            kfprintf(file, "\n");
        }

	/* 
	 * print out usage for optional arguments 
	 */
	if (kgen_opt_num > 0)
	{
	    indx = 0;
	    while (indx < kgen_opt_num)
	    {
                selection = kgen_opt_sels[indx++];

                if (selection->type == KUIS_TOGGLE)
                    print_toggle_expls(file, selection, FALSE, FALSE);

                else if ((selection->type == KUIS_LIST)        ||
                         (selection->type == KUIS_DISPLAYLIST) ||
                         (selection->type == KUIS_CYCLE))
                    print_list_expls(file, selection, FALSE, FALSE);

                else if ((selection->type == KUIS_MUTEXCL) ||
                         (selection->type == KUIS_MUTINCL) ||
                         (selection->type == KUIS_GROUP))
                    print_group_expls(file, selection, FALSE, FALSE);

                else print_arg_expls(file, selection, FALSE, FALSE);
	    }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: print_arg_expls
|
|       Purpose: Prints the usage for the "regular" arguments
|                of a program (everything but toggles, lists,
|                groups, etc).
|
|         Input: file      - file to which to write usage (may be kstderr)
|                selection - selection representing argument
|                required  - pass TRUE if argument is required
|                nested    - pass FALSE if argument is not nested,
|                            pass 1 if argument is nested once,
|                            pass 2 if argument is nested twice
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/

static void print_arg_expls(
    kfile      *file,
    kselection *selection,
    int        required, 
    int        nested)
{
	int    i;
	char   *title;
	char   *description;
	char   *ascii_default = NULL;
	char   *datatype = NULL;
	char   desc_buffer[KLENGTH];
	char   temp[KLENGTH];
	char   title_buffer[KLENGTH];

	kvf_get_attributes(selection->back_kformstruct,
			   KVF_DESCRIPTION, &description,
			   KVF_TITLE,       &title,
			   NULL);

	if (description == NULL) 
	    description = kstrdup("no description");

        if (description != NULL)
	    sprintf(desc_buffer, "%s", description);
	else sprintf(desc_buffer, " ");

	if (title != NULL)
	    sprintf(title_buffer, "%s", title);
	else sprintf(title_buffer, " ");

	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/*
	 *  required argument
	 */
	if (required)
	{
	    kstring_cleanup(title, temp);
	    kfprintf(file, "\"%s\"", temp);

	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "  (required %s", datatype);

            if ((selection->type == KUIS_DOUBLE) ||
                (selection->type == KUIS_FLOAT)  ||
                (selection->type == KUIS_INTEGER))
	    {
	        kfprintf(file, ", ");
                kgen_clui_print_bounds(file, selection, KGEN_CLUI_PRINT_USAGE);
	    }
	    kfprintf(file, ")\n");

	    print_spaces(file, 0, 6+2*nested);
	    kfprintf(file, "%s\n\n", desc_buffer);
	}

        /*
         *  optional argument
         */
	else
	{
	    kstring_cleanup(title, temp);
	    kfprintf(file, "\"%s\"", temp);

	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "  (optional %s", datatype);


            if ((selection->type == KUIS_FLOAT)  ||
                (selection->type == KUIS_DOUBLE) ||
                (selection->type == KUIS_INTEGER))
	    {
		kfprintf(file, ", ");
                kgen_clui_print_bounds(file, selection, KGEN_CLUI_PRINT_USAGE);
	    }
	    kfprintf(file, ")\n");

	    print_spaces(file, 0, 6+2*nested);
	    kfprintf(file, "%s\n", desc_buffer);

            ascii_default = kgen_ascii_def_val(selection);
	    if (ascii_default != NULL)
	    {
	        print_spaces(file, 0, 6+2*nested);
                kfprintf(file, "[default = %s]\n", ascii_default);
	    }
	    kfprintf(file, "\n");
	}

	kfree(description);
	kfree(title);
}

/*-----------------------------------------------------------
|
|  Routine Name: print_toggle_expls
|
|       Purpose: Prints the usage for the optional toggle arguments
|                of a program; variable formatting for usage.
|
|         Input: file      - file to which to write usage (may be kstderr)
|                selection - selection representing argument
|                required  - pass TRUE if argument is required
|                nested    - pass FALSE if argument is not nested,
|                            pass 1 if argument is nested once,
|                            pass 2 if argument is nested twice
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: May 16, 1993
| Modifications:
|
------------------------------------------------------------*/
static void print_toggle_expls(
    kfile      *file,
    kselection *selection,
    int        required,
    int        nested)
{
	int   i;
	char  *description;
	char  *title;
	char  *ascii_default = NULL;
	char  desc_buffer[KLENGTH];
        char  title_buffer[KLENGTH];
        char  temp[KLENGTH];

        kvf_get_attributes(selection->back_kformstruct,
                           KVF_DESCRIPTION, &description,
                           KVF_TITLE,       &title,
                           NULL);

	if (description == NULL) 
	    description = kstrdup("no description");

        if (description != NULL)
            sprintf(desc_buffer, "%s", description);
        else sprintf(desc_buffer, " ");

        if (title != NULL)
            sprintf(title_buffer, "%s", title);
        else sprintf(title_buffer, " ");

	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	kstring_cleanup(title, temp);
        kfprintf(file, "\"%s\"", title);

	/*
	 *  required toggle argument
	 */
	if (required)
	{
            kfprintf(file, " (required toggle)\n");

            kgen_clui_print_toggle_vals(file, selection, KGEN_CLUI_PRINT_USAGE, 
	    		               FALSE);
	    print_spaces(file, 0, 6+2*nested);
            kfprintf(file, "%s\n\n", desc_buffer);
	}

        /*
         *  optional toggle argument
         */
	else
	{
            kfprintf(file, " (optional toggle)\n");

            kgen_clui_print_toggle_vals(file, selection,
                                        KGEN_CLUI_PRINT_USAGE, nested);

	    print_spaces(file, 0, 6+2*nested);
            kfprintf(file, "%s\n", desc_buffer);

            ascii_default = kgen_ascii_def_val(selection);
	    if (ascii_default != NULL)
	    {
	        print_spaces(file, 0, 6+2*nested);
                kfprintf(file, "[default = %s]\n", ascii_default);
	    }

	    kfprintf(file, "\n");
	}
}



/*-----------------------------------------------------------
|
|  Routine Name: print_list_expls
|
|       Purpose: Prints the usage for the optional toggle arguments
|                of a program; variable formatting for usage.
|
|         Input: file      - file to which to write usage (may be kstderr)
|                selection - selection representing argument
|                required  - pass TRUE if argument is required
|                nested    - pass FALSE if argument is not nested,
|                            pass 1 if argument is nested once,
|                            pass 2 if argument is nested twice
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: May 16, 1993
| Modifications:
|
------------------------------------------------------------*/
static void print_list_expls(
    kfile      *file,
    kselection *selection,
    int        required,
    int        nested)
{
	int    i;
        char  *description;
        char  *title;
        char  *ascii_default = NULL;
        char  *datatype      = NULL;
        char  desc_buffer[KLENGTH];
        char  title_buffer[KLENGTH];
        char  temp[KLENGTH];

        kvf_get_attributes(selection->back_kformstruct,
                           KVF_DESCRIPTION, &description,
                           KVF_TITLE,       &title,
                           NULL);

	if (description == NULL) 
	    description = kstrdup("no description");

        if (description != NULL)
            sprintf(desc_buffer, "%s", description);
        else sprintf(desc_buffer, " ");

        if (title != NULL)
            sprintf(title_buffer, "%s", title);
        else sprintf(title_buffer, " ");


	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/* 
	 * required list argument 
         */
	if (required)
	{
	    kstring_cleanup(title, temp);
            kfprintf(file, "\"%s\"", temp);
	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "  (required %s)\n", datatype);

	    print_spaces(file, 0, 6+2*nested);
	    kfprintf(file, "%s\n", desc_buffer);

            kgen_clui_print_list_vals(file, selection, KGEN_CLUI_PRINT_USAGE, 
			              FALSE);
            kfprintf(file, "\n");
	}
	
	/* 
	 * optional list argument 
         */
	else
	{
	    kstring_cleanup(title, temp);
            kfprintf(file, "\"%s\"", temp);
	    datatype = kgen_ascii_datatype(selection->type);
	    kfprintf(file, "  (optional %s)\n", datatype);

	    print_spaces(file, 0, 6+2*nested);
	    kfprintf(file, "%s\n", desc_buffer);

            kgen_clui_print_list_vals(file, selection,
                                      KGEN_CLUI_PRINT_USAGE, nested);

            if (nested)
                kfprintf(file, "\t\t\t");
            kfprintf(file, "\t\t");

            ascii_default = kgen_ascii_def_val(selection);
	    if (ascii_default != NULL)
                kfprintf(file, "[default = %s]\n", ascii_default);
	    kfprintf(file, "\n");
	}

}

/*-----------------------------------------------------------
|
|  Routine Name: print_group_expls
|
|       Purpose: Prints the usage for an ME/MI group; variable
|                formatting for usage.
|
|         Input: file      - file to which to write usage (may be kstderr)
|                group_start - header of group list
|                required    - pass TRUE if group is a required ME group
|                nested      - pass FALSE if argument is not nested,
|                              pass 1 if argument is nested once,
|                              pass 2 if argument is nested twice
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications: 
|
------------------------------------------------------------*/

static void print_group_expls(
    kfile      *file,
    kselection *group_start,
    int        required,
    int        nested)
{
	int        i, group_size; 
	kselection *groupmember;

	kfprintf(file, "\n");

	/*
	 *  indent appropriate for nesting (if any)
	 */
	for (i = 0; i < 2+2*nested; i++)
	   kfprintf(file, " ");

	/*
	 *  print introductory stmt about group
	 */
	if (required)
	    kfprintf(file, "Required ");
	else kfprintf(file, "[Optional ");

	group_size = kgen_get_group_size(group_start);
	if (group_start->type == KUIS_MUTEXCL)
	{
	    kfprintf(file, "Mutually Exclusive Group - Specify ");
	    if (required)
	       kfprintf(file, "exactly one of the %d selections:", group_size);
	    else kfprintf(file, "one or none of the %d selections:", group_size);
	}
	else if (group_start->type == KUIS_MUTINCL)
	{
	    kfprintf(file, "Mutually Inclusive Group - "
		     "Specify all or none of the %d selection:", group_size);
	}
	else if (group_start->type == KUIS_GROUP)
        {
            kfprintf(file, "Loose Group - "
		     "Specify at least one of the %d selection:", group_size);
        }
	if (!required)
	    kfprintf(file, "]\n");
	else kfprintf(file, "\n");

	/*
	 *  print usage for all members in the group
	 */
	groupmember = group_start->group_next;
	while (groupmember != NULL)
	{
	    /* ignore blanks */
	    if (groupmember->type == KUIS_BLANK)
	    {
		groupmember= groupmember->next;
		continue;
	    }

	    /* toggle in group */
	    if (groupmember->type == KUIS_TOGGLE)
	 	print_toggle_expls(file, groupmember, FALSE, nested + 1);

	    /* list in group */
            else if ((groupmember->type == KUIS_LIST)        ||
                     (groupmember->type == KUIS_DISPLAYLIST) ||
                     (groupmember->type == KUIS_CYCLE))
                print_list_expls(file, groupmember, FALSE, nested + 1);
	  
	    /* group nested in group */
            else if ((groupmember->type == KUIS_MUTEXCL) ||
                     (groupmember->type == KUIS_MUTINCL) ||
                     (groupmember->type == KUIS_GROUP))
		print_group_expls(file, groupmember, FALSE, nested + 1);

	    /* "regular" argument in group */
	    else print_arg_expls(file, groupmember, FALSE, nested + 1);

	    groupmember = groupmember->next;
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: print_spaces
|
|       Purpose: Used to make the usage statement pretty and lined
|                up, this routine takes the number of spaces that have
|                already been used in the column, and then the number
|                of spaces in the column, and pads out to the end of
|                the column with spaces.
|
|         Input: file        - file to which to write usage (may be kstderr)
|                used_spaces - number of the spaces in the column already used
|                spacenum    - total number of the spaces in the column
|        Output: none
|       Returns: The length of the longest variable name in selection list
|    Written By: Danielle Argiro
|          Date: April 19, 1993
| Modifications:
|
------------------------------------------------------------*/
static void print_spaces(
    kfile *file,
    int   used_spaces,
    int   spacenum)
{
	int i;

        for (i = 0; i < (spacenum - used_spaces); i++)
            kfprintf(file, " ");
}

