/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Initialize Khoros Command Line Interface Routines
   >>>>
   >>>>	 Static:
   >>>>  Private:
   >>>>             kgen_initialize()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


static int write_answer_outfile PROTO((char *, kform *, char *));
static int read_answer_infile   PROTO((char *, kform *, char *));

/************************************************************
*
*  Routine Name: kgen_initialize - initialize kcodegen library
*
*       Purpose: Initializes the kcodegen library
*
*         Input: pane_path - path to software object's *.pane file
*                tbname    - name of toolbox in which software object exists
*                oname     - name of software object 
*                obj_type  - type of software object, one of:
*			     KGEN_KROUTINE, KGEN_XVROUTINE, or KGEN_PANE
*                usage_additions - usage_additions() routine of kroutine or
*                                  xvroutine; NULL otherwise
*        Output:
*       Returns: TRUE if successful, FALSE otherwise.
*    Written By: Danielle Argiro and Mark Young
*          Date: April 8, 1994
*
*************************************************************/

kform *kgen_initialize(
   char       *pane_path,
   int        obj_type,
   char       *tbname,
   char       *oname,
   void      (*usage_additions)(void))
{
        int   usage, fullusage, version, prompting; 
	int   ansin, ansout, ansprint, gui, status;
        kform *form = NULL;
	kselection *selection;
	char  temp[KLENGTH], command[KLENGTH], *args[KLENGTH];
	char  *arguments, *ansinfile, *ansoutfile, *ansprintfile;

	/*
	 *  Create the clui from the pane specification
	 */
	if ((obj_type != KGEN_KROUTINE) &&
	    (obj_type != KGEN_XVROUTINE) &&
	    (obj_type != KGEN_PANE) &&
	    (obj_type != KGEN_NONE))
	{
	    kerror(KCODEGEN, "kgen_initialize",
		   "Software objects supported are limited to kroutines, xvroutines, and pane objects");
	    return(NULL);
	}

	if ((form = kvf_create_form(pane_path, NONE, NULL, NULL)) == NULL)
	{
	   if (!(ktoolbox_query_tbname(tbname)))
	   {
	       kstring_upper(tbname, temp);
	       kerror("kcodegen", "kgen_initialize", "Cannot open the $%s toolbox, in which %s exists.  Please check the Toolbox File(s) specified by your KHOROS_TOOLBOX environment variable and make sure that the $%s toolbox is properly referenced before trying again.", temp, oname, temp);
	       kexit(KEXIT_FAILURE);
	   }
	   return(NULL);
	}

	/*
	 *  Build a command argument string from argv[] & argc;
         *  uses quotes for spaces, tabs, newlines, etc.
	 */
	kclui_arguments(temp);


	if ((kclui_consolidate_args(form) > 1 && obj_type != KGEN_NONE) ||
	    (obj_type == KGEN_XVROUTINE))
	{
	   /*
	    * for kroutines & panes, use extended pane w/ std args for kroutines
	    */
	   if ((obj_type == KGEN_KROUTINE) || (obj_type == KGEN_PANE))
	      status = kvf_append_selections(form, KGEN_KSTDARGS_PANEPATH);

	   /*
	    * for xvroutines, use extended pane w/ std args for xvroutines
	    */
	   else
	       status = kvf_append_selections(form, KGEN_XVSTDARGS_PANEPATH);

	   if (status == FALSE)
              return(NULL);

	   /*
	    *  modify form tree associated w/ pane from the command string given
	    *  this sets the -P, -V, -P, -gui, -A, and -a arguments, as well as
            *  assigns values in the form tree to any other program arguments
            *  that may appear on the command line.
	    */
	   if (!(kvf_modify_form_from_cmd(form, temp, TRUE)))
	       kexit(KEXIT_FAILURE);

	   /*
	    *  get any standardized command line arguments
	    */
	   (void) kclui_get_flag(form, "U",     &usage);
	   (void) kclui_get_flag(form, "usage", &fullusage);
	   (void) kclui_get_flag(form, "V",     &version);
	   (void) kclui_get_flag(form, "P",     &prompting);
	   (void) kclui_get_flag(form, "g""ui",   &gui);
	   (void) kclui_get_outfile(form, "A",  &ansoutfile, &ansout);
	   (void) kclui_get_infile (form, "a",  &ansinfile,  &ansin);
	   (void) kclui_get_infile (form, "ap", &ansprintfile, &ansprint);
	}
	else
	{
	   /*
	    *  modify form tree associated w/ pane from the command string
	    *  in order to assign values in the form tree for the program
	    *  arguments that may appear on the command line.
	    */
	   if (!(kvf_modify_form_from_cmd(form, temp, TRUE)))
	       kexit(KEXIT_FAILURE);

	   ansoutfile = ansinfile = ansprintfile = NULL;
	   usage = fullusage = version = prompting = gui = ansout =
	   ansin = ansprint = FALSE;
	}

	/*
         *  enforce mutual exclusion of [-gui] and everything else
         */
        if (gui && (prompting || ansout || ansin ))
        {
            kfprintf(kstderr, "Error: The [-gui] option may not be combined with interactive prompting [-P],\n       answer file input [-a], or answer file output [-A]\n");
            kexit(KEXIT_FAILURE);
        }

	/*
	 * enforce mutual exclusion of [-U] and everything else
	 */
	if (usage && (prompting || gui || ansout || ansin))
	{
	    kfprintf(kstderr, "Error: The [-U] option may not be combined with interactive prompting [-P],\nGUI display [-gui], answer file input [-a], or answer file output [-A]\n");
            kexit(KEXIT_FAILURE);

	}

	/*
	 * enforce mutual exclusion of [-P] and [-a]
	 */
        if (prompting && ansin)
        {
            kfprintf(kstderr, "Error: Use either the [-a] option to specify an answer file, or [-P] for interactive prompting, not both\n");
            kexit(KEXIT_FAILURE);
	}

	/*
	 *  [-V]: print the version, using #define in kcodegen.h, exit
	 */
	if (version)
	{
	    kfprintf(kstderr, "This version of %s is from %s\n\n",
		     oname, KGEN_VERSION);
	    if (!(usage || prompting || gui || ansin || ansout || ansprint))
		kexit(KEXIT_SUCCESS);
	}

	/*
	 *  [-usage]: print usage according to CLUI specified in *.pane file
	 *            plus standard command line arguments, exit
	 */
	if (fullusage)
	{
	    kgen_interpret_clui_usage(kstderr, form, tbname, oname, 
				      usage_additions);
	    kexit(KEXIT_SUCCESS);
	}

        /*
         *  [-U]: print usage according to CLUI specified in *.pane file
         *        (without standard command line arguments), exit
         */
	if (usage)
	{
	    /* 
	     * have to destroy form w/ std cmd line args appended, 
	     * recreate it with only the arguments in the *.pane file
	     */
	    kvf_destroy_form(form);
            form = kvf_create_form(pane_path, NONE, NULL, NULL);
	    kfprintf(kstderr,"(Brief usage;  use [-usage] for standardized Khoros arguments)\n\n");
	    kgen_interpret_clui_usage(kstderr, form, tbname, oname,
                                      usage_additions);
            kexit(KEXIT_SUCCESS);

	}

	/*
	 * [-ap]: read in values from the answer file, print them, and exit.
	 */
	if (ansprint)
	{
	    /* read in the answer file, modify form tree from stored values */
	    if (!(read_answer_infile(ansprintfile, form, oname)))
                kexit(KEXIT_FAILURE);

	    /* 
	     *  -ap and -A will appear as TRUE in form:  -ap because it
	     *  had to have been on the command line, and -A because it 
	     *  got stored in the answer file along with everything else.
	     *  but the user really doesn't need to see either of these,
	     *  it'll confuse them;  so set them to unselected before 
	     *  constructing the command line interpretation of the form tree
	     */
	    selection = kvf_variable_sel_search(form, "ap");
	    kvf_set_attribute(selection->back_kformstruct, KVF_OPTSEL, FALSE);

	    selection = kvf_variable_sel_search(form, "A");
	    kvf_set_attribute(selection->back_kformstruct, KVF_OPTSEL, FALSE);

	    /* construct command line from form settings */
	    arguments = kvf_form_to_cmd(form, form->subform, NULL, TRUE);

	    /* print the command line interpretation, and exit */
	    kinfo(KFORCE, "%s\n", arguments);	

	    /*
	     * Freeing arguments allocated above. I think this is the correct
	     * place (SJ)
	     */
	    kfree(arguments);

	    kexit(KEXIT_SUCCESS);
	}

	/*
 	 *  [-a]: first, read in values from the answer file. then, over-ride
         *        answer file values with anything given on command line.
	 */
	if (ansin)
	{
	    if (!(read_answer_infile(ansinfile, form, oname)))
	    	kexit(KEXIT_FAILURE);
	}

	/*
	 *  [-A]: special case when -A appears alone on the command line
	 *        (with or without filename following) go ahead and assume -P.
	 */
	if (ansout)
	{
	    int  argc;
	    argc = kprog_get_argc();
	    if (((argc == 2) && (ansoutfile == NULL)) ||
		 ((argc == 3) && (ansoutfile != NULL)))
		prompting = TRUE;
	}


	/*
 	 * [-gui]: display the GUI specified by the *.pane for the program 
	 */
	if (gui)
	{
	   int  argc;

	   /* [-gui] simply calls xvrun with toolbox & object name */
	   ksprintf(command, "xvrun -tb %s -oname %s", tbname, oname);

	   /* arguments should contain "{binary} -gui " */
	   arguments = kvf_form_to_cmd(form, form->subform, NULL, TRUE);

	   /* 
	    * if they specified any other arguments, need to print a warning
	    * that the additional arguments will be ignored.  this is because
	    * you can't correctly highlight optional boxes *both* as specified
            * in the *.pane file, AND as specified on the command line.  so
            * play it safe, always give the GUI as per the *.pane.
	    */
	   argc = kprog_get_argc();
	   if (argc > 2)
		kwarn(NULL, "kgen_initialize", 
		      "All arguments besides [-gui] being ignored...");

	   /* 
	    * translate xvrun -tb %s -oname %s into argument 
	    * structure for kexecvp, then execute xvrun
	    */
	   kcommand_to_args(kstrcpy(temp, command), args);
	   kexecvp(args[0], args);

	   /* kexecvp should never return; if it does, there was an error */
	   kerror(KCODEGEN, "main()", "Unable to execute command:\n '%s'",command);
	   /*
	    * Freeing arguments allocated above. I think this is the correct
	    * place (SJ)
	    */
	   kfree(arguments);

	   kexit(KEXIT_FAILURE);
	}

	/*
 	 *  [-P]: first, read in values from the command line. next, 
         *        interactively prompt for values not given on command line.
	 */
	else if (prompting)
	{
	   if (!kgen_interpret_clui_args(tbname, oname, usage_additions, 
				         form, TRUE))
	      return(NULL);

	   (void) kclui_consolidate_args(form);
	}

	/*
	 *  values only appear on command line.  update the kform for the pane
         *  according to those values, error check.
	 */
	else
	{
	   if (!kgen_interpret_clui_args(tbname, oname, usage_additions,
					 form, FALSE))
	      return(NULL);

	   (void) kclui_consolidate_args(form);
	}

	/*
	 *  [-A]: however values got set, write them out into answer file. 
	 */
	if (ansout)
	{
	    if (!(write_answer_outfile(ansoutfile, form, oname)))
	    	kexit(KEXIT_FAILURE);
	}

	return(form);
}


/*-----------------------------------------------------------
|
|  Routine Name: write_answer_outfile
|
|       Purpose: Answer files for the CLUI of Khoros programs are
|                supported via gdbm.  All Khoros programs use the
|                same answer file by default, $HOME/khoros.ans, if
|		 the answer file is not explicitly specified, or specified
|		 via the KHOROS_ANSFILE environment variable.
|                This routine writes out the command line options used
|                with a program to the Khoros answer file using gdbm.
|
|         Input: 
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Mark Young & Danielle Argiro
|          Date: April 11, 1994
| Modifications: 
|
------------------------------------------------------------*/

static int write_answer_outfile(
    char  *ansoutfile,
    kform *form,
    char  *oname)
{
        kdbm   *dbm;
        kdatum  key;
        kdatum  contents;
	char   *arguments;
	char   *cmdline;

	/* by default, look for the answer file in $HOME */
        if (ansoutfile == NULL &&
	    (ansoutfile = kgetenv("KHOROS_ANSFILE")) == NULL)
            ansoutfile = "$HOME/khoros.ans";

	/* open the gdbm file */
        dbm = kdbm_open(ansoutfile, O_WRONLY|O_CREAT, 0666);
        if (dbm != NULL)
        {
	    /* the key for the answer file data base is the program name */
            key.dptr  = oname;
            key.dsize = kstrlen(key.dptr)+1;

	    /* construct command line call from modified pane kform struct */
	    cmdline = kvf_form_to_cmd(form, form->subform, NULL, TRUE);

	    /* scan past program name to first argument */
	    arguments = kstrstr(cmdline, " -");

	    /* store the rest of the command line as the data */
            contents.dptr  = arguments;
            contents.dsize = kstrlen(contents.dptr)+1;
            if (kdbm_store(dbm, key, contents, KDBM_REPLACE) !=  0)
	    {
                kerror(KCODEGEN, "kgen_initialize",
                       "kdbm_store failed to store entry in %s",
                       ansoutfile);
                kdbm_close(dbm);
	        return(FALSE);
	    }
            kdbm_close(dbm);
	    return(TRUE);
       	}
       	else 
	{
	    kerror(KCODEGEN, "kgen_initialize",
                   "Unable to create/write to '%s' answer file",
                   ansoutfile);
	    return(FALSE);
	}

}

/*-----------------------------------------------------------
|
|  Routine Name: read_answer_infile
|
|       Purpose: Answer files for the CLUI of Khoros programs are
|                supported via gdbm.  All Khoros programs use the
|                same answer file by default, $HOME/khoros.ans, if
|		 the answer file is not explicitly specified, or specified
|		 via the KHOROS_ANSFILE environment variable.
|                This routine reads in the command line options used
|                with a program to the Khoros answer file using gdbm.
|
|         Input: 
|
|        Output: Returns TRUE on success, FALSE on failure
|    Written By: Mark Young & Danielle Argiro
|          Date: April 11, 1994
| Modifications: 
|
------------------------------------------------------------*/
static int read_answer_infile(
    char  *ansinfile,
    kform *form,
    char  *oname)
{
        kdbm   *dbm;
        kdatum key;
        kdatum contents;

        if (ansinfile == NULL &&
	    (ansinfile = kgetenv("KHOROS_ANSFILE")) == NULL)
            ansinfile = "$HOME/khoros.ans";

        dbm = kdbm_open(ansinfile, O_RDONLY, 0666);
        if (dbm != NULL)
        {
            key.dptr  = oname;
            key.dsize = kstrlen(key.dptr)+1;
            contents = kdbm_fetch(dbm, key);
            if (contents.dptr == NULL)
	    {
                kerror(KCODEGEN, "kgen_initialize",
                        "There is no entry in your answer file for %s", oname);
                kdbm_close(dbm);
		return(FALSE);
	    }
            else kvf_modify_form_from_cmd(form, contents.dptr, FALSE);
            kdbm_close(dbm);
	    return(TRUE);
        }
        else 
	{
	    kerror(KCODEGEN, "kgen_initialize",
                    "'%s' answer file not found", ansinfile);
	    return(FALSE);
	}
}
