/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Toolbox dbm utility routines
   >>>>
   >>>>   Static:
   >>>>         _kcms_tb_store_attribute()
   >>>>  Private:
   >>>>         kcms_tb_sync()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "toolboxP.h"
#include "databaseP.h"

static int _kcms_tb_save_fobj PROTO((kobject, kdbm *, kstring));

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_store_attribute - write a toolbox attr to database
|
| Purpose:	This routine takes a toolbox object structure, and
|		writes a corresponding .cm file out to disk
|
| Input:	toolbox  - The toolbox object we're saving.
|		database - The database which holds toolbox info.
|
| Written By:	Neil Bowers
| Date:		4-apr-94
------------------------------------------------------------*/
void
_kcms_tb_save_object_list(
   kobject   toolbox,
   kdbm     *database)
{
   klist    *list;
   kobject   object;
   char	    *oname;
   char	    *typestring;
   int	     otype;
   char     *str            = NULL;
   char	     field[KLENGTH];
   char     *tmp;


   for (list=toolbox->sw_objects; list != NULL; list = klist_next(list))
   {
      object = (kobject) klist_clientdata(list);
      kcms_get_attribute(object, KCMS_CMOBJ_TYPE, &otype);
      kcms_get_attribute(object, KCMS_NAME, &oname);

      typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);
      ksprintf(field, "%s:%s\n", oname, typestring);
      tmp = kstring_cat(str, field, NULL);
      kfree(str);
      str = tmp;
   }

   kcms_db_update_string(database, str, KCMS_KEY_TB_SOFTWARE_OBJECTS);
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_store_attribute - write a toolbox attr to database
|
| Purpose:	This routine takes a toolbox object structure, and
|		writes a corresponding .cm file out to disk
|
| Input:	database	- The database we are storing the toolbox in.
|		keystr		- Database key to store attribute under
|		attribute	- The identifier for attribute to store
|		list		- A list holding the attribute's value
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		Sun Aug 29
------------------------------------------------------------*/
static int
_kcms_tb_store_attribute(
   kdbm   *database,
   char   *keystr,
   int     attribute,
   klist  *list)
{
   static char data[KLENGTH * 20];


   *data = '\0';
   obj_list_to_string(list, attribute, data);
   kcms_db_update_string(database, data, keystr);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_save_fobj - save a file object to database
|
| Purpose:	This routine takes a toolbox object structure, and
|		writes a corresponding .cm file out to disk
|
| Input:	file_object - The file object to save in the database.
|		database    - The database we are storing the toolbox in.
|		keystring   - Database key to store attribute under
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		Sun Aug 29
------------------------------------------------------------*/
static int
_kcms_tb_save_fobj(
   kobject   file_object,
   kdbm     *database,
   kstring   keystring)
{
   kstring  path = NULL;


   if (file_object != NULL)
      kcms_get_attribute(file_object, KCMS_PATH, &path);
   kcms_db_update_string(database, path, keystring);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_tb_sync - synchronize a toolbox object with database
|
| Purpose:	This routine takes a toolbox object, and stores the
|		attributes in the toolbox's database.
|
| Input:	toolbox	- toolbox object we are storing the
|		database for
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steven Jorgensen and Neil Bowers
| Date:		Nov 23, 1992 13:38
------------------------------------------------------------*/
int
kcms_tb_sync(
   kobject toolbox)
{
   kstring   routine    = "Synchronize Toolbox";
   kdbm     *database;
   char      dbpath[KLENGTH];
   char      directory[KLENGTH];


   if (toolbox->update == KCMS_UPDATE_NONE)
      return TRUE;

   ksprintf(dbpath, "$%s/repos/db/cms", toolbox->name);
   if (kdirname(dbpath, directory) == NULL)
      return FALSE;

   /*-- check to see whether we can access the database directory -----*/
   if (kaccess(directory, (W_OK | X_OK)) == -1)
   {
      if (errno == EEXIST && !kmake_dir(directory, 0777))
      {
	 kerror(KCMS, routine, "Unable to create database directory: %s\n",
		directory);
	 return FALSE;
      }
      else
      {
	 kerror(KCMS, routine, "Could not access the directory to write "
		"the toolbox database.\n"
		"Toolbox:   %s\n"
		"Directory: %s\n",
		toolbox->name, directory);
	 return FALSE;
      }
   }

   /*--------------------------------------------------------------------
   |	open dbm file
   --------------------------------------------------------------------*/
   if ((database = kdbm_open(dbpath, O_WRONLY | O_CREAT, 0666)) == NULL)
   {
      kerror(KCMS, routine, "Cannot open %s for writing\n", dbpath);
      errno = KCMS_CMOPENERR;
      return FALSE;
   }

   kcms_db_update_string(database, toolbox->name,      KCMS_KEY_TB_NAME);
   kcms_db_update_string(database, toolbox->poc_email, KCMS_KEY_AUTHOR_EMAIL);
   kcms_db_update_string(database, toolbox->poc_name,  KCMS_KEY_AUTHOR);
   kcms_db_update_string(database, toolbox->title,     KCMS_KEY_TITLE);
   kcms_db_update_string(database, toolbox->version,   KCMS_KEY_VERSION);

   if (!toolbox->default_short_copyright)
      kcms_db_update_string(database, toolbox->short_copyright,
			    KCMS_KEY_SHORT_COPYRIGHT);

   if (!toolbox->default_long_copyright)
      kcms_db_update_string(database, toolbox->long_copyright,
			 KCMS_KEY_LONG_COPYRIGHT);

   /*-- save list of keywords -----------------------------------------*/
   kcms_db_update_string_list(database, toolbox->keywords, KCMS_KEY_KEYWORDS);

   /*--------------------------------------------------------------------
   |	Save CMS Version number
   --------------------------------------------------------------------*/
   kcms_db_update_string(database, CM_VERSION, KCMS_KEY_CMS_VERSION);

   /*--------------------------------------------------------------------
   |	Save list of software objects
   --------------------------------------------------------------------*/
   _kcms_tb_save_object_list(toolbox, database);

   /*--------------------------------------------------------------------
   |	Save CMS File object info
   --------------------------------------------------------------------*/
   _kcms_tb_store_attribute(database, KCMS_KEY_MISC_FILES,
			    KCMS_PATH, toolbox->misc_files);
   _kcms_tb_store_attribute(database, KCMS_KEY_APP_DEFAULTS,
			    KCMS_PATH, toolbox->app_defaults);
   _kcms_tb_save_fobj(toolbox->info_file, database, KCMS_KEY_INFO_FILE);
   _kcms_tb_save_fobj(toolbox->config_file, database, KCMS_KEY_CONFIG_FILE);
   _kcms_tb_save_fobj(toolbox->include_file, database, KCMS_KEY_INCLUDE_FILE);
   _kcms_tb_save_fobj(toolbox->alias_file, database, KCMS_KEY_ALIAS_FILE);
   _kcms_tb_save_fobj(toolbox->todo_file, database, KCMS_KEY_TODO_FILE);
   _kcms_tb_save_fobj(toolbox->changelog_file, database,
		      KCMS_KEY_CHANGELOG_FILE);

   /*-- save the three dates associated with a toolbox ----------------*/
   if (toolbox->times.modification != NULL)
      kfree(toolbox->times.modification);
   toolbox->times.modification = kcms_get_date();
   kcms_db_write_dates(database, toolbox->times.creation,
		       toolbox->times.modification, toolbox->times.generation);

   /*--------------------------------------------------------------------
   |	Save CMS State info
   --------------------------------------------------------------------*/
   kcms_db_update_int(database, toolbox->status, KCMS_KEY_STATUS);

   kdbm_close(database);

   toolbox->update = KCMS_UPDATE_NONE;

   return TRUE;
}
