/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Program library object routines
   >>>>
   >>>>  Private:
   >>>>		kcms_tb_attr_table()
   >>>>
   >>>>   Public:
   >>>>
   >>>>   Static:
   >>>>		_kcms_tb_get_opath()
   >>>>		_kcms_tb_get_obj()
   >>>>		_kcms_merge_lists()
   >>>>		_kcms_tb_get_subobj_list()
   >>>>		_kcms_tb_set_subobj_name()
   >>>>		_kcms_tb_get_ci()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "toolboxP.h"


static kbool _kcms_tb_get_subobj_list	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_ci		PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_string_attr	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_int_attr	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_fobj          PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_stringlist	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_ulong_attr	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_get_date		PROTO((kobject, int, kva_list *));

static kbool _kcms_tb_set_string_attr	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_set_int_attr	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_set_stringlist	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_set_ulong_attr	PROTO((kobject, int, kva_list *));
static kbool _kcms_tb_set_date		PROTO((kobject, int, kva_list *));


AttrTbl tbattr_tbl[] =
{
   {0,				NULL,
				NULL				},

   {KCMS_NAME,			_kcms_tb_get_string_attr,
				NULL				},

   {KCMS_TYPE,			_kcms_tb_get_int_attr,
				NULL				},

   {KCMS_PATH,			_kcms_tb_get_string_attr,
				NULL				},

   {KCMS_PARENT,		NULL,
				NULL				},

   {KCMS_DATE,		        _kcms_tb_get_date,
				_kcms_tb_set_date	     	},

   {KCMS_TB_SOFTWARE_OBJECTS,	_kcms_tb_get_subobj_list,
				NULL				},

   {KCMS_TB_CATEGORYINFO,	_kcms_tb_get_ci,
				NULL				},

   {KCMS_TB_MISC_FILES,		_kcms_tb_get_subobj_list,
				NULL				},

   {KCMS_TB_AUTHOR_EMAIL,	_kcms_tb_get_string_attr,
				_kcms_tb_set_string_attr	},

   {KCMS_TB_AUTHOR,		_kcms_tb_get_string_attr,
				_kcms_tb_set_string_attr	},

   {KCMS_TB_TITLE,		_kcms_tb_get_string_attr,
				_kcms_tb_set_string_attr	},

   {KCMS_TB_INFO_FILE,		_kcms_tb_get_fobj,
                                NULL                            },

   {KCMS_TB_APP_DEFAULTS,	_kcms_tb_get_subobj_list,
				NULL				},

   {KCMS_TB_FLAGS,		_kcms_tb_get_ulong_attr,
				_kcms_tb_set_ulong_attr		},

   {KCMS_TB_STATUS,		_kcms_tb_get_int_attr,
				_kcms_tb_set_int_attr		},

   {KCMS_TB_UPDATE_FLAG,	_kcms_tb_get_int_attr,
				_kcms_tb_set_int_attr		},

   {KCMS_TB_COPYRIGHT_SHORT,    _kcms_tb_get_string_attr,
                                _kcms_tb_set_string_attr        },

   {KCMS_TB_COPYRIGHT_LONG,     _kcms_tb_get_string_attr,
                                _kcms_tb_set_string_attr        },

   {KCMS_TB_KEYWORDS,           _kcms_tb_get_stringlist,
                                _kcms_tb_set_stringlist         },

   {KCMS_TB_CONFIG_FILE,        _kcms_tb_get_fobj,
                                NULL                            },

   {KCMS_TB_INCLUDE_FILE,       _kcms_tb_get_fobj,
                                NULL                            },

   {KCMS_TB_VERSION,            _kcms_tb_get_string_attr,
                                _kcms_tb_set_string_attr        },

   {KCMS_TB_ALIAS_FILE,         _kcms_tb_get_fobj,
                                NULL                            },

   {KCMS_TB_TODO_FILE,          _kcms_tb_get_fobj,
                                NULL                            },

   {KCMS_TB_CHANGELOG_FILE,     _kcms_tb_get_fobj,
                                NULL                            }

};

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_get_int_attr - get an integer toolbox attribute
|
| Purpose:	This function is an attribute handler, used in the
|		table of toolbox attributes.  It is used to retrieve
|		the value (get-attribute) of a simple integer attribute.
|
| Input:	toolbox	  - toolbox object to get attribute from.
|		attribute - identifier for the attribute to get.
|
| Output:	valist	  - varargs list which contains pointers
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		10-nov-93
------------------------------------------------------------*/
static kbool
_kcms_tb_get_int_attr(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   int    *int_ptr;
   kbool   result    = TRUE;


   int_ptr = kva_arg(*valist,int *);
   switch (attribute)
   {
      case KCMS_TYPE:
	 *int_ptr = toolbox->type;
	 break;

      case KCMS_TB_STATUS:
	 *int_ptr = toolbox->status;
	 break;

      case KCMS_TB_UPDATE_FLAG:
	 *int_ptr = toolbox->update;
	 break;

      default:
	 kerror(KCMS,"Get Integer Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
   }

   return result;
}

static kbool
_kcms_tb_get_ulong_attr(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   unsigned long  *pul;
   kbool           result = TRUE;


   pul = kva_arg(*valist, unsigned long *);
   *pul = (unsigned long) 0;
   switch (attribute)
   {
      case KCMS_TB_FLAGS:
	 *pul = toolbox->flags;
	 break;

      default:
	 kerror(KCMS, "_kcms_tb_get_ulong_attr()",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_get_fobj - attribute handler for getting file object
|
| Purpose:	This function is an attribute handler, used in the
|		table of toolbox attributes.  It is used to retrieve
|		the value (get-attribute) of a file object attribute
|
| Input:	toolbox	  - toolbox object to get attribute from.
|		attribute - identifier for the attribute to get.
|
| Output:	valist	  - varargs list which contains pointers
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		10-nov-93
------------------------------------------------------------*/
static kbool
_kcms_tb_get_fobj(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kstring   routine = "_kcms_tb_get_fobj()";
   kbool     result  = TRUE;
   kobject  *objptr;


   objptr = kva_arg(*valist, kobject *);
   switch (attribute)
   {
      case KCMS_TB_INFO_FILE:
	 *objptr = toolbox->info_file;
	 break;

      case KCMS_TB_CONFIG_FILE:
	 *objptr = toolbox->config_file;
	 break;

      case KCMS_TB_INCLUDE_FILE:
	 *objptr = toolbox->include_file;
	 break;

      case KCMS_TB_ALIAS_FILE:
	 *objptr = toolbox->alias_file;
	 break;

      case KCMS_TB_TODO_FILE:
	 *objptr = toolbox->todo_file;
	 break;

      case KCMS_TB_CHANGELOG_FILE:
	 *objptr = toolbox->changelog_file;
	 break;

      default:
	 kerror(KCMS, routine, "Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_get_stringlist - get toolbox string attribute
|
| Purpose:	This function is an attribute handler, used in the
|		table of toolbox attributes.  It is used to retrieve
|		the value (get-attribute) of a simple string attribute.
|
| Input:	toolbox	  - toolbox object to get attribute from.
|		attribute - identifier for the attribute to get.
|
| Output:	valist    - varargs list which contains pointers
|			    to places to put the attribute values.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		10-nov-93
------------------------------------------------------------*/
static kbool
_kcms_tb_get_stringlist(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kbool       result  = TRUE;
   klist     **list;


   list = kva_arg(*valist, klist **);
   switch (attribute)
   {
      case KCMS_TB_KEYWORDS:
	 *list = toolbox->keywords;
	 break;

      default:
	 kerror(KCMS,"Get String Array Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

static kbool
_kcms_tb_set_stringlist(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kbool      result  = TRUE;
   klist     *list;


   list = kva_arg(*valist, klist *);
   switch (attribute)
   {
      case KCMS_TB_KEYWORDS:
	 toolbox->keywords = list;
	 toolbox->update |= KCMS_UPDATE_SYNC;
	 break;

      default:
	 kerror(KCMS,"Get String Array Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_get_string_attr - get toolbox string attribute
|
| Purpose:	This function is an attribute handler, used in the
|		table of toolbox attributes.  It is used to retrieve
|		the value (get-attribute) of a simple string attribute.
|
| Input:	toolbox	  - toolbox object to get attribute from.
|		attribute - identifier for the attribute to get.
|
| Output:	valist    - varargs list which contains pointers
|			    to places to put the attribute values.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		10-nov-93
------------------------------------------------------------*/
static kbool
_kcms_tb_get_string_attr(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kobject    bootstrap;
   char     **strptr;
   char      *tmp;
   kbool      result     = TRUE;


   strptr = kva_arg(*valist,char **);
   switch (attribute)
   {
      case KCMS_NAME:
	 *strptr = toolbox->name;
	 break;

      case KCMS_PATH:
	 *strptr = toolbox->path;
	 break;

      case KCMS_TB_AUTHOR_EMAIL:
	 *strptr = toolbox->poc_email;
	 break;

      case KCMS_TB_AUTHOR:
	 *strptr = toolbox->poc_name;
	 break;

      case KCMS_TB_TITLE:
	 *strptr = toolbox->title;
	 break;

      case KCMS_TB_COPYRIGHT_SHORT:
	 if (toolbox->short_copyright == NULL
	     && !toolbox->default_short_copyright
	     && kstrcasecmp(toolbox->name, KCMS_COPYRIGHT_TOOLBOX) != 0)
	 {
	    bootstrap = kcms_open_toolbox(KCMS_COPYRIGHT_TOOLBOX);
	    if (bootstrap != NULL)
	    {
	       kcms_get_attribute(bootstrap, KCMS_TB_COPYRIGHT_SHORT, &tmp);
	       toolbox->short_copyright = kstrdup(tmp);
	       kcms_close(bootstrap);
	    }
	    toolbox->default_short_copyright = TRUE;
	 }
	 *strptr = toolbox->short_copyright;
	 break;

      case KCMS_TB_COPYRIGHT_LONG:
	 if (toolbox->long_copyright == NULL
	     && !toolbox->default_long_copyright
	     && kstrcasecmp(toolbox->name, KCMS_COPYRIGHT_TOOLBOX) != 0)
	 {
	    bootstrap = kcms_open_toolbox(KCMS_COPYRIGHT_TOOLBOX);
	    if (bootstrap != NULL)
	    {
	       kcms_get_attribute(bootstrap, KCMS_TB_COPYRIGHT_LONG, &tmp);
	       toolbox->long_copyright = kstrdup(tmp);
	       kcms_close(bootstrap);
	    }
	    toolbox->default_long_copyright = TRUE;
	 }
	 *strptr = toolbox->long_copyright;
	 break;

      case KCMS_TB_VERSION:
	 *strptr = (toolbox->version != NULL
		    ? toolbox->version
		    : "2.0.1");
	 break;

      default:
	 kerror(KCMS,"Get String Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_set_int_attr - set a string toolbox attribute
|
| Purpose:	This function is an attribute handler used in the table
|		of toolbox attributes.  It handles the setting of an
|		integer toolbox attribute.
|
| Input:	toolbox		- the toolbox object being modified.
|		attribute	- identifier for the integer attribute.
|		valist		- varargs list which contains the arguments
|				  for the attribute.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		10-nov-93
------------------------------------------------------------*/
static kbool
_kcms_tb_set_int_attr(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   int	  newvalue;
   kbool  result    = TRUE;


   newvalue = kva_arg(*valist,int);
   switch (attribute)
   {
      case KCMS_TB_UPDATE_FLAG:
	 toolbox->update = newvalue;
	 break;

      case KCMS_TB_STATUS:
	 toolbox->status = newvalue;
	 toolbox->update = KCMS_UPDATE_SYNC;
	 break;

      default:
	 kerror(KCMS,"Set Integer Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

static kbool
_kcms_tb_set_ulong_attr(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   unsigned long  ulValue;
   kbool          result   = TRUE;


   ulValue = kva_arg(*valist, unsigned long);
   switch (attribute)
   {
      case KCMS_TB_FLAGS:
	 toolbox->flags = ulValue;
	 break;

      default:
	 kerror(KCMS,"Set Integer Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_set_string_attr - set a string toolbox attribute
|
| Purpose:	This should be a complete description that anyone
|		could understand;  it should have acceptable grammar
|		and correct spelling.
|
| Input:	toolbox		- the kcms toolbox object.
|		attribute	- identifier for the attribute being changed.
|		valist		- varargs lists which contains arguments
|				  to the attribute.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		1-apr-94
------------------------------------------------------------*/
static kbool
_kcms_tb_set_string_attr(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kbool    result	= TRUE;
   kbool    need_update	= TRUE;
   char   **valueptr    = NULL;
   char    *newvalue;


   newvalue = kva_arg(*valist,char *);
   switch (attribute)
   {
      case KCMS_TB_AUTHOR_EMAIL:
	 valueptr = &toolbox->poc_email;
	 break;

      case KCMS_TB_AUTHOR:
	 valueptr = &toolbox->poc_name;
	 break;

      case KCMS_TB_TITLE:
	 valueptr = &toolbox->title;
	 break;

      case KCMS_TB_COPYRIGHT_SHORT:
	 valueptr = &toolbox->short_copyright;
	 toolbox->default_short_copyright = FALSE;
	 break;

      case KCMS_TB_COPYRIGHT_LONG:
	 valueptr = &toolbox->long_copyright;
	 toolbox->default_long_copyright = FALSE;
	 break;

      case KCMS_TB_VERSION:
	 valueptr = &toolbox->version;
	 break;

      default:
	 kerror(KCMS,"Set String Attributes",
		"Illegal attribute identifier (%d)", attribute);
	 result = FALSE;
   }

   if (result && valueptr != NULL)
   {
      kfree(*valueptr);
      *valueptr = kstrdup(newvalue);
      if (need_update)
	 toolbox->update	= KCMS_UPDATE_SYNC;
   }

   return result;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_get_subobj_list - get list of sub objects from toolbox
|
| Purpose:	This routine gets a list of all sub objects, program objects,
|		or library objects depending on which attribute the
|		calling routine specifies.
|
| Input:	toolbox   - object to get attribute for
|		attribute - attribute to get
|		valist    - a variable argument list that will hold the answer
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen & Neil Bowers
| Date:		28-nov-93
------------------------------------------------------------*/
static kbool
_kcms_tb_get_subobj_list(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kstring    routine  = "_kcms_tb_get_subobj_list";
   klist    **vlist;
   kbool      result   = TRUE;


   vlist = kva_arg(*valist, klist **);

   switch (attribute)
   {
      case KCMS_TB_SOFTWARE_OBJECTS:
	 *vlist = toolbox->sw_objects;
	 break;

      case KCMS_TB_MISC_FILES:
	 *vlist = toolbox->misc_files;
	 break;

      case KCMS_TB_APP_DEFAULTS:
	 *vlist = toolbox->app_defaults;
	 break;

      default:
	 kerror(KCMS,routine, "Invalid attribute identifier (%d).", attribute);
	 result = FALSE;
	 break;
   }

   return result;
}

static kbool
_kcms_tb_set_date(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kstring  string;


   if ((string = kva_arg(*valist, kstring)) != NULL)
   {
      toolbox->times.creation = string;
      toolbox->update |= KCMS_UPDATE_SYNC;
   }

   if ((string = kva_arg(*valist, kstring)) != NULL)
   {
      toolbox->times.modification = string;
      toolbox->update |= KCMS_UPDATE_SYNC;
   }

   if ((string = kva_arg(*valist, kstring)) != NULL)
   {
      toolbox->times.generation = string;
      toolbox->update |= KCMS_UPDATE_SYNC;
   }

   return TRUE;
}

static kbool
_kcms_tb_get_date(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kstring  *string_ptr;


   if ((string_ptr = kva_arg(*valist, kstring *)) != NULL)
      *string_ptr = toolbox->times.creation;
   if ((string_ptr = kva_arg(*valist, kstring *)) != NULL)
      *string_ptr = toolbox->times.modification;
   if ((string_ptr = kva_arg(*valist, kstring *)) != NULL)
      *string_ptr = toolbox->times.generation;

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_tb_get_ci - parse category info file for users
|
| Purpose:	This routine sucks information from the objects cache
|		and returns arrays of information about the objects in the
|		toolbox passed.
|
| Input:	toolbox	  - The toolbox object which the attribute is
|			    requested for.
|		attribute - Attribute identifier.
|		valist	  - variable argument list
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen & Neil Bowers
| Date:		25-jan-94
------------------------------------------------------------*/
static kbool
_kcms_tb_get_ci(
   kobject    toolbox,
   int	      attribute,
   kva_list  *valist)
{
   kstring        routine  = "Get Category-Info";
   kdbm          *database;
   kdatum         key;
   kdatum         keydata;
   char	          cachepath[KLENGTH];
   char        ***onames;
   char        ***icons;
   char        ***pane;
   char        ***workspaces;
   char        ***cat;
   char        ***subcat;
   char        ***short_desc;
   int          **otypes;
   int          **incantata;
   int           *prog_cnt;
   int	          nkeys;
   int	          i;
   int	          j;
   int	          count;
   char          *head;
   char          *tail;
   unsigned int	  sblocksize;
   unsigned int	  iblocksize;
   char	          elements[8][KLENGTH];


   if (attribute != KCMS_TB_CATEGORYINFO)
   {
      errno = KCMS_EINVALIDATTR;
      kerror(KCMS, routine,
	     "Unexpected attribute, expected KCMS_TB_CATEGORYINFO.");
      return FALSE;
   }

   onames	= kva_arg(*valist,char ***);
   otypes	= kva_arg(*valist,int **);
   icons	= kva_arg(*valist,char ***);
   cat		= kva_arg(*valist,char ***);
   subcat	= kva_arg(*valist,char ***);
   pane		= kva_arg(*valist,char ***);
   workspaces   = kva_arg(*valist,char ***);
   incantata	= kva_arg(*valist,int **);
   short_desc	= kva_arg(*valist,char ***);
   prog_cnt	= kva_arg(*valist,int *);

   if (onames)
      *onames = NULL;
   if (otypes)
      *otypes = NULL;
   if (pane)
      *pane = NULL;
   if (workspaces)
      *workspaces = NULL;
   if (icons)
      *icons = NULL;
   if (cat)
      *cat = NULL;
   if (incantata)
      *incantata = NULL;
   if (short_desc)
      *short_desc = NULL;
   if (prog_cnt)
      *prog_cnt = 0;

   ksprintf(cachepath,"$%s/repos/db/cache/objects",toolbox->name);

   /*UPDATE:-------------------------------------------------------------
   |	Need to be smarter here - might be no categories
   |	or we should perhaps generate the cantata cache
   |	it should exist, just be empty
   +-------------------------------------------------------------------*/
   if ((database = kdbm_open(cachepath,O_RDONLY,0666)) == NULL)
      return FALSE;

   /*-- free previously cached CATEGORY INFO information --*/
   if (toolbox->progs != NULL)
   {
      for (i=0; i<toolbox->ci_size; i++)
      {
	 kfree(toolbox->progs[i]);
	 kfree(toolbox->icon_names[i]);
	 kfree(toolbox->cats[i]);
	 kfree(toolbox->subcats[i]);
	 kfree(toolbox->panes[i]);
	 kfree(toolbox->workspaces[i]);
	 kfree(toolbox->short_descriptions[i]);
      }
      kfree(toolbox->progs);
   }

   /*-- find out how many keys there are in the database --*/
   key = kdbm_firstkey(database);
   for (nkeys=0; key.dptr; nkeys++, key = kdbm_nextkey(database))
      continue;

   /*-- allocate one big chunk of memory --*/
   sblocksize	= nkeys * sizeof(char *);
   iblocksize	= nkeys * sizeof(int);

   toolbox->progs		= (char **) kmalloc(sblocksize);
   toolbox->icon_names		= (char **) kmalloc(sblocksize);
   toolbox->cats		= (char **) kmalloc(sblocksize);
   toolbox->subcats		= (char **) kmalloc(sblocksize);
   toolbox->panes		= (char **) kmalloc(sblocksize);
   toolbox->workspaces		= (char **) kmalloc(sblocksize);
   toolbox->short_descriptions	= (char **) kmalloc(sblocksize);
   toolbox->otypes		= (int *)   kmalloc(iblocksize);
   toolbox->incantata		= (int *)   kmalloc(iblocksize);

   toolbox->ci_size		= nkeys;

   /*-- foreach key, get key data, split and fill in info --*/
   key = kdbm_firstkey(database);
   for (i=0; key.dptr != NULL; key = kdbm_nextkey(database),i++)
   {
      if ((keydata = kdbm_fetch(database,key)),keydata.dptr == NULL)
      {
	 kwarn(KCMS,routine,
	       "Couldn't get cache info for key %s",key.dptr);
	 continue;
      }

      head = tail = kstring_ncopy(keydata.dptr,keydata.dsize,NULL);

      for (j=0; j<7; j++)
      {
	 for (count=0; *tail != ':' && *tail != '\0'; count++)
	    tail++;

	 kstring_ncopy(head,count,elements[j]);
	 tail++;
	 head = tail;
      }
      
      toolbox->progs[i]  = kstrdup(key.dptr);
      toolbox->otypes[i] = kcms_attr_string2int(KCMS_CMOBJ_PROGTYPE,
						elements[0]);
      toolbox->icon_names[i]         = kstrdup(elements[1]);
      toolbox->cats[i]               = kstrdup(elements[2]);
      toolbox->subcats[i]            = kstrdup(elements[3]);
      toolbox->panes[i]              = kstrdup(elements[4]);
      toolbox->workspaces[i]         = kstrdup(elements[5]);
      toolbox->incantata[i]          = (kstrcmp(elements[6],"YES") == 0);
      toolbox->short_descriptions[i] = kstrdup(head);
   }

   kdbm_close(database);

   toolbox->ci_set = TRUE;

   if (onames)
      *onames = toolbox->progs;
   if (otypes != NULL)
      *otypes = toolbox->otypes;
   if (icons != NULL)
      *icons = toolbox->icon_names;
   if (cat)
      *cat = toolbox->cats;
   if (subcat)
      *subcat = toolbox->subcats;
   if (pane)
      *pane = toolbox->panes;
   if (workspaces)
      *workspaces = toolbox->workspaces;
   if (incantata)
      *incantata = toolbox->incantata;
   if (short_desc)
      *short_desc = toolbox->short_descriptions;
   if (prog_cnt)
      *prog_cnt = toolbox->ci_size;

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_tb_attr_table() - get pointer to toolbox attribute table
|
| Purpose:	This routine returns the address of the attribute table.
|		It is called by the initialize routine so that it can
|		initialized the kobject's internal attribute pointer.
|
| Returns:	A pointer to the attribute table which holds information
|		on the attributes defined for kcms toolbox objects.
| Written By:	Steven Jorgensen
| Date:		25-apr-93
------------------------------------------------------------*/
AttrTbl *
kcms_tb_attr_table(void)
{
   return tbattr_tbl;
}
