/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            CMS File Utility Routines
   >>>>
   >>>>   Static:
   >>>>		_kcms_fobj_initialize()
   >>>>
   >>>>  Private:
   >>>>		kcms_fobj_free()
   >>>>		kcms_fobj_duplicate()
   >>>>		kcms_fobj_destroy()
   >>>>
   >>>>   Public:
   >>>>		kcms_open_fileobj()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "fileP.h"

static kobject _kcms_fobj_initialize PROTO((kobject, char *, char *));

/************************************************************
* Routine Name:	kcms_open_fileobj - create a file object for an existing file
*
* Purpose:	This routine allocates and initializes a kcms file object
*		structure for an existing file on disk.
*
* Input:	parent	 - The parent kobject for the new file object.
*		filename - path in cm or Imakefile if prefix is NULL
*		           otherwise assumed to be fullname path
*		           beginning with $TOOLBOX/.../filename
*		path	 - The $TOOLBOX/... part of the fullname path;
*		           set to NULL if name uses the fullname path.
*		ftype	 - The file type.  Preprocessor symbols for
*		           legal file types can be found in the include
*		           file "kcms/fileobj.h"
*
*		           Note, if type is set to -1, this routine will
*		           try to figure out the type and subtype given
*		           the path and relative path.
*
*		subtype	 - The subtype of the file depending on type.
*		           Preprocessor symbols for legal file subtypes
*		           can be found in the include
*		           file "kcms/fileobj.h"
*
*		           Note, if type is set to -1, this value is
*		           ignored.
*
*		source	 - A flag indicating which tool generated it.
*		           Preprocessor symbols for legal source flags
*		           can be can be found in the include
*		           file "kcms/fileobj.h"
*
*		perms	 - A flag indicating the user permissions on
*		           the file it can be one of:
*		           ! KCMS_FACCESS_RDWR - User can read and write the
*		                                 file
*		           ! KCMS_FACCESS_RDONLY - User can only read the file
* Output:	
* Returns:	The opened file object, or NULL on failure.
* Written By:	Steven Jorgensen and Neil Bowers
* Date:		13-aug-94
*************************************************************/
kobject
kcms_open_fileobj(
   kobject  parent,
   char    *name,
   char    *path,
   int      type,
   int      subtype,
   int      source,
   int      perms)
{
   kstring  routine = "Open File Object";
   kobject  fileobj;


   if (name == NULL)
   {
      kinfo(KSYSLIB, "kcms_open_fileobj: name parameter is NULL");
      errno = KCMS_NULLNAME;
      return NULL;
   }

   if (!kcms_legal_object(parent, routine)
       || (fileobj = _kcms_fobj_initialize(parent, name, path)) == NULL)
      return NULL;

   /*UPDATE: there is some kinda kludge here, set file_attr.c for details --*/
   if (source < KCMS_FOBJ_GEN_NONE || source > KCMS_FOBJ_GEN_SCRIPT)
   {
      errno = KCMS_RANGEERR;
      return NULL;
   }

   fileobj->generated = source;
    
   if (!kcms_set_attributes(fileobj,
			    KCMS_FOBJ_TYPE,      type,
			    KCMS_FOBJ_SUBTYPE,   subtype,
			    KCMS_FOBJ_ACCESS,    perms,
			    KCMS_END))
   {
      return NULL;
   }

   return fileobj;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_fobj_initialize - initialize the CMS file object
|
| Purpose:	This routine will initialize the CMS file object.
|
| Input:	name   - pathname given in the CM file or fullpath name.
|		prefix - prefix to prepend to "name"
|		parent - parent kobject of the file
| Output:	
| Returns:	new data object on success, NULL otherwise
| Written By:	Steven Jorgensen
| Date:		Apr 21, 1993 13:53
------------------------------------------------------------*/
static kobject
_kcms_fobj_initialize(
   kobject  parent,
   char    *name,
   char    *prefix)
{
   kstring  routine         = "_kcms_fobj_initialize()";
   kobject  fileobj;
   char     nname[KLENGTH];
   kstring  tmp;
   char     cmname[KLENGTH];
   char     temp[KLENGTH];


   errno = KCMS_OK;
   if ((fileobj = (kobject) kcalloc(1, sizeof(struct _kobject))) == NULL)
   {
      errno = KCMS_CANNOTMALLOCOBJ;
      kerror(KCMS, routine, "Unable to kmalloc (%d) bytes.\n",
	     sizeof(struct _kobject));
      return NULL;
   }

   if (prefix != NULL)
   {
      if (((unsigned int)prefix[kstrlen(prefix) - 1]) != '/')
	 kstring_3cat(prefix, "/", name, nname);
      else
	 kstring_cat(prefix, name, nname);
      name = nname;
   }
   tmp = kstrchr(name, '/');
   if (tmp != NULL)
      kstring_cleanup(tmp + 1, cmname);
   else
      kstring_cleanup(name, cmname);
   if (((unsigned int)name[0]) != '$')
   {
      errno = KCMS_INVALIDPATH;
      kerror(KCMS, routine, "Path does not begin with $TOOLBOX/.\n"
	     "File objects must exist within a toolbox");
      kfree(fileobj);
      return NULL;
   }
   kstring_cleanup(name, temp);
   fileobj->type          = KOBJ_CMSFILE;
   fileobj->fullname      = kstring_to_token(temp);
   fileobj->cm_name       = kstring_to_token(cmname);
   fileobj->attributes    = kcms_fobj_attr_tbl();
   fileobj->attribute_cnt = KCMS_FOBJ_MAX;

   fileobj->add_sub_obj   = NULL;
   fileobj->del_sub_obj   = NULL;
   fileobj->mod_sub_obj   = NULL;
   fileobj->parent        = parent;

   fileobj->callbacks     = NULL;

   return fileobj;
}


/*-----------------------------------------------------------
| Routine Name:	kcms_fobj_free - free memory associated with a file object
|
| Purpose:	This routine frees the memory associated with a
|		CMS file object
|
| Input:	object - the object to free
| Output:	TRUE (1) on success, FALSE (0) on failure.
| Written By:	Steven Jorgensen
| Date:		Apr 21, 1993 22:46
------------------------------------------------------------*/
int
kcms_fobj_free(
   kobject fileobj)
{
   if (fileobj == NULL)
      return FALSE;

   kfree(fileobj);
   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	kcms_fobj_duplicate - duplicate a file object in one
|		parent object into another.
|
| Purpose:	This routine copies the physical file on disk,
|		as well as the file information structure to
|		a new parent object.  The new parent must
|		be of the same type as the old one. (i.e
|		if the old parent is a program object, you can't
|		copy it to a toolbox object.)  Finally, it
|		informs the parent object that it has arrived.
|
| Input:	src  - the object to copy
|		dest - the parent object destination
|
| Returns:	the kobject duplicated on success, NULL on error
|
| Written By:	Steven Jorgensen and Neil Bowers
| Date:		24-aug-94
------------------------------------------------------------*/
kobject
kcms_fobj_duplicate(
   kobject  src,
   kobject  dest)
{
   kobject  otb;
   kobject  ntb;
   kobject  parent;
   kobject  new;
   kfile   *file;
   kstring  old_tbname;
   kstring  new_tbname;
   char     oldstring[11][KLENGTH];
   char     newstring[11][KLENGTH];
   char     old_tblower[KLENGTH];
   char     new_tblower[KLENGTH];
   char     otbcpyrt[KLENGTH];
   char     ntbcpyrt[KLENGTH];
   char     temp[KLENGTH];
   char    *tmp1;
   char    *tmp2 = NULL;
   char    *tmp3 = NULL;
   char    *fullname = NULL;
   int      gen;
   int      acc;
   int      type;
   int      subtype;
   int      itmp;


   if (kcms_get_attribute(src, KCMS_PARENT, &parent) == FALSE)
      return NULL;
   if (parent->type != dest->type)
   {
      errno = KCMS_OBJTYPEMISMATCH;
      return NULL;
   }
   if (parent->type == KOBJ_CMSOBJ)
   {
      if (!kcms_get_attribute(parent, KCMS_PARENT, &otb))
	 return NULL;

      if (!kcms_get_attribute(dest, KCMS_PARENT, &ntb))
	 return NULL;
   }
   else
   {
      otb = parent;
      ntb = dest;
   }

   if (!kcms_get_attribute(otb, KCMS_NAME, &old_tbname)
       || !kcms_get_attribute(ntb, KCMS_NAME, &new_tbname)
       || kstring_lower(old_tbname, old_tblower) == NULL
       || kstring_lower(new_tbname, new_tblower) == NULL)
      return NULL;

   ksprintf(oldstring[0], "\"%s\"", old_tbname);
   ksprintf(newstring[0], "\"%s\"", new_tbname);

   ksprintf(oldstring[1], "\"%s\"", old_tblower);
   ksprintf(newstring[1], "\"%s\"", new_tblower);

   ksprintf(oldstring[2], "$%s/", old_tbname);
   ksprintf(newstring[2], "$%s/", new_tbname);

   ksprintf(oldstring[3], "$%sBIN/", old_tbname);
   ksprintf(newstring[3], "$%sBIN/", new_tbname);

   ksprintf(oldstring[4], ".syntax %s", old_tbname);
   ksprintf(newstring[4], ".syntax %s", new_tbname);

   ksprintf(oldstring[5], ".syntax %s", old_tblower);
   ksprintf(newstring[5], ".syntax %s", new_tblower);

   ksprintf(oldstring[6], "<%s.h>", old_tblower);
   ksprintf(newstring[6], "<%s.h>", new_tblower);

   /*-- we could check, and only do this for pane objects -------------*/
   ksprintf(oldstring[7], "kexec -tb %s", old_tbname);
   ksprintf(newstring[7], "kexec -tb %s", new_tbname);

   /*-- similarly, this need only be done for info files --------------*/
   ksprintf(oldstring[8], "^Toolbox:(\\s+)$%s$", old_tbname);
   ksprintf(newstring[8], "Toolbox:\\1$%s\n", new_tbname);

   /*-- only needs to be done for Imakefile ---------------------------*/
   ksprintf(oldstring[9], "^#include %s_INCLUDE$", old_tbname);
   ksprintf(newstring[9], "#include %s_INCLUDE\n", new_tbname);

   /*-- only needs to be done for Imakefile ---------------------------*/
   ksprintf(oldstring[10], "^TOOLBOX_NAME\\s+=\\s+%s$", old_tblower);
   ksprintf(newstring[10], "TOOLBOX_NAME = %s\n", new_tblower);


   ksprintf(ntbcpyrt, "$%s/repos/copyright/Copyright", new_tbname);
   if (kcms_get_attribute(src, KCMS_PATH, &fullname) == FALSE)
      return NULL;


   /*UPDATE: we could check to see if reparenting in the same tb -- */
   ksprintf(temp, "$%s/%s", new_tbname, kstrchr(fullname, '/') + 1);
   if (!kcms_get_attributes(src,
			    KCMS_FOBJ_TYPE,      &type,
			    KCMS_FOBJ_SUBTYPE,   &subtype,
			    KCMS_FOBJ_GENERATED, &gen,
			    KCMS_FOBJ_ACCESS,    &acc,
			    KCMS_END))
      return NULL;

   if ((new = kcms_create_fileobj(dest, temp, NULL, type, subtype,
				  gen, acc)) == NULL)
      return NULL;
   new->update_routine = src->update_routine;
   if (kcms_get_attribute(new, KCMS_PATH, &tmp1) == FALSE)
   {
      itmp = errno;
      kcms_destroy(new);
      errno = itmp;
      return NULL;
   }

   if ((file = kfinput(fullname)) != NULL)
   {
      if (kparse_file_search(file, "$.*/repos/copyright/Copyright",
			     KLITERAL, &tmp2) == KPARSE_OK)
      {
	 kstrcpy(otbcpyrt, tmp2);
	 kfree(tmp2);
	 tmp2 = ntbcpyrt;
	 tmp3 = otbcpyrt;
      }
      kfclose(file);
   }
   if (!ksedfile(fullname, tmp1, TRUE, KFILE_UPDATE, NULL,
		 oldstring[0],   newstring[0],
		 oldstring[1],   newstring[1],
		 oldstring[2],   newstring[2],
		 oldstring[3],   newstring[3],
		 oldstring[4],   newstring[4],
		 oldstring[5],   newstring[5],
		 oldstring[6],   newstring[6],
		 oldstring[7],   newstring[7],
		 oldstring[8],   newstring[8],
		 oldstring[9],   newstring[9],
		 oldstring[10],  newstring[10],
		 tmp2,          tmp3,
		 NULL))
   {
      kcms_destroy(new);
      errno = KCMS_PERMISSION;
      return NULL;
   }
   return new;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_fobj_destroy - destroy a file object and it's associated
|		file.
|
| Purpose:	This routine kunlinks the file represented by the cms file
|		object, and then destroy's the file object info structure.
|
| Input:	object - the object to destroy
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen & Neil Bowers
| Date:		10-jul-94
------------------------------------------------------------*/
int
kcms_fobj_destroy(
   kobject  fileobj)
{
   kstring  routine = "Destroy File Object";
   kstring  tmp;
   int      result  = FALSE;


   if (kcms_get_attribute(fileobj, KCMS_PATH, &tmp))
   {
      /*-- if the file is there, unlink (delete) it ------------------*/
      if (kaccess(tmp, R_OK) == 0 && kunlink(tmp) == -1)
         kerror(KCMS, routine, "Unable to unlink (kunlink) file %s.", tmp);

      /*-- remove the file object from its parent --------------------*/
      if (fileobj->parent->del_sub_obj(fileobj->parent, fileobj))
      {
         kcms_fobj_free(fileobj);
         result = TRUE;
      }
   }

   return result;
}
