/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Image Display Utility Routines
   >>>>
   >>>>   Static:
   >>>>                 _kcms_cmobj_free_header_list
   >>>>  Private:
   >>>>                 kcms_cmobj_free_comb_lists
   >>>>                 kcms_cmobj_partial_close
   >>>>                 kcms_cmobj_free
   >>>>   Public:
   >>>>		kcms_attr_string2int
   >>>>		kcms_attr_int2string
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "cmobjP.h"

typedef struct
{
   int      type;
   kstring  typestring;
} SWtype;

static SWtype otypes[] =
{
   {KCMS_NONE,      "unknown"},
   {KCMS_KROUTINE,  "kroutine"},
   {KCMS_XVROUTINE, "xvroutine"},
   {KCMS_EXAMPLE,   "example"},
   {KCMS_XEXAMPLE,  "xexample"},
   {KCMS_TESTSUITE, "testsuite"},
   {KCMS_LIBRARY,   "library"},
   {KCMS_SCRIPT,    "script"},
   {KCMS_PANE,      "pane"},
   {KCMS_MANUAL,    "manual"},
   {KCMS_MINDEX,    "mindex"},
   {KCMS_MGLOSSARY, "mglossary"},
   {KCMS_DIR,       "dir"},
   {KCMS_MDIR,	   "mdir"}
};
#define KCMS_CMOBJ_MAXPROGTYPES	knumber(otypes)

static char *ltypes[] =
{
   "unknown",
   "none",
   "fortran",
   "perl",
   "csh",
   "sh",
   "ksh"
};
#define KCMS_CMOBJ_MAXLANGTYPES	knumber(ltypes)

#define _GUARDED_CLOSE_OBJECT(object)	if (object != NULL)		\
					{				\
						kcms_close(object);	\
						object = NULL;		\
					}

/************************************************************
* Routine Name: kcms_attr_int2string - translate integer attribute to string
* Purpose:	This function is used to map an integer value to a string,
*		where the integer is interpreted in the context of a
*		specified attribute.
*
*		For example, the attribute KCMS_CMOBJ_TYPE is of type integer,
*		and can take values KCMS_KROUTINE, KCMS_XVROUTINE,
*		KCMS_PANE, KCMS_LIBRARY, and KCMS_SCRIPT.
*		The string "kroutine" corresponds to the value KCMS_KROUTINE.
*
* Input:	attribute - An attribute identifier, which provides a
*			    a context for the attribute value.
*		value     - An integer value, which is a legal value for the
*			    specified attribute.
* Returns:	A string corresponding to the integer value passed,
*		or NULL if an invalid value is passed.
* Written By:	Neil Bowers
* Date:		27-aug-94
*************************************************************/
kstring
kcms_attr_int2string(
   int    attribute,
   int    value)
{
   int      i;
   kstring  string = NULL;


   switch (attribute)
   {
      case KCMS_CMOBJ_TYPE:
	 string = otypes[0].typestring;
	 for (i = KCMS_CMOBJ_MAXPROGTYPES - 1; i > 0; i--)
	    if (otypes[i].type == value)
	       string = otypes[i].typestring;
	 break;

      case KCMS_CMOBJ_LANGTYPE:
	 if (value > KCMS_CMOBJ_MAXLANGTYPES || value < 1)
	    string = ltypes[0];
	 else
	    string = ltypes[value];
	 break;

      case KCMS_CMOBJ_CODETYPE:
	 string = (value == KCMS_KNR) ? "KNR" : "ANSI";
	 break;

      default:
	 errno = KCMS_EINVALIDATTR;
	 break;
   }

   return string;
}

/************************************************************
* Routine Name: kcms_attr_string2int - translate string to integer value
* Purpose:	This function is used to map a string to an integer value,
*		where the string is interpreted in the context of a
*		specified attribute.
*
*		For example, the attribute KCMS_CMOBJ_TYPE is of type integer,
*		and can take values KCMS_KROUTINE
* Input:	attribute - An attribute identifier, which provides a
*			    a context for the attribute value.
*		value     - An integer value, which is a legal value for the
*			    specified attribute.
* Returns:	A string corresponding to the integer value passed.
*		For example, if attribute is KCMS_CMOBJ_TYPE, and
*		value is KCMS_KROUTINE, this function will return the
*		string "kroutine".
* Written By:	Neil Bowers
* Date:		27-aug-94
*************************************************************/
int
kcms_attr_string2int(
   int      attribute,
   kstring  string)
{
   int  i;
   int  value = 0;


   /*-- if NULL string, return 0 to signify error (should be -1?) -----*/
   if (string == NULL)
      return 0;

   switch (attribute)
   {
      case KCMS_CMOBJ_TYPE:
	 for (i = KCMS_CMOBJ_MAXPROGTYPES - 1; i > 0; i--)
	    if (kstrcasecmp(string, otypes[i].typestring) == 0)
	       value = otypes[i].type;
	 break;

      case KCMS_CMOBJ_LANGTYPE:
	 for (i = KCMS_CMOBJ_MAXLANGTYPES - 1; i > 0; i--)
	    if (kstrcasecmp(string, ltypes[i]) == 0)
	       value = i;
	 break;

      case KCMS_CMOBJ_CODETYPE:
	 if (kstrcasecmp(string, "KNR") == 0)
	    value = KCMS_KNR;
	 else
	    value = KCMS_ANSI;
	 break;

      default:
	 errno = KCMS_EINVALIDATTR;
	 break;
   }

   return value;
}

/*-----------------------------------------------------------
| Routine Name:	_kcms_cmobj_free_header_list - free a list of header_info
|		structures
|
| Purpose:	This routine frees a klist of header_info structure, along
|		with each klist element structure in the list.
|
| Input:	list - a klist of header_info structures.
|
| Written By:	Steven Jorgensen
| Date:		Aug 27, 1993
------------------------------------------------------------*/
static void
_kcms_cmobj_free_header_list(
   klist  *list)
{
   header_info  *header;


   if (list == NULL)
      return;
   header = (header_info *) klist_clientdata(list);
   kcms_header_info_free(header);
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_free_comb_lists - free combined file object lists
|
| Purpose:	This routine free's an objects allfiles, allsrc, alluis,
|		alldoc combined file object lists, and resets the
|		structure pointers to NULL.
|
| Input:	object - object to free
|
| Written By:	Steven Jorgensen
| Date:		Jun 09, 1993 12:00
------------------------------------------------------------*/
void
kcms_cmobj_free_comb_lists(
   kobject cmobj)
{
   if (cmobj == NULL || cmobj->type != KOBJ_CMSOBJ)
      return;
   klist_free(cmobj->allfiles, NULL);
   klist_free(cmobj->alluis, NULL);
   klist_free(cmobj->allsrc, NULL);
   klist_free(cmobj->alldoc, NULL);
   klist_free(cmobj->allconfig, NULL);
   cmobj->allfiles = NULL;
   cmobj->alluis = NULL;
   cmobj->allsrc = NULL;
   cmobj->alldoc = NULL;
   cmobj->allconfig = NULL;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_partial_close - close the partially open kobject
|
| Purpose:	This routine closes the cmobj passed in.  This routine
|		handles the special toolbox special partially open kobjects,
|		and it truly frees all parts of the kobject.  It
|		is called during the close of a toolbox.
|
| Input:	cmobj - kobject to free
| Output:	
| Returns:	TRUE (1) on success, FALSE (0) otherwise
| Written By:	Steven Jorgensen
| Date:		Jun 08, 1993 10:44
------------------------------------------------------------*/
int
kcms_cmobj_partial_close(
   kobject cmobj)
{
   if (cmobj == NULL)
      return TRUE;
   if (cmobj->type != KOBJ_CMSOBJ)
      return FALSE;
   if (cmobj->fullyopen == TRUE)
      (void)kcms_close(cmobj);
   kfree(cmobj->opath);
   kfree(cmobj->topsrc);
   kfree(cmobj->category);
   kfree(cmobj->subcategory);
   _GUARDED_CLOSE_OBJECT(cmobj->pane);
   _GUARDED_CLOSE_OBJECT(cmobj->database);
   _GUARDED_CLOSE_OBJECT(cmobj->imakefile);
   _GUARDED_CLOSE_OBJECT(cmobj->makefile);
   kfree(cmobj);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_free - free a cmobj
|
| Purpose:	This routine frees all the memory associated
|		with a cmobj.
|
| Input:	cmobj - program cmobj to free
|
| Written By:	Steven Jorgensen
| Date:		Oct 29, 1992 13:11
------------------------------------------------------------*/
void
kcms_cmobj_free(
   kobject cmobj)
{
   if (cmobj == NULL || cmobj->fullyopen == FALSE)
      return;

   kfree(cmobj->bname);
   kfree(cmobj->filename);
   kfree(cmobj->assoc_tbname);
   kfree(cmobj->assoc_oname);
   klist_free(cmobj->header_list, _kcms_cmobj_free_header_list);
   cmobj->header_list = NULL;
   klist_free(cmobj->y_files, close_n_free);
   klist_free(cmobj->l_files, close_n_free);
   klist_free(cmobj->c_files, close_n_free);
   klist_free(cmobj->f_files, close_n_free);
   klist_free(cmobj->cplus_files, close_n_free);
   klist_free(cmobj->inc_files, close_n_free);
   klist_free(cmobj->script_files, close_n_free);
   klist_free(cmobj->libman_files, close_n_free);
   klist_free(cmobj->help_files, close_n_free);
   cmobj->y_files = NULL;
   cmobj->l_files = NULL;
   cmobj->c_files = NULL;
   cmobj->f_files = NULL;
   cmobj->cplus_files = NULL;
   cmobj->inc_files = NULL;
   cmobj->script_files = NULL;
   cmobj->libman_files = NULL;
   cmobj->help_files = NULL;
   kcms_cmobj_free_comb_lists(cmobj);
   _GUARDED_CLOSE_OBJECT(cmobj->form);
   cmobj->pane = NULL;
   cmobj->form = NULL;
   kfree(cmobj->library);
   kfree(cmobj->short_description);
   kfree(cmobj->poc_email);
   kfree(cmobj->poc_name);
   kfree(cmobj->version);

   _GUARDED_CLOSE_OBJECT(cmobj->man1file);
   _GUARDED_CLOSE_OBJECT(cmobj->man3file);

   /*-- close file objects referring to info files --*/
   _GUARDED_CLOSE_OBJECT(cmobj->todo);
   _GUARDED_CLOSE_OBJECT(cmobj->done);
   _GUARDED_CLOSE_OBJECT(cmobj->bugs);
   _GUARDED_CLOSE_OBJECT(cmobj->changelog);

   cmobj->cfile = NULL;
   cmobj->hfile = NULL;
   cmobj->lfile = NULL;
   cmobj->helpfile = NULL;
   cmobj->scriptfile = NULL;

   karray_free((void *)cmobj->prog_spec, cmobj->prog_cnt, NULL);
   cmobj->prog_spec = NULL;
   cmobj->prog_cnt = 0;
   kfree(cmobj->prog_valid);
   klist_free(cmobj->misc_files, close_n_free);
   klist_free(cmobj->app_defaults, close_n_free);
   cmobj->misc_files = NULL;
   cmobj->fullyopen = FALSE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_update_cache - update objects cache
|
| Purpose:	This function is used to update the object cache database
|		which holds information used by cantata and other applications.
|
| Input:	cmobj - the object being cached
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		21-nov-93
------------------------------------------------------------*/
int
kcms_cmobj_update_cache(
   kobject cmobj)
{
   kstring         routine   = "Update Objects Cache";
   kobject         paneobj   = NULL;
   kobject         workspace = NULL;
   kstring         subcat    = NULL;
   kstring         pane      = NULL;
   kstring         icon      = NULL;
   kstring         wksp_path = NULL;
   unsigned long   update;
   char            path[KLENGTH];
   char            cinfo[KLENGTH];
   char           *tbpath;
   kdbm           *database;
   char           *cat;
   int             otype;
   int             incantata;
   kstring         typestring;
   kstring         description;
   kdatum          key;
   kdatum          keydata;


   if (!kcms_get_attribute(cmobj, KCMS_CMOBJ_UPDATE_DB, &update))
      return FALSE;

   if ((update & KCMS_UPDATE_CACHE) != KCMS_UPDATE_CACHE)
      return TRUE;

   if (!kcms_get_attribute(cmobj->parent, KCMS_PATH, &tbpath)
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_CATEGORY, &cat)
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_SUBCATEGORY, &subcat)
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_UIS_PANE, &paneobj)
       || (paneobj != NULL
	   && !kcms_get_attribute(paneobj, KCMS_PATH, &pane))
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_WORKSPACE, &workspace)
       || (workspace != NULL
	   && !kcms_get_attribute(workspace, KCMS_PATH, &wksp_path))
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_ICON_NAME, &icon)
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_SHORT_DESC, &description)
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_INCANTATA, &incantata)
       || !kcms_get_attribute(cmobj, KCMS_CMOBJ_PROGTYPE, &otype))
      return FALSE;

   ksprintf(path, "%s/repos/db/cache/objects", tbpath);
   if ((database = kdbm_open(path, (O_RDWR | O_CREAT), 0666)) == NULL)
   {
      kerror(KCMS, routine, "Failed to open objects cache.");
      return FALSE;
   }

   if (cat == NULL)
      cat = "";
   if (subcat == NULL)
      subcat = "";
   if (pane == NULL)
      pane = "";
   if (icon == NULL)
      icon = "";
   if (wksp_path == NULL)
      wksp_path = "";
   if (description == NULL)
      description = "";
   typestring = kcms_attr_int2string(KCMS_CMOBJ_TYPE, otype);

   key.dptr = cmobj->oname;
   key.dsize = kstrlen(cmobj->oname) + 1;

   ksprintf(cinfo, "%s:%s:%s:%s:%s:%s:%s:%s",
	    typestring, icon, cat, subcat,
	    pane, wksp_path, (incantata ? "YES" : "NO"),
	    description);
   keydata.dptr = cinfo;
   keydata.dsize = kstrlen(cinfo) + 1;
   kdbm_store(database, key, keydata, KDBM_REPLACE);

   kdbm_close(database);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	kcms_cmobj_set_pane_attribute - update pane attribute
|
| Purpose:	This routine updates the short description in a software
|		object's pane UIS file.
|
| Input:	object      - The software object being updated.
|		description - The new short description
|
| Written By:	Neil Bowers and Danielle Argiro
| Date:		15-nov-94
------------------------------------------------------------*/
void
kcms_cmobj_set_pane_title(
   kobject  object,
   kstring  description)
{
   kobject      file_object;
   kstring      file_path;
   kform       *form;
   int          force_flag  = TRUE;
   int          make_backup = FALSE;


   if (!kcms_get_attribute(object, KCMS_CMOBJ_UIS_PANE, &file_object)
       || file_object == NULL
       || !kcms_get_attribute(file_object, KCMS_PATH, &file_path))
      return;

   if ((form = kvf_create_form(file_path, NONE, NULL, NULL)) == NULL)
       return;

   /*-- should only be one pane, but better safe than sorry! ----------*/
   if (!kvf_check_for_single_pane(form, NULL))
      return;

   kvf_set_attribute(form->subform->back_kformstruct, KVF_TITLE, description);

   kvf_print_UIS_file(form, file_path, force_flag, make_backup);
}
