 /*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

#include "copyright.h"		/* Khoros copyright */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         Routines for generating a manual
   >>>>
   >>>> Private:
   >>>>         run_kgenmanual()
   >>>>         expand_file() 
   >>>> Static:
   >>>>         _process_so()
   >>>>         _process_include()
   >>>>         _process_pspic()
   >>>>         _process_help()
   >>>>         _process_code()
   >>>> Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "kgenmanual.h"

struct token_list
{
   kstring  key;
   int      minargs;
   int      maxargs;
   kbool    (*function) PROTO((kfile *, int, kstring *, kstring));
};


static kbool	_process_function   PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_so	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_ci	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_include    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_help	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_pspic	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_code	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_whatis	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_syntax	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_man1	    PROTO((kfile *, int, kstring *, kstring));
static kbool	_process_exec	    PROTO((kfile *, int, kstring *, kstring));

static kbool	expand_file		PROTO((kstring, kfile *));


	/* 
	 * This table contains the search key and function to call
	 * when that search key is found. To add to this table
	 * list the search key before the NULL statement and
	 * create the appropriate function at the bottom of this
	 * file. The key will automatically be search for
	 * when processing the manual files.
	 */

static struct token_list map[] =
{
   {"^\\.[ \t]*include[ \t]+.*$",  1, 1,   _process_include},
   {"^\\.[ \t]*help[ \t]+.*$",	   2, 2,   _process_help},
   {"^\\.[ \t]*function[ \t]+.*$", 3, 4,   _process_function},
   {"^\\.[ \t]*syntax[ \t]+.*$",   2, 2,   _process_syntax},
   {"^\\.[ \t]*so[ \t]+.*$",	   1, 1,   _process_so},
   {"^\\.[ \t]*cI[ \t]+.*$",       1, 1,   _process_ci},
   {"^\\.[ \t]*PSPIC[ \t]+.*$",	   1, 3,   _process_pspic},
   {"^\\.[ \t]*code[ \t]+.*$",	   1, 1,   _process_code},
   {"^\\.[ \t]*whatis[ \t]+.*$",   3, 3,   _process_whatis},
   {"^\\.[ \t]*man1[ \t]+.*$",	   2, 3,   _process_man1},
   {"^\\.[ \t]*exec[ \t]+.*$",	  -1, -1, _process_exec},
   {NULL, -1, -1, NULL}
};


/*-----------------------------------------------------------
| Routine Name:	run_kgenmanual - main routine for bldmanual
|
| Purpose:	This routine will open the toolbox object and the
|		output file, then call the expand_file routine
|		to parse the input files.		 
|
| Input:	input_file	- the input file to parse
|		output_file	- the name of the file to write the output to.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Tom Sauer and Neil Bowers
| Date:		Nov 30, 1992 15:47
------------------------------------------------------------*/
kbool
run_kgenmanual(
   kstring  input_file,
   kstring  output_file)
{
   kstring   routine         = "run_kgenmanual()";
   kobject   toolbox         = NULL;
   kstring   short_copyright = NULL;
   kstring   long_copyright  = NULL;
   kfile    *optr;
   int       chmod_status;


   if ((toolbox = kcms_open_toolbox(clui_info->tb_string)) == NULL)
   {
      if (clui_info->tb_string == NULL)
	 kerror(NULL, routine, "NULL toolbox specified\n");
      else
         kerror(NULL, routine, "Unable to open toolbox \"%s\"\n",
	        clui_info->tb_string);
      return FALSE;
   }
   if (!kcms_get_attributes(toolbox,
			    KCMS_TB_COPYRIGHT_SHORT, &short_copyright,
			    KCMS_TB_COPYRIGHT_LONG,  &long_copyright,
			    NULL))
   {
      kerror(NULL, routine,
	     "Failed to get copyright attributes from toolbox \"%s\",\n"
	     "so the manual will not be (re)generated.\n",
	     clui_info->tb_string);
      return FALSE;
   }


   /* 
    * chmod the output file to writable in case it already
    * exists and needs to be overwritten
    */

   chmod_status = chmod(output_file, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
   if (chmod_status == -1 && errno == EACCES)
   {
      kerror(NULL, routine, "Cannot generate output file '%s'\n", output_file);
      return FALSE;
   }

   if ((optr = kfopen(output_file, "w")) == NULL)
   {
      kerror(NULL, routine, "Cannot open the output file '%s'\n", output_file);
      return FALSE;
   }

   kfprintf(optr, ".\\\" ************************************************\n");
   kfprintf(optr, ".\\\" **                                            **\n");
   kfprintf(optr, ".\\\" **    WARNING!  WARNING!  WARNING!  WARNING!  **\n");
   kfprintf(optr, ".\\\" **                                            **\n");
   kfprintf(optr, ".\\\" ** This file was automatically generated by   **\n");
   kfprintf(optr, ".\\\" ** kgenmanual from the .sec and .man files in **\n");
   kfprintf(optr, ".\\\" ** current directory.  Do not edit this file! **\n");
   kfprintf(optr, ".\\\" **                                            **\n");
   kfprintf(optr, ".\\\" ************************************************\n");

   kfprintf(optr,
	    ".\n"
	    ".\\\" toolbox's short copyright - can modify from Craftsman\n"
	    ".ds khoros*short-copyright %s\n"
	    ".\n", short_copyright);
   kcms_close(toolbox);

   if (!expand_file(input_file, optr))
   {
      kfclose(optr);
      kunlink(output_file);
      return FALSE;
   }

   /*
    * Change the output file (*.ms) permissions to read only
    * to prevent the user from editing this file.
    */

   chmod_status = chmod(output_file, S_IRUSR | S_IRGRP | S_IROTH);

   if ((chmod_status == -1) && (errno == EACCES))
   {
      kerror(NULL, routine, "Cannot access output file '%s'\n", output_file);
      return FALSE;
   }

   kfclose(optr);

   /*-- this tells get_object() to close any currently open objects ---*/
   (void)get_object(NULL, NULL);

   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	expand_file - expand all pseudo macros in file to output
|
| Purpose:	This function preprocesses a file, expanding any pseudo
|		macros found within the file, by calling the handler
|		function associated with each macro.
|
| Input:	filename - The name/path of the file to expand.
|		file     - A filehandle for the output file.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		25-sep-94
------------------------------------------------------------*/
static kbool
expand_file(
   kstring   filename,
   kfile    *file)
{
   kstring   file_contents;


   file_contents = (kstring)kreadfile(filename, NULL);

   if (file_contents == NULL)
      return FALSE;

   expand_string(file_contents, filename, file);
   kfree(file_contents);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	expand_string - expand all pseudo macros in a string
|
| Purpose:	This function preprocesses a string, expanding any pseudo
|		macros found within the file, by calling the handler
|		function associated with each macro.
|
| Input:	string   - The string to expand.
|		filename - The name of the file the string came from.
|		optr     - A filehandle for the output file.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		25-sep-94
------------------------------------------------------------*/
kbool
expand_string(
   kstring   string,
   kstring   filename,
   kfile    *optr)
{
   kstring   routine        = "expand_string()";
   kstring  *argv;
   int       argc;
   kstring   delimiter      = " \t:,;";
   kstring   file_part;
   kstring   start;
   kstring   currentline;
   kstring   cptr;
   kstring   fptr;
   kstring   temp;
   kstring   final;
   kstring   line = NULL;
   int       minargs;
   int       maxargs;
   int       index;
   int       parse_status;
   int       status;
   int       done;
   int       length;
   kbool     retval        = TRUE;
   kbool    (*key_function) PROTO((kfile *, int, kstring *, kstring)) = NULL;


   done = FALSE;
   fptr = string;
   /*
    * start scanning the file search for the keys in the
    * map structure above. Call the appropriate routine
    * based on the key found. Terminate when we have hit
    * the end of the file.
    */
   while (!done)
   {
      start = fptr;
      final = NULL;
      currentline = NULL;
      status = KPARSE_NOKEY;
      index = 0;

      /* 
       * since the kparser does not support an "or" function
       * we must do it ourselves. So, find the first occurance of
       * one of the keys listed in the map structure above.
       */
      while (map[index].key != NULL)
      {
	 line = NULL;
	 if ((temp = kparse_string_search(fptr, map[index].key,
			       KIGNORE_CASE, &line, &parse_status)) == NULL)
	 {
	    index++;
	    kfree(line);
	    continue;
	 }
	 else if ((temp != NULL && temp < final) || final == NULL)
	 {
	    final = temp;
	    status = parse_status;
	    kfree(currentline);
	    currentline  = line;
	    key_function = map[index].function;
	    minargs      = map[index].minargs;
	    maxargs      = map[index].maxargs;
	 }
	 else
	    kfree(line);
	 index++;
      }
      fptr = final;

      /*
       * Now check the status from the key search above and call the
       * appropriate routine for that key if the status was OK.
       * Otherwise hopefully there was no error and we hit the end of the file.
       */
      switch (status)
      {
	 case KPARSE_OK:

	    /*
	     * print out the contents of the file we are processing
	     * up to the current point in the file.
	     */
	    length = fptr - (kstrlen(currentline) + 1) - start;
	    if (length > 0)
	    {
	       file_part = kstring_ncopy(start, length, NULL);
	       output_string(optr, file_part, TRUE);
	       kfree(file_part);
	    }

	    /*-- skip initial period, optional whitespace, macro name --*/
	    cptr = currentline+1;
	    while (isspace(*cptr))
	       cptr++;
	    while (!isspace(*cptr))
	       cptr++;
	    while (isspace(*cptr))
	       cptr++;

	    /*-- split the line into arguments ------------------------*/
	    argv = kparse_string_delimit(cptr, delimiter, KDELIM_CLEAN, &argc);

	    if (minargs == -1)
	    {
	       if (key_function != NULL
		   && !key_function(optr, 1, &cptr, currentline))
	       {
		  done   = TRUE;
		  retval = FALSE;
	       }
	    }
	    else if (minargs == maxargs && argc != minargs)
	    {
	       kfprintf(kstderr,
			"kgenmanual: [%s] expecting %d arguments, saw %d:\n"
			"                %s\n",
			filename, minargs, argc, currentline);
	    }
	    else if (argc < minargs)
	    {
	       kfprintf(kstderr,
			"kgenmanual: [%s] expecting AT LEAST %d arguments:\n"
			"                %s\n",
			filename, minargs, currentline);
	    }
	    else if (argc > maxargs)
	    {
	       kfprintf(kstderr,
			"kgenmanual: syntax error in file %s, on line:\n"
			"                %s\n"
			"            Was expecting AT MOST %d arguments, "
			"saw %d.\n", filename, currentline, maxargs, argc);
	    }
	    else
	    {
	       /*-- call the routine to process the key found --*/
	       if (key_function != NULL
		   && !key_function(optr, argc, argv, currentline))
	       {
		  done   = TRUE;
		  retval = FALSE;
	       }
	    }

	    karray_free(argv, argc, NULL);
	    kfree(currentline);
	    break;

	 case KPARSE_PARTKEY:
	 case KPARSE_SYNTAXKEY:
	    kerror(NULL, routine, "A file parsing error occurred, "
		   "check the '.man', '.sec' and 'help files' for errors\n");
	    done   = TRUE;
	    retval = FALSE;
	    break;

	 case KPARSE_NOKEY:
	    /* 
	     * probably hit the end of the file or there was not a key 
	     * in the file check to see if we hit the end of the file.
	     * If so then write out the remaining part of the
	     * file being processed.
	     */

	    kinfo(KDEBUG, "Processing file `%s`\n", filename);
	    output_string(optr, start, TRUE);
	    done = TRUE;
	    break;
      }
   }
   return retval;
}

/*-----------------------------------------------------------
| Routine Name:	_process_function - routine to process a .function line
|
| Purpose:	This routine will parse a .function line.
|		The .function line is in the form 
|		.function {program_object_name} {function_name}.
|		The program object that the function belongs to is opened 
|		and the functions header is grabbed. Then the
|		header is formatted for the manual and written to the
|		output file.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  toolbox name
|			! argv[1]  object name
|			! argv[2]  function name
|			! argv[3]  section-level (optional)
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_function(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kstring        routine     = "_process_function()";
   header_info   *header;
   kobject        object;
   int            level       = DEFAULT_SECTION_LEVEL;


   if (argc == 4)
   {
      level = atoi(argv[3]);
      if (level < 1 || level > 4)
      {
	 kwarn(NULL, routine,
	       "Invalid section level specified for .function (%d).\n"
	       "Legal values are 1 through 4.\n"
	       "Using default value of %d.", level, DEFAULT_SECTION_LEVEL);
	 level = DEFAULT_SECTION_LEVEL;
      }
   }

   if ((object = get_object(argv[0], argv[1])) == NULL)
   {
      kerror(NULL, routine,
	     "A .function line in one of the .sec or .help files has an "
	     "incorrect toolbox name or object name, the line in error is:\n"
	     "    %s\n", line);
      return FALSE;
   }

   kcms_set_attribute(object, KCMS_CMOBJ_ROUTINENAME, argv[2]);

   if (!kcms_get_attribute(object, KCMS_CMOBJ_HEADERINFO, &header))
   {
      kerror(NULL, routine,
	     "Unable to read the source code header for the function '%s'\n"
	     "The error occurred on the line:\n%s\nPlease check the source "
	     "code header for errors and make sure\nit conforms with the "
	     "Khoros coding conventions", argv[2], line);
      return FALSE;
   }

   return output_header(file, header, argv[1], level);
}

/*-----------------------------------------------------------
| Routine Name:	_process_cI - routine to process a .cI line
|
| Purpose:	This routine handles the .cI (conditional include) macro.
|		We currently just print a line which gives the file name.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  name of file to include
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_ci(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kfprintf(file, "\\fIinclude file\\fP \\f(CW\\s-2%s\\s+2\\fP\n", argv[0]);

   return TRUE;
}


/*-----------------------------------------------------------
| Routine Name:	_process_so - routine to process a .so line
|
| Purpose:	This routine will expand the filename for a .so line
|		and write it to the output file as 
|		.so {expanded file}
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  name of file to include
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_so(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kfprintf(file, ".line 0 \"%s\"\n", argv[0]);

   return expand_file(argv[0], file);
}

/*-----------------------------------------------------------
| Routine Name:	_process_pspic - routine to process a .pspic line
|
| Purpose:	This routine will expand the filename for a .pspic line
|		and write it to the output file as 
|		.pspic {expanded file}
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  name of file to include
|			! argv[1]  width (optional)
|			! argv[2]  height (optional)
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_pspic(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kstring  routine  = "_process_pspic()";
   int      i;
   char     fullpath[KLENGTH];


   if (kfullpath(argv[0], NULL, fullpath) == NULL)
   {
      kerror(NULL, routine, "Could not expand path for %s\n", argv[0]);
      return FALSE;
   }

   kfprintf(file, ".PSPIC %s ", fullpath);
   for (i = 1; i < argc; i++)
      kfprintf(file, "%s ", argv[i]);
   kfprintf(file, "\n");

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_process_help - routine to process a .help line
|
| Purpose:	This routine will strip off the .help key and expand 
|		the filename, then call expand_file to process the file.
|		The on-line help file will be preprocessed to strip off
|		some of the commands specific to on-line help.
|		The result is the file passed in and all files it calls
|		will be literally included in the output file. 
|		For example, if "file.man" has a .sec {filename}
|		line, then this routine will call expand_file with
|		the .sec's file. This file will be parsed and included
|		in the output file.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  section level offset
|			! argv[1]  path to help file
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_help(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kstring   routine  = "_process_help()";
   kbool     result;
   char      fullpath[KLENGTH];


   if (kfullpath(argv[1], NULL, fullpath) == NULL)
   {
      kerror(NULL, routine, "Could not expand path for %s\n", argv[1]);
      return FALSE;
   }

   /*-- see description of the .line macro in the documentation system doc --*/
   kfprintf(file, ".nr khoros:section-offset %s\n", argv[0]);
   kfprintf(file, ".line 0 \"%s\"\n", argv[1]);

   /*--------------------------------------------------------------------
   | Recursivly call expand_file to process the current file
   --------------------------------------------------------------------*/
   result = expand_file(fullpath, file);

   kfprintf(file, ".nr khoros:section-offset 0\n");

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_process_code - routine to process a .code line
|
| Purpose:	This routine will strip off the .code key and expand 
|		the filename, then read the code file. Some special
|		roff commands are placed around the code then the
|		code is output to the document or ,ms file.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  path to file with code in it
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_code(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kstring  file_contents;


   if ((file_contents = kreadfile(argv[0], NULL)) == NULL)
      return FALSE;

   kfprintf(file, ".begin code\n");
   output_string(file, file_contents, TRUE);
   kfprintf(file, ".end code\n");

   kfree(file_contents);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_process_include - routine to process a .sec file
|
| Purpose:	This routine will strip off the .sec key and expand 
|		the filename, then call call expand_file to process the file.
|		The result is the file passed in and all files it calls
|		will be literally included in the output file. 
|		For example, if "file.man" has a .sec {filename}
|		line, then this routine will call expand_file with
|		the .sec's file. This file will be parsed and included
|		in the output file.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  path to file to include
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_include(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kfprintf(file, ".line 0 \"%s\"\n", argv[0]);

   return expand_file(argv[0], file);
}

/*-----------------------------------------------------------
| Routine Name:	_process_whatis - routine to process a .whatis line
|
| Purpose:	This routine will strip off the .whatis key and expand 
|		the object name, then replace retrieve the object's
|		whatis entry. Some special roff commands are placed
|		around the whatis entry and then the whatis is output
|		to the document or, ms file.
|
|		The following is the form of the generated whatis
|		section for the routine kcadd:
|
|		\(bu
|		\fIkcadd()\fP -  add two complex numbers
|		.br
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  toolbox name
|			! argv[1]  object name
|			! argv[2]  function name
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Steve Jorgensen
| Date:		8-dec-92
------------------------------------------------------------*/
static kbool
_process_whatis(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kstring       routine     = "_process_whatis()";
   header_info  *header;
   kobject       object;

   
   if ((object = get_object(argv[0], argv[1])) == NULL)
   {
      kerror(NULL, routine,
	     "A .whatis line in one of the .sec or .help files has an "
	     "incorrect toolbox name or object name, the line in error is:\n"
	     "    %s\n", line);
      return FALSE;
   }

   kcms_set_attribute(object, KCMS_CMOBJ_ROUTINENAME, argv[2]);

   if (!kcms_get_attribute(object, KCMS_CMOBJ_HEADERINFO, &header))
   {
      kerror(NULL, routine,
	     "Unable to read the source code header for the whatis '%s'\n"
	     "The error occurred on the line:\n"
             "    %s\n"
             "Please check the source code header for errors and make sure\n"
             "it conforms with the Khoros coding conventions",
             argv[2], line);
      return FALSE;
   }

   if (header->shortdesc == NULL)
      kwarn(NULL, routine, "The short description for the routine '%s' "
	    "is empty.  kgenmanual will put this entry in your manual, "
	    "but you will need to update the header for this routine, "
	    "and rerun both kgenman and kgenmanual.", header->name);

   kfprintf(file, "\\(bu\n\\fI%s()\\fP - %s\n.br\n", header->name,
	    (header->shortdesc == NULL) ? "" : header->shortdesc);

   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_process_syntax - routine to process a .syntax line
|
| Purpose:	This routine will call kgen_interpret_clui_usage()
|		when it sees a ".syntax" line.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0] toolbox name
|			! argv[1] object name
|			!
|		line - The line from the input file which is being interpreted.
|
| Returns:	TRUE (1) on success, FALSE (0) otherwise
|
| Written By:	Danielle Argiro and Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
/* ARGSUSED */
static kbool
_process_syntax(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kstring   routine      = "_process_syntax()";
   kform    *form        = NULL;
   kobject   object;
   kobject   file_object;
   char     *panepath;
   int       progtype;


   if ((object = get_object(argv[0], argv[1])) == NULL)
   {
      kerror(NULL, routine,
	     "A .syntax line in one of the .sec or .help files has an "
	     "incorrect toolbox name or object name, the line in error is:\n"
	     "    %s\n", line);
      return FALSE;
   }
 
   if (!kcms_get_attribute(object, KCMS_CMOBJ_UIS_PANE, &file_object)
       || file_object == NULL
       || !kcms_get_attribute(file_object, KCMS_FOBJ_FULLNAME, &panepath)
       || panepath == NULL
       || !kcms_get_attribute(object, KCMS_CMOBJ_PROGTYPE, &progtype))
   {
      return FALSE;
   }

   if ((form = kvf_create_form(panepath, NONE, NULL, NULL)) == NULL)
      return FALSE;

   if (progtype == KCMS_KROUTINE || progtype == KCMS_PANE)
      kvf_append_selections(form, KGEN_KSTDARGS_PANEPATH); 
   else if (progtype == KCMS_XVROUTINE)
      kvf_append_selections(form, KGEN_XVSTDARGS_PANEPATH);

   kfprintf(file, ".nf\n");
   kgen_interpret_clui_usage(file, form, argv[0], argv[1], NULL);
   kfprintf(file, ".fi\n");
 
   kvf_destroy_form(form);
   return TRUE;
}

/*-----------------------------------------------------------
| Routine Name:	_process_man1 - routine to process a .function line
|
| Purpose:	This routine will parse a .function line.
|		The .function line is in the form 
|		.function {program_object_name} {function_name}.
|		The program object that the function belongs to is opened 
|		and the functions header is grabbed. Then the
|		header is formatted for the manual and written to the
|		output file.
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0]  toolbox name
|			! argv[1]  object name
|			! argv[2]  function name
|			! argv[3]  section-level (optional)
|			!
|		line - The line from the input file which is being interpreted.
|
| Written By:	Neil Bowers
| Date:		4-sep-94
------------------------------------------------------------*/
static kbool
_process_man1(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   kobject        object;
   kstring        level       = DEFAULT_MAN1_SECTION_LEVEL;


   if ((object = get_object(argv[0], argv[1])) == NULL)
      return FALSE;

   if (argc == 3)
      level = argv[2];

   return generate_man1(object, file, level);
}

/*-----------------------------------------------------------
| Routine Name:	_process_exec - handler routine for the .exec macro
|
| Purpose:	This routine execs the specifed command.
|		The entire line after the .exec is passed as argv[0],
|		to make life easier :-)
|
| Input:	file - A file handler for the file being generated.
|		argc - The number of \fIarguments\fP to the macro.
|		       For this macro, argc will always be 1.
|		argv - The array of arguments to the macro.
|			!
|			! argv[0] the rest of the line after the .exec
|			!
|		line - The line from the input file which is being interpreted.
|
| Written By:	Neil Bowers
| Date:		26-sep-94
------------------------------------------------------------*/
/* ARGSUSED */
static kbool
_process_exec(
   kfile    *file,
   int       argc,
   kstring  *argv,
   kstring   line)
{
   ksystem(argv[0]);
   return TRUE;
}
