/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { Fragment, VNode, h } from "preact";
import { CreatePage } from "./CreatePage.js";
import {
  assertUnreachable,
  Duration,
  HttpStatusCode,
  TalerError,
} from "@gnu-taler/taler-util";
import { useInstanceDetails } from "../../../../hooks/instance.js";
import { Loading } from "@gnu-taler/web-util/browser";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";

interface Props {
  onBack?: () => void;
  onConfirm: () => void;
}

export default function CreateTemplate({ onConfirm, onBack }: Props): VNode {
  const result = useInstanceDetails();
  
  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  return (
    <>
      <CreatePage
        onBack={onBack}
        onCreated={onConfirm}
        defaultSettingsDuration={Duration.fromTalerProtocolDuration(
          result.body.default_pay_delay,
        )}
      />
    </>
  );
}
