/*
 This file is part of GNU Taler
 (C) 2021-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Martin Schanzenbach
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { StateUpdater, useState } from "preact/hooks";
import { StatSlug } from "./index.js";


const TALER_SCREEN_ID = 60;

type Entity = StatSlug;

interface Props {
  counterStatSlugs: Entity[];
  amountStatSlugs: Entity[];
  onSelectAmountStat: (e: string) => void;
  onSelectCounterStat: (e: string) => void;
  onLoadMoreBefore?: () => void;
  onLoadMoreAfter?: () => void;
}

export function CardTable({
  counterStatSlugs,
  amountStatSlugs,
  onSelectAmountStat,
  onSelectCounterStat,
}: Props): VNode {
  const [rowSelection, rowSelectionHandler] = useState<string[]>([]);

  const { i18n } = useTranslationContext();

  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-counter" />
          </span>
          <i18n.Translate>Available statistics</i18n.Translate>
        </p>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {amountStatSlugs.length > 0 ? (
              <Table
                counterStats={counterStatSlugs}
                amountStats={amountStatSlugs}
                onSelectCounterStat={onSelectCounterStat}
                onSelectAmountStat={onSelectAmountStat}
                rowSelection={rowSelection}
                rowSelectionHandler={rowSelectionHandler}
              />
            ) : (
              <EmptyTable />
            )}
              <div style="width: 800px;"><canvas id="orderschartcanvas"></canvas></div>
          </div>
        </div>
      </div>
    </div>
  );
}
interface TableProps {
  rowSelection: string[];
  amountStats: Entity[];
  counterStats: Entity[];
  onSelectAmountStat: (e: string) => void;
  onSelectCounterStat: (e: string) => void;
  rowSelectionHandler: StateUpdater<string[]>;
}

function Table({
  amountStats,
  counterStats,
  onSelectAmountStat,
  onSelectCounterStat
}: TableProps): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="table-container">
      <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
        <thead>
          <tr>
            <th>
              <i18n.Translate>Description</i18n.Translate>
            </th>
            <th>
            </th>
            <th />
          </tr>
        </thead>
        <tbody>
          {amountStats.map((i) => {
            return (
              <tr key={i.slug}>
                <td
                  onClick={(): void => onSelectAmountStat(i.slug)}
                  style={{ cursor: "pointer" }}
                >
                  {i.text}
                </td>
                <td
                  onClick={(): void => onSelectAmountStat(i.slug)}
                  style={{ cursor: "pointer" }}
                >
                  <span class="mdi mdi-arrow-right"></span>
                </td>
              </tr>
            );
          })}
          {counterStats.map((i) => {
            return (
              <tr key={i.slug}>
                <td
                  onClick={(): void => onSelectCounterStat(i.slug)}
                  style={{ cursor: "pointer" }}
                >
                  {i.text}
                </td>
                <td
                  onClick={(): void => onSelectCounterStat(i.slug)}
                  style={{ cursor: "pointer" }}
                >
                  <span class="mdi mdi-arrow-right"></span>
                </td>
              </tr>
            );
          })}
        </tbody>
      </table>
    </div>
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-magnify mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>
          There are no statistics to list
        </i18n.Translate>
      </p>
    </div>
  );
}
