/*
 * @(#)DirectoryClassChannelLogReader.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.logger;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import net.sourceforge.groboutils.codecoverage.v2.IChannelLogRecord;
import net.sourceforge.groboutils.codecoverage.v2.IClassChannelLogReader;
import net.sourceforge.groboutils.codecoverage.v2.util.HexUtil;

/**
 * Reads logs written by DirectoryChannelLogger.
 * <p>
 * As of 2004-Jul-03, the output format has changed the way the indices
 * numbers are written.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/07/07 09:39:10 $
 * @since     December 17, 2002
 */
public class DirectoryClassChannelLogReader implements IClassChannelLogReader
{
    private File classLogFile;
    private String classSig;
    private BufferedReader in;
    
    
    DirectoryClassChannelLogReader( File log, String classSig )
    {
        if (log == null || classSig == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        if (!log.exists() || !log.isFile())
        {
            throw new IllegalArgumentException( "File "+log+
                " is not a file or doesn't exist." );
        }
        this.classLogFile = log;
        this.classSig = classSig;
    }
    
    
    /**
     * Reads the next record from the log.  If there are no more records, then
     * <tt>null</tt> is returned.  These do not need to be returned in any
     * specific order.
     *
     * @return the next log record.
     * @exception IOException thrown if there was an underlying problem reading
     *        from the log.
     */
    public synchronized IChannelLogRecord nextRecord()
            throws IOException
    {
        if (this.classLogFile == null)
        {
            return null;
        }
        if (this.in == null)
        {
            this.in = new BufferedReader(
                new FileReader( this.classLogFile ) );
        }
        
        String s = this.in.readLine();
        if (s == null)
        {
            close();
            return null;
        }
        
        // method value then mark value
        HexUtil.TwoShorts ts = new HexUtil.TwoShorts();
        if (!HexUtil.getInstance().parseTwoHex( s.trim(), ts, ' ', 0))
        {
            close();
            throw new IOException( "Invalid file format in '"+
                this.classLogFile+"'." );
        }
        
        return new DefaultChannelLogRecord(
            this.classSig, ts.a, ts.b );
        
    }
    
    
    public void close()
            throws IOException
    {
        if (this.in != null)
        {
            this.in.close();
            this.in = null;
        }
        this.classLogFile = null;
    }
    
    
    protected void finalize() throws Throwable
    {
        close();
        super.finalize();
    }
}

