package org.apache.commons.openpgp;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

/**
 * An interface for updating an OpenPGP signature on the fly with streaming data.
 *
 * @author <a href="mailto:brett@apache.org">Brett Porter</a>
 */
public interface OpenPgpStreamingSignatureVerifier
{
    String ROLE = OpenPgpStreamingSignatureVerifier.class.getName();

    /**
     * Update the signature with the next block from the data buffer.
     *
     * @param buf the buffer
     * @throws org.apache.commons.openpgp.OpenPgpException if the buffer is not valid for updating the signature
     */
    void update( byte[] buf )
        throws OpenPgpException;

    /**
     * Update the signature with the next block from the data buffer.
     *
     * @param buf    the buffer
     * @param offset offset within the buffer to start from
     * @param length number of bytes in the buffer to read from
     * @throws org.apache.commons.openpgp.OpenPgpException if the buffer is not valid for updating the signature
     */
    void update( byte[] buf, int offset, int length )
        throws OpenPgpException;

    /**
     * Finish and verify the signature that has been obtained.
     *
     * @return the status of the signature
     * @throws org.apache.commons.openpgp.OpenPgpException if the signature is not in a consistent or complete state
     */
    SignatureStatus finish()
        throws OpenPgpException, IOException;
}
