/*
 *	This is a Print Symbiont to run under VAX/VMS for the Apple
 *	LaserWriter printer.
 *
 *	It is designed to read in PostScript files and output them
 *	the the LaserWriter. It monitors the LaserWriter status
 *	and deals with any errors that the LaserWriter reports during
 *	the processing of a file.
 *
 *	This Print Symbiont is a full-blown symbiont implementation,
 *	designed to be controlled by the VMS Job Controller. As many
 *	features as possible of a Good Print Symbiont have been
 *	implemented. Many standard VMS symbiont features, such as
 *	margin control and page headings, cannot be reliably
 *	implemented for the LaserWriter, and have no effect if specified.
 *
 *	Other needed features, however, are hard to do. The most
 *	difficult is the detection of when a completed page is being
 *	ejected from the LaserWriter. Page ejection is handled by the
 *	application that generated the PostScript file, not by the
 *	symbiont. It might be suggested that we just scan for the
 *	string that causes a page ejection, much like you might scan
 *	for imbedded form feeds? Well, you can't. That's because the
 *	page ejection command could be imbedded in a procedure that has
 *	a different name; it would require us having to know too much
 *	PostScript for a lowly symbiont (we have to know quite a bit as it
 *	is just to handle error processing and flag and trailer pages).
 *	It would have been possible to assert that files passed to the
 *	symbiont adhere to certain rules and restrictions, such as the
 *	double-percent notation suggested in the PostScript manuals.
 *	Unfortunately, this is too imprecise (and arbitrary) to use in
 *	the terribly precise VMS environment (many of the PostScript
 *	conventions seem to have come from a popular AT&T-developed
 *	operating system noted for its looseness).
 *
 *	Because of the inability to isolate the command that generates
 *	a new page, checkpointing and recovery from checkpoint are not
 *	possible. This really wouldn't make sense to try anyway, since
 *	page descriptions often depend on commands and data processed
 *	on previous pages, such as downloaded fonts and procedure and
 *	object definitions.
 *
 *	This symbiont makes no attempt to implement a line-printer-like
 *	capability for the LaserWriter. This capability is easily
 *	implemented using the Diablo 630 emulation mode and the standard
 *	VMS-supplied print symbiont and form definition tools. It deals
 *	solely with PostScript files.
 *
 *	"LaserWriter" is a trademark of Apple Computer, Inc.
 *	"PostScript" is a trademark of Adobe Systems Incorporated.
 *	"Times" and "Helvetica" are trademarks of Allied Corporation.
 *	"VAX", "VMS", et. al. are trademarks of Digital Equipment Corporation.
 */

#include "symbiont.h"
#include "ast.h"
#include opcdef

/*
 *	The message processing routine leaves a global variable set
 *	indicating the type of message just processed. This is useful
 *	to determine additional control-level functions that must be
 *	performed after a message has been processed:
 */

unsigned long Last_Message_Type = 0;

char Scratch_String[256];

/*
 *	We begin. First, we must tell VMS some things about us and
 *	enable AST processing. The main program is called from a Macro
 *	routine to eliminate linking in all of the C language run-time
 *	stuff, which is not needed:
 */

unsigned long Main_Program ()
{
	auto   unsigned long Sys_Status, Stream, Count;
	static unsigned long Arg, SecArg;
	static unsigned long Max_Threads = sizeof (Device_Thread) / sizeof (struct Thread);
	static unsigned long Level = SMBMSG$K_STRUCTURE_LEVEL;
	extern unsigned long Smb$Initialize(), Sys$SetAST(), Set_Privileges();
	extern unsigned long Smb_Message_AST_Handler(), DeQueue_AST();
	extern unsigned long Process_JobCtl_Message(), Process_IO_Complete();
	extern unsigned long Process_LW_Message(), Process_Carrier_Loss();
	extern unsigned long Process_LW_Input(), Process_Control_D();
	extern unsigned long Process_TimeOut();
	globalvalue SS$_NORMAL;

	Initialize_AST_Queue ();
	Initialize_Threads ();
	if (((Sys_Status = Set_Privileges ()) & 0x01) != 0 &&
	    ((Sys_Status = Sys$SetAST (1)) & 0x01) != 0)
		Sys_Status = Smb$Initialize (&Level, &Smb_Message_AST_Handler,
					     &Max_Threads);
/*
 *	The main processing loop of the program is pretty simple. We
 *	do everything using AST's, so the processing at this level
 *	only is concerned with AST's. The different types of AST's that
 *	can occur are:
 *
 *		a) We have gotten a message from the Job Controller
 *		b) An I/O has completed on the LaserWriter
 *		c) Input has been received from the LaserWriter
 *		d) The input processor has detected a Control-D
 *		e) A timeout has occurred on Control-D wait
 *		f) Some other terminal status change has occurred
 *
 *	The only thing not done using an AST is reading in data from the
 *	input file (or object modules).
 *
 *	Initially, only one AST can occur: getting a message from the
 *	Job Controller. After a stream has been established, the other
 *	types of AST's can occur.
 */
	while ((Sys_Status & 0x01) != 0) {
		switch (DeQueue_AST (&Arg, &SecArg)) {
/*
 *	When there is nothing currently going on, we sleep; if there
 *	are currently no active tasks, purge out the working set:
 */
		case AST_NULL:
			Count = 0;
			for (Stream = 0; Stream < sizeof (Device_Thread) / sizeof (struct Thread); Stream++)
			if (Device_Thread[Stream].Task_Request_Status != 0)
				Count++;
			if (Count > 0)
				Purge_Working_Set ();
			Sys_Status = Sys$Hiber (); continue;
/*
 *	A write I/O completion is processed here. The special return
 *	code of zero means that we have stopped a task in response
 *	to a reset stream request:
 */
		case AST_DEVICE_WRITE_IO:
			Sys_Status = Process_IO_Complete (Arg, SecArg);
			if (Sys_Status == 0) {
				Sys_Status = SS$_NORMAL;
				Last_Message_Type = SMBMSG$K_RESET_STREAM;
				break;
			}
			continue;
/*
 *	Occasionally, the LaserWriter will mutter something to us that
 *	may require our attention, like running out of memory. This is
 *	done in two steps: first, the input is detected and a read is
 *	issued to get the message; second, the read operation is completed
 *	by checking what was sent:
 */
		case AST_DEVICE_INPUT: Sys_Status = Process_LW_Input (Arg); continue;
/*
 *	A simulated Control-D interrupt is generated by the input routine
 *	when it sees a Control-D. It is handled at this level because
 *	the task needs to be resumed in a controlled manner; a timeout can
 *	also occur if the Control-D is not received:
 */
		case AST_CONTROL_D: Sys_Status = Process_Control_D (Arg, SecArg); continue;
		case AST_TIMER:     Sys_Status = Process_TimeOut (Arg); continue;
/*
 *	In case the LaserWriter is hooked up over a modem, we must
 *	check for loss of carrier; broadcast messages are ignored:
 */
		case AST_CARRIER_LOSS: Sys_Status = Process_Carrier_Loss (Arg); continue;

		case AST_BROADCAST: Sys_Status = SS$_NORMAL;
/*
 *	A message from the Job Controller is processed here. This
 *	is a little bit more complicated because we would like important
 *	things done at this level rather than one level down; this is
 *	the only case that falls through to the end of the switch (all
 *	others continue to the next while-loop iteration):
 */
		case AST_JOBCTL_MESSAGE: Sys_Status = Process_JobCtl_Message (); break;
/*
 *	This next thing should never happen, but we should check even
 *	so:
 */
		default: Sys_Status = SS$_NORMAL;

		}
/*
 *	We do some extra checking when a Job Controller message has been
 *	processed: This includes:
 *
 *		a) Checking if all threads have been terminated. If so,
 *		   terminate with successful status.
 */
		if (Last_Message_Type == SMBMSG$K_STOP_STREAM || Last_Message_Type == SMBMSG$K_RESET_STREAM) {
			Count = 0;
			for (Stream = 0; Stream < sizeof (Device_Thread) / sizeof (struct Thread); Stream++)
			if (Device_Thread[Stream].Dcb != 0)
				Count++;
			if (Count == 0)
				return (SS$_NORMAL);
		}
	}
/*
 *	Exit off the bottom of this loop means a fatal program error.
 *	Normally, it should "never happen":
 */
	return (Sys_Status);
}

/*
 *	Routine Set_Privileges sets the required privileges needed. All
 *	symbionts initially have only the SETPRV privilege.
 *
 *	The privileges required are:
 *
 *	    ALLSPOOL - allocate a spooled device. Since most system
 *	               managers will set the device to allow automatic
 *	               spooling, this privilege is necessary.
 *	    TMPMBX   - create a temporary mailbox. A mailbox is created
 *	               whenever a device is assigned.
 *	    READALL  - necessary to access any file on the system in
 *	               order to print it. The symbiont has uic [1,4],
 *	               making it (usually) world relative to file
 *	               protections. READALL overrides uic and ACL
 *	               protections
 *	    SHARE    - another privilege needed if the output device
 *	               is already in use by another process.
 */

#include prvdef

unsigned long Set_Privileges ()
{
	auto   int Index;
	auto   unsigned char Priv_Code;
	static unsigned char Privileges[] = {
		PRV$V_ALLSPOOL,
		PRV$V_TMPMBX,
		PRV$V_READALL,
		PRV$V_SHARE
	};
	static unsigned long Privilege_Mask[2];
	extern unsigned long Sys$SetPrv();

	Privilege_Mask[0] = Privilege_Mask[1] = 0;
	for (Index = 0; Index < sizeof (Privileges); Index++) {
		Priv_Code = Privileges[Index];
		Privilege_Mask[Priv_Code >> 5] |= (1 << (Priv_Code & 0x1F));
	}
	return (Sys$SetPrv (1, Privilege_Mask, 0, 0));
}

/*
 *	Routine Process_JobCtl_Message obtains the message from the
 *	the Job Controller and dispatches further duties to
 *	subordinates:
 */

unsigned long Process_JobCtl_Message ()
{
	auto   struct Thread *Thread_Ptr;
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc = { 0, 0, 0, 0 };
	static unsigned long Stream, Request;
	extern unsigned long Smb$Read_Message();
	extern unsigned long Start_Stream(), Stop_Stream(), Reset_Stream();
	extern unsigned long Start_Task(), Stop_Task(), Pause_Task(), Resume_Task();
	globalvalue SMB$_INVSTMNBR, SS$_NORMAL;
/*
 *	First, allocate a dynamic string descriptor to be used henceforth
 *	if it has not already been done. The VMS Run-Time routines will
 *	automatically maintain this for us:
 */
	if (Buf_Desc.dsc$b_class == 0)
		Set_Dynamic_VMS_Descriptor (&Buf_Desc);
/*
 *	Get the whole message body from the Job Controller:
 */
	Sys_Status = Smb$Read_Message (&Stream, &Buf_Desc, &Request);
/*
 *	Determine the message type. In accordance with DEC wishes,
 *	unknown types are ignored:
 */
	if ((Sys_Status & 0x01) != 0) {
		Last_Message_Type = Request;
		if (Stream >= sizeof (Device_Thread) / sizeof (struct Thread))
			Sys_Status = SMB$_INVSTMNBR;
		else {
			Thread_Ptr = &Device_Thread[Stream];
			switch (Request) {

			case SMBMSG$K_START_STREAM:
				Sys_Status = Start_Stream (&Buf_Desc, Thread_Ptr);
				break;

			case SMBMSG$K_STOP_STREAM:
				Sys_Status = Stop_Stream (&Buf_Desc, Thread_Ptr);
				break;

			case SMBMSG$K_RESET_STREAM:
				Sys_Status = Reset_Stream (&Buf_Desc, Thread_Ptr);
				break;

			case SMBMSG$K_START_TASK:
				Sys_Status = Start_Task (&Buf_Desc, Thread_Ptr);
				break;

			case SMBMSG$K_STOP_TASK:
				Sys_Status = Stop_Task (&Buf_Desc, Thread_Ptr);
				break;

			case SMBMSG$K_PAUSE_TASK:
				Sys_Status = Pause_Task (&Buf_Desc, Thread_Ptr);
				break;

			case SMBMSG$K_RESUME_TASK:
				Sys_Status = Resume_Task (&Buf_Desc, Thread_Ptr);
				break;

			default:
				Sys_Status = SS$_NORMAL;
			}
		}
	}
	return (Sys_Status);
}

/*
 *	Routine Start_Stream begins a new stream on a new device:
 */

unsigned long Start_Stream (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	auto   int Modem;
	static char *Characteristics = "type=unknown,eightbit=off,escape=off,hostsync=on,lower=on,mbxdsabl=off,\
formfeed=on,tab=on,brdcst=off,echo=off,typeahd=on,altypeahd=on,scope=off,ttsync=on,wrap=off,editing=off,\
parity=off,pasthru=off,xon=on";
	static unsigned long Status;
	extern char *Connect_Device();
	extern unsigned long ReInitialize_Thread(), Read_Message_Items();
	extern unsigned long Set_Device_Characteristics();
	extern unsigned long Send_JBC_Message();
	extern int Is_a_Terminal(), On_a_Modem();
	globalvalue SS$_NORMAL, SHR$_NOTTERM, SS$_DEVACTIVE;
/*
 *	Initialize thread data; read items for this message; connect the device;
 *	formulate return codes based on success or failure; set device characteristics
 *	to those required for successful operation:
 */
	if (Thread_Ptr->Dcb != 0)
		Sys_Status = SS$_DEVACTIVE;
	else if (((Sys_Status = ReInitialize_Thread (Thread_Ptr)) & 0x01) != 0 &&
	    ((Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr)) & 0x01) != 0) {
		Copy_Desc_to_String (&Thread_Ptr->Device_Name, Scratch_String, sizeof (Scratch_String));
		Thread_Ptr->Device_Status = 0;
		if ((Thread_Ptr->Dcb = Connect_Device (Scratch_String, BUFFER_SIZE, 1, &Status)) == 0) {
			Thread_Ptr->Device_Status |= SMBMSG$M_UNAVAILABLE | SMBMSG$M_STOP_STREAM;
			Add_Condition_Value (Status, Thread_Ptr);
		} else if (Is_a_Terminal (Thread_Ptr->Dcb) == 0) {
			Add_Condition_Value (SHR$_NOTTERM, Thread_Ptr);
			Thread_Ptr->Device_Status |= SMBMSG$M_STOP_STREAM;
		} else {
			Thread_Ptr->Device_Status |= SMBMSG$M_TERMINAL | SMBMSG$M_LOWERCASE;
			if ((Modem = On_a_Modem (Thread_Ptr->Dcb)) != 0)
				Thread_Ptr->Device_Status |= SMBMSG$M_REMOTE;
			Sys_Status = Set_Device_Characteristics (Characteristics, Thread_Ptr->Dcb);
		}
		Sys_Status = Send_JBC_Message (SMBMSG$K_START_STREAM,
					       JBC_DVST | JBC_ERR, Thread_Ptr);
	}
	return (Sys_Status);
}

/*
 *	Routine Stop_Stream terminates a stream and disconnects the device
 *	associated with it:
 */

unsigned long Stop_Stream (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Read_Message_Items(), Send_JBC_Message();

	Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr);
	if ((Sys_Status & 0x01) != 0) {
		if (Thread_Ptr->Dcb != 0) {
			Disconnect_Device (Thread_Ptr->Dcb);
			Thread_Ptr->Dcb = 0;
		}
		Sys_Status = Send_JBC_Message (SMBMSG$K_STOP_STREAM, 0, Thread_Ptr);
	}
	return (Sys_Status);
}

/*
 *	Routine Reset_Stream resets a stream. A special set of actions are
 *	required if a task is active: set the RESET flag to signal routine
 *	Process_IO_Complete that the stream is to be reset, and the current
 *	task aborted, upon the next I/O completion. Otherwise, this acts
 *	like a STOP_STREAM request.
 */

unsigned long Reset_Stream (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Read_Message_Items(), Send_JBC_Message(), Cancel_Device_IO();

	Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr);
	if ((Sys_Status & 0x01) != 0) {
		if (Thread_Ptr->Dcb != 0) {
			if (Thread_Ptr->Task_Request_Status != 0 &&
			    (Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) == 0) {	/* Active task */
				Thread_Ptr->Task_Status |= TASK_M_RESET;
				Sys_Status = Cancel_Device_IO (Thread_Ptr->Dcb);
				return (Sys_Status);
			}
			if ((Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) != 0)
				Cancel_Timer (Thread_Ptr);
			Thread_Ptr->Task_Request_Status = 0;
			Disconnect_Device (Thread_Ptr->Dcb);
			Thread_Ptr->Dcb = 0;
		}
		Sys_Status = Send_JBC_Message (SMBMSG$K_STOP_STREAM, 0, Thread_Ptr);
	}
	return (Sys_Status);
}

/*
 *	Routine Start_Task begins the printing of a file:
 */

unsigned long Start_Task (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long ReInitialize_Task(), Read_Message_Items();
	extern unsigned long Start_Subtask(), Resume_Subtask();
	extern unsigned long Sys$GetTim();
	extern int Test_Item();
	globalvalue RMS$_EOF, SS$_NORMAL;

	if (((Sys_Status = ReInitialize_Task (Thread_Ptr)) & 0x01) != 0 &&
	    ((Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr)) & 0x01) != 0) {
		Thread_Ptr->Incarnation++;
		Sys_Status = Sys$GetTim (&Thread_Ptr->Time_Printed);
		Build_Qualifier_Sentence (Thread_Ptr);
		Build_Job_Sentence (Thread_Ptr);
		Build_File_Sentence (Thread_Ptr);
		if (Test_Item (SMBMSG$K_NOTE, Thread_Ptr) != 0)
			Format_Sentence (&Thread_Ptr->Note);
		Setup_Subtasks (Thread_Ptr);
		Sys_Status = Start_Subtask (Thread_Ptr);
		if ((Sys_Status & 0x01) != 0) {
			Sys_Status = Resume_Subtask (Thread_Ptr);
			if (Sys_Status == RMS$_EOF)
				Sys_Status = SS$_NORMAL;
		}
	}
	return (Sys_Status);
}

/*
 *	Routine Setup_Subtasks sets up the bit vector which controls
 *	the processing of subtasks:
 */

Setup_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Req_Stat, Sep_Ctl;
	extern int Test_Item();
/*
 *	Start with the basics, done for every file:
 */
	Req_Stat = TASK_M_START | TASK_M_FILE_PREFIX | TASK_M_PRINT_FILE |
		   TASK_M_FILE_POSTFIX | TASK_M_AFTER_PRINT | TASK_M_FINISH;
/*
 *	Check for file setup (and reset) parameters:
 */
	if (Test_Item (SMBMSG$K_FORM_SETUP_MODULES, Thread_Ptr) != 0)
		Req_Stat |= TASK_M_FORM_SETUP;
	if (Test_Item (SMBMSG$K_FILE_SETUP_MODULES, Thread_Ptr) != 0)
		Req_Stat |= TASK_M_FILE_SETUP;
	if (Test_Item (SMBMSG$K_PAGE_SETUP_MODULES, Thread_Ptr) != 0)
		Req_Stat |= TASK_M_PAGE_SETUP_PREFIX | TASK_M_PAGE_SETUP | TASK_M_PAGE_SETUP_POSTFIX;
/*
 *	Check for file separation parameters:
 */
	if (Test_Item (SMBMSG$K_SEPARATION_CONTROL, Thread_Ptr) != 0) {
		Sep_Ctl = Thread_Ptr->Separation_Control;
		if ((Sep_Ctl & SMBMSG$M_JOB_BURST) != 0)
			Req_Stat |= TASK_M_JOB_BURST;
		if ((Sep_Ctl & SMBMSG$M_JOB_FLAG) != 0)
			Req_Stat |= TASK_M_JOB_FLAG;
		if ((Sep_Ctl & SMBMSG$M_JOB_TRAILER) != 0)
			Req_Stat |= TASK_M_JOB_TRAILER | TASK_M_JOB_TRAILER_PREFIX | TASK_M_JOB_TRAILER_POSTFIX;
		if ((Sep_Ctl & SMBMSG$M_JOB_RESET) != 0)
			Req_Stat |= TASK_M_JOB_RESET;
		if ((Sep_Ctl & SMBMSG$M_FILE_BURST) != 0)
			Req_Stat |= TASK_M_FILE_BURST;
		if ((Sep_Ctl & SMBMSG$M_FILE_FLAG) != 0)
			Req_Stat |= TASK_M_FILE_FLAG;
		if ((Sep_Ctl & SMBMSG$M_FILE_TRAILER) != 0)
			Req_Stat |= TASK_M_FILE_TRAILER | TASK_M_FILE_TRAILER_PREFIX | TASK_M_FILE_TRAILER_POSTFIX;
		if ((Req_Stat & (TASK_M_JOB_BURST | TASK_M_JOB_FLAG | TASK_M_FILE_BURST |
		     TASK_M_FILE_FLAG)) != 0)
			Req_Stat |= TASK_M_FLAG_PREFIX | TASK_M_FLAG_POSTFIX;
	}
/*
 *	Check for job statistics wanted:
 */
	if ((Thread_Ptr->Task_Options & TASK_M_STATISTICS) != 0)
		Req_Stat |= TASK_M_JOB_TRAILER_PREFIX | TASK_M_JOB_STATS | TASK_M_JOB_TRAILER_POSTFIX;
	Thread_Ptr->Task_Request_Status = Req_Stat;
/*
 *	Set the first subtask to be performed, initialize:
 */
	Thread_Ptr->Current_Subtask = TASK_V_START;
	Thread_Ptr->Task_Completion_Status = 0;
	Thread_Ptr->Task_Status = 0;
}

/*
 *	Routine Setup_Abort_Subtasks sets up additional subtasks if
 *	the task completed abnormally:
 */

Setup_Abort_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Req_Stat, Sep_Ctl;
	extern int Test_Item();

	Req_Stat = 0;
	if (Test_Item (SMBMSG$K_SEPARATION_CONTROL, Thread_Ptr) != 0) {
		Sep_Ctl = Thread_Ptr->Separation_Control;
		if ((Sep_Ctl & SMBMSG$M_JOB_TRAILER_ABORT) != 0)
			Req_Stat |= TASK_M_JOB_TRAILER | TASK_M_JOB_TRAILER_PREFIX | TASK_M_JOB_TRAILER_POSTFIX;
		if ((Sep_Ctl & SMBMSG$M_JOB_RESET_ABORT) != 0 &&
		    Test_Item (SMBMSG$K_JOB_RESET_MODULES, Thread_Ptr) != 0)
			Req_Stat |= TASK_M_JOB_RESET;
		if ((Sep_Ctl & SMBMSG$M_FILE_TRAILER_ABORT) != 0)
			Req_Stat |= TASK_M_FILE_TRAILER | TASK_M_FILE_TRAILER_PREFIX | TASK_M_FILE_TRAILER_POSTFIX;
	}
	Thread_Ptr->Task_Request_Status |= Req_Stat;
}

/*
 *	Routine Setup_After_Subtasks sets up any after-the-file-is-printed
 *	subtasks. This MUST wait for the Control-D to be seen in order to
 *	faithfully determine what, if any, outputs or errors have occurred
 *	as a result of the printing of the file:
 */

Setup_After_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	if (Thread_Ptr->Error_Name.dsc$w_length != 0 || Thread_Ptr->N_Condition_Values > 0)
		Thread_Ptr->Task_Request_Status |= (TASK_M_JOB_ERRORS | TASK_M_FILE_TRAILER_PREFIX |
						    TASK_M_FILE_TRAILER_POSTFIX);
	if (Thread_Ptr->Misc_Output_Head != 0)
		Thread_Ptr->Task_Request_Status |= (TASK_M_JOB_OUTPUT | TASK_M_FILE_TRAILER_PREFIX |
						    TASK_M_FILE_TRAILER_POSTFIX);
}

/*
 *	Routine Stop_Task does an emergency stop of a printing.
 *	To accomplish this, a Control-C (interrupt) is sent to
 *	the device after cancelling any outstanding I/O, done only
 *	if the file is printing or if the file postfix is being
 *	printed. There are several possible situations that can
 *	occur that may cause simple task termination (via I/O
 *	cancellation and subsequent clean-up) to be less than
 *	timely:
 *
 *	    a) The file is being printed, but the QIO is stuck in
 *	       XOFF and a compute intensive procedure is executing
 *	       on the LaserWriter (or perhaps an endless loop).
 *	       For this situation, the Control-C interrupt will
 *	       cause normal task termination, with the 'interrupt'
 *	       error seeming like a user error at the device.
 *
 *	    b) The whole file has been sent, but the file postfix
 *	       is similarly stuck in XOFF (there are only two
 *	       characters in the postfix string, but this
 *	       situation is still possible). In this situation
 *	       the Control-D in the file postfix may not reach
 *	       the device, so we send one synchronously.
 *
 *	    c) The file postfix has been sent, and I/O completed,
 *	       but the LaserWriter is stuck in a loop with the
 *	       terminating Control-D inside it's buffer, not yet
 *	       seen. The symbiont's state in this situation will
 *	       be Control-D Wait.
 *
 *	After cancelling outstanding I/O, a Control-C, possibly
 *	followed by a Control-D, is immediately sent to the
 *	device (the output I/O AST will [I think] execute
 *	immediately after the call to Cancel_Device_IO). This
 *	causes an 'interrupt' error to occur, causing the
 *	LaserWriter to flush its buffer until the Control-D is
 *	seen. It is imperative that a Control-D is presented in
 *	the input stream after the Control-C.
 *
 *	For this operation, the normal I/O completion routines
 *	handle the rest of the task stoppage. Any subtask other
 *	than printing the file and setup subtasks are allowed to
 *	complete.
 */

unsigned long Stop_Task (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Read_Message_Items(), Cause_Interrupt();
	extern unsigned long Write_Device_Immediate();

	Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr);
	if ((Sys_Status & 0x01) != 0 && Thread_Ptr->Task_Request_Status != 0) {
		Add_Condition_Value (Thread_Ptr->Stop_Condition, Thread_Ptr);
		Thread_Ptr->Task_Status &= ~TASK_M_PAUSED;
		Thread_Ptr->Task_Status |= TASK_M_STOP | TASK_M_STOPPED;
		if (Thread_Ptr->Current_Subtask == TASK_V_PRINT_FILE ||
		    Thread_Ptr->Current_Subtask == TASK_V_FILE_POSTFIX ||
		    Thread_Ptr->Current_Subtask == TASK_V_AFTER_PRINT)
			Sys_Status = Cause_Interrupt (Thread_Ptr);
			if ((Sys_Status & 0x01) != 0 && Thread_Ptr->Current_Subtask == TASK_V_FILE_POSTFIX)
				Sys_Status = Write_Device_Immediate ("\004", 1, Thread_Ptr->Dcb);
	}
	return (Sys_Status);
}

unsigned long Cause_Interrupt (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Set_Device_Characteristics(), Cancel_Device_IO();
	extern unsigned long Write_Device_Immediate();
	globalvalue SS$_NORMAL;

	if ((Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) == 0)
		Sys_Status = Cancel_Device_IO (Thread_Ptr->Dcb);	/* AST occurs here */
	else
		Sys_Status = SS$_NORMAL;
	if ((Sys_Status & 0x01) != 0) {
		Sys_Status = Set_Device_Characteristics ("xon=on", Thread_Ptr->Dcb);
		if ((Sys_Status & 0x01) != 0)
			Sys_Status = Write_Device_Immediate ("\003", 1, Thread_Ptr->Dcb);
	}
	return (Sys_Status);
}

/*
 *	Routine Pause_Task suspends the printing of a file:
 */

unsigned long Pause_Task (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Read_Message_Items(), Send_JBC_Message();

	Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr);
	if ((Sys_Status & 0x01) != 0) {
		Thread_Ptr->Task_Status |= TASK_M_PAUSED;
		Sys_Status = Send_JBC_Message (SMBMSG$K_PAUSE_TASK, 0, Thread_Ptr);
	}
	return (Sys_Status);
}

/*
 *	Routine Resume_Task resumes printing from a paused task:
 */

unsigned long Resume_Task (Buf_Desc, Thread_Ptr)
struct dsc$descriptor *Buf_Desc;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Read_Message_Items(), Resume_Subtask(), Send_JBC_Message();
	globalvalue RMS$_EOF, SS$_NORMAL;

	Sys_Status = Read_Message_Items (Buf_Desc, Thread_Ptr);
	if ((Sys_Status & 0x01) != 0) {
		if ((Thread_Ptr->Task_Status & (TASK_M_PAUSED | TASK_M_CONTROL_D_WAIT)) != 0)
			Thread_Ptr->Task_Status &= ~TASK_M_PAUSED;
		else {
			Sys_Status = Resume_Subtask (Thread_Ptr);
			if (Sys_Status == RMS$_EOF)
				Sys_Status = SS$_NORMAL;
		}
	}
	if ((Sys_Status & 0x01) != 0)
		Sys_Status = Send_JBC_Message (SMBMSG$K_RESUME_TASK, 0, Thread_Ptr);
	return (Sys_Status);
}

#include "iostatus.h"

/*
 *	Routine Process_IO_Complete is executed when an output I/O to the
 *	LaserWriter has finished.
 */

unsigned long Process_IO_Complete (Dev_Block, Buf_Block)
char *Dev_Block, *Buf_Block;
{
	auto   struct Thread *Thread_Ptr;
	extern struct Thread *Find_Thread_Using_Dcb();
	extern unsigned long Process_Completion();
	static struct Write_IO_Status_Block IO_Status;
	globalvalue SS$_NORMAL, SS$_CANCEL;
/*
 *	Determine the thread associated with this device:
 */
	if ((Thread_Ptr = Find_Thread_Using_Dcb (Dev_Block)) == 0)
		return (SS$_NORMAL);	/* (not really) */
/*
 *	Check the I/O completion status:
 */
	Set_Device_Write_Status (Buf_Block, &IO_Status);
	if (IO_Status.Status != SS$_CANCEL)
		Thread_Ptr->Accounting_Data.smbmsg$l_qio_puts++;
	return (Process_Completion (IO_Status.Status, Thread_Ptr));
}

/*
 *	Routine Process_Control_D resumes a subtask that has been
 *	hung up waiting for the Control-D from the LaserWriter:
 *
 *	There is a possible synchronization problem between processing
 *	of the Control-D and the timeout. If they occur simultaneously,
 *	I'm not sure what would happen. Theoretically, the Control-D
 *	will be processed first. Even if it did happen, all that would
 *	occur is an abrupt cancellation of the task.
 */

unsigned long Process_Control_D (Thread_Ptr, Check)
struct Thread *Thread_Ptr;
unsigned long Check;
{
	auto   unsigned long Sys_Status;
	static struct Time End_Time;
	extern unsigned long Process_Completion(), Sys$GetTim();
	globalvalue SS$_NORMAL;

	if (Thread_Ptr->Task_Request_Status == 0 || Thread_Ptr->Incarnation != Check ||
	    (Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) == 0)
		Sys_Status = SS$_NORMAL;
	else {
		Cancel_Timer (Thread_Ptr);
		if (Thread_Ptr->Timer_Count < MAX_TIMER) {
			Sys$GetTim (&End_Time);
			Subtract_Quad (&End_Time, &Thread_Ptr->Timer[Thread_Ptr->Timer_Count].Total_Time);
			Thread_Ptr->Timer_Count++;
		}
		Thread_Ptr->Task_Status |= TASK_M_CONTROL_D;
		Thread_Ptr->Task_Status &= ~TASK_M_CONTROL_D_WAIT;
		Sys_Status = Process_Completion (SS$_NORMAL, Thread_Ptr);
	}
	return (Sys_Status);
}

/*
 *	A timeout occurs if the wait for a Control-D from the LaserWriter
 *	takes too long to appear (this could mean a lot of things - someone
 *	turned off the LaserWriter, an XON hang has occurred [see known bugs
 *	in LaserWriter], and others). The task is unconditionally terminated.
 *
 *	By definition, no other things can be happening if a timeout has
 *	occurred, since all task activity has stopped pending the Control-D.
 */

unsigned long Process_TimeOut (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct Time End_Time;
	extern unsigned long Process_Completion(), Sys$GetTim();
	globalvalue SS$_TIMEOUT, SS$_NORMAL;

	if (Thread_Ptr->Task_Request_Status == 0 ||
	    (Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) == 0)
		Sys_Status = SS$_NORMAL;
	else {
		if (Thread_Ptr->Timer_Count < MAX_TIMER) {
			Sys$GetTim (&End_Time);
			Subtract_Quad (&End_Time, &Thread_Ptr->Timer[Thread_Ptr->Timer_Count].Total_Time);
			Thread_Ptr->Timer_Count++;
		}
		Thread_Ptr->Task_Status &= ~TASK_M_CONTROL_D_WAIT;	/* In case of simultaneous completion */
		Sys_Status = Process_Completion (SS$_TIMEOUT, Thread_Ptr);
	}
	return (Sys_Status);
}

/*
 *	Routine Process_Completion performs those actions required when
 *	an event that allows continuation of activity to the device
 *	occurs:
 */

unsigned long Process_Completion (Status, Thread_Ptr)
unsigned long Status;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Device_Control(), Stop_Task_and_Reset(), Stop_Task_Soft();
	extern unsigned long Resume_Subtask(), Start_Subtask(), Finish_Subtask();
	extern int Advance_Subtask();
	globalvalue SS$_NORMAL, SS$_CANCEL, SS$_ABORT, RMS$_EOF;
/*
 *	Check the status value. If anything abnormal has happened,
 *	set up to finish up the task without performing any further
 *	I/O:
 */
	Sys_Status = SS$_NORMAL;
	if (Status != SS$_ABORT && Status != SS$_CANCEL && (Status & 0x01) == 0) {
		Add_Condition_Value (Status, Thread_Ptr);
		Thread_Ptr->Task_Status |= TASK_M_DEVERROR;
	}
/*
 *	Check if the task is to be stopped in response to a stream
 *	reset directive; if so, stop all activity and return. Note
 *	that this cannot happen for Control_D or timeout events,
 *	since the reset is performed synchronously in those cases.
 */
	if ((Thread_Ptr->Task_Status & TASK_M_RESET) != 0) {
                Sys_Status = Stop_Task_and_Reset (Thread_Ptr);
		if ((Sys_Status & 0x01) != 0)
			Sys_Status = 0;		/* Signal to main program */
		return (Sys_Status);
	}
/*
 *	Check for a device error. This can occur if the line is lost,
 *	the LaserWriter is unresponsive, a modem hang-up occurs, or a
 *	QIO error has occurred. In all cases, terminate all further
 *	activity on the device. Note that a timeout error can occur
 *	if the program executing on the LaserWriter exceeds the timeout
 *	period in returning a Control-D; this may be normal - the
 *	LaserWriter may be executing in some long-winded loop. Currently
 *	there is no way around this; we must make sure that the task
 *	(from our perspective) eventually finishes, regardless of
 *	circumstances.
 */
	if ((Thread_Ptr->Task_Status & TASK_M_DEVERROR) != 0) {
/*		Thread_Ptr->Device_Status |= SMBMSG$M_STALLED;	*/
		Finish_Subtask (Thread_Ptr);
		Thread_Ptr->Task_Status |= TASK_M_CONTROL_D;	/* Simulated Control-D */
		Thread_Ptr->Current_Subtask = TASK_V_FINISH;
		Start_Subtask (Thread_Ptr);
	}
/*
 *	Check for an error occuring in the program execution at the
 *	LaserWriter. Cancel printing of the file, since there is no
 *	point in continuing (unless some fool has output the string
 *	identical to that used by the LaserWriter to signal errors!)
 *	Start_Subtask is ok since the next subtask must be FILE_POSTFIX,
 *	which does not require a Control-D wait, and cannot generate
 *	an error.
 */
	if ((Thread_Ptr->Task_Status & TASK_M_USERERROR) != 0 &&
	    Thread_Ptr->Current_Subtask == TASK_V_PRINT_FILE) {
		Finish_Subtask (Thread_Ptr);
		Advance_Subtask (Thread_Ptr);
		Start_Subtask (Thread_Ptr);
	}
/*
 *	Check if the current task is to be stopped. Mostly, this will
 *	occur while printing the file, but can occur anywhere during
 *	the task. Routine Stop_Task_Soft ensures subtask continuity.
 *	Note that if a device error has occurred (above), routine
 *	Stop_Task_Soft has no effect, since the current task is now
 *	FINISH -- after any trailer pages that may be set up.
 */
	if ((Thread_Ptr->Task_Status & TASK_M_STOP) != 0) {
		Thread_Ptr->Task_Status &= ~TASK_M_STOP;
		if ((Sys_Status = Stop_Task_Soft (Thread_Ptr)) == 0)
			return (SS$_NORMAL);
		else if ((Sys_Status & 0x01) == 0)
			return (Sys_Status);
	}
/*
 *	Check if the task is being paused. If so, stop doing anything
 *	on the task; otherwise, continue the current subtask or start
 *	up a new one until the last subtask is done:
 */
	if ((Thread_Ptr->Task_Status & TASK_M_PAUSED) != 0)
		Thread_Ptr->Task_Status &= ~TASK_M_PAUSED;
	else {
		Sys_Status = Resume_Subtask (Thread_Ptr);
		if (Sys_Status == RMS$_EOF)
			Sys_Status = SS$_NORMAL;
	}
	return (Sys_Status);
}

/*
 *	Routine Stop_Task_and_Reset performs an emergency stop of the
 *	currently executing task, then performs those operations
 *	necessary to reset the queue. This routine is called from the
 *	I/O completion handlers.
 */

unsigned long Stop_Task_and_Reset (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Finish_Subtask(), Send_JBC_Message();

	if ((Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) != 0)
		Cancel_Timer (Thread_Ptr);
	Finish_Subtask (Thread_Ptr);
	Thread_Ptr->Task_Request_Status = 0;
	Sys_Status = Send_JBC_Message (SMBMSG$K_STOP_TASK, JBC_ACNT | JBC_ERR, Thread_Ptr);
	Disconnect_Device (Thread_Ptr->Dcb);
	Thread_Ptr->Dcb = 0;
	Sys_Status = Send_JBC_Message (SMBMSG$K_STOP_STREAM, 0, Thread_Ptr);
	return (Sys_Status);
}

/*
 *	Routine Stop_Task_Soft performs those actions necessary to
 *	discontinue a task, but to allow for the printing of trailer pages
 *	and reset modules. A return value of zero from Stop_Task_Soft means
 *	that the task is in Control-D wait.
 *
 *	Most subtasks, once started, cannot be reliably
 *	stopped. Even stopping the printing of the file can generate
 *	errors from the LaserWriter, if say, we interrupt it in the
 *	middle of a command, which is quite likely. Therefore, we
 *	eliminate (and, if necessary, abort) the printing of the file,
 *	but allow most other subtasks to continue until completion.
 */

unsigned long Stop_Task_Soft (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Finish_Subtask(), Start_Subtask();
	extern int Advance_Subtask();
	globalvalue SS$_NORMAL;

	Thread_Ptr->Task_Status &= ~TASK_M_PAUSED;
	Cancel_All_Subtasks (Thread_Ptr);
	Setup_Abort_Subtasks (Thread_Ptr);
/*
 *	If the current subtask has to do with printing the file,
 *	finish it up and continue to the next subtask; otherwise just
 *	continue on. After_Print is cancelled because aborting the
 *	printing of the file will almost surely generate an error
 *	at the LaserWriter, and we don't want people to see the
 *	mess we've left.
 */
	Cancel_File_Subtasks (Thread_Ptr);
	if (Thread_Ptr->Current_Subtask == TASK_V_PAGE_SETUP || Thread_Ptr->Current_Subtask == TASK_V_FILE_SETUP ||
	    Thread_Ptr->Current_Subtask == TASK_V_PRINT_FILE) {
		Finish_Subtask (Thread_Ptr);
		if (Thread_Ptr->Current_Subtask == TASK_V_PAGE_SETUP)
			Thread_Ptr->Task_Request_Status &= ~(TASK_M_FILE_SETUP | TASK_M_PRINT_FILE | TASK_M_AFTER_PRINT);
		else if (Thread_Ptr->Current_Subtask == TASK_V_FILE_SETUP)
			Thread_Ptr->Task_Request_Status &= ~(TASK_M_PRINT_FILE | TASK_M_AFTER_PRINT);
		else
			Thread_Ptr->Task_Request_Status &= ~TASK_M_AFTER_PRINT;
		Advance_Subtask (Thread_Ptr);
		Sys_Status = Start_Subtask (Thread_Ptr);
	} else
		Sys_Status = SS$_NORMAL;
	return (Sys_Status);
}

unsigned long Resume_Subtask (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Start_Subtask(), Continue_Subtask(), Finish_Subtask();
	extern int Advance_Subtask();
	globalvalue RMS$_EOF, SS$_NORMAL;

	while ((Sys_Status = Continue_Subtask (Thread_Ptr)) == RMS$_EOF) {
		if (Thread_Ptr->Current_Subtask == TASK_V_FINISH) {
			Thread_Ptr->Task_Request_Status = 0;
			break;
		}
		Finish_Subtask (Thread_Ptr);
		Advance_Subtask (Thread_Ptr);
		if (((Sys_Status = Start_Subtask (Thread_Ptr)) & 0x01) == 0)
			break;
	}
	return ((Sys_Status == 0) ? SS$_NORMAL : Sys_Status);
}

/*
 *	Routine Start_Subtask performs any initialization required
 *	for the next subtask of the task, such as opening files,
 *	etc.
 */

unsigned long Start_Subtask (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   struct dsc$descriptor *Desc_Ptr;
	auto   char **String_Ptr;
	auto   unsigned long Sys_Status, FAO_Status, Request;
	auto   unsigned long Transmit_Time;
	static struct dsc$descriptor FAO_Desc, Buf_Desc, Form_Desc;
	static union {	/* forces OPC buffer large enough */
		struct OPC Opc_Buf;
		char Msg_Buf[168];
	} Opc_Msg;
	static char *FAO_String = "\
Manual feed form request for LaserWriter !AS (queue !AS)!/!/\
!_User:!_!AS!/!_Job:!_!AS!/!_Form:!_!AS";
	static unsigned long Status[2];
	static unsigned short Length;
	extern char *Open_File(), *Open_Text_Library();
	extern unsigned long Send_JBC_Message(), Set_Timer();
	extern unsigned long Sys$SndOpr(), Sys$GetTim();
	extern unsigned long Compute_Transmit_Time(), Append_Stat();
	extern unsigned long SMB_StartFile();
	extern int Next_Word();
	extern char *Flag_Prefix[], *Flag_Postfix[], *File_Prefix[];
	extern char *File_Postfix[], *Job_Burst[], *File_Burst[];
	extern char *Page_Setup_Prefix[], *Page_Setup_Postfix[];
	extern char *Manual_Feed;
	globalvalue SS$_NORMAL, SS$_NOSUCHFILE;

	Sys_Status = Sys$GetTim (&Thread_Ptr->Subtask_Time[Thread_Ptr->Current_Subtask]);
	switch (Thread_Ptr->Current_Subtask) {

	case TASK_V_START:

		Sys_Status = Send_JBC_Message (SMBMSG$K_START_TASK, 0, Thread_Ptr);
		if ((Sys_Status & 0x01) != 0 && (Thread_Ptr->Task_Options & TASK_M_MANUAL_FEED) != 0) {
			if ((Thread_Ptr->Task_Options & TASK_M_FORM) != 0)
				Copy_Desc_to_Fixed_Desc (&Thread_Ptr->Form_Type, &Form_Desc);
			else
				Make_VMS_Descriptor ("UNSPECIFIED", &Form_Desc);
			Make_VMS_Descriptor (FAO_String, &FAO_Desc);
			Set_VMS_Descriptor (&Opc_Msg.Opc_Buf.opc$l_ms_text, sizeof (Opc_Msg) - 8, &Buf_Desc);
			FAO_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc, &Thread_Ptr->Device_Name,
					      &Thread_Ptr->Executor_Queue, &Thread_Ptr->User_Name,
					      &Thread_Ptr->Job_Name, &Form_Desc);
			if ((FAO_Status & 0x01) != 0) {
				Opc_Msg.Opc_Buf.opc$b_ms_type = OPC$_RQ_RQST;
				Opc_Msg.Opc_Buf.opc$b_ms_target = OPC$M_NM_PRINT;
				Set_VMS_Descriptor (&Opc_Msg, Length+8, &Buf_Desc);
				FAO_Status = Sys$SndOpr (&Buf_Desc, 0);
			}
		}
		break;

	case TASK_V_ALIGNMENT:

		Thread_Ptr->Special_Page_Count = 0;
		Sys_Status = SS$_NORMAL;
		break;

	case TASK_V_FORM_SETUP:

		Desc_Ptr = &Thread_Ptr->Form_Setup_Modules;
		goto Do_Setup;

	case TASK_V_PAGE_SETUP:

		Desc_Ptr = &Thread_Ptr->Page_Setup_Modules;
		goto Do_Setup;

	case TASK_V_FILE_SETUP:

		Desc_Ptr = &Thread_Ptr->File_Setup_Modules;
		goto Do_Setup;

	case TASK_V_JOB_RESET:

		Desc_Ptr = &Thread_Ptr->Job_Reset_Modules;
Do_Setup:	Copy_Desc_to_Fixed_Desc (Desc_Ptr, &Thread_Ptr->Whole_Desc[0]);
		Set_VMS_Descriptor (Thread_Ptr->Whole_Desc[0].dsc$a_pointer, 0, &Thread_Ptr->Partial_Desc);
		Copy_Desc_to_String (&Thread_Ptr->Library_Specification, &Scratch_String[0], sizeof (Scratch_String));
		Thread_Ptr->Lcb = Open_Text_Library (Scratch_String, "SYS$LIBRARY:SYSDEVCTL", "r", &Status[0]);
		if (Thread_Ptr->Lcb == 0)
			Append_Stat (Thread_Ptr, "Can't open device control library !AS",
				     &Thread_Ptr->Library_Specification);
		Thread_Ptr->Fcb = 0;
		Sys_Status = SS$_NORMAL;
		break;

	case TASK_V_FILE_TRAILER_PREFIX:
	case TASK_V_JOB_TRAILER_PREFIX:
	case TASK_V_FLAG_PREFIX:

		String_Ptr = &Flag_Prefix[0];
		goto Do_String;

	case TASK_V_JOB_FLAG:

		Build_Job_Flag (Thread_Ptr);
		goto Do_Special;

	case TASK_V_JOB_BURST:

		String_Ptr = &Job_Burst[0];
		goto Do_String;

	case TASK_V_FILE_FLAG:

		Build_File_Flag (Thread_Ptr);
		goto Do_Special;

	case TASK_V_FILE_BURST:

		String_Ptr = &File_Burst[0];
		goto Do_String;

	case TASK_V_FLAG_POSTFIX:
	case TASK_V_FILE_TRAILER_POSTFIX:
	case TASK_V_JOB_TRAILER_POSTFIX:

		String_Ptr = &Flag_Postfix[0];
		goto Do_String;

	case TASK_V_PAGE_SETUP_PREFIX:

		String_Ptr = &Page_Setup_Prefix[0];
		goto Do_String;

	case TASK_V_PAGE_SETUP_POSTFIX:

		String_Ptr = &Page_Setup_Postfix[0];
		goto Do_String;

	case TASK_V_FILE_PREFIX:

		String_Ptr = &File_Prefix[0];
		goto Do_String;

	case TASK_V_PRINT_FILE:

		Thread_Ptr->Fcb = Open_File (0, &Thread_Ptr->File_Identification[0], "r", "\n");
		if (Thread_Ptr->Fcb == 0)
			Add_Condition_Value (SS$_NOSUCHFILE, Thread_Ptr);
		Sys_Status = SMB_StartFile (&Thread_Ptr->File_Specification, &Thread_Ptr->Job_Name,
					    &Thread_Ptr->User_Name, &Thread_Ptr->Uic, &Thread_Ptr->Time_Printed,
					    &Thread_Ptr->Queue, &Thread_Ptr->Executor_Queue,
					    &Thread_Ptr->Device_Name, &Thread_Ptr->Entry_Number,
					    &Thread_Ptr->File_Count);
		break;

	case TASK_V_FILE_POSTFIX:

		String_Ptr = &File_Postfix[0];
Do_String:	Thread_Ptr->Desc_Count = 0;
		while (*String_Ptr != 0 &&
		   Thread_Ptr->Desc_Count < sizeof (Thread_Ptr->Whole_Desc) / sizeof (struct dsc$descriptor))
			Make_VMS_Descriptor (*String_Ptr++, &Thread_Ptr->Whole_Desc[Thread_Ptr->Desc_Count++]);
/*
 *	Check for the special case of having to put the string to
 *	set manual feed when the subtask is the file prefix:
 */
		if (Thread_Ptr->Current_Subtask == TASK_V_FILE_PREFIX &&
		    (Thread_Ptr->Task_Options & TASK_M_MANUAL_FEED) != 0 &&
		    Thread_Ptr->Desc_Count < sizeof (Thread_Ptr->Whole_Desc) / sizeof (struct dsc$descriptor))
			Make_VMS_Descriptor (Manual_Feed, &Thread_Ptr->Whole_Desc[Thread_Ptr->Desc_Count++]);
		goto Set_String;

	case TASK_V_AFTER_PRINT:

		Sys_Status = SS$_NORMAL;
		break;

	case TASK_V_JOB_OUTPUT:
	case TASK_V_JOB_STATS:

		Set_VMS_Descriptor (0, 0, &Thread_Ptr->Whole_Desc[0]);
		Thread_Ptr->Desc_Count = 1;
		Thread_Ptr->Special_Page_Count = 0;
		goto Set_String;

	case TASK_V_JOB_ERRORS:

		Build_Error_Page (Thread_Ptr);
		goto Do_Special;

	case TASK_V_FILE_TRAILER:

		Build_File_Trailer (Thread_Ptr);
		goto Do_Special;

	case TASK_V_JOB_TRAILER:

		Build_Job_Trailer (Thread_Ptr);
Do_Special:	Copy_Desc_to_Fixed_Desc (&Thread_Ptr->Page_Desc, &Thread_Ptr->Whole_Desc[0]);
		Thread_Ptr->Desc_Count = 1;
Set_String:	Thread_Ptr->Desc_Index = 0;
		Thread_Ptr->Partial_Desc = Thread_Ptr->Whole_Desc[0];
		Sys_Status = SS$_NORMAL;
		break;

	case TASK_V_FINISH:

		Sys_Status = SS$_NORMAL;
		break;
	}
/*
 *	There is a problem with subtask synchronization that requires
 *	that certain subtasks cannot be started without the Control-D
 *	from the previous task being detected. The subtasks that are
 *	affected are:
 *
 *	FILE_PREFIX must wait for FLAG_POSTFIX
 *	AFTER_PRINT must wait for FILE_POSTFIX
 *	JOB_TRAILER_PREFIX must wait for FILE_TRAILER_POSTFIX
 *	FINISH must wait for JOB_TRAILER_POSTFIX.
 *
 *	The special return value of zero is returned if a subtask cannot
 *	be started because of a previous task (which has had I/O completion
 *	but, due to buffering by the LaserWriter, has not processed all the
 *	device commands) that has not as yet finished at the LaserWriter.
 *
 *	Note that 'AFTER_PRINT' is always scheduled, so trailer subtasks need
 *	not worry about anything before it.
 *
 *	This section of code attempts to compute reasonable values for
 *	the amount of time that we should wait for the Control-D that we
 *	send to be echoed. Since we don't know how compute intensive the
 *	PostScript program contained in the file is, we allow 2 minutes
 *	for that. For the other types of waits, we allow an amount of time
 *	it takes to absorb two full buffers, execute it, and to print one page
 *	(on an 8 page per minute printer, that is about 8 seconds). These
 *	values are admitted quite arbitrary. The assumption is made that the
 *	LaserWriter buffer can hold the entire flag/trailer command sequence
 *	prior to execution. The reason for doing this is to minimize the
 *	amount of time it takes to kill a job if a STOP_TASK directive is
 *	received from the Job Controller and we are hung up waiting for a
 *	Control-D.
 */

	if ((Sys_Status & 0x01) != 0 && (Thread_Ptr->Task_Status & TASK_M_CONTROL_D) == 0)
	switch (Thread_Ptr->Current_Subtask) {

	case TASK_V_FILE_PREFIX:

		if ((Thread_Ptr->Task_Completion_Status & TASK_M_FLAG_POSTFIX) != 0) {
			Transmit_Time = Compute_Transmit_Time (2*BUFFER_SIZE, Thread_Ptr);
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_FLAG) != 0)
				Transmit_Time += 35 + 8;
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_BURST) != 0)
				Transmit_Time += 5 + 8;
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_FILE_FLAG) != 0)
				Transmit_Time += 35 + 8;
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_FILE_BURST) != 0)
				Transmit_Time += 5 + 8;
			goto Set_Wait;
		}
		break;

	case TASK_V_AFTER_PRINT:

		if ((Thread_Ptr->Task_Completion_Status & (TASK_M_FLAG_POSTFIX | TASK_M_FILE_POSTFIX)) != 0) {
			Transmit_Time = 120;
			goto Set_Wait;
		}
		break;

	case TASK_V_JOB_TRAILER_PREFIX:

		if ((Thread_Ptr->Task_Completion_Status & TASK_M_FILE_TRAILER_POSTFIX) != 0) {
			Transmit_Time = Compute_Transmit_Time (2*BUFFER_SIZE, Thread_Ptr);
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_OUTPUT) != 0) {
				Transmit_Time += 20 + 8;
				if (Thread_Ptr->Output_Page_Count > 1)
					Transmit_Time += 20 + 8;
			}
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_ERRORS) != 0)
				Transmit_Time += 15 + 8;
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_FILE_TRAILER) != 0)
				Transmit_Time += 35 + 8;
			goto Set_Wait;
		}
		break;

	case TASK_V_FINISH:

		if ((Thread_Ptr->Task_Completion_Status &
		    (TASK_M_FILE_TRAILER_POSTFIX | TASK_M_JOB_TRAILER_POSTFIX)) != 0) {
			Transmit_Time = Compute_Transmit_Time (2*BUFFER_SIZE, Thread_Ptr);
			if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_TRAILER_POSTFIX) != 0) {
				if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_RESET) != 0)
					Transmit_Time += 30;
				if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_TRAILER) != 0)
					Transmit_Time += 35 + 8;
				if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_STATS) != 0)
					Transmit_Time += 20 + 8;
			} else {
				if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_OUTPUT) != 0) {
					Transmit_Time += 20 + 8;
					if (Thread_Ptr->Output_Page_Count > 1)
						Transmit_Time += 20 + 8;
				}
				if ((Thread_Ptr->Task_Completion_Status & TASK_M_JOB_ERRORS) != 0)
					Transmit_Time += 15 + 8;
				if ((Thread_Ptr->Task_Completion_Status & TASK_M_FILE_TRAILER) != 0)
					Transmit_Time += 35 + 8;
			}
			goto Set_Wait;
		}
		break;
/*
 *	Show that this task is waiting for a Control-D; start the timer:
 */

Set_Wait:	Thread_Ptr->Task_Status |= TASK_M_CONTROL_D_WAIT;
		if (Thread_Ptr->Timer_Count < MAX_TIMER) {
			Sys$GetTim (&Thread_Ptr->Timer[Thread_Ptr->Timer_Count].Total_Time);
			Thread_Ptr->Timer[Thread_Ptr->Timer_Count].Task_Type = Thread_Ptr->Current_Subtask;
		}
		Append_Stat (Thread_Ptr, "Setting timer for !UL seconds for subtask type !UL",
			     Transmit_Time, Thread_Ptr->Current_Subtask);
		Sys_Status = Set_Timer (1000*Transmit_Time, Thread_Ptr);
		if ((Sys_Status & 0x01) != 0)
			Sys_Status = 0;
	}
	Thread_Ptr->Task_Status &= ~TASK_M_CONTROL_D;
	return (Sys_Status);
}

/*
 *	Routine Continue_Subtask continues processing on the
 *	current subtask of the current task.
 */

unsigned long Continue_Subtask (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   int Count;
	static unsigned long Stats[2];
	extern char *Open_Library_File();
	extern unsigned long Send_JBC_Message(), Append_Stat();
	extern int Transfer_to_Device(), Next_Word();
	extern unsigned char Read_Character(), Read_Library_Character();
	extern unsigned char Read_String_Character();
	globalvalue SS$_NORMAL, RMS$_EOF;

	switch (Thread_Ptr->Current_Subtask) {

	case TASK_V_START:
	case TASK_V_ALIGNMENT:

		Count = 0;
		break;

	case TASK_V_FINISH:

		Send_JBC_Message (((Thread_Ptr->Task_Status & TASK_M_STOPPED) == 0) ?
				    SMBMSG$K_TASK_COMPLETE : SMBMSG$K_STOP_TASK, JBC_ACNT | JBC_ERR,
				   Thread_Ptr);
		Count = 0;
		break;

	case TASK_V_FORM_SETUP:
	case TASK_V_PAGE_SETUP:
	case TASK_V_FILE_SETUP:
	case TASK_V_JOB_RESET:

		if (Thread_Ptr->Lcb == 0)
			Count = 0;
		else do {
			while (Thread_Ptr->Fcb == 0)
			if (Next_Word (&Thread_Ptr->Partial_Desc, &Thread_Ptr->Whole_Desc[0], ',') == 0) {
				Count = 0;
				goto Setup_Done;
			} else {
				Copy_Desc_to_String (&Thread_Ptr->Partial_Desc, &Scratch_String[0],
						     sizeof (Scratch_String));
				Thread_Ptr->Fcb = Open_Library_File (&Scratch_String[0], "\n", Thread_Ptr->Lcb,
								     &Stats[0]);
				if (Thread_Ptr->Fcb == 0)
					Append_Stat (Thread_Ptr, "Can't open library module !AS",
						     &Thread_Ptr->Partial_Desc);
				else
					Thread_Ptr->Library_Module_Count++;
			}
			if ((Count = Transfer_to_Device (&Read_Library_Character, Thread_Ptr->Fcb, 0,
							 Thread_Ptr)) == 0) {
				Get_Library_Stats (Thread_Ptr->Fcb, &Stats[0]);
				Thread_Ptr->Accounting_Data.smbmsg$l_rms_gets += Stats[0];
				Close_Library_File (Thread_Ptr->Fcb);
				Thread_Ptr->Fcb = 0;
			}
		} while (Count == 0);
Setup_Done:	break;

	case TASK_V_JOB_OUTPUT:
	case TASK_V_JOB_STATS:

		while ((Count = Transfer_to_Device (&Read_String_Character, &Thread_Ptr->Partial_Desc, 1,
						    Thread_Ptr)) == 0) {
			Thread_Ptr->Special_Page_Count++;
			if (Thread_Ptr->Current_Subtask == TASK_V_JOB_OUTPUT)
				Build_Output_Page (Thread_Ptr);
			else
				Build_Statistics_Page (Thread_Ptr);
			Copy_Desc_to_Fixed_Desc (&Thread_Ptr->Page_Desc, &Thread_Ptr->Whole_Desc[0]);
			Thread_Ptr->Partial_Desc = Thread_Ptr->Whole_Desc[0];
			if (Thread_Ptr->Partial_Desc.dsc$w_length == 0) {
				Thread_Ptr->Special_Page_Count--;
				break;
			}
		};
		break;

	case TASK_V_FLAG_PREFIX:
	case TASK_V_JOB_FLAG:
	case TASK_V_JOB_BURST:
	case TASK_V_FILE_FLAG:
	case TASK_V_FILE_BURST:
	case TASK_V_FLAG_POSTFIX:
	case TASK_V_PAGE_SETUP_PREFIX:
	case TASK_V_PAGE_SETUP_POSTFIX:
	case TASK_V_FILE_PREFIX:
	case TASK_V_FILE_POSTFIX:
	case TASK_V_FILE_TRAILER_PREFIX:
	case TASK_V_JOB_ERRORS:
	case TASK_V_FILE_TRAILER:
	case TASK_V_FILE_TRAILER_POSTFIX:
	case TASK_V_JOB_TRAILER_PREFIX:
	case TASK_V_JOB_TRAILER:
	case TASK_V_JOB_TRAILER_POSTFIX:

		while ((Count = Transfer_to_Device (&Read_String_Character, &Thread_Ptr->Partial_Desc,
						    1, Thread_Ptr)) == 0) {
			if (++Thread_Ptr->Desc_Index >= Thread_Ptr->Desc_Count)
				break;
			Thread_Ptr->Partial_Desc = Thread_Ptr->Whole_Desc[Thread_Ptr->Desc_Index];
		}
		break;

	case TASK_V_PRINT_FILE:

		if (Thread_Ptr->Fcb == 0)
			Count = 0;
		else
			Count = Transfer_to_Device (&Read_Character, Thread_Ptr->Fcb, 0, Thread_Ptr);
		break;

	case TASK_V_AFTER_PRINT:

		Setup_After_Subtasks (Thread_Ptr);
		Count = 0;
	}
	return ((Count == 0) ? RMS$_EOF : SS$_NORMAL);
}

/*
 *	Routine Finish_Subtask wraps up processing on the
 *	current subtask of the current task.
 */

unsigned long Finish_Subtask (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct Time End_Time;
	static unsigned long Stats[2];
	extern unsigned long Send_JBC_Message(), Sys$GetTim();
	globalvalue SS$_NORMAL;

	switch (Thread_Ptr->Current_Subtask) {

	case TASK_V_START:
	case TASK_V_ALIGNMENT:
	case TASK_V_AFTER_PRINT:
	case TASK_V_FINISH:

		break;

	case TASK_V_FORM_SETUP:
	case TASK_V_PAGE_SETUP:
	case TASK_V_FILE_SETUP:
	case TASK_V_JOB_RESET:

		if (Thread_Ptr->Lcb != 0) {
			if (Thread_Ptr->Fcb != 0) {
				Get_Library_Stats (Thread_Ptr->Fcb, &Stats[0]);
				Thread_Ptr->Accounting_Data.smbmsg$l_rms_gets += Stats[0];
				Close_Library_File (Thread_Ptr->Fcb);
				Thread_Ptr->Fcb = 0;
			}
			Close_Library (Thread_Ptr->Lcb);
			Thread_Ptr->Lcb = 0;
		}
		break;

	case TASK_V_JOB_FLAG:
	case TASK_V_JOB_BURST:
	case TASK_V_FILE_FLAG:
	case TASK_V_FILE_BURST:
	case TASK_V_JOB_ERRORS:
	case TASK_V_FILE_TRAILER:
	case TASK_V_JOB_TRAILER:

		Thread_Ptr->Accounting_Data.smbmsg$l_pages_printed++;
		goto Set_Desc;

	case TASK_V_JOB_OUTPUT:

		Thread_Ptr->Output_Page_Count = Thread_Ptr->Special_Page_Count;

	case TASK_V_JOB_STATS:

		Thread_Ptr->Accounting_Data.smbmsg$l_pages_printed += Thread_Ptr->Special_Page_Count;

	case TASK_V_FLAG_PREFIX:
	case TASK_V_FLAG_POSTFIX:
	case TASK_V_PAGE_SETUP_PREFIX:
	case TASK_V_PAGE_SETUP_POSTFIX:
	case TASK_V_FILE_PREFIX:
	case TASK_V_FILE_POSTFIX:
	case TASK_V_FILE_TRAILER_PREFIX:
	case TASK_V_FILE_TRAILER_POSTFIX:
	case TASK_V_JOB_TRAILER_PREFIX:
	case TASK_V_JOB_TRAILER_POSTFIX:

Set_Desc:	Thread_Ptr->Partial_Desc.dsc$w_length = 0;
		Thread_Ptr->Desc_Index = Thread_Ptr->Desc_Count;
		break;

	case TASK_V_PRINT_FILE:

		if (Thread_Ptr->Fcb != 0) {
			Get_File_Stats (Thread_Ptr->Fcb, Stats);
			Thread_Ptr->Accounting_Data.smbmsg$l_rms_gets += Stats[0];
			Close_File (Thread_Ptr->Fcb);
			Thread_Ptr->Fcb = 0;
		}
		break;
	}
	Thread_Ptr->Task_Completion_Status |= (1 << Thread_Ptr->Current_Subtask);
	Sys_Status = Sys$GetTim (&End_Time);
	Subtract_Quad (&End_Time, &Thread_Ptr->Subtask_Time[Thread_Ptr->Current_Subtask]);
	return (SS$_NORMAL);
}

/*
 *	Routine Transfer_to_Device transfers characters from an input
 *	stream to the output device. It returns the number of characters
 *	written:
 */

int Transfer_to_Device (Get_Func, Get_Arg, Incl_Flag, Thread_Ptr)
unsigned char (*Get_Func)();
char *Get_Arg;
int Incl_Flag;
struct Thread *Thread_Ptr;
{
	auto   char *Ptr;
	auto   int Count;
	static unsigned char c;
	extern unsigned long Write_Device(), Flush_Device_IO();

	for (Count = 0; Count < BUFFER_SIZE && (c = (*Get_Func) (Get_Arg)) != '\0'; )
	if (c != '\004' || Incl_Flag != 0) {	/* DO NOT allow user to pass Control-D */
		Write_Device (&c, 1, Thread_Ptr->Dcb);
		Count++;
	}
	if (Count > 0 && (Flush_Device_IO (Thread_Ptr->Dcb) & 0x01) == 0)
		Count = 0;
	return (Count);
}

int Advance_Subtask (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	do
		Thread_Ptr->Current_Subtask++;
	while ((Thread_Ptr->Task_Request_Status & (1 << Thread_Ptr->Current_Subtask)) == 0);
	return (Thread_Ptr->Current_Subtask);
}

Add_Condition_Value (Value, Thread_Ptr)
unsigned long Value;
struct Thread *Thread_Ptr;
{
	auto   unsigned int Index;

	for (Index = MAX_CONDITION_VALUES-1; Index > 0; Index--)
		Thread_Ptr->Condition_Vector[Index] = Thread_Ptr->Condition_Vector[Index-1];
	Thread_Ptr->Condition_Vector[0] = Value;
	if (Thread_Ptr->N_Condition_Values < MAX_CONDITION_VALUES)
		Thread_Ptr->N_Condition_Values++;
}

/*
 *	Routine Process_Carrier_Loss does the necessary things to
 *	shut down the processing due to loss of the channel to the
 *	output device.
 */

unsigned long Process_Carrier_Loss (Dev_Block)
char *Dev_Block;
{
	auto   unsigned long Sys_Status;
	auto   struct Thread *Thread_Ptr;
	extern struct Thread *Find_Thread_Using_Dcb();
	extern unsigned long Cancel_Device_IO();
	globalvalue SS$_NORMAL, SS$_HANGUP;

	Sys_Status = SS$_NORMAL;
	if ((Thread_Ptr = Find_Thread_Using_Dcb (Dev_Block)) != 0 &&
	    Thread_Ptr->Task_Request_Status != 0)
		Add_Condition_Value (SS$_HANGUP, Thread_Ptr);
		Thread_Ptr->Task_Status |= TASK_M_DEVERROR;
/*		Thread_Ptr->Device_Status |= SMBMSG$M_STOP_STREAM;	*/
		if ((Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) == 0)
			Sys_Status = Cancel_Device_IO (Thread_Ptr->Dcb);
	return (Sys_Status);
}

/*
 *	Routines Cancel_All_Subtasks is to cancel all but necessary subtasks.
 *	Cancel is used when processing has been deliberately halted by the
 *	job controller - but our exit must not generate errors.
 */

Cancel_All_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	Cancel_Flag_Subtasks (Thread_Ptr);
	Cancel_File_Subtasks (Thread_Ptr);
	Cancel_Trailer_Subtasks (Thread_Ptr);
}

Cancel_Flag_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	if (Thread_Ptr->Current_Subtask < TASK_V_FLAG_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_FLAG_PREFIX | TASK_M_JOB_FLAG | TASK_M_JOB_BURST |
						     TASK_M_FILE_FLAG | TASK_M_FILE_BURST | TASK_M_FLAG_POSTFIX);
	else if (Thread_Ptr->Current_Subtask == TASK_V_FLAG_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_JOB_FLAG | TASK_M_JOB_BURST | TASK_M_FILE_FLAG |
						     TASK_M_FILE_BURST);
}

Cancel_File_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	if (Thread_Ptr->Current_Subtask < TASK_V_FILE_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_FILE_PREFIX | TASK_M_PAGE_SETUP_PREFIX |
						     TASK_M_PAGE_SETUP | TASK_M_PAGE_SETUP_POSTFIX |
						     TASK_M_FILE_SETUP | TASK_M_PRINT_FILE |
						     TASK_M_AFTER_PRINT | TASK_M_FILE_POSTFIX);
	else if (Thread_Ptr->Current_Subtask == TASK_V_FILE_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_PAGE_SETUP_PREFIX | TASK_M_PAGE_SETUP |
						     TASK_M_PAGE_SETUP_POSTFIX | TASK_M_FILE_SETUP |
						     TASK_M_PRINT_FILE | TASK_M_AFTER_PRINT);
	else if (Thread_Ptr->Current_Subtask == TASK_V_PAGE_SETUP_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_PAGE_SETUP | TASK_M_FILE_SETUP |
						     TASK_M_PRINT_FILE | TASK_M_AFTER_PRINT);
	else if (Thread_Ptr->Current_Subtask == TASK_V_FILE_SETUP)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_PRINT_FILE | TASK_M_AFTER_PRINT);
}

Cancel_Trailer_Subtasks (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	if (Thread_Ptr->Current_Subtask < TASK_V_FILE_TRAILER_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_FILE_TRAILER_PREFIX | TASK_M_JOB_OUTPUT |
						     TASK_M_JOB_ERRORS | TASK_M_FILE_TRAILER |
						     TASK_M_FILE_TRAILER_POSTFIX | TASK_M_JOB_RESET |
						     TASK_M_JOB_TRAILER_PREFIX | TASK_M_JOB_TRAILER | 
						     TASK_M_JOB_STATS | TASK_M_JOB_TRAILER_POSTFIX);
	else if (Thread_Ptr->Current_Subtask == TASK_V_FILE_TRAILER_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_JOB_OUTPUT | TASK_M_JOB_ERRORS | TASK_M_FILE_TRAILER |
						     TASK_M_JOB_RESET | TASK_M_JOB_TRAILER_PREFIX |
						     TASK_M_JOB_STATS | TASK_M_JOB_TRAILER |
						     TASK_M_JOB_TRAILER_POSTFIX);
	else if (Thread_Ptr->Current_Subtask < TASK_V_JOB_RESET)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_JOB_RESET | TASK_M_JOB_TRAILER_PREFIX |
						     TASK_M_JOB_STATS | TASK_M_JOB_TRAILER |
						     TASK_M_JOB_TRAILER_POSTFIX);
	else if (Thread_Ptr->Current_Subtask < TASK_V_JOB_TRAILER_PREFIX)	/* Doing JOB_RESET */
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_JOB_TRAILER_PREFIX | TASK_M_JOB_TRAILER | 
						     TASK_M_JOB_STATS | TASK_M_JOB_TRAILER_POSTFIX);
	else if (Thread_Ptr->Current_Subtask == TASK_V_JOB_TRAILER_PREFIX)
		Thread_Ptr->Task_Request_Status &= ~(TASK_M_JOB_STATS | TASK_M_JOB_TRAILER);
}

struct Thread *Find_Thread_Using_Dcb (Device_Control_Block)
char *Device_Control_Block;
{
	auto   unsigned int Stream;
/*
 *	Determine the thread associated with this device:
 */
	for (Stream = 0; Stream < sizeof (Device_Thread) / sizeof (struct Thread); Stream++)
	if (Device_Control_Block == Device_Thread[Stream].Dcb)
		return (&Device_Thread[Stream]);
	return (0);
}

unsigned long Compute_Transmit_Time (Char_Count, Thread_Ptr)
unsigned long Char_Count;
struct Thread *Thread_Ptr;
{
	auto   unsigned long Baud_Rate;
	extern unsigned long Device_Speed();

	if ((Baud_Rate = Device_Speed (Thread_Ptr->Dcb)) == 0)
		Baud_Rate = 1200;
	Baud_Rate /= 10;
	return ((Char_Count + Baud_Rate - 1) / Baud_Rate);
}

/*
 *	This next routine purges the working set when the symbiont is
 *	not processing a job. The peculiar variable name consisting of
 *	all underscores generates a program section at or very near
 *	the highest program address. This routine will purge the pages
 *	containing the memory that is allocated dynamically.
 */

Purge_Working_Set ()
{
	auto   unsigned long High_Addr;
	static unsigned long Addr_Range[2];
	extern long _______________________________;
	extern unsigned long Sys$PurgWS();

	High_Addr = (long) &_______________________________;
	Addr_Range[0] = (High_Addr + 511) & ~0x1FF;
	Addr_Range[1] = 0x3FFFFFFF;	/* All of user P0 space */
	Check_System_Status (Sys$PurgWS (Addr_Range));
}

/*
 *	This is the AST Handler for messages from the Job Controller.
 *	It queues an AST and wakes the main process:
 */

unsigned long Smb_Message_AST_Handler ()
{
	extern unsigned long Sys$Wake();
	globalvalue SS$_NORMAL;

	EnQueue_AST (AST_JOBCTL_MESSAGE, 0, 0, PRI_JOBCTL_MESSAGE);
	Sys$Wake (0, 0);
	return (SS$_NORMAL);
}

int Check_System_Status (Status)
unsigned long Status;
{
	if ((Status & 0x01) == 0)
		return (0);
	return (1);
}
