/*
 *	Miscellaneous routines used by the symbiont. Most deal with
 *	PostScript related things.
 */

#include "symbiont.h"

/*
 *	Routines Build_Job_Flag, Build_File_Flag, Build_Job_Trailer,
 *	Build_File_Trailer, Build_Output_Page and Build_Error_Page construct
 *	the complete string necessary to generate the specified page. All
 *	five routines output to the 'Page_Desc' descriptor.
 */

char Scratch_String[256];

Build_Job_Flag (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc;
	static struct dsc$descriptor User_Desc, Job_Desc, Announce_Desc;
	static unsigned short Length;
	static char FAO_String[] = "\
LM 30 add TM 30 sub moveto(!UL)54 54 6 jobbox \
RM 30 sub TM 30 sub moveto(!UL)54 54 6 jobbox \
mediumfont XC TM 30 sub moveto !AS RM LM sub 150 sub 54 6 boxedtext \
smallfont LM TM 100 sub moveto !AS \
largefont XC TM YC sub 3 idiv YC add moveto !AS cshow \
hugefont XC YC moveto !AS cshow \
largefont XC YC YC BM sub 3 idiv sub moveto(Entry !UL)cshow \
smallfont LM BM 100 add LS add moveto !AS \
footer \
showpage ";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO();
	extern char *Mem_Alloc();
/*
 *	Format all strings which we have no content control over:
 */
	Generate_Job_Params (&Announce_Desc, &User_Desc, &Job_Desc, Thread_Ptr);
/*
 *	Determine the approximate length of the output string:
 */
	Length = FAO_Desc.dsc$w_length - 7 * 3 + 3 * 5 + Announce_Desc.dsc$w_length +
		 Thread_Ptr->Note.dsc$w_length + User_Desc.dsc$w_length +
		 Job_Desc.dsc$w_length + Thread_Ptr->Job_Sentence.dsc$w_length;
/*
 *	Allocate what is needed for the output string:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Format the string:
 */
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc,
			      Thread_Ptr->Job_Count, Thread_Ptr->Job_Count,
			      &Announce_Desc, &Thread_Ptr->Note,
			      &User_Desc, &Job_Desc, Thread_Ptr->Entry_Number,
			      &Thread_Ptr->Job_Sentence);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
	Mem_Free (Buf_Desc.dsc$a_pointer);
}

Build_File_Flag (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc;
	static struct dsc$descriptor User_Desc, Job_Desc, Announce_Desc;
	static struct dsc$descriptor Node_Desc, Dev_Desc, Dir_Desc;
	static struct dsc$descriptor Name_Desc, Ext_Desc, Version_Desc;
	static unsigned short Length;
	static char FAO_String[] = "\
LM 30 add TM 30 sub moveto(!UL)54 54 6 jobbox \
LM 102 add TM 30 sub moveto(!UL)54 54 6 filebox \
RM 102 sub TM 30 sub moveto(!UL)54 54 6 filebox \
RM 30 sub TM 30 sub moveto(!UL)54 54 6 jobbox \
mediumfont XC TM 30 sub moveto !AS RM LM sub 294 sub 54 6 boxedtext \
smallfont LM TM 100 sub moveto !AS \
hugefont XC TM YC sub 3 idiv YC add moveto !AS cshow \
largefont XC YC moveto(!AS)cshow \
XC YC LS sub moveto mediumfont(!AS!AS)cshow \
smallfont LM BM 100 add LS 6 mul add moveto !AS \
cr lf lf !AS \
footer \
showpage ";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO();
	extern char *Mem_Alloc();
/*
 *	Format all strings which we have no content control over;
 *	extract VMS filename, extension, and version from the filespec:
 */
	Generate_Job_Params (&Announce_Desc, &User_Desc, &Job_Desc, Thread_Ptr);
	Generate_File_Params (&Node_Desc, &Dev_Desc, &Dir_Desc, &Name_Desc,
			      &Ext_Desc, &Version_Desc, Thread_Ptr);
/*
 *	Determine the approximate length of the output string:
 */
	Length = FAO_Desc.dsc$w_length - 12 * 3 + 4 * 5 + Announce_Desc.dsc$w_length +
		 Thread_Ptr->Note.dsc$w_length + User_Desc.dsc$w_length +
		 Name_Desc.dsc$w_length + Ext_Desc.dsc$w_length + Version_Desc.dsc$w_length +
		 Thread_Ptr->File_Sentence.dsc$w_length + Thread_Ptr->Job_Sentence.dsc$w_length;
/*
 *	Allocate what is needed for the output string:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Format the string:
 */
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc,
			      Thread_Ptr->Job_Count, Thread_Ptr->File_Count,
			      Thread_Ptr->File_Count, Thread_Ptr->Job_Count,
			      &Announce_Desc, &Thread_Ptr->Note,
			      &User_Desc, &Name_Desc, &Ext_Desc, &Version_Desc,
			      &Thread_Ptr->File_Sentence, &Thread_Ptr->Job_Sentence);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
	Mem_Free (Buf_Desc.dsc$a_pointer);
}

Build_File_Trailer (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc;
	static struct dsc$descriptor User_Desc, Job_Desc, Announce_Desc;
	static struct dsc$descriptor Node_Desc, Dev_Desc, Dir_Desc;
	static struct dsc$descriptor Name_Desc, Ext_Desc, Version_Desc;
	static unsigned short Length;
	static char FAO_String[] = "\
LM 30 add TM 30 sub moveto(!UL)54 54 6 jobbox \
LM 102 add TM 30 sub moveto(!UL)54 54 6 filebox \
RM 102 sub TM 30 sub moveto(!UL)54 54 6 filebox \
RM 30 sub TM 30 sub moveto(!UL)54 54 6 jobbox \
hugefont XC TM 30 sub moveto(End of File)cshow \
XC TM YC sub 3 idiv YC add moveto !AS cshow \
largefont XC YC moveto(!AS)cshow \
XC YC LS sub moveto mediumfont(!AS!AS)cshow \
smallfont LM BM 100 add LS 8 mul add moveto !AS \
cr lf lf !AS \
cr lf lf !AS \
footer \
showpage ";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO();
	extern char *Mem_Alloc();
/*
 *	Format all strings which we have no content control over;
 *	extract VMS filename, extension, and version from the filespec:
 */
	Generate_Job_Params (&Announce_Desc, &User_Desc, &Job_Desc, Thread_Ptr);
	Generate_File_Params (&Node_Desc, &Dev_Desc, &Dir_Desc, &Name_Desc,
			      &Ext_Desc, &Version_Desc, Thread_Ptr);
/*
 *	Determine the approximate length of the output string:
 */
	Length = FAO_Desc.dsc$w_length - 11 * 3 + 4 * 5 + User_Desc.dsc$w_length +
		 Name_Desc.dsc$w_length + Ext_Desc.dsc$w_length + Version_Desc.dsc$w_length +
		 Thread_Ptr->Qualifier_Sentence.dsc$w_length + Thread_Ptr->File_Sentence.dsc$w_length +
		 Thread_Ptr->Job_Sentence.dsc$w_length;
/*
 *	Allocate what is needed for the output string:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Format the string:
 */
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc,
			      Thread_Ptr->Job_Count, Thread_Ptr->File_Count,
			      Thread_Ptr->File_Count, Thread_Ptr->Job_Count,
			      &User_Desc, &Name_Desc, &Ext_Desc, &Version_Desc,
			      &Thread_Ptr->Qualifier_Sentence, &Thread_Ptr->File_Sentence,
			      &Thread_Ptr->Job_Sentence);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
	Mem_Free (Buf_Desc.dsc$a_pointer);
}

Build_Job_Trailer (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc;
	static struct dsc$descriptor User_Desc, Job_Desc, Announce_Desc;
	static unsigned short Length;
	static char FAO_String[] = "\
LM 30 add TM 30 sub moveto(!UL)54 54 6 jobbox \
RM 30 sub TM 30 sub moveto(!UL)54 54 6 jobbox \
hugefont XC TM 30 sub moveto(End of Job)cshow \
smallfont LM TM 100 sub moveto !AS \
largefont XC TM YC sub 3 idiv YC add moveto !AS cshow \
hugefont XC YC moveto !AS cshow \
largefont XC YC YC BM sub 3 idiv sub moveto(Entry !UL)cshow \
mediumfont XC 24 add \
LM 12 add BM 100 add LS 3 mul add moveto \
RM 12 sub(I have reviewed this printout and certify it is classified:)textrule \
LM 12 add BM 100 add LS 2 mul add moveto dup 12 sub(Received by:)textrule \
dup BM 100 add LS 2 mul add moveto RM 12 sub(Date:)textrule \
LM 12 add BM 100 add LS add moveto dup 12 sub(Operator:)textrule \
BM 100 add LS add moveto(Printed on !17%D)show \
XC BM 104 add LS 2 mul add moveto RM LM sub 4 sub LS 4 mul 4 crect \
footer \
showpage ";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO();
	extern char *Mem_Alloc();
/*
 *	Format all strings which we have no content control over:
 */
	Generate_Job_Params (&Announce_Desc, &User_Desc, &Job_Desc, Thread_Ptr);
/*
 *	Determine the approximate length of the output string:
 */
	Length = FAO_Desc.dsc$w_length - 6 * 3 + 3 * 5 + Thread_Ptr->Job_Sentence.dsc$w_length +
		 User_Desc.dsc$w_length + Job_Desc.dsc$w_length + 23;
/*
 *	Allocate what is needed for the output string:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Format the string:
 */
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc,
			      Thread_Ptr->Job_Count, Thread_Ptr->Job_Count,
			      &Thread_Ptr->Job_Sentence, &User_Desc, &Job_Desc,
			      Thread_Ptr->Entry_Number, &Thread_Ptr->Time_Printed);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
	Mem_Free (Buf_Desc.dsc$a_pointer);
}

#define LINE_CNT 25

Build_Output_Page (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	auto   int Index, Line_Count;
	static struct dsc$descriptor Buf_Desc;
	static unsigned short Length;
	static char FAO_String[] = "\
largefont XC TM 30 sub moveto(Job output, Page !UL)cshow \
LM TM 60 sub moveto RM TM 60 sub lineto CP 2 setlinewidth stroke moveto \
mediumfont cr lf !#(AS) \
footer \
showpage ";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	static struct dsc$descriptor Str_Desc[LINE_CNT];
	static unsigned long FAO_List[LINE_CNT+2];
	extern unsigned long Str$Free1_DX(), Str$Copy_DX(), Sys$FAOL();
	extern int Build_Line_List();
	extern char *Mem_Alloc();

	Set_VMS_Descriptor (0, 0, &Buf_Desc);
/*
 *	Construct the strings for each line in the output:
 */
	Line_Count = Build_Line_List (&Thread_Ptr->Misc_Output_Head, LINE_CNT, Str_Desc);
	if (Line_Count == 0) {	/* none left */
		Sys_Status = Str$Free1_DX (&Thread_Ptr->Page_Desc);
		return;
	}
/*
 *	Allocate enough space for the final result:
 */
	Length = FAO_Desc.dsc$w_length - 3 - 6 + 5;
	for (Index = 0; Index < Line_Count; Index++)
		Length += Str_Desc[Index].dsc$w_length;
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Build the argument list for $FAOL, format final string:
 */
	FAO_List[0] = Thread_Ptr->Special_Page_Count;
	FAO_List[1] = Line_Count;
	for (Index = 0; Index < Line_Count; Index++)
		FAO_List[Index+2] = (unsigned long) &Str_Desc[Index];
	Sys_Status = Sys$FAOL (&FAO_Desc, &Length, &Buf_Desc, FAO_List);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
/*
 *	Copy completed string into page descriptor; free up memory
 *	no longer needed:
 */
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
	if (Buf_Desc.dsc$a_pointer != 0)
		Mem_Free (Buf_Desc.dsc$a_pointer);
	for (Index = 0; Index < Line_Count; Index++)
		Mem_Free (Str_Desc[Index].dsc$a_pointer);
}

/*
 *	Routine Build_Statistics_Page builds the page containing various
 *	task statistics and debug output.
 */

Build_Statistics_Page (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   struct dsc$descriptor *FAO_Desc;
	auto   unsigned long Sys_Status;
	auto   int Index, Line_Count, Arg_Index, Task_Len;
	auto   int Subtask_Count;
	static struct dsc$descriptor Buf_Desc, Temp_Desc;
	static unsigned short Length;
	static char FAO_String1[] = "\
largefont XC TM 30 sub moveto(Job statistics, Page !UL)cshow \
LM TM 60 sub moveto RM TM 60 sub lineto CP 2 setlinewidth stroke moveto \
/tabskip 162 def mediumfont cr lf \
cr lf(Number of pages printed (excluding statistics): !UL)show \
cr lf(Number of RMS gets: !UL)show \
cr lf(Number of QIO writes: !UL)show \
cr lf(Number of setup and reset library modules processed: !UL)show \
cr lf lf(Subtask Name)show tab(Elapsed Time)show tab(Control-D Wait)show \
!#(AS) \
cr lf !#(AS) \
footer \
showpage ";
	static char FAO_Stringn[] = "\
largefont XC TM 30 sub moveto(Job statistics, Page !UL)cshow \
LM TM 60 sub moveto RM TM 60 sub lineto CP 2 setlinewidth stroke moveto \
mediumfont cr lf \
!#(AS) \
footer \
showpage ";
	static $DESCRIPTOR (FAO_Desc1, FAO_String1);
	static $DESCRIPTOR (FAO_Descn, FAO_Stringn);
	static struct dsc$descriptor Str_Desc_List[LINE_CNT], Task_Desc_List[32];
	static unsigned long FAO_List[7+LINE_CNT+32];
	extern unsigned long Str$Free1_DX(), Str$Copy_DX(), Sys$FAOL();
	extern int Build_Line_List(), Build_Subtask_List();
	extern char *Mem_Alloc();

	Set_VMS_Descriptor (0, 0, &Buf_Desc);
/*
 *	Build task list string, compute string size:
 */
	FAO_List[0] = Thread_Ptr->Special_Page_Count;
	if (Thread_Ptr->Special_Page_Count <= 1) {
		FAO_List[1] = Thread_Ptr->Accounting_Data.smbmsg$l_pages_printed;
		FAO_List[2] = Thread_Ptr->Accounting_Data.smbmsg$l_rms_gets;
		FAO_List[3] = Thread_Ptr->Accounting_Data.smbmsg$l_qio_puts;
		FAO_List[4] = Thread_Ptr->Library_Module_Count;
		FAO_List[5] = Subtask_Count = Build_Subtask_List (Task_Desc_List, Thread_Ptr);
		Arg_Index = 6;
		Length = FAO_Desc1.dsc$w_length - 5 * 3 - 2 * 6 + 5 * 5;
		for (Index = 0; Index < Subtask_Count; Index++) {
			Copy_Desc_to_Fixed_Desc (&Task_Desc_List[Index], &Temp_Desc);
			Length += Temp_Desc.dsc$w_length;
			FAO_List[Arg_Index++] = (unsigned long) &Task_Desc_List[Index];
		}
		FAO_Desc = &FAO_Desc1;
		if ((Line_Count = LINE_CNT - Subtask_Count - 7) < 0)
			Line_Count = 0;
	} else {
		Subtask_Count = 0;
		Length = FAO_Descn.dsc$w_length - 3 - 6 + 5;
		FAO_Desc = &FAO_Descn;
		Arg_Index = 1;
		Line_Count = LINE_CNT;
	}
/*
 *	Construct the strings for each line in the output:
 */
	Line_Count = Build_Line_List (&Thread_Ptr->Stats_Output_Head, Line_Count, Str_Desc_List);
	if (Line_Count == 0 && Thread_Ptr->Special_Page_Count > 1) {	/* none left */
		Sys_Status = Str$Free1_DX (&Thread_Ptr->Page_Desc);
		return;
	}
/*
 *	Allocate enough space for the final result:
 */
	for (Index = 0; Index < Line_Count; Index++)
		Length += Str_Desc_List[Index].dsc$w_length;
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Build the remaining argument list for SYS$FAOL, format the
 *	final string:
 */
	FAO_List[Arg_Index++] = Line_Count;
	for (Index = 0; Index < Line_Count; Index++)
		FAO_List[Arg_Index++] = (unsigned long) &Str_Desc_List[Index];
	Sys_Status = Sys$FAOL (FAO_Desc, &Length, &Buf_Desc, FAO_List);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
/*
 *	Copy completed string into page descriptor; free up memory
 *	no longer needed:
 */
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
	if (Buf_Desc.dsc$a_pointer != 0)
		Mem_Free (Buf_Desc.dsc$a_pointer);
	for (Index = 0; Index < Line_Count; Index++)
		Mem_Free (Str_Desc_List[Index].dsc$a_pointer);
	for (Index = 0; Index < Subtask_Count; Index++)
		Mem_Free (Task_Desc_List[Index].dsc$a_pointer);
}
#undef LINE_CNT

int Build_Line_List (Output_Head, Max_Lines, Str_Desc)
struct Misc_Output **Output_Head;
int Max_Lines;
struct dsc$descriptor Str_Desc[];
{
	auto   struct Misc_Output *Misc_Ptr;
	auto   char *Ptr;
	auto   unsigned long Sys_Status;
	auto   int Line_Count, Str_Length, Length;
	static char Pre[] = "cr lf", Post[] = "show ";
	static struct dsc$descriptor Line_Desc;
	static char Temp_Str[4];
	extern char *Mem_Alloc(), *stringcpy();
	extern unsigned long Str$Free1_DX();
	extern int Format_String();

	for (Line_Count = 0; Line_Count < Max_Lines && (Misc_Ptr = *Output_Head) != 0; Line_Count++) {
		Copy_Desc_to_Fixed_Desc (&Misc_Ptr->Misc_Output_Desc, &Line_Desc);
		Str_Length = Format_String (&Line_Desc, Temp_Str, sizeof (Temp_Str)) +
			     sizeof (Pre) + sizeof (Post) - 2 + 1;
		Ptr = Mem_Alloc (Str_Length);
		Set_VMS_Descriptor (Ptr, Str_Length-1, &Str_Desc[Line_Count]);
		stringcpy (Ptr, Pre, Str_Length);
		Ptr = &Ptr[sizeof(Pre)-1];
		Str_Length -= sizeof (Pre) - 1;
		Length = Format_String (&Line_Desc, Ptr, Str_Length);
		Ptr = &Ptr[Length];
		Str_Length -= Length;
		stringcpy (Ptr, Post, Str_Length);
		Sys_Status = Str$Free1_DX (&Misc_Ptr->Misc_Output_Desc);
		*Output_Head = Misc_Ptr->Link;
		Mem_Free (Misc_Ptr);
	}
	return (Line_Count);
}

int Build_Subtask_List (Str_Desc, Thread_Ptr)
struct dsc$descriptor Str_Desc[];
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	auto   int Index, Subtask_Count, Subtask;
	static char *Task_Names[] = {	/* List must match TASK_V definitions */
		"Start",               "Alignment",
		"Form setup",          "Flag prefix",
		"Job flag",            "Job burst",
		"File flag",           "File burst",
		"Flag postfix",        "File prefix",
		"Page setup prefix",   "Page setup",
		"Page setup postfix",  "File setup",
		"Print file",          "File postfix",
		"After print",         "File trailer prefix",
		"Job output",          "Job errors",
		"File trailer",        "File trailer postfix",
		"Job reset",           "Job trailer prefix",
		"Job trailer",         "Job stats",
		"Job trailer postfix", "Finish"
	};
	static struct dsc$descriptor CtlD_Time_Desc, Elapsed_Time_Desc, Buf_Desc;
	static struct dsc$descriptor Task_Name_Desc;
	static unsigned short Length;
	static char FAO_String[] = "cr lf(!AS)show tab(!AS)show tab(!AS)show ";
	static char CtlD_Time_Buf[18], Elapsed_Time_Buf[18];
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO(), Sys$AscTim();
	extern char *Mem_Alloc();

	Subtask_Count = 0;
	for (Subtask = TASK_V_START; Subtask <= TASK_V_FINISH; Subtask++)
	if ((Thread_Ptr->Task_Completion_Status & (1 << Subtask)) != 0) {
/*
 *	First, search for a Control-D timer entry for the subtask:
 */
		Set_VMS_Descriptor (0, 0, &CtlD_Time_Desc);
		for (Index = 0; Index < Thread_Ptr->Timer_Count; Index++)
		if (Thread_Ptr->Timer[Index].Task_Type == Subtask) {
			Set_VMS_Descriptor (CtlD_Time_Buf, sizeof (CtlD_Time_Buf), &CtlD_Time_Desc);
			Sys_Status = Sys$AscTim (&Length, &CtlD_Time_Desc,
						 &Thread_Ptr->Timer[Index].Total_Time, 0);
			if ((Sys_Status & 0x01) == 0)
				CtlD_Time_Desc.dsc$w_length = 0;
			else
				Set_VMS_Descriptor (&CtlD_Time_Buf[5], Length-5, &CtlD_Time_Desc);
			break;
		}
/*
 *	Generate the task time completion value:
 */
		Set_VMS_Descriptor (Elapsed_Time_Buf, sizeof (Elapsed_Time_Buf), &Elapsed_Time_Desc);
		Sys_Status = Sys$AscTim (&Length, &Elapsed_Time_Desc, &Thread_Ptr->Subtask_Time[Subtask], 0);
		if ((Sys_Status & 0x01) == 0)
			Elapsed_Time_Desc.dsc$w_length = 0;
		else
			Set_VMS_Descriptor (&Elapsed_Time_Buf[5], Length-5, &Elapsed_Time_Desc);
/*
 *	Format the completed string and copy it to the output:
 */
		Set_VMS_Descriptor (Scratch_String, sizeof (Scratch_String), &Buf_Desc);
		Make_VMS_Descriptor (Task_Names[Subtask], &Task_Name_Desc);
		Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc, &Task_Name_Desc,
				      &Elapsed_Time_Desc, &CtlD_Time_Desc);
		if ((Sys_Status & 0x01) != 0) {
			Buf_Desc.dsc$w_length = Length;
			Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Str_Desc[Subtask_Count]);
			Sys_Status = Str$Copy_DX (&Str_Desc[Subtask_Count], &Buf_Desc);
			if ((Sys_Status & 0x01) != 0)
				Subtask_Count++;
			else
				Mem_Free (Str_Desc[Subtask_Count].dsc$a_pointer);
		}
	}
	return (Subtask_Count);
}

Build_Error_Page (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor PS_Desc, VMS_Desc, Msg_Desc, Buf_Desc;
	static unsigned short Length;
	static char PS_Error_String[] = "\
mediumfont cr lf lf(A PostScript error was detected by the LaserWriter for this job.)show \
cr lf lf(The PostScript error code is: !AS.)show \
cr lf(The command that generated the error is )show(\"!AS\".)pw \
cr lf lf(All data after the command generating the error was ignored.)show";
	static char VMS_Error_String[] = "\
mediumfont cr lf lf(This job terminated abnormally. The VMS status is)show cr lf !AS";
	static char FAO_String[] = "hugefont XC TM 30 sub moveto(Job Status)cshow !AS !AS footer showpage ";
	static $DESCRIPTOR (PS_Error_Desc, PS_Error_String);
	static $DESCRIPTOR (VMS_Error_Desc, VMS_Error_String);
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Str$Free1_DX(), Sys$FAO();
	extern char *Mem_Alloc();

	Set_VMS_Descriptor (0, 0, &PS_Desc);
	Set_VMS_Descriptor (0, 0, &VMS_Desc);
/*
 *	Format the PostScript error string:
 */
	if ((Thread_Ptr->Task_Status & TASK_M_USERERROR) != 0) {
		Length = PS_Error_Desc.dsc$w_length - 2 * 3 + Thread_Ptr->Error_Name.dsc$w_length +
			 Thread_Ptr->Offending_Command.dsc$w_length;
		Set_VMS_Descriptor (Mem_Alloc (Length), Length, &PS_Desc);
		Sys_Status = Sys$FAO (&PS_Error_Desc, &Length, &PS_Desc, &Thread_Ptr->Error_Name,
				      &Thread_Ptr->Offending_Command);
		PS_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	}
/*
 *	Format the VMS error string:
 */
	if (Thread_Ptr->N_Condition_Values > 0) {
		Set_Dynamic_VMS_Descriptor (&Msg_Desc);
		Get_VMS_Message (Thread_Ptr->Condition_Vector[0], Scratch_String, sizeof (Scratch_String));
		Make_VMS_Descriptor (Scratch_String, &Buf_Desc);
		Sys_Status = Str$Copy_DX (&Msg_Desc, &Buf_Desc);
		Format_Sentence (&Msg_Desc);
		Length = VMS_Error_Desc.dsc$w_length - 3 + Msg_Desc.dsc$w_length;
		Set_VMS_Descriptor (Mem_Alloc (Length), Length, &VMS_Desc);
		Sys_Status = Sys$FAO (&VMS_Error_Desc, &Length, &VMS_Desc, &Msg_Desc);
		VMS_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
		Sys_Status = Str$Free1_DX (&Msg_Desc);
	}
/*
 *	Determine the approximate length of the	output string:
 */
	Length = FAO_Desc.dsc$w_length - 2 * 3 + PS_Desc.dsc$w_length + VMS_Desc.dsc$w_length;
/*
 *	Allocate what is needed for the output string, then format it:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc, &PS_Desc, &VMS_Desc);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Page_Desc, &Buf_Desc);
/*
 *	Free up all dynamic string storage used:
 */
	Mem_Free (Buf_Desc.dsc$a_pointer);
	if (PS_Desc.dsc$a_pointer != 0)
		Mem_Free (PS_Desc.dsc$a_pointer);
	if (VMS_Desc.dsc$a_pointer != 0)
		Mem_Free (VMS_Desc.dsc$a_pointer);
}

Generate_Job_Params (Announce_Desc, User_Desc, Job_Desc, Thread_Ptr)
struct dsc$descriptor *Announce_Desc, *User_Desc, *Job_Desc;
struct Thread *Thread_Ptr;
{
	static struct dsc$descriptor Temp_Desc;
	static char User_Str[20], Job_Str[16], Announce_Str[80];
	static char System_Announce[80];
	extern int Translate_VMS_Logical_Name(), Format_String();

	if (Translate_VMS_Logical_Name ("PSM$ANNOUNCE", System_Announce, sizeof (System_Announce)) != 0) {
		Make_VMS_Descriptor (System_Announce, Announce_Desc);
		Trim_String (Announce_Desc);
	} else
		Make_VMS_Descriptor ("DEC VAX/VMS", Announce_Desc);
	Format_String (Announce_Desc, Announce_Str, sizeof (Announce_Str));
	Make_VMS_Descriptor (Announce_Str, Announce_Desc);

	Temp_Desc = Thread_Ptr->User_Name;
	Trim_String (&Temp_Desc);
	Format_String (&Temp_Desc, User_Str, sizeof (User_Str));
	Make_VMS_Descriptor (User_Str, User_Desc);

	Temp_Desc = Thread_Ptr->Job_Name;
	Trim_String (&Temp_Desc);
	Format_String (&Temp_Desc, Job_Str, sizeof (Job_Str));
	Make_VMS_Descriptor (Job_Str, Job_Desc);
}

Generate_File_Params (Node_Desc, Dev_Desc, Dir_Desc, Name_Desc, Ext_Desc,
		      Version_Desc, Thread_Ptr)
struct dsc$descriptor *Node_Desc, *Dev_Desc, *Dir_Desc;
struct dsc$descriptor *Name_Desc, *Ext_Desc, *Version_Desc;
struct Thread *Thread_Ptr;
{
	static char *Filespec_Desc[12];
	extern int Parse_VMS_Filespec();

	Copy_Desc_To_String (&Thread_Ptr->File_Specification, Scratch_String, sizeof (Scratch_String));
	Parse_VMS_Filespec (Scratch_String, Filespec_Desc);
	Set_File_Portion_Descriptor (&Filespec_Desc[0], Scratch_String, Node_Desc, Thread_Ptr);
	Set_File_Portion_Descriptor (&Filespec_Desc[2], Scratch_String, Dev_Desc, Thread_Ptr);
	Set_File_Portion_Descriptor (&Filespec_Desc[4], Scratch_String, Dir_Desc, Thread_Ptr);
	Set_File_Portion_Descriptor (&Filespec_Desc[6], Scratch_String, Name_Desc, Thread_Ptr);
	Set_File_Portion_Descriptor (&Filespec_Desc[8], Scratch_String, Ext_Desc, Thread_Ptr);
	Set_File_Portion_Descriptor (&Filespec_Desc[10], Scratch_String, Version_Desc, Thread_Ptr);
}

Set_File_Portion_Descriptor (Filespec_Desc, Base, Out_Desc, Thread_Ptr)
char *Filespec_Desc[2], *Base;
struct dsc$descriptor *Out_Desc;
struct Thread *Thread_Ptr;
{
	if (Filespec_Desc[0] == 0)
		Make_VMS_Descriptor ("", Out_Desc);
	else
		Set_VMS_Descriptor (&Thread_Ptr->File_Specification.dsc$a_pointer[Filespec_Desc[0]-Base],
				    Filespec_Desc[1] - Filespec_Desc[0] + 1, Out_Desc);
}

/*
 *	Routine Build_Job_Sentence builds the 'Job Sentence' for a
 *	particular stream:
 */

Build_Job_Sentence (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc, Temp_Desc;
	static unsigned short Length;
	static char FAO_String[] = "\
Job !AS (!UL) queued to !AS on !17%D by user !AS, UIC !%I, under account !AS \
at priority !UL, started on printer !AS on !17%D from queue !AS, copy !UL of !UL.";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO();
	extern char *Mem_Alloc();
/*
 *	Set up the descriptors; Determine the approximate length of the
 *	output string:
 */
	Temp_Desc = Thread_Ptr->User_Name;
	Trim_String (&Temp_Desc);
	Length = FAO_Desc.dsc$w_length - 13 * 3 + 2 * 23 + 33 + 4 * 5 + 2 +
		 Thread_Ptr->Job_Name.dsc$w_length + Thread_Ptr->Queue.dsc$w_length +
		 Temp_Desc.dsc$w_length + Thread_Ptr->Account_Name.dsc$w_length +
		 Thread_Ptr->Device_Name.dsc$w_length + Thread_Ptr->Executor_Queue.dsc$w_length;
/*
 *	Allocate what is needed for the output string:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Format the string:
 */
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc,
			      &Thread_Ptr->Job_Name, Thread_Ptr->Entry_Number,
			      &Thread_Ptr->Queue, &Thread_Ptr->Time_Queued,
			      &Temp_Desc, Thread_Ptr->Uic,
			      &Thread_Ptr->Account_Name, Thread_Ptr->Priority,
			      &Thread_Ptr->Device_Name, &Thread_Ptr->Time_Printed,
			      &Thread_Ptr->Executor_Queue, Thread_Ptr->Job_Count,
			      Thread_Ptr->Job_Copies);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->Job_Sentence, &Buf_Desc);
	Mem_Free (Buf_Desc.dsc$a_pointer);
	Format_Sentence (&Thread_Ptr->Job_Sentence);
}

/*
 *	Routine Build_File_Sentence builds the 'File Sentence' for a
 *	task:
 */

Build_File_Sentence (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor Buf_Desc;
	static struct dsc$descriptor Node_Desc, Dev_Desc, Dir_Desc;
	static struct dsc$descriptor Name_Desc, Ext_Desc, Version_Desc;
	static unsigned short Length;
	static char FAO_String[] = "File !AS !AS !AS !AS !AS !AS, copy !UL of !UL.";
	static $DESCRIPTOR (FAO_Desc, FAO_String);
	extern unsigned long Str$Copy_DX(), Sys$FAO();
	extern char *Mem_Alloc();
/*
 *	Parse file name into component parts:
 */
	Generate_File_Params (&Node_Desc, &Dev_Desc, &Dir_Desc, &Name_Desc,
			      &Ext_Desc, &Version_Desc, Thread_Ptr);
/*
 *	Determine the approximate length of the output string:
 */
	Length = FAO_Desc.dsc$w_length - 8 * 3 + 2 * 5 + 5 +
		 Thread_Ptr->File_Specification.dsc$w_length;
/*
 *	Allocate what is needed for the output string:
 */
	Set_VMS_Descriptor (Mem_Alloc (Length), Length, &Buf_Desc);
/*
 *	Format the string:
 */
	Sys_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc, &Node_Desc, &Dev_Desc,
			      &Dir_Desc, &Name_Desc, &Ext_Desc, &Version_Desc,
			      Thread_Ptr->File_Count, Thread_Ptr->File_Copies);
	Buf_Desc.dsc$w_length = ((Sys_Status & 0x01) != 0) ? Length : 0;
	Sys_Status = Str$Copy_DX (&Thread_Ptr->File_Sentence, &Buf_Desc);
	Mem_Free (Buf_Desc.dsc$a_pointer);
	Format_Sentence (&Thread_Ptr->File_Sentence);
}

/*
 *	Routine Build_Qualifier_Sentence generates the task qualifier
 *	string:
 */

Build_Qualifier_Sentence (Thread_Ptr)
struct Thread *Thread_Ptr;
{

}

/*
 *	Routine Format_Sentence generates a string that will print
 *	an arbitrarily long string with proper breaks when the end
 *	of line is reached. The input descriptor must be a dynamic
 *	string descriptor.
 */

Format_Sentence (Sentence_Desc)
struct dsc$descriptor *Sentence_Desc;
{
	auto   unsigned long Sys_Status;
	auto   int Length, Word_Length;
	auto   char *Ptr0, *Ptr;
	static struct dsc$descriptor Sen_Desc, Buf_Desc, Word_Desc;
	static char Temp_Word[4];
	extern char *Mem_Alloc(), *stringcpy();
	extern unsigned long Str$Copy_R();
	extern int Next_Word(), Format_String();

	Copy_Desc_to_Fixed_Desc (Sentence_Desc, &Sen_Desc);
/*
 *	Count the number of characters in the words of the sentence:
 */
	Length = 1;
	Set_VMS_Descriptor (Sen_Desc.dsc$a_pointer, 0, &Word_Desc);
	while (Next_Word (&Word_Desc, &Sen_Desc, ' ') != 0)
		Length += Format_String (&Word_Desc, Temp_Word, sizeof (Temp_Word)) + 3;
/*
 *	Allocate enough memory to store the whole partitioned sentence;
 *	then build it from the individual words:
 */
	Ptr0 = Ptr = Mem_Alloc (Length);
	Set_VMS_Descriptor (Sen_Desc.dsc$a_pointer, 0, &Word_Desc);
	while (Next_Word (&Word_Desc, &Sen_Desc, ' ') != 0) {
		Word_Length = Format_String (&Word_Desc, Ptr, Length);
		Length -= Word_Length;
		Ptr = &Ptr[Word_Length];
		stringcpy (Ptr, "pw ", Length);
		Length -= 3;
		Ptr = &Ptr[3];
	}
/*
 *	Copy the re-formatted string back into the original:
 */
	Length = Ptr - Ptr0;
	Sys_Status = Str$Copy_R (Sentence_Desc, &Length, Ptr0);
	Mem_Free (Ptr0);
}

/*
 *	Routine Next_Word returns the next "word" in a string
 *	(the word descriptor must be initialized to point to the
 *	first character in the string, length = 0):
 */

int Next_Word (Word_Desc, Str_Desc, Delimiter)
struct dsc$descriptor *Word_Desc, *Str_Desc;
char Delimiter;
{
	auto   char *Ptr0, *Ptr;
	auto   unsigned short Length;
	extern char *StrChr();

	Ptr0 = &Word_Desc->dsc$a_pointer[Word_Desc->dsc$w_length];
	Length = &Str_Desc->dsc$a_pointer[Str_Desc->dsc$w_length] - Ptr0;
	if (Ptr0 != Str_Desc->dsc$a_pointer && Length > 0 && *Ptr0 == Delimiter) {
		Ptr0++;
		Length--;
	}
	for (; Length > 0 && *Ptr0 == ' '; Ptr0++, Length--)
		;
	for (Ptr = Ptr0; Length > 0 && *Ptr != Delimiter; Ptr++, Length--)
		;
	if (Ptr == Ptr0 && Length == 0)
		return (0);
	Set_VMS_Descriptor (Ptr0, Ptr-Ptr0, Word_Desc);
	return (1);
}

/*
 *	Routine Format_String formats a string for output using the
 *	PostScript 'show' command. It puts on the enclosing parentheses,
 *	and substitutes the proper escape codes for special characters.
 *	It returns the length of the completed string, whether or not it
 *	fit into the output buffer.
 */

int Format_String (Input_Desc, Out_Ptr, Max_Size)
struct dsc$descriptor *Input_Desc;
char *Out_Ptr;
unsigned short Max_Size;
{
	auto   unsigned char c;
	auto   char *In_Ptr, *Ptr, *End_Ptr;
	auto   int Size;
	auto   unsigned short Count;

	Ptr = Out_Ptr;
	End_Ptr = &Ptr[Max_Size - 2];
	In_Ptr = Input_Desc->dsc$a_pointer;
	Size = 0;
	*Ptr++ = '(';
	for (Count = Input_Desc->dsc$w_length; Count > 0; Count--)
	if ((c = *In_Ptr++) == '(' || c == ')' || c == '\\' || c == '%') {
		Size += 2;
		if (Ptr <= End_Ptr - 2) {
			*Ptr++ = '\\';
			*Ptr++ = c;
		}
	} else if (c < 040 || c > 0176) {
		Size += 4;
		if (Ptr <= End_Ptr - 4) {
			*Ptr++ = '\\';
			*Ptr++ = ((c >> 6) & 07) + '0';
			*Ptr++ = ((c >> 3) & 07) + '0';
			*Ptr++ = (c & 07) + '0';
		}
	} else {
		Size++;
		if (Ptr < End_Ptr)
			*Ptr++ = c;
	}
	*Ptr++ = ')';
	*Ptr = '\0';
	return (Size + 2);
}

/*
 *	Routine Append_Stat appends a statistics message to the end
 *	of the current list:
 */

unsigned long Append_Stat (Thread_Ptr, FAO_Str, FAO_Arg)
struct Thread *Thread_Ptr;
char *FAO_Str;
unsigned long FAO_Arg;
{
	auto   unsigned long Sys_Status;
	static struct dsc$descriptor FAO_Desc, Msg_Desc;
	static unsigned short Length;
	extern unsigned long Sys$FAOL(), Append_Output_String();

	Make_VMS_Descriptor (FAO_Str, &FAO_Desc);
	Set_VMS_Descriptor (Scratch_String, sizeof (Scratch_String), &Msg_Desc);
	Sys_Status = Sys$FAOL (&FAO_Desc, &Length, &Msg_Desc, &FAO_Arg);
	if ((Sys_Status & 0x01) != 0) {
		Set_VMS_Descriptor (Scratch_String, Length, &Msg_Desc);
		Sys_Status = Append_Output_String (&Msg_Desc, &Thread_Ptr->Stats_Output_Head);
	}
	return (Sys_Status);
}

/*
 *	Routine 'Append_Output_String' appends a message string from the
 *	LaserWriter to the current list, which are printed after the
 *	job is completed:
 */

unsigned long Append_Output_String (Desc_Ptr, Output_Head)
struct dsc$descriptor *Desc_Ptr;
struct Misc_Output **Output_Head;
{
	auto   struct Misc_Output *Misc_Ptr;
	extern char *Mem_Alloc();
	extern unsigned long Str$Copy_DX();

	if (Desc_Ptr->dsc$w_length > 0) {
		for (Misc_Ptr = (struct Misc_Output *) Output_Head; Misc_Ptr->Link != 0; Misc_Ptr = Misc_Ptr->Link)
			;
		Misc_Ptr = Misc_Ptr->Link = (struct Misc_Output *) Mem_Alloc (sizeof (struct Misc_Output));
		Misc_Ptr->Link = 0;
		Set_Dynamic_VMS_Descriptor (&Misc_Ptr->Misc_Output_Desc);
		return (Str$Copy_DX (&Misc_Ptr->Misc_Output_Desc, Desc_Ptr));
	}
}
