/*
** Datei: DVISPLIN.C
** Autor: Ingo Eichenseher
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#include <string.h>
#include "dvi.h"
#include "dvisplin.h"
#include "dvidraw.h"

static real dotsize = 0;
static real arrow_width, arrow_length, arrow_t, arrow_l, arrow_d;
static line_startstyle = 0, line_endstyle = 0;

void setlinecap(int start, int end)
{
    line_startstyle = start;
    line_endstyle = end;
}

void setarrowsize(real width, real length)
{
    arrow_length = length;
    arrow_width = width;
}

void setarrowshape(real t, real l)
{
    arrow_t = t;
    arrow_l = l;
}

void setarrowline(real d)
{
    arrow_d = d;
}

void setdotsize(real diam)
{
    dotsize = diam/2.0;
}

void setlinewidth(real width)
{
    gstate.line_width = width;
}

static void arrow(coord *p1, coord *p2, int style)
{
    coord p[3], q[3], pa, pb;
    real x, y, n, px_m, py_m;
    gstate_t save;

    if (arrow_length<1 || arrow_width<1) return;

    x = (p2->x - p1->x);
    y = (p2->y - p1->y)/aspect_ratio;
    n = sqrt(x*x+y*y);
    if (n<0.05) return;
    x /= n;
    y /= n;

    p[0] = *p2;
    p[1].x = p[0].x - x*arrow_length-y*arrow_width;
    p[1].y = p[0].y - (y*arrow_length-x*arrow_width)*aspect_ratio;
    p[2].x = p[0].x - x*arrow_length+y*arrow_width;
    p[2].y = p[0].y - (y*arrow_length+x*arrow_width)*aspect_ratio;

    px_m = (p[0].x+p[1].x+p[2].x)/3;
    py_m = (p[0].y+p[1].y+p[2].y)/3;

    q[2].x = px_m*arrow_t + (p[1].x+p[0].x)*(1-arrow_t)/2;
    q[2].y = py_m*arrow_t + (p[1].y+p[0].y)*(1-arrow_t)/2;
    q[1].x = px_m*arrow_t + (p[2].x+p[0].x)*(1-arrow_t)/2;
    q[1].y = py_m*arrow_t + (p[2].y+p[0].y)*(1-arrow_t)/2;
    q[0].x = px_m*arrow_l + (p[1].x+p[2].x)*(1-arrow_l)/2;
    q[0].y = py_m*arrow_l + (p[1].y+p[2].y)*(1-arrow_l)/2;

    save = gstate;
    gstate.dash_len = 0;
    gstate.dash_color = 1;

    if (style!=2)
    {
	pa.x = (p[1].x+p[2].x)/8 + 0.75*q[0].x;
	pa.y = (p[1].y+p[2].y)/8 + 0.75*q[0].y;
	pb.x = (p[1].x+p[2].x)/2;
	pb.y = (p[1].y+p[2].y)/2;

	if (arrow_l>0)  { g_line(&pb,&pa); *p2 = pb; }
	else            { g_line(&pa,&pb); *p2 = pa; }

	x = (p2->x - p->x);
	y = (p2->y - p->y)/aspect_ratio;
	if (sqrt(x*x+y*y)>n) *p1 = *p2; 

	gstate.line_width = arrow_d;

	clip_init();    
	g_bezier(p,q+2,q+2,p+1,MAX_RECURSE);
	g_bezier(p+1,q,q,p+2,MAX_RECURSE);
	g_bezier(p+2,q+1,q+1,p,MAX_RECURSE);
	if (fill_with_pattern(style==1 ? 0x00:0xFF))
	    halt("Internal error filling arrow");
    }

    if (style==1)
	g_bezier(p+1,q,q,p+2,MAX_RECURSE);
    if (style>0)
    {
	g_bezier(p,q+2,q+2,p+1,MAX_RECURSE);
	g_bezier(p+2,q+1,q+1,p,MAX_RECURSE);
	g_join(p);
	g_join(p+1);
	g_join(p+2);
    }

    gstate = save;
}

static void startline(coord *p1, coord *p2)
{
    if (fill_mode) return;
    switch(line_startstyle)
    {
	case 1 : 
	    if (gstate.line_width>=3.0) g_join(p1);
	    break;
	case 2 : 
	case 3 :
	case 4 :
	    arrow(p2,p1,line_startstyle-2); 
	    break;
    }
}

static void endline(coord *p1, coord *p2)
{
    if (fill_mode) return;
    switch(line_endstyle)
    {
	case 1 : 
	    if (gstate.line_width>=3.0) g_join(p2);
	    break;
	case 2 :
	case 3 :
	case 4 :
	    arrow(p1,p2,line_endstyle-2); 
	    break;
    }
}

void g_dot(const coord *p)
{
    g_circle(p,dotsize);
}

void g_dpoly(const coord *p, const coord *edge, int n, int closed)
{
    if (n>2)
    {
	if (closed)
	{
	    int i;
	    for (i=0; i<n-1; i++)
	    {
		g_line(p+i,p+i+1);
		g_join(p+i);
	    }
	    g_line(p+n-1,p);
	    g_join(p+n-1);
	}
	else
	{
	    coord r[2];
	    r[0] = edge[0];
	    r[1] = *++p;
	    startline(r,r+1);
	    g_line(r,r+1);
	    for (n -= 3; n--; p++)
	    {
		g_line(p,p+1);
		g_join(p);
	    }
	    r[0] = p[0];
	    r[1] = edge[1];
	    endline(r,r+1);
	    g_line(r,r+1);
	    g_join(p);
	}
    }
    else if (n==2)
    {
	coord r[2];
	r[0] = edge[0];
	r[1] = edge[1];
	startline(r,r+1);
	endline(r,r+1);
	g_line(r,r+1);
    }
}

void g_draw(const coord *p, const coord *q, coord *r, int n, int closed)
{
    if (n-- > 0)
    {
	if (!closed) startline(r,r+1);
	g_bezier(r,r+1,r+2,p+1,MAX_RECURSE);
	p++, q++;
	g_join(p);
	while(n-- > 0)
	{
	    coord r;
	    r.x = 2*p[1].x-q[1].x;
	    r.y = 2*p[1].y-q[1].y;
	    g_bezier(p,q,&r,p+1,MAX_RECURSE);
	    p++, q++;
	    g_join(p);
	}
	if (!closed) endline(r+4,r+5);
	g_bezier(p,r+3,r+4,r+5,MAX_RECURSE);
	if (closed) g_join(r+5);
    }
    else
    {
	startline(r,r+3);
	endline(r+4,r+5);
	g_bezier(r,r+3,r+4,r+5,MAX_RECURSE);
    }
}

