/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* HANDLER.C --- Handlers for drawing canvases. */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <graphics.h>
#include <stdarg.h>
#include <string.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"
#include "resource.h"
#include "env.h"

enum { env_displacement_cursor = bit(cLINE) };
#define Cursor ((CURSOR)e->mouse.window)
#define Draw   ((WINDOW)e->mouse.window->parent)
#define Canvas ((CANVAS)e->mouse.window->parent->parent)
#define ContPos (&Canvas->cont_pos)

static LINE_THICKNESS thickness = ltTHIN;

void handle_thickness(LINE_THICKNESS *lt)
{
  thickness = *lt;
}

#define PICK_BUF_SIZE 20

int pick_size, pick_ptr;
static PICK_REC pick_buf[PICK_BUF_SIZE];

/* Predicates for objects elligible for picking. */
static int is_picked_p(OBJECT o)        { return o->picked_p; }
static int is_not_picked_p(OBJECT o)    { return !o->picked_p; }
static int is_text_object_p(OBJECT o)   { return bit(o->type) & TEXT_OBJECT_TYPES; }
static int is_thin_p(OBJECT o)          { return bit(o->type) & LINE_OBJECT_TYPES && o->line_thickness == ltTHIN; }
static int is_thick_p(OBJECT o)         { return bit(o->type) & LINE_OBJECT_TYPES && o->line_thickness == ltTHICK; }
static int is_not_origin_p(OBJECT o)    { return o->type != oORIGIN; }
#pragma argsused
static int is_modifiable_p(OBJECT o)    { return 1; }

/* Do a quick pick of one object.  When this
   is called, all objects are unpicked. */
LOCAL(void) do_quick_pick(EVENT e)
{
  PICK_REC quick_pick_buf[1];
  CANVAS c = Canvas;
  assert(n_picked(c, 1) == 0);
  if (do_point_pick(c, is_not_picked_p, quick_pick_buf, 1, e->mouse.x, e->mouse.y, settings->pick_radius) > 0) 
    pick_objects(c, &quick_pick_buf[0].o, 1, 1);
}

/* Do standard pick or unpick based on pick_p. */
LOCAL(void) do_pick(EVENT e, int pick_p)
{
  CANVAS c = Canvas;

  if (e->type == eBUTTON_PRESS) {
    if (e->mouse.button == env_start_button) {
      pick_size = do_point_pick(c, pick_p ? is_not_picked_p : is_picked_p, pick_buf, PICK_BUF_SIZE, e->mouse.x, e->mouse.y, settings->pick_radius);
      if (pick_size > 1)
	enable_button(&tool_selector_buttons[tsCYCLPICK]);
      else
	disable_button(&tool_selector_buttons[tsCYCLPICK]);
      pick_ptr = 0;
      if (pick_size > 0)
	pick_objects(c, &pick_buf[0].o, 1, pick_p);
    }
    else if (e->mouse.button == env_cont_button)
      env_set_cursor(bit(cBOX), e->mouse.x, e->mouse.y);
  }
  else {
    unsigned mask;
    int x0, y0, x1, y1;
    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);
    env_set_cursor(bit(cMOUSE));
    if (mask & bit(cBOX)) 
      do_rect_pick(c, pick_p, x0, y0, x1, y1);
  }
}

/* Check picked objects in a command for any off screen.
   If any, ask the user if they should be unpicked and
   handle the response.  A response of CANCEL means destroy
   the command completely.  Return the command or NULL if
   it's been destroyed. */
LOCAL(COMMAND) check_command(COMMAND cmd, char *op)
{
  CP_REC vp_sizes;
  int n, reply, offset;
  char *s, *them;

  if (settings->status & bit(sOFF_SCREEN_PICK_WARN)) {
    offset = binary_command_type_p(cmd->type) ? cmd->n_obj : 0;
    vp_sizes.x = edit_canvas->vp_x_size;
    vp_sizes.y = vp_y_size(edit_canvas);
    n = mark_outside(edit_canvas, cmd->obj + offset, cmd->n_obj, edit_canvas->vp_org, vp_sizes);
    if (n > 0) {
      if (n == 1) {
	s = "";
	them = "it";
      }
      else {
	s = "s";
	them = "them";
      }
      reply = message(bit(mYES) | bit(mNO) | bit(mCANCEL), root_window, 
		      "About to %s %d pick%s off screen.\n"
		      "Unpick %s now?", op, n, s, them);
      switch (reply) {
	case mYES:
	  n = unpick_outside(edit_canvas, cmd->obj + offset, cmd->n_obj);
	  if (n > 0)
	    cmd = cmd_realloc(cmd, n);
	  else {
	    free(cmd);
	    cmd = NULL;
	  }
	  break;

	case mCANCEL:
	  free(cmd);
	  cmd = NULL;
	  break;
      }
    }
  }
  return cmd;
}

LOCAL(int) do_copy(CANVAS c, int dx, int dy)
{
  int n;
  COMMAND cmd;

  if ((n = n_picked(c, 0)) == 0)
    return 0;

  cmd = cmd_alloc(cADD_AND_UNPICK, c, n);

  /* Get picks into command vector unpick slots. */
  get_picks(c, cmd->obj + n, is_not_origin_p);

  /* Check command for picks outside window. */
  if ((cmd = check_command(cmd, "copy")) == NULL)
    return 0;

  /* Copy picks into add slots. 
     Save displacement for repeat operations. */
  copy_objects(c, cmd->obj + n, cmd->obj, n, dx, dy);

  if (s_status_p(settings, sAUTO_DISPLACEMENT)) {
    c->copy_displacement.x = dx;
    c->copy_displacement.y = dy;
  }

  exec_command(cmd);
  edit();
  return 1;
}

void handle_copy(EVENT e)
{
  CANVAS c = Canvas;

  if (e->type == eBUTTON_PRESS) {
    if (e->mouse.button != env_start_button) 
      return;
    if (n_picked(c, 0) == 0) 
      do_quick_pick(e);
    env_set_cursor(env_displacement_cursor|env_graphic_ptr, e->mouse.x, e->mouse.y);
  }
  else {

    int x0, y0, x1, y1;
    unsigned mask;

    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);

    if (mask & env_displacement_cursor) {
      env_set_cursor(env_graphic_ptr);
      do_copy(c, sc2ccdx(c, x1, x0), sc2ccdy(c, y1, y0));
    }
  }
}

#pragma argsused
void handle_repeat_copy(ENV env)
{
  CANVAS c = env;
  do_copy(c, c->copy_displacement.x, c->copy_displacement.y);
}

LOCAL(int) do_move(CANVAS c, int dx, int dy)
{
  int n;
  COMMAND cmd;

  if ((n = n_picked(c, 1)) == 0)
    return 0;
  cmd = cmd_alloc(cMOVE, c, n, dx, dy);
  get_picks(c, cmd->obj, is_picked_p);
  if ((cmd = check_command(cmd, "move")) == NULL)
    return 0;
  
  exec_command(cmd);
  edit();
  return 1;
}

void handle_move(EVENT e)
{
  CANVAS c = Canvas;

  if (e->type == eBUTTON_PRESS) {
    if (e->mouse.button != env_start_button) 
      return;
    if (n_picked(c, 1) == 0) 
      do_quick_pick(e);
    env_set_cursor(env_displacement_cursor|env_graphic_ptr, e->mouse.x, e->mouse.y);
  }
  else {

    int x0, y0, x1, y1;
    unsigned mask;

    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);
    if (mask & env_displacement_cursor) {
      env_set_cursor(env_graphic_ptr);
      do_move(c, sc2ccdx(c, x1, x0), sc2ccdy(c, y1, y0));
    }
  }
}

/* Delete the current pick set in 
   response to tool button press. */
void handle_delete_button(CANVAS c)
{
  COMMAND cmd;
  int n;

  if ((n = n_picked(c, 1)) == 0)
    return;

  cmd = cmd_alloc(cDELETE, c, n);
  get_picks(c, cmd->obj, is_picked_p);
  if ((cmd = check_command(cmd, "delete")) == NULL)
    return;
  exec_command(cmd);
  edit();
}


/* Quick pick and delete an item in response to 
   a mouse click on the canvas.  The pick set
   must be empty to get here. */
void handle_delete(EVENT e)
{
  CANVAS c = Canvas;
  PICK_REC pick_buf[1];
  COMMAND cmd;

  assert(n_picked(c, 1) == 0);

  if (e->type != eBUTTON_PRESS || e->mouse.button != env_start_button) 
    return;
  /* Quick pick makes trouble here. */
  if (do_point_pick(c, is_not_picked_p, pick_buf, 1, 
		    e->mouse.x, e->mouse.y, settings->pick_radius) > 0) {
    cmd = cmd_alloc(cDELETE, c, 1);
    cmd->obj[0] = pick_buf->o;
    exec_command(cmd);
    edit();
  }
}

/* Change thickness of the current pick 
   set in response to tool button press. */
void handle_change_thickness_button(CANVAS c)
{
  int n;
  COMMAND cmd;

  n = n_picked(c, 0);
  cmd = cmd_alloc(thickness == ltTHIN ? cMAKE_THIN : cMAKE_THICK, c, n);
  n = get_picks(c, cmd->obj, thickness == ltTHIN ? is_thick_p : is_thin_p);
  if (n == 0) {
    free(cmd);
    return;
  }
  cmd = cmd_realloc(cmd, n);
  if ((cmd = check_command(cmd, "change line weight of")) == NULL)
    return;
  exec_command(cmd);
  edit();
}

/* Quick pick and change thickness of an item
   in response to a mouse click on the canvas. 
   The pick set must be empty to get here. */
void handle_change_thickness(EVENT e)
{
  CANVAS c = Canvas;
  PICK_REC pick_buf[1];
  COMMAND cmd;

  if (e->type != eBUTTON_PRESS || e->mouse.button != env_start_button)
    return;
  /* Quick pick makes trouble here. */
  if (do_point_pick(c, thickness == ltTHIN ? is_thick_p : is_thin_p, pick_buf, 1, 
		    e->mouse.x, e->mouse.y, settings->pick_radius) > 0) {
    cmd = cmd_alloc(thickness == ltTHIN ? cMAKE_THIN : cMAKE_THICK, c, 1);
    cmd->obj[0] = pick_buf->o;
    exec_command(cmd);
    edit();
  }
}

/* Line-vector slope tables. */
#include "lvslopes.c"

typedef struct {
  struct slope *tbl;
  int tbl_size;
  int slope_num, slope_den;
} SLOPE_CONSTRAINT_ENV_REC, *SLOPE_CONSTRAINT_ENV;

TYPEDEF_LOCAL(void) (*CONSTRAINT_ENV_SETTER)(SLOPE_CONSTRAINT_ENV);

LOCAL(void) set_line_constraint(SLOPE_CONSTRAINT_ENV env)
{
  env->tbl = line_slopes;
  env->tbl_size = N_LINE_SLOPES;
}

LOCAL(void) set_vector_constraint(SLOPE_CONSTRAINT_ENV env)
{
  env->tbl = vect_slopes;
  env->tbl_size = N_VECT_SLOPES;
}

/* Keep cursor at line or vector slope. */
static void constrain_cursor_slope(CURSOR c)
{
  int dx, dy, sx, sy, n, d, s, i, p0, p1;
  SLOPE_CONSTRAINT_ENV env = c->constraint_env;
  struct slope *tbl = env->tbl;

  dx = c->x - c->x0;
  dy = c->y - c->y0;

  if (dx == 0 && dy == 0)
    return;

  sx = 1;
  sy = -1;
  if (dx < 0) {
    dx = -dx;
    sx = -1;
  }
  if (dy < 0) {
    dy = -dy;
    sy = 1;
  }

  if (dx > dy) {
    n = dy;
    d = dx;
  }
  else {
    n = dx;
    d = dy;
  }

  p0 = 0;
  p1 = env->tbl_size;

  while (p0 + 1 < p1) {

    i = (p0 + p1) >> 1;

    s = tbl[i].snum * d - tbl[i].sden * n;
    if (s <= 0) {
      p0 = i;
      if (s == 0)
	break;
    }
    else if (s > 0)
      p1 = i;
  }

  if (dx > dy) {
    env->slope_den = tbl[p0].den * sx;
    c->y = c->y0 - dx * (env->slope_num = tbl[p0].num * sy)/ tbl[p0].den;
  }
  else {
    env->slope_num = tbl[p0].den * sy;
    c->x = c->x0 + dy * (env->slope_den = tbl[p0].num * sx) / tbl[p0].den;
  }
}

static SLOPE_CONSTRAINT_ENV_REC slope_constraint_env[1];

LOCAL(void) handle_linear(EVENT e,      
			  LINEAR_CONSTRUCTOR *make, 
			  CONSTRAINT_ENV_SETTER set_env,
			  unsigned mask)
{
  int x0, y0, x1, y1;
  index linear;
  COMMAND cmd;
  CP_REC p1;
  CANVAS c = Canvas;

  if (e->type == eBUTTON_PRESS) {
    if (e->mouse.button == env_start_button) {
      ContPos->x = sc2ccx(c, e->mouse.x);
      ContPos->y = sc2ccy(c, e->mouse.y);
    }
    else if (e->mouse.button != env_cont_button || !cont_pos_valid_p(ContPos)) 
      return;
    (*set_env)(slope_constraint_env);
    set_cursor_constraint(Cursor, constrain_cursor_slope, slope_constraint_env);
    env_set_cursor(mask|env_graphic_ptr, 
		   cc2scx(c, ContPos->x), cc2scy(c, ContPos->y));
  }
  else if (get_cursor(Cursor, &x1, &y1, &x0, &y0) & mask) {
    env_set_cursor(env_graphic_ptr);
    unset_cursor_constraint(Cursor);
    linear = (*make)(c, thickness, 
		     ContPos->x, ContPos->y,
		     sc2cc(c, abs(x0 == x1 ? y1 - y0 : x1 - x0)),
		     slope_constraint_env->slope_num, slope_constraint_env->slope_den, &p1);
    cmd = cmd_alloc(cADD_AND_MOVE_CONT_POS, c, 1, p1);
    cmd->obj[0] = linear;
    exec_command(cmd);
    edit();
  }
}

void handle_line(EVENT e)
{
  handle_linear(e, make_line, set_line_constraint, bit(cLINE));
}

void handle_vector(EVENT e)
{
  handle_linear(e, make_vector, set_vector_constraint, bit(cVECTOR));
}

/* General handler for circles. */
LOCAL(void) handle_circles(EVENT e, CIRCLE_CONSTRUCTOR *make)
{
  CANVAS c = Canvas;

  if (e->type == eBUTTON_PRESS) {

    unsigned cursor_mask;

    if (e->mouse.button == env_start_button) 
      cursor_mask = bit(cLINE)|bit(cCIRCLE_RAD)|env_graphic_ptr;
    else if (e->mouse.button == env_cont_button)
      cursor_mask = bit(cLINE)|bit(cCIRCLE_DIAM)|env_graphic_ptr;
    else return;

    env_set_cursor(cursor_mask, e->mouse.x, e->mouse.y);
  }
  else {

    COMMAND cmd;
    int x0, y0, x1, y1;
    unsigned mask;
    CP_REC p0, p1, center;
    CC dist, rad;

    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);
    env_set_cursor(env_graphic_ptr);

    if (mask & CIRCLE_CURSORS) {
      p0 = sp2cp(c, x0, y0);
      p1 = sp2cp(c, x1, y1);
      dist = isqrt(sum_sqr(p1.x - p0.x, p1.y - p0.y));
      if (mask & bit(cCIRCLE_RAD)) {
	center = p0;
	rad = dist;
      }
      else { /* cursor is diameter */
	center.x = ((long)p0.x + p1.x) / 2;
	center.y = ((long)p0.y + p1.y) / 2;
	rad = dist/2;
      }
      cmd = cmd_alloc(cADD_AND_MOVE_CONT_POS, c, 1, center);
      cmd->obj[0] = (*make)(c, thickness, center.x, center.y, rad);
      exec_command(cmd);
      edit();
    }
  }
}

LOCAL(void) handle_ovals(EVENT e, unsigned smask, unsigned cmask)
{
  CANVAS c = Canvas;
  unsigned cursor_mask;

  if (e->type == eBUTTON_PRESS) {
    if (e->mouse.button == env_start_button) 
      cursor_mask = smask;
    else if (e->mouse.button == env_cont_button)
      cursor_mask = cmask;
    else return;

    env_set_cursor(cursor_mask, e->mouse.x, e->mouse.y, cc2sc(c, env_oval_rad));
  }
  else {

    COMMAND cmd;
    int x0, y0, x1, y1;
    unsigned oval_mask;
    CP_REC p0, p1;

    cursor_mask = get_cursor(Cursor, &x1, &y1, &x0, &y0, &oval_mask);
    env_set_cursor(env_graphic_ptr);
    if (cursor_mask & OVAL_CURSORS) {
      p0 = sp2cp(c, x0, y0);
      p1 = sp2cp(c, x1, y1);
      cmd = cmd_alloc(strchr("\1\2\4\10", cursor_mask) ? 
			cADD_AND_MOVE_CONT_POS : cADD, 
		      c, 1, p1);
      diagonalize(&p0, &p1);
      cmd->obj[0] = make_oval(c, thickness,
			      p0.x, p0.y, 
			      p1.x - p0.x, p1.y - p0.y, 
			      env_oval_rad, 
			      som2com(oval_mask));
      exec_command(cmd);
      edit();
    }
  }
}

typedef enum { QUERY_ON_CONT, ALWAYS_QUERY, NEVER_QUERY } QUERY_P;

LOCAL(void) handle_boxes(EVENT e, BOX_CONSTRUCTOR *make, QUERY_P q)
{
  static int do_query_p;

  if (e->type == eBUTTON_PRESS) {
    do_query_p = (q == ALWAYS_QUERY) || 
		 (q == QUERY_ON_CONT && e->mouse.button == env_cont_button);
    env_set_cursor(bit(cBOX)|env_graphic_ptr, e->mouse.x, e->mouse.y);
  }
  else {

    COMMAND cmd;
    int x0, y0, x1, y1;
    unsigned mask;
    CP_REC p0, p1;
    char *str;
    CANVAS c = Canvas;

    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);
    env_set_cursor(env_graphic_ptr);
    if (mask & bit(cBOX)) {
      p0 = sp2cp(c, x0, y0);
      p1 = sp2cp(c, x1, y1);
      diagonalize(&p0, &p1);
      cmd = cmd_alloc(cADD, c, 1);
      str = NULL;
      if (!do_query_p || (str = get_text(NULL)) != 0) {
	cmd->obj[0] = (*make)(c, thickness,
			      p0.x, p0.y, 
			      p1.x - p0.x, p1.y - p0.y,
			      str, get_hjust(), get_vjust(), 
			      env_dash_len);
	exec_command(cmd);
	edit();
      }
    }
  }
}

LOCAL(void) handle_text_edit(EVENT e)
{
  PICK_REC text_pick[1];
  CANVAS c = Canvas;
  char *str;
  COMMAND cmd;
  OBJECT op;

  if (e->type == eBUTTON_PRESS)
    if (do_point_pick(c, is_text_object_p, text_pick, 1, e->mouse.x, e->mouse.y, settings->pick_radius) > 0) {
      edit_pick_object(c, text_pick->o, 1);
      op = dereference(c, text_pick->o);
      set_text(op);
      str = get_text(NULL);
      edit_pick_object(c, text_pick->o, 0);
      if (str != NULL) {
	cmd = cmd_alloc(cREPLACE, c, 1);
	cmd->obj[0] = make_text_edited_box(c, text_pick->o, str, get_hjust(), get_vjust(), op->picked_p);
	cmd->obj[1] = text_pick->o;
	exec_command(cmd);
	edit();
      }
    }
}

void handle_text(EVENT e)
{
  if (e->mouse.button == env_cont_button)
    handle_text_edit(e);
  else
    handle_boxes(e, make_text, ALWAYS_QUERY);
}

void handle_rect(EVENT e)
{
  handle_boxes(e, make_frame_box, QUERY_ON_CONT);
}

void handle_dashrect(EVENT e)
{
  handle_boxes(e, make_dash_box, QUERY_ON_CONT);
}

void handle_fillrect(EVENT e)
{
  handle_boxes(e, make_fill_box, NEVER_QUERY);
}

void handle_circle(EVENT e)
{
  handle_circles(e, make_circle);
}

void handle_fillcircle(EVENT e)
{
  handle_circles(e, make_fill_circle);
}

void handle_oval(EVENT e)
{
  handle_ovals(e, env_graphic_ptr|bit(cOVAL), env_graphic_ptr);
}

void handle_oval2(EVENT e)
{
  handle_ovals(e, env_graphic_ptr|bit(cOVAL2H), env_graphic_ptr|bit(cOVAL2V));
}

void handle_oval4(EVENT e)
{
  handle_ovals(e, env_graphic_ptr|bit(cOVAL4H), env_graphic_ptr|bit(cOVAL4V));
}

void handle_origin(EVENT e)
{
  if (e->type == eBUTTON_PRESS) {
    if (e->mouse.button != env_start_button)
      return;
    env_set_cursor(bit(cBOX)|bit(cMOUSE), e->mouse.x, e->mouse.y);
  }
  else {

    COMMAND cmd;
    int x0, y0, x1, y1;
    unsigned mask;
    CP_REC p0, p1;
    CANVAS c = Canvas;
    index o;

    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);
    env_set_cursor(env_graphic_ptr);
    if (mask & bit(cBOX)) {
      p0 = sp2cp(c, x0, y0);
      p1 = sp2cp(c, x1, y1);
      diagonalize(&p0, &p1);
      o = make_origin(c, p0.x, p0.y, p1.x - p0.x, p1.y - p0.y, 0);
      cmd = cmd_alloc(cADD, c, 1);
      cmd->obj[0] = o;
      exec_command(cmd);
      edit();
    }
  }
}

LOCAL(int) nearest_pwr_of_2(int x)
{
  int next, xx;

  if (x <= 0)
    return 1;

  for (xx = x; (next = xx ^ (xx & -xx)) != 0; xx = next) 
    /* skip */;

  return (x - xx < 2 * xx - x) ? xx : 2 * xx;
}

static void constrain_cursor_aspect(CURSOR c)
{
  int dx, dy, dxp, dyp;
  char sx, sy;

  dx = c->x - c->x0;
  dy = c->y - c->y0;

  sx = sy = 1;
  if (dx < 0) {
    dx = -dx;
    sx = -1;
  }
  if (dy < 0) {
    dy = -dy;
    sy = -1;
  }

  /* Compute height of cursor box based on width and vice-versa. */
  dyp = scale_int(dx, c->window.height, c->window.width);
  dxp = scale_int(dy, c->window.width, c->window.height);

  /* Make box higher if it's too flat and wider if too high. */
  if (dyp > dy) 
    dy = dyp;
  else
    dx = dxp;
    
  if (!s_status_p(settings, sZOOM_SMOOTH)) {

    /* Constrain size of cursor so viewport measured
       in canvas coords is pwr of 2 multiple of number 
       of window pixels.  This leads to perfect scaling. */

    CANVAS can = (CANVAS)c->window.parent->parent;
    int n_pixels, vp_width_cc, multiple, new_dx;

    n_pixels = can->draw.width;
    vp_width_cc = sc2cc(can, dx);
    multiple = nearest_pwr_of_2(scale_int(vp_width_cc, 1, n_pixels));
    vp_width_cc = multiple * n_pixels;
    new_dx = cc2sc(can, vp_width_cc);
    dy = scale_int(new_dx, c->window.height, c->window.width);
    dx = new_dx;
  }

  c->x = c->x0 + dx * sx;
  c->y = c->y0 + dy * sy;
}

#define MAX_ZOOMS 8
static struct zoom { CC size; CP_REC org; } zoom[MAX_ZOOMS];
int zoom_stk_ptr = 0;

void reset_zoom(void)
{
  if (zoom_stk_ptr == 0)
    return;
  set_canvas_viewport(edit_canvas, zoom[0].size, zoom[0].org.x, zoom[0].org.y, 1);
  zoom_stk_ptr = 0;
  disable_menu_entry(zoom_out_menu_entry);
}

void handle_zoom(EVENT e)
{
  /* Stack of active zooms. */
  CP_REC p0, p1;
  CC vp_size;

  struct zoom *z;
  CANVAS c = Canvas;

  if (e->type == eBUTTON_PRESS) {

    if (e->mouse.button == env_start_button) {

      /* Start cursing zoom box. */
      env_set_cursor(bit(cMOUSE)|bit(cBOX), e->mouse.x, e->mouse.y);
      set_cursor_constraint(Cursor, constrain_cursor_aspect, settings);

    }
    else if (e->mouse.button == env_cont_button && zoom_stk_ptr > 0) {

      /* Restore last viewport from stack. */
      z = &zoom[--zoom_stk_ptr];
      set_canvas_viewport(c, z->size, z->org.x, z->org.y, 1);
      if (zoom_stk_ptr == 0)
	disable_menu_entry(zoom_out_menu_entry);
    }

  }
  else {

    int x0, y0, x1, y1;
    unsigned mask;

    mask = get_cursor(Cursor, &x1, &y1, &x0, &y0);
    env_set_cursor(bit(cMOUSE));
    unset_cursor_constraint(Cursor);

    if (mask & bit(cBOX)) {

      /* Save current Canvas state on stack and set next. */
      z = &zoom[zoom_stk_ptr];
      if (zoom_stk_ptr < MAX_ZOOMS-1)
	++zoom_stk_ptr;
      enable_menu_entry(zoom_out_menu_entry);

      z->size = c->vp_x_size;
      z->org = c->vp_org;
      p0 = sp2cp(c, x0, y0);
      p1 = sp2cp(c, x1, y1);
      diagonalize(&p0, &p1);
      vp_size = p1.x - p0.x;
      if (!s_status_p(settings, sZOOM_SMOOTH)) {
	/* Ensure viewport size is pwr of 2 multiple 
	   of window size so scaling is precise. */
	int n_pixels = c->draw.width - 1;
	int multiple = nearest_pwr_of_2(scale_int(vp_size, 1, n_pixels));
	vp_size = multiple * n_pixels + 1;
	p0.x -= p0.x % multiple;
	p0.y -= p0.y % multiple;
      }
      set_canvas_viewport(c, vp_size, p0.x, p0.y, 1);
    }
  }
}

static int picking_p;

void handle_pick(EVENT e)
{
  do_pick(e, picking_p = 1);
}

void handle_unpick(EVENT e)
{
  do_pick(e, picking_p = 0);
}

void handle_cycle_pick(ENV env)
{
  if (pick_size > 1) {
    pick_objects(env, &pick_buf[pick_ptr].o, 1, 1 - picking_p);
    if (++pick_ptr == pick_size)
      pick_ptr = 0;
    pick_objects(env, &pick_buf[pick_ptr].o, 1, picking_p);
  }
}

/* Handle canvas buttons when modify tool is selected.
   This calls the canvas edit info and edit routines
   for the object.  The only type-consciousness it needs
   is to determine what cursor constraints should be
   set, because this module manages cursor constraints. */
void handle_modify(EVENT e)
{
  static index o;
  static EDIT_INFO_REC info[1];

  int x0, y0, x1, y1;
  unsigned release_cursor_mask;
  PICK_REC modify_pick[1];
  COMMAND cmd;
  CANVAS c;

  if (e->mouse.button != env_start_button)
    return;

  c = Canvas;

  if (e->type == eBUTTON_PRESS) {
    if (do_point_pick(c, is_modifiable_p, modify_pick, 1, e->mouse.x, e->mouse.y, settings->pick_radius) > 0) {
      o = modify_pick->o;
      /* Do the edit pick of the closest object. */
      edit_pick_object(c, o, 1);
      /* Retrieve edit information about the object. */
      edit_info(c, o, sp2cp(c, e->mouse.x, e->mouse.y), info);
      /* Set cursor constraint as required. */
      switch (info->cursor_mask) {

	case bit(cLINE):
	  set_line_constraint(slope_constraint_env);
	  set_cursor_constraint(Cursor, constrain_cursor_slope, slope_constraint_env);
	  break;

	case bit(cVECTOR):
	case bit(cVARIANT)|bit(cVECTOR):
	  set_vector_constraint(slope_constraint_env);
	  set_cursor_constraint(Cursor, constrain_cursor_slope, slope_constraint_env);
	  break;
      }
      /* Turn on the edit cursor. */
      env_set_cursor(info->cursor_mask|env_graphic_ptr, cc2scx(c, info->p0.x), cc2scy(c, info->p0.y), cc2sc(c, env_oval_rad));
    }
    else
      o = INULL;
  }
  else if (o != INULL) {
    /* Unmark the edited object. */
    edit_pick_object(c, o, 0);
    /* Get the cursor at button up. */
    release_cursor_mask = get_cursor(Cursor, &x1, &y1, &x0, &y0, &info->oval_mask);
    /* Undo the constraint, if any. */
    unset_cursor_constraint(Cursor);
    /* Go back to routine cursor. */
    env_set_cursor(bit(cMOUSE));
    /* If this isn't a bogus button release 
       (have to "and out" the crosshairs, if any). */
    if ((EDIT_INFO_CURSORS & release_cursor_mask) == info->cursor_mask) {
      info->p1 = sp2cp(c, x1, y1);
      cmd = cmd_alloc(cREPLACE, c, 1);
      info->slope_num = slope_constraint_env->slope_num;
      info->slope_den = slope_constraint_env->slope_den;
      cmd->obj[0] = make_edited_object(c, o, info);
      cmd->obj[1] = o;
      exec_command(cmd);
      edit();
    }
  }
}
