/* makempx.c - Make an MPX file
   Time-stamp: "97/07/26 21:59:51 mik"

   Copyright (C) 1997
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <io.h>
#include <direct.h>
#include <process.h>
#include "sys/stat.h"

#include "miktex.h"
#include "makempx.rc"

#if defined (_WIN32)
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#endif /* _WIN32 */

static int		verbose_p;
static int		debug_p;
static int		quiet_p;
static int		print_only_p;

#if 0
static const char *	destdir;
#endif

static char *		mpname;
static char *		mpxname;

/* _________________________________________________________________________

   USAGE / HELP
   _________________________________________________________________________ */


static void
show_version (void)

{
  printf ("This is makempx version %s (%s %s)\n",
	  VER_FILEVERSION_STR, VER_PRODUCTNAME_STR,
	  VER_PRODUCTVERSION_STR);
}

static void
usage (void)

{
  puts ("Usage: makempx [OPTION]... MPFILE MPXFILE");
  puts ("");
  puts ("This program makes an MPX file from an MP file.");
  puts ("");
  puts ("Options:");
  puts ("--debug, -d                         Print debugging information.");
  puts ("--dest-dir DIRECTORY, -D DIRECTORY  Specify the destination fmt directory.");
  puts ("--help, -h                          Print this help screen and exit.");
  puts ("--print-only, -n                    Print what commands would be executed.");
  puts ("--verbose, -v                       Print information on what is being done.");
  puts ("--version, -V                       Print the version number and exit.");
  puts ("");
  puts ("Email problem reports to cschenk@berlin.snafu.de");
}

/* _________________________________________________________________________

   GETTING COMMAND LINE OPTIONS.
   _________________________________________________________________________ */


static struct option	long_options[] =

{
  "debug",		no_argument,		0,	'd',
  "help",		no_argument,		0,	'h',
  "print-only",		no_argument,		0,	'n',
#if 0
  "dest-dir",		required_argument,	0,	'D',
#endif
  "quiet",		no_argument,		0,	'q',
  "verbose",		no_argument,		0,	'v',
  "version",		no_argument,		0,	'V',
  0,			no_argument,		0,	0,
};

static void
get_options (int	argc,
	     char **	argv)

{
  int option_index;
  int c;

#if 0
  destdir = 0;
#endif
  print_only_p = 0;
  quiet_p = 0;
#if defined (DEBUG)
  verbose_p = 1;
  debug_p = 1;
#else
  verbose_p = 0;
  debug_p = 0;
#endif

  while ((c=getopt_long (argc, argv, "d:hnqvV", long_options, &option_index))
	 != EOF)
    {
      switch (c)
	{
	case 'h':
	  usage ();
	  exit (0);
	  break;
	case 'n':
	  print_only_p = 1;
	  break;
#if 0
	case 'D':
	  destdir = strdup (optarg);
	  break;
#endif
	case 'd':
	  debug_p = 1;
	  break;
	case 'v':
	  verbose_p = 1;
	  break;
	case 'q':
	  quiet_p = 1;
	  break;
	case 'V':
	  show_version ();
	  exit (0);
	  break;
	default:
	  usage ();
	  exit (1);
	  break;
	}
    }

  if (argc - optind != 2)
    {
      usage ();
      exit (1);
    }

  mpname = argv[optind++];
  mpxname = argv[optind++];
}

/* _________________________________________________________________________

   INVOKING MPTOTEX.
   _________________________________________________________________________ */


static int
invoke_mptotex (const char *	mpname,
		const char *	texname)

{
  char command_line[3 * _MAX_PATH + 100];
  int rc;

  if (! find_executable ("mptotex.exe", command_line))
    {
      fprintf (stderr, "mptotex.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  sprintf (command_line + strlen (command_line), " %s >%s", mpname, texname);

  if (! quiet_p || print_only_p)
    puts (command_line);

  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = system (command_line);
    }

  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      exit (1);
    }

  if (rc > 0)
    {
      fprintf (stderr, "mptotex failed on %s! Return code: %d\n", mpname, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  return (rc);
}

/* _________________________________________________________________________

   INVOKING TEX.
   _________________________________________________________________________ */


static int
invoke_tex (const char * texname)

{
  char command_line[3 * _MAX_PATH + 100];
  int rc;

  if (! find_executable ("tex.exe", command_line))
    {
      fprintf (stderr, "tex.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  sprintf (command_line + strlen (command_line),
	   " \\batchmode\\input %s> nul", texname);

  if (! quiet_p || print_only_p)
    puts (command_line);

  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = system (command_line);
    }

  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      exit (1);
    }

  if (rc != 0)
    {
      fprintf (stderr, "TeX failed on %s! Return code: %d\n", texname, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  return (rc);
}

/* _________________________________________________________________________

   INVOKING DVITOMP.
   _________________________________________________________________________ */


static int
invoke_dvitomp (const char *dviname,
		const char *mpxname,
		const char *destdir)

{
  char command_line[3 * _MAX_PATH + 100];
  int rc;

  if (! find_executable ("dvitomp.exe", command_line))
    {
      fprintf (stderr, "dvitomp.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  sprintf (command_line + strlen (command_line),
	   " %s %s\\%s", dviname, destdir, mpxname);

  if (! quiet_p || print_only_p)
    puts (command_line);

  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = system (command_line);
    }

  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      exit (1);
    }

  if (rc > 0)
    {
      fprintf (stderr, "dvitomp failed on %s! Return code: %d\n", dviname, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  return (rc);
}

/* _________________________________________________________________________

   MAKING FILE NAMES.
   _________________________________________________________________________ */


static char *
make_tex_filename (const char *basename)

{
  char current_working_directory[_MAX_PATH];
  char result[_MAX_PATH];
  size_t i, len;

  if (_getcwd (current_working_directory, sizeof (current_working_directory))
      == 0)
    {
      perror ("_getcwd");
      exit (1);
    }

  _makepath (result, 0, current_working_directory, basename, "tex");
  len = strlen (result);
  for (i = 0; i < len; i++)
    {
      if (result[i] == '\\')
	result[i] = '/';
    }

  return (strdup (result));
}

/* _________________________________________________________________________

   MAIN.
   _________________________________________________________________________ */


static int
newer (const char *file1,
       const char *file2)

{
  struct stat x, y;
  if (stat (file1, &x) < 0)
    return (1);
  else if (stat (file2, &y) < 0)
    return (0);
  else if (x.st_mtime < y.st_mtime)
    return (1);
  else
    return (0);
}

int
main (int argc,
      char ** argv)

{
  char current_working_directory[_MAX_PATH];
  char *texname;

  /* Get command line options. */
  c4pargv[0] = argv[0];
  get_options (argc, argv);

  /* Return if MPXNAME exists and is newer than MPNAME. */
  if (newer (mpname, mpxname))
    {
      exit (0);
    }

  if (_getcwd (current_working_directory,
	       sizeof (current_working_directory))
      == 0)
    {
      perror ("_getcwd");
      exit (1);
    }

  if (verbose_p && ! print_only_p)
    printf ("Creating %s...\n", mpxname);

  /* Create temporary directory; make it current. */
  enter_temporary_directory ("mpx", print_only_p);

  texname = make_tex_filename ("mpx314");

  /* Invoke mptotex to make a TeX file. */
  invoke_mptotex (mpname, texname);

  /* Invoke TeX. */
  invoke_tex (texname);

  /* Invoke dvitomp. */
  invoke_dvitomp ("mpx314.dvi", mpxname, current_working_directory);

  /* Remove temporary directory. */
  leave_temporary_directory ();

  return (0);
}

/* makempx.c ends here */
