#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <endian.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>

#include "snd.h"
#include "conf.h"
#include "misc.h"

#undef SND_DEBUG

/* ---------------------------------------------------------------------- */

#define	SOUND_BUF_LEN	16384
#define	SOUND_VOL	0.5

static const char *sounddev = NULL;

static int eightbit = FALSE;
static int stereo = FALSE;
static int testmode = FALSE;
static int fullduplex = FALSE;

static int sound_fd = -1;

static int16_t snd_16_buffer[2 * SOUND_BUF_LEN];
static u_int8_t snd_8_buffer[2 * SOUND_BUF_LEN];

static char snd_error_str[1024];

/* ---------------------------------------------------------------------- */

static void snderr(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(snd_error_str, sizeof(snd_error_str), fmt, args);
	snd_error_str[sizeof(snd_error_str) - 1] = 0;
	va_end(args);

#ifdef SND_DEBUG
	fprintf(stderr, "%s\n", snd_error_str);
#endif
}

#ifdef SND_DEBUG
static void dprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vfprintf(stderr, fmt, args);
	va_end(args);
}
#endif

/* ---------------------------------------------------------------------- */

static int opensnd(int direction, int samplerate)
{
#ifdef SND_DEBUG
	audio_buf_info info;
	char *str;
#endif
	unsigned int sndparam, wanted;
	int fd;

#ifdef SND_DEBUG
	switch (direction) {
	case O_RDONLY:
		str = "reading";
		break;
	case O_WRONLY:
		str = "writing";
		break;
	case O_RDWR:
		str = "read/write";
		break;
	default:
		str = "???";
		break;
	}
	dprintf("Opening %s for %s .. ", sounddev, str);
#endif

	if ((fd = open(sounddev, direction)) < 0) {
		snderr("open: %s: %m", sounddev);
		return -1;
	}

#ifdef SND_DEBUG
	if (eightbit)
		str = "8 bit unsigned";
	else
		str = "16 bit signed, native byteorder";

	dprintf("ok\nSetting sample format (%s) .. ", str);
#endif

	if (eightbit)
		wanted = AFMT_U8;	/* 8 bit unsigned */
	else
		wanted = AFMT_S16_NE;	/* 16 bit signed, native byteorder */

	sndparam = wanted;
	if (ioctl(fd, SNDCTL_DSP_SETFMT, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_SETFMT: %m");
		goto error;
	}
	if (sndparam != wanted) {
		snderr("Requested sample format not supported");
		goto error;
	}

#ifdef SND_DEBUG
	dprintf("ok\nSetting %s audio .. ", stereo ? "stereo" : "mono");
#endif

	if (stereo)
		wanted = 1;		/* stereo */
	else
		wanted = 0;		/* mono */

	sndparam = wanted;
	if (ioctl(fd, SNDCTL_DSP_STEREO, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_STEREO: %m");
		goto error;
	}
	if (sndparam != wanted) {
		snderr("Cannot set %s audio", stereo ? "stereo" : "mono");
		goto error;
	}

#ifdef SND_DEBUG
	dprintf("ok\nSetting samplerate to %u .. ", samplerate);
#endif

	sndparam = samplerate;
	if (ioctl(fd, SNDCTL_DSP_SPEED, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_SPEED: %m");
		goto error;
	}
	if (sndparam != samplerate) {
		snderr("Warning: Sampling rate is %u, requested %u .. ",
			sndparam,
			samplerate);
	}

#ifdef SND_DEBUG
	dprintf("ok\n");
#endif

	/* Request a buffer size of 512 samples */
	if (eightbit)
		sndparam = 0x00000009;
	else
		sndparam = 0x0000000A;

	if (stereo)
		sndparam += 1;

	/* Unlimited amount of buffers for RX, four for TX */
	if (direction == O_RDONLY)
		sndparam |= 0x7FFF0000;
	else
		sndparam |= 0x00040000;

#ifdef SND_DEBUG
	dprintf("Setting fragment size (param = 0x%08X) .. ", sndparam);
#endif

	if (ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_SETFRAGMENT: %m");
		goto error;
	}

#ifdef SND_DEBUG
	dprintf("ok\n");

	if (direction == O_RDONLY) {
		if (ioctl(fd, SNDCTL_DSP_GETISPACE, &info) < 0) {
			dprintf("ioctl: SNDCTL_DSP_GETISPACE: %m");
		}
	} else {
		if (ioctl(fd, SNDCTL_DSP_GETOSPACE, &info) < 0) {
			dprintf("ioctl: SNDCTL_DSP_GETOSPACE: %m");
		}
	}

	dprintf("Audio buffer size: %d bytes, number of buffers: %d\n",
		info.fragsize, info.fragstotal);
#endif

#if 0
	/* non-blocking io if requested */
	if (nonblock && ioctl(fd, SNDCTL_DSP_NONBLOCK, 0) < 0) {
		snderr("ioctl: SNDCTL_DSP_NONBLOCK: %m");
		goto error;
	}
#endif

#ifdef SND_DEBUG
	dprintf("-- \n");
#endif

	return fd;

error:
	close(fd);
	return -1;
}

/* ---------------------------------------------------------------------- */

int sound_open_for_write(int srate)
{
	/* get current config */
	sounddev = conf_get_sounddev();
	eightbit = conf_get_eightbit();
	stereo = conf_get_stereo();
	fullduplex = conf_get_fulldup();
	testmode = conf_get_testmode();

	/* use stdout for sound output */
	if (testmode == TRUE) {
		sound_fd = 1;
		return 1;
	}

	if (fullduplex == TRUE) {
		if (sound_fd < 0)
			sound_fd = opensnd(O_RDWR, srate);

		return sound_fd;
	}

	sound_fd = opensnd(O_WRONLY, srate);

	return sound_fd;
}

int sound_open_for_read(int srate)
{
	/* get current config */
	sounddev = conf_get_sounddev();
	eightbit = conf_get_eightbit();
	stereo = conf_get_stereo();
	fullduplex = conf_get_fulldup();
	testmode = conf_get_testmode();

	/* use stdin for sound input */
	if (testmode == TRUE) {
		sound_fd = 0;
		return 0;
	}

	if (fullduplex == TRUE) {
		if (sound_fd < 0)
			sound_fd = opensnd(O_RDWR, srate);

		return sound_fd;
	}

	sound_fd = opensnd(O_RDONLY, srate);

	return sound_fd;
}

void sound_close(void)
{
	if (testmode == TRUE) {
		sound_fd = -1;
		return;
	}

	if (fullduplex == TRUE)
		return;

	/* never close stdin/out/err */
	if (sound_fd > 2) {
		if (ioctl(sound_fd, SNDCTL_DSP_SYNC, 0) < 0)
			snderr("ioctl: SNDCTL_DSP_SYNC: %m");
		close(sound_fd);
		sound_fd = -1;
	}
}

char *sound_error(void)
{
	return snd_error_str;
}

/* ---------------------------------------------------------------------- */

int sound_write(float *buf, int count)
{
	void *p;
	int i, j;

	if (sound_fd < 0) {
		snderr("sound_write: fd < 0");
		return -1;
	}

	if (count > SOUND_BUF_LEN) {
		snderr("sound_write: count > SOUND_BUF_LEN");
		return -1;
	}

	if (eightbit) {
		for (i = j = 0; i < count; i++) {
			snd_8_buffer[j++] = (buf[i] * 127.0 * SOUND_VOL) + 128.0;
			if (stereo)
				snd_8_buffer[j++] = 128;
		}

		count *= sizeof(u_int8_t);
		p = snd_8_buffer;
	} else {
		for (i = j = 0; i < count; i++) {
			snd_16_buffer[j++] = buf[i] * 32767.0 * SOUND_VOL;
			if (stereo)
				snd_16_buffer[j++] = 0;
		}

		count *= sizeof(int16_t);
		p = snd_16_buffer;
	}

	if (stereo)
		count *= 2;

	if ((i = write(sound_fd, p, count)) < 0)
		snderr("sound_write: write: %m");

	return i;
}

int sound_read(float *buf, int count)
{
	int len, i, j;

	if (sound_fd < 0) {
		snderr("sound_read: fd < 0");
		return -1;
	}

	if (count > SOUND_BUF_LEN) {
		snderr("sound_read: count > SOUND_BUF_LEN");
		return -1;
	}

	if (stereo)
		count *= 2;

	if (eightbit) {
		count *= sizeof(u_int8_t);

		if ((len = read(sound_fd, snd_8_buffer, count)) < 0)
			goto error;

		len /= sizeof(u_int8_t);

		if (stereo)
			len /= 2;

		for (i = j = 0; i < len; i++) {
			buf[i] = (snd_8_buffer[j++] - 128) / 128.0;
			if (stereo)
				j++;
		}
	} else {
		count *= sizeof(int16_t);

		if ((len = read(sound_fd, snd_16_buffer, count)) < 0)
			goto error;

		len /= sizeof(int16_t);

		if (stereo)
			len /= 2;

		for (i = j = 0; i < len; i++) {
			buf[i] = snd_16_buffer[j++] / 32768.0;
			if (stereo)
				j++;
		}
	}

	return len;

error:
	if (errno != EAGAIN)
		snderr("sound_read: read: %m");

	return len;
}

/* ---------------------------------------------------------------------- */

