#========================================================================
#
# PROGRAM FUNCTIONS
#

# Clean-up tmp and lock files
#
function cleanup() {
	if [ "$ERROR" != "" ]; then
	        echo -e "         
\n==============================================================================
WARNING!        WARNING!        WARNING!        WARNING!        WARNING!
==============================================================================
One or more errors occurred while slackpkg was running.                       
One or more packages most likely could not be installed, or your mirror
is having problems. It's a good idea recheck your mirror and run slackpkg
again.
=============================================================================="
	fi    
	echo
	if [ "$DELALL" = "1" ] && [ "$NAMEPKG" != "" ]; then
		rm $TEMP/$NAMEPKG &>/dev/null
	fi		
	( rm -f /var/lock/slackpkg.$$ && rm -rf $TMPDIR ) &>/dev/null
	exit
}
trap 'cleanup' 2 14 15 		# trap CTRL+C and kill

# Syntax Checking
#
function system_checkup() {
	# Checking if another instance of slackpkg is running
	#
	if [ "`ls /var/lock/slackpkg.* 2>/dev/null`" ] && \
		[ "$CMD" != "search" ]; then
		echo -e "\
\nAnother instance of slackpkg is running. If this isn't true, you can remove\n\
/var/lock/slackpkg.* files and run slackpkg again.\n"
		cleanup
	else        
		ls /var/lock/slackpkg.* &>/dev/null || \
			touch /var/lock/slackpkg.$$
	fi

	# Checking if the we can create TMPDIR
	#
	if [ "$TMPDIR" = "FAILED" ]; then
		echo -e "\
\nSome problem writing slackpkg's temporary dir in /tmp.\n\
Probably you don't had permissions to wrote on /tmp or it's full.\n\
Fix it and run slackpkg again.\n"
		cleanup
	fi

	# Checking if is the first time running slackpkg
	#                                               
	if ! [ -f ${CONF}/pkglist ] && [ "$CMD" != "update" ]; then
		echo -e "\
\nHey! Probably this is the first time you are running slackpkg.\n\
Before install|upgrade|reinstall anything, you need to uncomment\n\
one mirror in ${CONF}/mirrors and run:\n\n\
\t# slackpkg update\n\n\
You can see more information about slackpkg functions in slackpkg manpage."
		cleanup
	fi                                                      


	# Checking if /etc/slackpkg/mirrors are in correct syntax.
	#                                                         
	if [ "$SOURCE" = "" ]; then
		echo -e "\
\nYou don't have any mirror selected in ${CONF}/mirrors\n\
Please, edit that file and uncomment one mirror. Slackpkg\n\
only works with ONE mirror selected.\n"
		cleanup
	else
		COUNT=`echo $SOURCE | wc -w | tr -d " "`
		if [ "$COUNT" != "1" ]; then
			echo -e "\n\
Slackpkg only works with ONE mirror selected. Please, edit your\n\
${CONF}/mirrors and comment one or more lines. Two or more\n\
mirrors uncommented isn't valid syntax.\n"
			cleanup
		fi
	fi

	# It will check if the mirror selected are ftp.slackware.com
	# if are "ftp.slackware.com" tell to the user to choice other
	#
	if echo ${SOURCE} | grep "^ftp://ftp.slackware.com" &>/dev/null ; then
		echo -e "\n\
Please, use one of the mirrors.\n\
ftp.slackware.com should be preserved so the\n\
mirrors can be kept up-to-date.\n"
		cleanup
	fi

	# Command line syntax checking 
	#
	if [ "$#" = "0" ]; then
		usage
	fi

	if [ "$#" = "1" ] && [ "$CMD" != "update" ] && \
		[ "$CMD" != "upgrade-all" ] && [ "$CMD" != "install-new" ] && \
		[ "$CMD" != "blacklist" ] && [ "$CMD" != "clean-system" ]; then
		usage
	fi
	
	# Checking if the user have the permissions to install/upgrade/update
	#                                                                    
	if [ "`id -u`" != "0" ] && [ "$CMD" != "search" ] && [ "$CMD" != "info" ]; then
		echo -e "\n\
Only root can install, upgrade or remove packages.\n\
Please, log as root or call your system administrator.\n"
		cleanup
	fi          

	# Check if the mirror are local (cdrom or file)
	#
	MEDIA=`echo ${SOURCE} | cut -f1 -d:`
	if [ "$MEDIA" = "cdrom" ] || [ "$MEDIA" = "file" ]; then
		SOURCE=/`echo ${SOURCE} | cut -f3- -d/`
		LOCAL=1
	fi

	# Check if we have md5sum in the PATH. Without md5sum, disables
	# md5sum checks
	#
	if ! [ `which md5sum 2>/dev/null` ]; then
		CHECKPKG=0
	elif ! [ -f ${CONF}/CHECKSUMS.md5 ] && \
		[ "$CMD" != "update" ] && \
		[ "$CHECKPKG" = "1" ]; then
		echo -e "\n\
No CHECKSUMS.md5 found! Please, disable md5sums checking\n\
on your ${CONF}/slackpkg.conf or run slackpkg update\n\
to download a new CHECKSUMS.md5 file.\n"
		cleanup
	fi

	# Check if awk is installed
	#
	if ! [ "$(which awk 2>/dev/null)" ]; then
		echo -e "\n\
No awk found! Please, install awk before you run slackpkg.\n\
Without awk, slackpkg doesn't work in any way.\n"
		cleanup
	fi

	# Check if gpg is enabled but no GPG command are found.
	#
	if ! [ "$(which gpg 2>/dev/null)" ] && [ "${NOGPG}" = "0" ]; then
		NOGPG=1
		echo -e "\n\
No gpg found!!! Please, disable GPG in ${CONF}/slackpkg.conf or install\n\
the gnupg package.\n\n\
To disable GPG, edit slackpkg.conf and add one line with NOGPG=1.\n\
You can see an example in slackpkg.conf.new.\n "
		sleep 5
	fi 

	# Check if the Slackware GPG key are found in the system
	#                                                       
	GPGFIRSTTIME="`gpg --list-keys \"$SLACKKEY\" 2>/dev/null \
			| grep -c \"$SLACKKEY\"`"
	if [ "$GPGFIRSTTIME" = "0" ] && [ "$CMD" != "search" ] && [ "$CMD" != "info" ] && \
			[ "$CMD" != "update" ] && [ "$NOGPG" != "1" ]; then
		echo -e "\n\
You need the GPG key of $SLACKKEY.\n\
To download and install that key, run:\n\n\
\t# slackpkg update gpg\n\n\
You can disable GPG checking, too. But it isn't a good idea.\n\
To disable GPG, edit slackpkg.conf and add one line with NOGPG=1.\n\
You can see an example in slackpkg.conf.new.\n"
		cleanup
	fi
	echo 
}

# Got the name of a package, without version-arch-release data
#
function cutpkg() {
	echo ${1/.tgz/} | awk -F- 'OFS="-" { 
				if ( NF > 3 ) { 
					NF=NF-3
					print $0 
				} else {
					print $0
				}
			}'
}

# Show the slackpkg usage
#
function usage() {
	echo -e "\
slackpkg - version $VERSION\n\
\nUsage: \tslackpkg update [gpg]\t\tdownload and update files and 
\t\t\t\t\tpackage indexes
\tslackpkg install package\tdownload and install packages 
\tslackpkg upgrade package\tdownload and upgrade packages
\tslackpkg reinstall package\tsame as install, but for packages 
\t\t\t\t\talready installed
\tslackpkg remove package\t\tremove installed packages
\tslackpkg clean-system\t\tremove all packages which are not 
\t\t\t\t\tpresent in slackware distribution.
\t\t\t\t\tGood to keep the house in order
\tslackpkg upgrade-all\t\tsync all packages installed in your 
\t\t\t\t\tmachine with the selected mirror. This
\t\t\t\t\tis the "true" way to upgrade all your 
\t\t\t\t\tmachine.
\tslackpkg install-new\t\tinstall packages which are added in
\t\t\t\t\tslackware distribution. 
\t\t\t\t\tRun this if you are upgrading to another
\t\t\t\t\tslackware version or using "current".
\tslackpkg blacklist\t\tBlacklist a package. Blacklisted
\t\t\t\t\tpackages cannot be upgraded, installed
\t\t\t\t\tor reinstalled by slackpkg
\tslackpkg download\t\tJust download (do not install) a package
\tslackpkg info package\t\tShow package information 
\t\t\t\t\t(works with only ONE package)
\tslackpkg search file\t\tSearch for a specific file in the
\t\t\t\t\tentire package collection
\nYou can see more information about slackpkg usage and some examples
in slackpkg's manpage. You can use partial package names (such as x11
instead x11-devel, x11-docs, etc), or even slackware series
(such as "n","ap","xap",etc) when searching for packages.
"
	cleanup
}

# Verify if the package was corrupted by checking md5sum
#
function checkpkg() {
	MD5ORIGINAL=`grep "${NAMEPKG}$" ${CONF}/CHECKSUMS.md5| cut -f1 -d \ `
	MD5DOWNLOAD=`md5sum ${TEMP}/${1} | cut -f1 -d \ `
	if [ "$MD5ORIGINAL" = "$MD5DOWNLOAD" ]; then
		echo 1 
	else
		echo 0 
	fi
}

function checkgpg() {
	gpg --verify $TEMP/${1}.asc $TEMP/$1 2>/dev/null && echo "1" || echo "0"
}

function check_blacklist_priority() {
	# That rule will check if the package are in 
	# blacklist. And later select the high priority pack
	TPKGNAME=`echo $PKGTEST | awk -F/ '{ print $NF }'`
	TDIRNAME=`dirname $PKGTEST | cut -d. -f2- `
	TBASENAME=`cutpkg $TPKGNAME`
	if grep -qx "\(${TBASENAME}\|${TDIRNAME}\)" ${CONF}/blacklist ; then 
		PKGFOUND=0
	else
		if [ "`echo $TDIRNAME | grep -e \"^/${FIRST}/\"`" != "" ] && [ $PKGFOUND -le 4 ]; then
			PKGNAME="$TPKGNAME"
			DIRNAME="$TDIRNAME"
			TBASENAME="$TBASENAME"
			PKGFOUND="4"
		elif [ "`echo $TDIRNAME | grep -e \"^/${SECOND}/\"`" != "" ] && [ $PKGFOUND -le 3 ]; then
			PKGNAME="$TPKGNAME"
			DIRNAME="$TDIRNAME"
			TBASENAME="$TBASENAME"
			PKGFOUND="3"
		elif [ "`echo $TDIRNAME | grep -e \"^/${THIRD}/\"`" != "" ] && [ $PKGFOUND -le 2 ]; then
			PKGNAME="$TPKGNAME"
			DIRNAME="$TDIRNAME"
			TBASENAME="$TBASENAME"
			PKGFOUND="2"
		elif [ "`echo $TDIRNAME | grep -e \"^/${FOURTH}/\"`" != "" ] && [ $PKGFOUND -le 1 ]; then
			PKGNAME="$TPKGNAME"
			DIRNAME="$TDIRNAME"
			TBASENAME="$TBASENAME"
			PKGFOUND="1"
		fi
	fi
}

# Function to make install/reinstall/upgrade lists
#
function makelist() {
	ls -1 /var/log/packages > $TMPDIR/tmplist
	case "$CMD" in
		install|reinstall|upgrade|download|blacklist)
			echo -n "Looking for $(echo $1 | tr -d '\\') in package list. Please, wait... "
			for i in `egrep -- /${1} ${CONF}/pkglist | \
				awk -F/ '{ print $NF }'`; do
				unset TEST
				BASENAME=`cutpkg $i`

				# That lines above choose de package with the 
				# higher priority
				#
				TEST=`grep -e "^\./[a-z]*\(\|/.*\)/${BASENAME}-[^-]\+-\(noarch\|${ARCH}\)" ${CONF}/pkglist` 
				PKGFOUND=0
				if [ -n "$TEST" ]; then
					for PKGTEST in $TEST ; do
						check_blacklist_priority
					done
				fi

				if [ "$PKGFOUND" != "0" ]; then
					PKGFOUND=`grep -e "^${BASENAME}-[^-]\+-\(noarch\|${ARCH}\)" $TMPDIR/tmplist`.tgz
					# Based in PKGFOUND info, put the package in
					# the right list.
					#
					if [ "$PKGFOUND" = "$PKGNAME" ]; then
						REINSTALLPKG="$REINSTALLPKG $PKGNAME"
					elif [ "$PKGFOUND" = ".tgz" ]; then
						INSTALLPKG="$INSTALLPKG $PKGNAME"
					else
						UPGRADEPKG="$UPGRADEPKG $PKGNAME"
					fi
				fi 
			done
			REINSTALLPKG=`echo -e $REINSTALLPKG | \
							tr \  "\n" |sort|uniq`
			INSTALLPKG=`echo -e $INSTALLPKG | tr \  "\n" |sort|uniq`
			UPGRADEPKG=`echo -e $UPGRADEPKG | tr \  "\n" |sort|uniq`
			;;
		remove)
			echo -n "Looking for packages to remove. Please, wait... "
			REMOVEPKG=`ls -1 /var/log/packages | \
                            egrep -i -- "^${PATTERN}.*-[^-]+-(${ARCH}|noarch)-"`
		;;
		clean-system)
			echo -n "Looking for packages to remove. Please, wait... "
	                for i in `cat $TMPDIR/tmplist`; do
				PKGNAME="`cutpkg $i | \
					sed -e 's/\+/\\\+/g' -e 's/\./\\\./g'`"
				if [ `egrep -c "${PKGNAME}-[^-]+-(${ARCH}|noarch)-" $CONF/pkglist` = "0" ] && [ `grep -cx "${PKGNAME}" $CONF/blacklist` = 0 ]; then
					REMOVEPKG="$REMOVEPKG $i"
                        	fi
                	done
		;;
		upgrade-all)
			echo -n "Looking for packages to upgrade. Please, wait... "
			for i in `cat $TMPDIR/tmplist`; do
				unset TEST
				PKGNAME=`cutpkg $i`
				TEST=`grep -e "^\./[a-z]*\(\|/.*\)/${PKGNAME}-[^-]\+-\(noarch\|${ARCH}\)" ${CONF}/pkglist` 
				PKGFOUND=0
				if [ -n "$TEST" ]; then
					for PKGTEST in $TEST ; do
						check_blacklist_priority
					done
				fi
				if [ "${PKGNAME/.tgz/}" != "$i" ] && [ -n "$TEST" ] && [ "$PKGFOUND" != "0" ]; then 
					UPGRADEPKG="$UPGRADEPKG $PKGNAME"
				fi
			done
			UPGRADEPKG=`echo -e $UPGRADEPKG | tr \  "\n"|sort|uniq`
		;;
		install-new)
			echo -n "Looking for NEW packages to install. Please, wait... "
			for i in `grep -v "testing" $CONF/ChangeLog.txt | \
				  awk -F: '/[a-z].*\/([a-z_]+)-.*:.*(Added|Split|Renamed).*/ \
				  { print $1 }'` \
				  a/acl a/attr ap/dmapi ap/xfsdump l/mm ; do    
				unset INSNAME
				PKGNAME=`echo $i | awk -F/ '{ print $NF }'`
				DIRNAME=`dirname $i`
				case $DIRNAME in
					a|ap|d|e|f|k|kde|kdei|l|n|t|tcl|x|xap|y)
						DIRNAME="/slackware/${DIRNAME}"
						;;
					*)
						DIRNAME="/${DIRNAME}"
						;;
				esac
				BASENAME=`cutpkg $PKGNAME`

				INSNAME=`ls -1 /var/log/packages | grep -e "^${BASENAME}-[^-]\+-\(${ARCH}\|noarch\)-"`
				if [ "${INSNAME}" = "" ] && \
				   [ `grep -cx "\(${BASENAME}\|${DIRNAME}\)" $CONF/blacklist` = 0 ]; then
					NAMEPKG=`basename $(grep -e "${DIRNAME}/${BASENAME}-[^-]\+-\(${ARCH}\|noarch\)-" $CONF/pkglist) 2>/dev/null`
					if [ "$NAMEPKG" != "" ] && \
					   [ `echo $INSTALLPKG | grep -c "$NAMEPKG"` = 0 ]; then
						INSTALLPKG="$INSTALLPKG $NAMEPKG"
					fi
				fi
			done
		;;
	esac
	echo -e "DONE\n"
}

# Function to count total of packages
#
function countpkg() {
	COUNTPKG=`echo -e "$1" | wc -w`
	if [ "$COUNTPKG" != "0" ]; then
		echo -e "Total of package(s): $COUNTPKG\n"
	fi
}

# Show the lists and asks if the user want to proceed with that action
# Return accepted list in $SHOWLIST
#
function showlist() {
	for i in $1; do echo $i; done | more 
	echo
	countpkg "$1"
	echo -e "Do you wish to $2 selected packages (Y/n)? \c"; read ANSWER
	if [ "$ANSWER" = "N" -o "$ANSWER" = "n" ]; then
		cleanup
	else
		SHOWLIST="$1"
		continue
	fi
}

function getfile() {
        if [ "$LOCAL" = "1" ]; then
                echo -e "\t\t\tCopying $1..."
                cp ${SOURCE}$1 $2 2>/dev/null
        else
                echo -e "\t\t\tDownloading $1..."
                wget $WGETFLAGS ${SOURCE}$1 -O $2
        fi
}                                                       

# Function to download the correct package and many "checks"
#
function getpkg() {
	ISOK="1"
	NAMEPKG=`grep -m 1 -- "/$1$" ${CONF}/pkglist`
	
	if [ "`ls ${TEMP}/$1 2>/dev/null`" = "" ]; then
		echo -e "\nPackage: $1"
		# Check if the mirror are local, if is local, copy files 
		# to $TEMP else, download packages from remote host and 
		# put then in $TEMP
		#
		if [ "$LOCAL" = "1" ]; then 
                	echo -e "\tCopying $1..."
			cp ${SOURCE}${NAMEPKG} $TEMP
			if [ "$NOGPG" != "1" ]; then
				cp ${SOURCE}${NAMEPKG}.asc $TEMP
			fi
		else
                	echo -e "\tDownloading $1..."
			wget $WGETFLAGS -P $TEMP -nd ${SOURCE}${NAMEPKG}
			if [ "$NOGPG" != "1" ]; then
				wget $WGETFLAGS -P $TEMP -nd ${SOURCE}${NAMEPKG}.asc
			fi
		fi
	else
		echo -e "\tPackage $1 is already in cache, not downloading" 
	fi

	# If MD5SUM checks are enabled in slackpkg.conf, check the
	# packages md5sum to detect if they are corrupt or not
	#
	if [ "$CHECKPKG" = "1" ]; then
		ISOK=`checkpkg $1`
		[ "$ISOK" = "0" ] && ERROR="md5sum"
	fi

	# Check the package against its .asc. If you don't like this
	# disable GPG checking in /etc/slackpkg/slackpkg.conf
	#
	if [ "$NOGPG" != "1" ]; then
		ISOK=`checkgpg $1`
		[ "$ISOK" = "0" ] && ERROR="gpg"
	fi

	if [ "$ISOK" = "1" ]; then
		case $2 in
			installpkg)
				echo -e "\tInstalling ${1/.tgz/}..."
			;;
			upgradepkg)
				echo -e "\tUpgrading ${1/.tgz/}..."
			;;
			*)
				echo -e "\c"
			;;
		esac	
		( cd $TEMP && $2 $1 ) 
	else
		rm $TEMP/$1 2>/dev/null
		echo -e "\tERROR - Package not installed! $ERROR error!" 
	fi

	# If DELALL is checked, all downloaded files will be erased
	# after installed/upgraded/reinstalled
	#
	if [ "$DELALL" = "1" ]; then
		rm $TEMP/$1 $TEMP/${1}.asc 2>/dev/null
	fi		
}

# Main logic to download and format package list, md5 etc.
#
function updatefilelists()
{
	if ! [ -e $CONF/ChangeLog.txt ]; then
		touch $CONF/ChangeLog.txt
	fi

	echo -e "\tDownloading..."
	#
	# Download ChangeLog.txt first of all and test if it's equal
	# or different from our already existent ChangeLog.txt 
	#
	getfile ChangeLog.txt $TMPDIR/ChangeLog.txt
	if diff --brief $CONF/ChangeLog.txt $TMPDIR/ChangeLog.txt ; then
		echo -e "\
\n\t\tNo changes in ChangeLog.txt between your last update and now.\n\
\t\tDo you really want to download all other files (y/N)? \c"
		read ANSWER
		if [ "$ANSWER" != "Y" ] && [ "$ANSWER" != "y" ]; then
			cleanup
		fi
	fi
	echo
	cp $TMPDIR/ChangeLog.txt $CONF/ChangeLog.txt

	#
	# Download MANIFEST, FILELIST.TXT and CHECKSUMS.md5
	#

	# That will be download MANIFEST.bz2 files in /slackware
	# /extra and /pasture. 
	#
	echo -e "\t\tList of all files"
	for i in patches slackware extra pasture; do
		getfile ${i}/MANIFEST.bz2 $TMPDIR/${i}-MANIFEST.bz2 && \
			DIRS="$DIRS $i"
	done

	echo -e "\t\tPackage List"
	getfile FILELIST.TXT $TMPDIR/FILELIST.TXT

	if [ "$CHECKPKG" = "1" ]; then
		echo -e "\t\tChecksums"
		getfile CHECKSUMS.md5 ${CONF}/CHECKSUMS.md5
	fi
		
	# Download all PACKAGES.TXT files
	# 
	echo -e "\t\tPackage descriptions"
	for i in $DIRS; do
		getfile ${i}/PACKAGES.TXT $TMPDIR/${i}-PACKAGES.TXT
	done

	# Format FILELIST.TXT
	#
	echo -e "\tFormatting lists to slackpkg style..."
	echo -e "\t\tPackage List"
	awk '/tgz/ && !/source\// && !/testing\// && !/.asc/ {print $NF}' $TMPDIR/FILELIST.TXT | tr -d "\r" > $CONF/pkglist
		
	# Format MANIFEST
	#
		
	# bunzip and concatenate all MANIFEST files
	#
	MANFILES=""
	for i in $DIRS; do
		bunzip2 -c $TMPDIR/${i}-MANIFEST.bz2 | awk -f $CONF/filelist.awk | \
			gzip > $CONF/${i}-filelist.gz
	done

	if [ -r ${CONF}/filelist.gz ]; then
		rm $CONF/filelist.gz
		ln -s $CONF/slackware-filelist.gz $CONF/filelist.gz
	fi

	# Concatenate PACKAGE.TXT files
	#
	echo -e "\t\tPackage descriptions"
	for i in $DIRS; do
		cat $TMPDIR/${i}-PACKAGES.TXT >> $TMPDIR/PACKAGES.TXT
	done
	cp $TMPDIR/PACKAGES.TXT $CONF/PACKAGES.TXT
}

function sanity_check() {
	for i in `ls -1 /var/log/packages | \
		egrep -- "^.*-(${ARCH}|noarch)-[^-]+-upgraded"`; do
		REVNAME=`echo ${i} | awk -F'-upgraded' '{ print $1 }'`
		mv /var/log/packages/${i} /var/log/packages/${REVNAME}
		mv /var/log/scripts/${i} /var/log/scripts/${REVNAME}
	done 
	for i in `ls -1 /var/log/packages | egrep '\-[^\.]+$'`; do
		cutpkg $i
	done | sort > $TMPDIR/list1
	cat $TMPDIR/list1 | uniq > $TMPDIR/list2
	FILES="`diff $TMPDIR/list1 $TMPDIR/list2 | grep '<' | cut -f2 -d\ `"
	if [ "$FILES" != "" ]; then
		for i in $FILES ; do
			grep -qx "${i}" ${CONF}/blacklist && continue
			DOUBLEFILES="$DOUBLEFILES $i"
		done
		unset FILES
	fi
	if [ "$DOUBLEFILES" != "" ]; then
		echo -e "\
You have a broked /var/log/packages, with two versions of the same package.\n\
The list of packages duplicated in your machine are show below, but don't\n\
worry about this list, when you select your action, slackpkg will show a\n\
better list:\n"
		for i in $DOUBLEFILES ; do
			ls -1 /var/log/packages |\
				egrep -i -- "^${i}-[^-]+-(${ARCH}|noarch)-"
		done
		echo -ne "\n\
You can (B)lacklist or (R)emove these packages.\n\
Select your action (B/R): "
		read ANSWER
		echo
		case "$ANSWER" in
			B|b)
				showlist "$DOUBLEFILES" blacklist
				blacklist_pkg
			;;
			R|r)
				for i in $DOUBLEFILES ; do
					FILE=`ls -1 /var/log/packages |\
						egrep -i -- "^${i}-[^-]+-(${ARCH}|noarch)-"`
					FILES="$FILES $FILE"
				done
				showlist "$FILES" remove
				remove_pkg
			;;
			*)
				echo -e "\n\
Ok! slackpkg won't do anything now. But, please, do something to fix it.\n"
				cleanup
			;;
		esac
	fi
}	

function blacklist_pkg() {
	BLACKLISTFILE="$TMPDIR/blacklist"
	echo $SHOWLIST | tr ' ' "\n" > ${BLACKLISTFILE}
					   
	grep "^#" ${CONF}/blacklist > ${BLACKLISTFILE}.new
	grep -v "^#" ${CONF}/blacklist >> ${BLACKLISTFILE}

	cat ${BLACKLISTFILE} | grep -v "^$" | sort | \
		uniq >> ${BLACKLISTFILE}.new
	mv ${BLACKLISTFILE}.new ${CONF}/blacklist

	echo -e "\nPackages added to your blacklist.\n\
If you want to remove those packages, edit ${CONF}/blacklist.\n"
}

function remove_pkg() {
	for i in $SHOWLIST; do
		echo -e "\nPackage: $i"
		echo -e "\tRemoving... "
		removepkg $i
        done
}

function upgrade_pkg() {
	if [ "$DOWNLOAD_ALL" = "1" ]; then
		OLDDEL="$DELALL"
		DELALL="0"
		for i in $SHOWLIST; do
			getpkg $i true
		done
		DELALL="$OLDDEL"
	fi
	for i in $SHOWLIST; do
		getpkg $i upgradepkg Upgrading
	done
}

function install_pkg() {
	if [ "$DOWNLOAD_ALL" = "1" ]; then
		OLDDEL="$DELALL"
		DELALL="0"
		for i in $SHOWLIST; do
			getpkg $i true
		done
		DELALL="$OLDDEL"
	fi
	for i in $SHOWLIST; do
		getpkg $i installpkg Installing
	done
}
