extern char *malloc(), *realloc();

# line 2 "ptlang.y"
/************************************************************************
 Version identification:
 @(#)ptlang.y	2.31	12/1/92

Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

-----------------------------------------------------------------------

Ptolemy "star language" preprocessor.  This version does not support
compiled-in galaxies yet and the language may still change slightly.
Caveat hacker.

Programmer: J. T. Buck and E. A. Lee

************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <time.h>

/* Symbols for special characters*/
#define LPAR '('
#define RPAR ')'
#define QUOTE '"'
#define ESC '\\'
#define NEWLINE '\n'

/* buffer sizes */
#define BIGBUFSIZE 200000
#define MEDBUFSIZE 50000
#define SMALLBUFSIZE 512

/* number of code blocks allowed */
#define NUMCODEBLOCKS 100

#define FLEN 256
#define NINC 10
#define NSEE 30
#define NSTR 20

/* chars allowed in "identifier" */
#define IDENTCHAR(c) (isalnum(c) || c == '.' || c == '_')

char yytext[BIGBUFSIZE];	/* lexical analysis buffer */
int yyline = 1;			/* current input line */
int bodyMode = 0;		/* special lexan mode flag to read bodies  */
int docMode = 0;		/* flag document bodies  */
int descMode = 0;		/* flag descriptor bodies  */
int codeMode = 0;		/* flag code block bodies  */
int codeModeBraceCount;		/* brace count in codeMode */
FILE* yyin;			/* stream to read from */

char* progName = "ptlang";	/* program name */
int nerrs = 0;			/* # syntax errors detected */

char* blockID;			/* identifier for code blocks */
char* blockNames[NUMCODEBLOCKS];
char* codeBlocks[NUMCODEBLOCKS];
int numBlocks = 0;

/* scratch buffers */
char str1[SMALLBUFSIZE];
char str2[SMALLBUFSIZE];
char consStuff[BIGBUFSIZE];
char publicMembers[MEDBUFSIZE];
char protectedMembers[MEDBUFSIZE];
char privateMembers[MEDBUFSIZE];
char inputDescriptions[MEDBUFSIZE];
char outputDescriptions[MEDBUFSIZE];
char stateDescriptions[MEDBUFSIZE];
char ccCode[BIGBUFSIZE];
char hCode[BIGBUFSIZE];
char miscCode[BIGBUFSIZE];
char codeBlock[MEDBUFSIZE];
char consCalls[BIGBUFSIZE];

/* state classes */
#define T_INT 0
#define T_FLOAT 1
#define T_COMPLEX 2
#define T_STRING 3
#define T_INTARRAY 4
#define T_FLOATARRAY 5
#define T_COMPLEXARRAY 6
#define T_FIX 7
#define T_FIXARRAY 8
#define T_STRINGARRAY 9
#define NSTATECLASSES 10

#define M_PURE 1
#define M_VIRTUAL 2
#define M_INLINE 4

/* note: names must match order of symbols above */
char* stateClasses[] = {
"IntState", "FloatState", "ComplexState", "StringState",
"IntArrayState", "FloatArrayState", "ComplexArrayState",
"FixState", "FixArrayState", "StringArrayState"
};

/* bookkeeping for state include files */
int stateMarks[NSTATECLASSES];

/* external functions */
char* save();			/* duplicate a string */
char* malloc();
char* ctime();
time_t time();
void exit();

/* forward declarations for functions */

char* whichMembers();
char* checkArgs();
char* stripQuotes();
char* portDataType();
int   stateTypeClass();

char* inputFile;		/* input file name */
char* idBlock;			/* ID block */
char* objName;			/* name of star or galaxy class being decld  */
char* objVer;			/* sccs version number */
char* objDate;			/* date of last update */
char* objDesc;			/* descriptor of star or galaxy */
char* objAuthor;		/* author of star or galaxy */
char* objAcknowledge;		/* acknowledgements (previous authors) */
char* objCopyright;		/* copyright */
char* objExpl;			/* long explanation */
char* objLocation;		/* location string */
int   galDef;			/* true if obj is a galaxy */
char* domain;			/* domain of object (if star) */
char* derivedFrom;		/* class obj is derived from */
char* portName;			/* name of porthole */
char* portType;			/* dataType of porthole */
char* portInherit;		/* porthole for inheritTypeFrom */
char* portNum;			/* expr giving # of tokens */
char* portDesc;			/* port descriptor */
int   portOut;			/* true if porthole is output */
int   portMulti;		/* true if porthole is multiporthole */
char* portAttrib;		/* attributes for porthole */
char* stateName;		/* name of state */
char* stateClass;		/* class of state */
char* stateDef;			/* default value of state */
char* stateDesc;		/* descriptor for state */
char* stateAttrib;		/* attributes for state */
char* instName;			/* star instance within galaxy */
char* instClass;		/* class of star instance */
char* methodName;		/* name of user method */
char* methodArgs;		/* arglist of user method */
char* methodAccess;		/* protection of user method */
char* methodType;		/* return type of user method */
char* methodCode;		/* body of user method */
int   methodMode;		/* modifier flags for a method */
int   pureFlag;			/* if true, class is abstract */
char* galPortName;		/* name of galaxy port */

/* codes for "standard methods".  To add more, add them at the end,
 * modify N_FORMS, and put types in the codeType array and names in
 * the codeFuncName array.
 */
#define C_CONS 0
#define C_EXECTIME 1
#define C_WRAPUP 2
#define C_INITCODE 3
#define C_DEST 4
#define C_SETUP 5
#define C_GO 6


#define N_FORMS 7
char* codeBody[N_FORMS];		/* code bodies for each entity */
int inlineFlag[N_FORMS];		/* marks which are to be inline */
char destNameBuf[MEDBUFSIZE];		/* storage for destructor name */

/* types and names of standard member funcs */
char* codeType[] = {"","int ","void ","void ","","void ","void "};
char* codeFuncName[] = {
"","myExecTime","wrapup","initCode",destNameBuf,"setup","go"};

int methKey;			/* signals which of the standard funcs */

#define consCode codeBody[C_CONS]	/* extra constructor code */

char* hInclude[NINC];		/* include files in .h file */
int   nHInclude;		/* number of such files */
char* ccInclude[NINC];		/* include files in .cc file */
int   nCcInclude;		/* number of such files */
char* seeAlsoList[NSEE];	/* list of pointers to other manual sections */
int   nSeeAlso;			/* number of such pointers */

/* all tokens with values are type char *.  Keyword tokens
 * have their names as values.
 */
typedef char * STRINGVAL;
#define YYSTYPE STRINGVAL

# define DEFSTAR 257
# define GALAXY 258
# define NAME 259
# define DESC 260
# define DEFSTATE 261
# define DOMAIN 262
# define NUMPORTS 263
# define NUM 264
# define VIRTUAL 265
# define DERIVED 266
# define CONSTRUCTOR 267
# define DESTRUCTOR 268
# define STAR 269
# define ALIAS 270
# define OUTPUT 271
# define INPUT 272
# define ACCESS 273
# define OUTMULTI 274
# define INMULTI 275
# define TYPE 276
# define DEFAULT 277
# define CLASS 278
# define SETUP 279
# define GO 280
# define WRAPUP 281
# define CONNECT 282
# define ID 283
# define CCINCLUDE 284
# define HINCLUDE 285
# define PROTECTED 286
# define PUBLIC 287
# define PRIVATE 288
# define METHOD 289
# define ARGLIST 290
# define CODE 291
# define BODY 292
# define IDENTIFIER 293
# define STRING 294
# define CONSCALLS 295
# define ATTRIB 296
# define LINE 297
# define VERSION 298
# define AUTHOR 299
# define ACKNOWLEDGE 300
# define COPYRIGHT 301
# define EXPLANATION 302
# define SEEALSO 303
# define LOCATION 304
# define CODEBLOCK 305
# define EXECTIME 306
# define PURE 307
# define INLINE 308
# define HEADER 309
# define INITCODE 310
#define yyclearin yychar = -1
#define yyerrok yyerrflag = 0
extern int yychar;
extern int yyerrflag;
#ifndef YYMAXDEPTH
#define YYMAXDEPTH 150
#endif
#ifndef YYSTYPE
#define YYSTYPE int
#endif
YYSTYPE yylval, yyval;
# define YYERRCODE 256

# line 629 "ptlang.y"

/* Reset for a new star or galaxy class definition.  If arg is TRUE
   we are defining a galaxy.
 */

clearDefs (g)
int g;
{
	int i;
	for (i = 0; i < NSTATECLASSES; i++) stateMarks[i] = 0;
	galDef = g;
	objName = objVer = objDesc = domain = derivedFrom =
		objAuthor = objCopyright = objExpl = objLocation = NULL;
	consStuff[0] = ccCode[0] = hCode[0] = codeBlock[0] = consCalls[0] = 0;
	publicMembers[0] = privateMembers[0] = protectedMembers[0] = 0;
	inputDescriptions[0] = outputDescriptions[0] = stateDescriptions[0] = 0;
	nCcInclude = nHInclude = nSeeAlso = 0;
	pureFlag = 0;
	for (i = 0; i < N_FORMS; i++) {
		codeBody[i] = 0;
		inlineFlag[i] = 0;
	}
}

/* Generate a state definition */
clearStateDefs ()
{
	stateName = stateClass = stateDef = stateDesc = stateAttrib = NULL;
}

char*
cvtToLower (name)
char* name;
{
	static char buf[128];
	char* p = buf, c;
	while ((c = *name++) != 0) {
		if (isupper(c)) *p++ = tolower(c);
		else *p++ = c;
	}
	*p = 0;
	return buf;
}

char*
cvtToUpper (name)
char* name;
{
	static char buf[128];
	char* p = buf, c;
	while ((c = *name++) != 0) {
		if (islower(c)) *p++ = toupper(c);
		else *p++ = c;
	}
	*p = 0;
	return buf;
}

/* get "real" state class name from an argument */
int
stateTypeClass (nameArg)
char* nameArg;
{
	char* name = cvtToLower (nameArg);
 
	if (strcmp (name, "int") == 0)
		return T_INT;
	if (strcmp (name, "float") == 0)
		return T_FLOAT;
	if (strcmp (name, "complex") == 0)
		return T_COMPLEX;
	if (strcmp (name, "string") == 0)
		return T_STRING;
	if (strcmp (name, "intarray") == 0)
		return T_INTARRAY;
	if (strcmp (name, "floatarray") == 0)
		return T_FLOATARRAY;
	if (strcmp (name, "complexarray") == 0)
		return T_COMPLEXARRAY;
	if (strcmp (name, "fix") == 0)
		return T_FIX;
	if (strcmp (name, "fixarray") == 0)
		return T_FIXARRAY;
	if (strcmp (name, "stringarray") == 0)
		return T_STRINGARRAY;
	fprintf (stderr, "state class %s\n", name);
	yyerror ("bad state class: assuming int");
		return T_INT;
}

/* generate code for a state defn */
genState ()
{
	char* stateDescriptor;
	char* stateDefault;
	/* test that all fields are known */
	if (stateName == NULL) {
		yyerror ("state name not defined");
		return;
	}
	if (stateClass == NULL) {
		yyerror ("state class not defined");
		return;
	}
	if (stateDef == NULL)
		stateDefault = "\"\"";
	else
		stateDefault = stateDef;
	if (stateDesc == NULL)
		stateDescriptor = stateName;
	else
		stateDescriptor = stateDesc;
	sprintf (str1,"\t%s %s;\n", stateClass, stateName);
	sprintf (str2,"\taddState(%s.setState(\"%s\",this,%s,\"%s\"",
		 stateName, stateName, stateDefault, stateDescriptor);
	if (stateAttrib) {
		strcat (str2, ",\n");
		strcat (str2, stateAttrib);
	}
	strcat (str2, "));\n");
	strcat (protectedMembers, str1);
	strcat (consStuff, str2);
}

/* describe the states */
describeState ()
{
	char descriptString[MEDBUFSIZE];

	sprintf(str1,".NE\n\\fI%s\\fR (%s)",stateName,stateClass);
	strcat(stateDescriptions,str1);
	if (stateDesc) {
	    if(unescape(descriptString, stateDesc, MEDBUFSIZE))
		yywarn("warning: Descriptor too long. May be truncated.");
	    sprintf(str1,": %s\n",descriptString);
	} else
	    sprintf(str1,"\n");
	strcat(stateDescriptions,str1);
	if (stateDef) {
		sprintf(str1,".DF %s\n",stateDef);
	}
	strcat(stateDescriptions,str1);
}

/* set up for port definition */
initPort (out, multi)
int out, multi;
{
	portOut = out;
	portMulti = multi;
	portName = portNum = portInherit = portDesc = portAttrib = NULL;
	portType = "ANYTYPE";
}

char* portDataType (name)
char* name;
{
	/* do better checking later */
	return save(name);
}

genPort ()
{
	/* test that all fields are known */
	char* dir = portOut ? "Out" : "In";
	char* m = portMulti ? "Multi" : "";
	char* d = galDef ? "" : domain;
	char* port = galDef ? "PortHole" : "Port";

	sprintf (str1,"\t%s%s%s%s %s;\n", m, dir, d, port, portName);
	if (portNum)
		sprintf (str2, "\taddPort(%s.setPort(\"%s\",this,%s,%s));\n",
			portName, portName, cvtToUpper(portType), portNum);
	else
		sprintf (str2, "\taddPort(%s.setPort(\"%s\",this,%s));\n",
			portName, portName, cvtToUpper(portType));
	strcat (publicMembers, str1);
	strcat (consStuff, str2);
	if (portAttrib) {
		sprintf (str2, "\t%s.setAttributes(\n%s);\n", portName,
			portAttrib);
		strcat (consStuff, str2);
	}
	if (portInherit) {
		sprintf (str2, "\t%s.inheritTypeFrom(%s);\n", portName,
			 portInherit);
		strcat (consStuff, str2);
	}
}

describePort ()
{
	char *dest;
	char descriptString[MEDBUFSIZE];
	if(portOut)
	    /* describe an output port */
	    dest = outputDescriptions;
	else
	    /* describe an input port */
	    dest = inputDescriptions;
	if (portMulti)
	    sprintf(str1,".NE\n\\fI%s\\fR (multiple), (%s)",portName,portType);
	else
	    sprintf(str1,".NE\n\\fI%s\\fR (%s)",portName,portType);
	strcat(dest,str1);

	if (portDesc) {
	    if(unescape(descriptString, portDesc, MEDBUFSIZE))
		yywarn("warning: Descriptor too long. May be truncated.");
	    sprintf(str1,": %s\n",descriptString);
	} else
	    sprintf(str1,"\n");
	strcat(dest,str1);
}

/* set up for user-supplied method */
clearMethodDefs (mode)
int mode;
{
	methodName = NULL;
	methodArgs = "()";
	methodAccess = "protected";
	methodCode = NULL;
	methodType = "void";
	methodMode = mode;
}

/* generate code for user-defined method */
genMethod ()
{
	char * p = whichMembers (methodAccess);
/* add decl to class body */
	if (methodMode == M_PURE) {
		if (methodCode) yyerror ("Code supplied for pure method");
		/* pure virtual function case */
		sprintf (str1, "\tvirtual %s %s %s = 0;\n",
			methodType, methodName, methodArgs);
		strcat (p, str1);
		pureFlag++;
		return;
	}
	if (methodCode == NULL) {
		yyerror ("No code supplied for method");
		methodCode = "";
	}
	/* form declaration in class body */
	sprintf (str1, "\t%s%s %s %s", (methodMode & M_VIRTUAL) ? "virtual " : "",
		 methodType, methodName, methodArgs);
	strcat (p, str1);
	/* handle inline functions */
	if (methodMode & M_INLINE) {
		strcat (p, " {\n");
		strcat (p, methodCode);
		strcat (p, "\n\t}\n");
		return;
	}
	/* not inline: put it into the .cc file */
	strcat (p, ";\n");
	sprintf (str2, "\n\n%s %s%s::%s %s\n{\n", methodType,
		 galDef ? "" : domain, objName,
		 methodName, methodArgs);
	strcat (miscCode, str2);
	strcat (miscCode, methodCode);
	strcat (miscCode, "\n}\n");
}

/* generate an instance of a block within a galaxy */
genInstance ()
{
	sprintf (str1, "\t%s $s;\n", instClass, instName);
	strcat (protectedMembers, str1);
	sprintf (str2,"addBlock(%s.setBlock(\"%s\",this)", instName, instName);
	strcat (consStuff, str2);
}


genAlias () {
	/* FILL IN */
}

genConnect () {
	/* FILL IN */
}

char* whichMembers (type)
char* type;
{
/* type must be "protected", "public", or "private" */
	switch (type[2]) {
	case 'o':
		return protectedMembers;
	case 'b':
		return publicMembers;
	case 'i':
		return privateMembers;
	default:
		fprintf (stderr, "Internal error in whichMembers\n");
		exit (1);
		/* NOTREACHED */
	}
}

/* add declarations of extra members to the class definition */
addMembers (type, defs)
char* type;
char* defs;
{
	char * p = whichMembers (type);
	strcat (p, defs);
	strcat (p, "\n");
	return;
}

/* This is the main guy!  It outputs the complete class definition. */
genDef ()
{
	FILE *fp;
	int i;
	char fname[FLEN], hname[FLEN], ccname[FLEN];
	char baseClass[SMALLBUFSIZE];
	char fullClass[SMALLBUFSIZE];
	char descriptString[MEDBUFSIZE];

/* temp, until we implement this */
	if (galDef) {
		fprintf (stderr, "Sorry, galaxy definition is not yet supported.\n");
		exit (1);
	}
	if (objName == NULL) {
		yyerror ("No class name defined");
		return;
	}
	if (!galDef && !domain) {
		yyerror ("No domain name defined");
		return;
	}
	sprintf (fullClass, "%s%s", galDef ? "" : domain, objName);

/***************************************************************************
			CREATE THE .h FILE
*/
	sprintf (hname, "%s.h", fullClass);
	if ((fp = fopen (hname, "w")) == 0) {
		perror (hname);
		exit (1);
	}
/* Surrounding ifdef stuff */
	fprintf (fp, "#ifndef _%s_h\n#define _%s_h 1\n", fullClass, fullClass);
	fprintf (fp, "// header file generated from %s by %s\n",
		 inputFile, progName);

/* Special GNU pragmas for increased efficiency */
	fprintf (fp, "\n#ifdef __GNUG__\n#pragma interface\n#endif\n\n");

/* copyright */
	if (objCopyright) {
	    if ( strncasecmp(objCopyright,"copyright",9)==0 ) {
		fprintf (fp, "/*\n%s\n */\n", objCopyright);
	    } else {
		fprintf (fp, "/*\n * copyright (c) %s\n */\n", objCopyright);
	    }
	}

/* ID block */
	if (idBlock)
		fprintf (fp, "%s\n", idBlock);
/* The base class */
/* For stars, we append the domain name to the beginning of the name,
   unless it is already there */
	if (derivedFrom) {
		if (domain &&
		    strncmp (domain, derivedFrom, strlen (domain)) != 0) {
			sprintf (baseClass, "%s%s", galDef ? "" : domain,
				 derivedFrom);
		}
		else
			(void) strcpy (baseClass, derivedFrom);
	}
/* Not explicitly specified: baseclass is Galaxy or XXXStar */
	else if (galDef)
		(void)strcpy (baseClass, "Galaxy");
	else
		sprintf (baseClass, "%sStar", domain);

/* Include files */
	fprintf (fp, "#include \"%s.h\"\n", baseClass);
	
	for (i = 0; i < nHInclude; i++) {
		fprintf (fp, "#include %s\n", hInclude[i]);
	}
/* Include files for states */
	for (i = 0; i < NSTATECLASSES; i++)
		if (stateMarks[i])
			fprintf (fp, "#include \"%s.h\"\n", stateClasses[i]);

/* extra header code */
	fprintf (fp, "%s\n", hCode);
/* The class template */
	fprintf (fp, "class %s : public %s\n{\n", fullClass, baseClass);
	sprintf (destNameBuf, "~%s", fullClass);
	fprintf (fp, "public:\n\t%s();\n", fullClass);
/* The makeNew function: only if the class isn't a pure virtual */
	if (!pureFlag)
		fprintf (fp, "\t/* virtual */ Block* makeNew() const;\n");
        fprintf (fp, "\t/* virtual*/ const char* className() const;\n");
	for (i=0; i<numBlocks; i++)
		fprintf (fp, "\tstatic CodeBlock %s;\n",blockNames[i]);
	for (i = C_EXECTIME; i <= C_DEST; i++)
		genStdProto(fp,i);
	if (publicMembers[0])
		fprintf (fp, "%s\n", publicMembers);
	fprintf (fp, "protected:\n");
	for (i = C_SETUP; i <= C_GO; i++)
		genStdProto(fp,i);
	if (protectedMembers[0])
		fprintf (fp, "%s\n", protectedMembers);
	if (privateMembers[0])
		fprintf (fp, "private:\n%s\n", privateMembers);
/* that's all, end the class def and put out an #endif */
	fprintf (fp, "};\n#endif\n");
	(void) fclose (fp);

/**************************************************************************
		CREATE THE .cc FILE
*/
	sprintf (ccname, "%s.cc", fullClass);
	if ((fp = fopen (ccname, "w")) == 0) {
		perror (ccname);
		exit (1);
	}
	fprintf (fp, "static const char file_id[] = \"%s\";\n", inputFile);
	fprintf (fp, "// .cc file generated from %s by %s\n",
		 inputFile, progName);
/* copyright */
	if (objCopyright) {
	    if ( strncasecmp(objCopyright,"copyright",9)==0 ) {
		fprintf (fp, "/*\n%s\n */\n", objCopyright);
	    } else {
		fprintf (fp, "/*\n * copyright (c) %s\n */\n", objCopyright);
	    }
	}

/* special GNU pragma for efficiency */
	fprintf (fp, "\n#ifdef __GNUG__\n#pragma implementation\n#endif\n\n");

/* ID block */
	if (idBlock)
		fprintf (fp, "%s\n", idBlock);
/* include files */
	if (!pureFlag)
		fprintf (fp, "#include \"KnownBlock.h\"\n");
	fprintf (fp, "#include \"%s.h\"\n", fullClass);
	for (i = 0; i < nCcInclude; i++)
		fprintf (fp, "#include %s\n", ccInclude[i]);
/* generate className and (optional) makeNew function */
/* also generate a global identifier with name star_nm_DDDNNN, where DDD is
   the domain and NNN is the name */
	fprintf (fp, "\nconst char *star_nm_%s = \"%s\";\n", fullClass, fullClass);
        fprintf (fp, "\nconst char* %s :: className() const {return star_nm_%s;}\n",
		fullClass, fullClass);
	if (!pureFlag) {
		fprintf (fp, "\nBlock* %s :: makeNew() const { LOG_NEW; return new %s;}\n",
			 fullClass, fullClass);
	}
/* generate the CodeBlock constructor calls */
	for (i=0; i<numBlocks; i++)
		fprintf (fp, "\nCodeBlock %s :: %s (\n%s);\n",
			fullClass,blockNames[i],codeBlocks[i]);
/* prefix code and constructor */
	fprintf (fp, "\n%s%s::%s ()", ccCode, fullClass, fullClass);
	if (consCalls[0])
		fprintf (fp, " :\n\t%s", consCalls);
	fprintf (fp, "\n{\n");
	if (objDesc)
		fprintf (fp, "\tsetDescriptor(\"%s\");\n", objDesc);
	/* define the class name */
	if (!consCode) consCode = "";
	fprintf (fp, "%s\n%s\n", consStuff, consCode);
	fprintf (fp, "}\n");
	for (i = C_EXECTIME; i <= C_GO; i++) {
		if (codeBody[i] && !inlineFlag[i])
			fprintf (fp, "\n%s%s::%s() {\n%s\n}\n",
			codeType[i], fullClass, codeFuncName[i], codeBody[i]);
	}
	if (miscCode[0])
		fprintf (fp, "%s\n", miscCode);
	if (pureFlag) {
		fprintf (fp,
			 "\n// %s is an abstract class: no KnownBlock entry\n",
			 fullClass);
	}
	else {
		fprintf (fp, "\n// prototype instance for known block list\n");
		fprintf (fp, "static %s proto;\n", fullClass);
		fprintf (fp, "static KnownBlock entry(proto,\"%s\");\n",
			 objName);
	}
	(void) fclose(fp);

/**************************************************************************
		CREATE THE DOCUMENTATION FILE
*/

	sprintf (fname, "%s.t", fullClass);
	if ((fp = fopen (fname, "w")) == 0) {
		perror (fname);
		exit (1);
	}

	fprintf (fp, ".\\\" documentation file generated from %s by %s\n",
		 inputFile, progName);

/* copyright */
	if (objCopyright) {
		char* p = objCopyright;
		while (*p) {
			/* make each line a comment */
			fprintf (fp, ".\\\" ");
			while (*p && *p != NEWLINE) fputc(*p++,fp);
			if (*p == NEWLINE) fputc(*p++,fp);
		}
		fputc(NEWLINE,fp);
	}

/* Name */
	fprintf (fp, ".NA \"%s\"\n", objName);

/* short descriptor */
	fprintf (fp, ".SD\n");
	if (objDesc) {
		/*
		 * print descriptor with "\n" replaced with NEWLINE,
		 * and "\t" replaced with a tab.
		 * Any other escaped character will be printed as is.
		 */
		if(unescape(descriptString, objDesc, MEDBUFSIZE))
		    yywarn("warning: Descriptor too long. May be truncated.");
		fprintf (fp, "%s\n", descriptString);
	}
	fprintf (fp, ".SE\n");

/* location */
	if (objLocation)
		fprintf (fp, ".LO \"%s\"\n",objLocation);

/* base class and domain */
	/* For stars, we append the domain name to the beginning of the name,
	   unless it is already there */
	if (derivedFrom) {
		if (domain &&
		    strncmp (domain, derivedFrom, strlen (domain)) != 0) {
			sprintf (baseClass, "%s%s", galDef ? "" : domain,
				 derivedFrom);
		}
		else
			(void) strcpy (baseClass, derivedFrom);
	}
	/* Not explicitly specified: baseclass is Galaxy or XXXStar */
	else if (galDef)
		(void)strcpy (baseClass, "Galaxy");
	else
		sprintf (baseClass, "%sStar", domain);

	fprintf (fp, ".DM %s %s\n", domain, baseClass);

/* version */
	fprintf (fp, ".SV %s %s\n", objVer, objDate);

/* author */
	if (objAuthor)
		fprintf (fp, ".AL \"%s\"\n", objAuthor);

/* acknowledge */
	if (objAcknowledge)
		fprintf (fp, ".AC \"%s\"\n", objAcknowledge);

/* inputs */
	if (strlen(inputDescriptions) > 0)
		fprintf (fp, ".IH\n%s.PE\n", inputDescriptions);

/* outputs */
	if (strlen(outputDescriptions) > 0)
		fprintf (fp, ".OH\n%s.PE\n", outputDescriptions);

/* states */
	if (strlen(stateDescriptions) > 0)
		fprintf (fp, ".SH\n%s.ET\n", stateDescriptions);

/* explanation */
	if (objExpl)
		fprintf (fp, ".LD\n%s\n", objExpl);

/* ID block (will appear in .h and .cc files only. */

/* See Also list */
	if (nSeeAlso > 0) fprintf (fp, ".SA\n");
	if (nSeeAlso > 2)
	    for (i = 0; i < (nSeeAlso - 2); i++)
		fprintf (fp, "%s,\n", seeAlsoList[i]);
	if (nSeeAlso > 1) fprintf (fp, "%s and\n", seeAlsoList[nSeeAlso-2]);
	if (nSeeAlso > 0) fprintf (fp, "%s.\n", seeAlsoList[nSeeAlso-1]);

/* end the final entry */
	fprintf (fp, ".ES\n");

/* close the file */
	(void) fclose (fp);
}



struct tentry {
	char* key;
	int code;
};

/* keyword table */
struct tentry keyTable[] = {
	"access", ACCESS,
	"acknowledge", ACKNOWLEDGE,
	"alias", ALIAS,
	"arglist", ARGLIST,
	"attrib", ATTRIB,
	"attributes", ATTRIB,
	"author", AUTHOR,
	"ccinclude", CCINCLUDE,
	"class", CLASS,
	"code", CODE,
	"codeblock", CODEBLOCK,
	"conscalls", CONSCALLS,
	"consCalls", CONSCALLS,
	"constructor", CONSTRUCTOR,
	"copyright", COPYRIGHT,
	"default", DEFAULT,
	"defstar", DEFSTAR,
	"defstate", DEFSTATE,
	"derived", DERIVED,
	"derivedFrom", DERIVED,
	"derivedfrom", DERIVED,
	"desc", DESC,
	"descriptor", DESC,
	"destructor", DESTRUCTOR,
	"domain", DOMAIN,
	"execTime", EXECTIME,
	"exectime", EXECTIME,
	"explanation", EXPLANATION,
	"galaxy", GALAXY,
	"go", GO,
	"header", HEADER,
	"hinclude", HINCLUDE,
	"ident", ID,
	"initCode", INITCODE,
	"initcode", INITCODE,
	"inmulti", INMULTI,
	"inline", INLINE,
	"input", INPUT,
	"location", LOCATION,
	"method", METHOD,
	"name", NAME,
	"num", NUM,
	"numports", NUMPORTS,
	"numTokens", NUM,
	"numtokens", NUM,
	"outmulti", OUTMULTI,
	"output", OUTPUT,
	"private", PRIVATE,
	"programmer", AUTHOR,
	"protected", PROTECTED,
	"public", PUBLIC,
	"pure", PURE,
	"seealso", SEEALSO,
	"setup", SETUP,
	"star", STAR,
	"start", SETUP,		/* backward compatibility */
	"state", DEFSTATE,
	"type", TYPE,
	"version", VERSION,
	"virtual", VIRTUAL,
	"wrapup", WRAPUP,
	0, 0,
};

#define input() ((c = getc(yyin))==10?(yyline++,c):c)

/* The lexical analyzer */
yylex () {
	static int c = 0;
	int key;
	char* p = yytext;
	if (c == EOF) return 0;
/*
 * In codeMode, we look for LINEs and return them.
 * A LINE is an exact copy of of line of input that
 * does not contain the closing '}'.
 * When a line is encountered that contains the closing '}'
 * that closing '}' is returned.  Anything else on the line is lost.
 */
	if (codeMode) {
	    int inQuote = 0;
	    /* eat spaces until a newline */
	    while (!c || (isspace(c) && c != NEWLINE))
		input();
	    /* now eat the newline */
	    if (c == NEWLINE) input();
	    /* now transfer characters to yytext until the next newline,
	       or the closing brace. */
	    while (c != NEWLINE) {
		*p++ = c;
		switch (c) {
			/* one backslash in input becomes two in output */
		  case ESC:
		    *p++ = c;
		    break;
		  case QUOTE:
		    /* quote in input is escaped */
		    p[-1] = ESC;
		    *p++ = c;
		    inQuote = !inQuote;
		    break;
		  case EOF:
		    yyerror ("Unexpected EOF in body!");
		    exit (1);
		  default:
		    if (!inQuote) {
			if (c == '{') codeModeBraceCount++;
			else if (c == '}') {
			    codeModeBraceCount--;
			    if (codeModeBraceCount == 0) {
				/* output doesn't include the '}' */
				p[0] = 0;
				c = 0;
				yylval = save(yytext);
				return '}';
			    }
			}
		    }
		}
		input();
	    }
	    /* output doesn't include the NEWLINE */
	    p[0] = 0;
	    yylval = save(yytext);
	    return LINE;
	}

/* bodyMode causes a whole function or document
 * body to be returned as a single token.
 * Leading and trailing spaces are removed
 */
	while (!c || isspace(c)) {
		input();
	}
	if (bodyMode) {
		int brace = 1;
		int inQuote = 0;
		int inComment = 0;
		int startLine = yyline;
		int startQLine = yyline;
/* if !docMode, put a "#line" directive in the token */
		if (!docMode) {
		   sprintf (yytext, "# line %d \"%s\"\n", yyline, inputFile);
		   p = yytext + strlen (yytext);
		}

		while (brace > 0) {
			*p++ = c;
			switch (c) {
			case ESC:
				input();
				*p++ = c;
				break;
			case QUOTE:
				if (!inComment) {
					inQuote = !inQuote;
					startQLine = yyline;
				}
				break;
			case EOF:
				sprintf (yytext,
			"Unterminated %s at EOF: it began on line %d",
			inQuote ? "string" : "code block",
			inQuote ? startQLine : startLine);
				yyerror (yytext);;
				exit (1);
			case '/':
				if (inQuote) break;
				c = getc(yyin);
				if (c == '/') {
					inComment = 1;
					*p++ = c;
				}
				else ungetc(c,yyin);
				break;
			case NEWLINE:
				inComment = 0;
				break;
			default:
				if (!inQuote && !inComment) {
				   if (c == '{') brace++;
				   else if (c == '}') brace--;
				}
			}
			input();
		}
/* The BODY token does not include the closing '}' though it is removed
 * from the input.
 */
		--p;
/* trim trailing whitespace */
		--p;
		while (isspace(*p)) --p;
		p[1] = 0;
		c = 0;
		yylval = save(yytext);
		return BODY;
	}

/* descMode causes a whole descriptor body to be returned as a single token
 * in the form of a string with newlines indicated as "\n" and quotes
 * escaped (\").
 */
	if (descMode) {
		int brace = 1;
		int inQuote = 0;
		while (brace > 0) {
			*p++ = c;
			switch (c) {
			case ESC:
				input();
				*p++ = c;
				break;
			case QUOTE:
				/* escape the quote */
				--p;
				*p++ = ESC;
				*p++ = QUOTE;
				inQuote = !inQuote;
				break;
			case NEWLINE:
				/* replace with "\n" */
				--p;
				*p++ = ESC;
				*p++ = 'n';
				break;
			case EOF:
				yyerror ("Unexpected EOF in descriptor!");
				exit (1);
			default:
				if (!inQuote) {
				  if (c == '{') brace++;
				  else if (c == '}') brace--;
				}
				break;
			}
			input();
		}
/* The BODY token does not include the closing '}' though it is removed
 * from the input.
 */
		--p;
/* trim trailing whitespace or '\n' */
		--p;
		while (isspace(*p) || (*p == 'n' && *(p-1) == ESC)) {
			if(*p == 'n') p -= 2;
			else --p;
		}
		p[1] = 0;
		c = 0;
		yylval = save(yytext);
		return BODY;
	}

/* regular code (not BODY mode) */

/* loop to eat up blanks and comments.  A comment starts with // and
 * continues for the rest of the line.  If a single / is seen in the
 * loop a '/' token is returned.
 */
	while (1) {
		if (c != '/') break;
		else {
			input();
			if (c != '/') {
				*yytext = '/';
				yytext[1] = 0;
				return '/';
			}
			/* comment -- eat rest of line */
			while (input() != NEWLINE && c != EOF);
		}
		while (isspace(c)) { input(); }
	}
	/*
	 * STRING token includes surrounding quotes
	 * If the STRING includes a NEWLINE, a warning is issued.
	 */
	if (c == EOF) return 0;		
	if (c == QUOTE) {
		p = yytext;
		*p++ = c;
		while (1) {
			*p++ = input();
			if (c == QUOTE) {
				*p = 0;
				break;
			}
			else if (c == ESC) {
				*p++ = input();
			}
			else if (c == NEWLINE) {
				yywarn ("warning: multi-line string");
			}
			else if (c == EOF) {
				yyerror ("Unexpected EOF in string");
				exit (1);
			}
		}
		c = 0;
		yylval = save(yytext);
		return STRING;
	}
	/* Token like <stdio.h> */
	else if (c == '<') {
		p = yytext;
		*p++ = c;
		while (1) {
			*p++ = input();
			if (c == '>') {
				*p = 0;
				break;
			}
			else if (c == EOF) {
				yyerror ("Unexpected EOF in string");
				exit (1);
			}
		}
		c = 0;
		yylval = save(yytext);
		return STRING;
	}
        else if (! IDENTCHAR(c) ) {
		yytext[0] = c;
		yytext[1] = 0;
		c = 0;
		return yytext[0];
	}
/* note: we also return numeric values as IDENTIFIER: digits and '.' allowed */
	else do {
		*p++ = c;
		input();
        } while ( IDENTCHAR(c) );
	*p = 0;
	yylval = save(yytext);
	if ((key = lookup (yytext)) != 0) {
		return key;
	}
	return IDENTIFIER;
}

int lookup (text)
char* text;
{
	struct tentry *p = keyTable;
	while (p->key && strcmp (p->key, text) != 0) p++;
	return p->code;
}

/* save token in dynamic memory */
char* save(in)
char* in;
{
	char* out = malloc((unsigned)strlen(in)+1);
	return strcpy(out,in);
}

/* strip quotes, save token in dynamic memory */
char* stripQuotes(in)
char* in;
{
	char* out;
	int l = strlen (in);
	if (l <= 2 || *in != QUOTE) return in;
	out = malloc((unsigned)l-1);
	return strncpy(out,in+1,l-2);
}

/* make sure a function is a valid arglist */
char* checkArgs(in)
char* in;
{
	if (*in != LPAR || in[strlen(in)-1] != RPAR)
		yyerror ("Invalid argument list");
	return in;
}

/*
 * copy one string into another, replacing the pattern "\n" with
 * NEWLINE, "\t" with tab, and "\x" with x for any other x.
 * The third argument is the size of the destination, which will not
 * be exceeded.
 */
int unescape(destination, source, dsize)
char* destination;
char* source;
int dsize;
{
	char* d = destination;
	char* s = source;
	int i = 1;
	while (*s != NULL) {
	    if (*s == ESC) {
		switch (*(s+1)) {
		    case 'n':
			*d++ = '\n';
			break;
		    case 't':
			*d++ = '\t';
			break;
		    default:
			*d++ = *(s+1);
		}
		s += 2;
	    } else
		*d++ = *s++;
	    if(i++ >= dsize) {
		*d = NULL; /* terminate the string */
		return(1);
	    }
	}
	*d = NULL; /* terminate the string */
	return(0);
}

/* main program, just calls parser */
main (argc, argv)
char **argv;
{
	if (argc != 2) {
		fprintf (stderr, "Usage: %s file\n", *argv);
		exit (1);
	}
	inputFile = argv[1];
	if ((yyin = fopen (inputFile, "r")) == NULL) {
		perror (inputFile);
		exit (1);
	}
	yyparse ();
	return nerrs;
}

yyerr2 (x, y)
char *x, *y;
{
	strcpy (str1, x);
	strcat (str1, y);
	yyerror (str1);
}

yyerror(s)
char *s;
{
	/* ugly: figure out if yacc is reporting syntax error at EOF */
	if (strcmp(s, "syntax error") == 0 && yychar == 0)
		s = "Unexpected EOF (mismatched curly braces?)";
	fprintf (stderr, "\"%s\", line %d: %s\n", inputFile, yyline, s);
	nerrs++;
	return;
}

yywarn(s)
char *s;
{
	fprintf (stderr, "\"%s\", line %d: %s\n", inputFile, yyline, s);
	return;
}

mismatch(s)
char *s;
{
	yyerr2 ("Extra token appears after valid input: ", s);
	exit (1);
}

/* fn to write out standard methods in the header */
genStdProto(fp,i)
FILE* fp;
int i;
{
	if (codeBody[i])
		fprintf (fp, "\t/* virtual */ %s%s()%s\n", codeType[i],
			 codeFuncName[i], inlineFlag[i] ? " {" : ";");
	if (inlineFlag[i])
		fprintf (fp, "%s\n\t}\n", codeBody[i]);
}
int yyexca[] ={
-1, 1,
	0, -1,
	-2, 0,
-1, 12,
	123, 9,
	-2, 141,
-1, 13,
	123, 11,
	-2, 142,
-1, 60,
	267, 49,
	268, 49,
	279, 49,
	280, 49,
	281, 49,
	306, 49,
	310, 49,
	-2, 0,
-1, 61,
	267, 49,
	268, 49,
	279, 49,
	280, 49,
	281, 49,
	306, 49,
	310, 49,
	-2, 0,
-1, 62,
	267, 49,
	268, 49,
	279, 49,
	280, 49,
	281, 49,
	306, 49,
	310, 49,
	-2, 0,
-1, 92,
	289, 51,
	-2, 50,
-1, 103,
	267, 49,
	268, 49,
	279, 49,
	280, 49,
	281, 49,
	306, 49,
	310, 49,
	-2, 0,
-1, 137,
	294, 130,
	125, 130,
	44, 130,
	-2, 0,
-1, 138,
	294, 134,
	125, 134,
	44, 134,
	-2, 0,
	};
# define YYNPROD 185
# define YYLAST 606
int yyact[]={

   261,     5,   154,   155,    14,    15,    16,    17,    18,   263,
   264,    19,    20,    21,    22,    23,    24,    25,    42,    26,
    27,    28,    29,   327,    30,    31,    32,    33,   325,    34,
    35,    36,    37,    38,    39,    40,    41,   285,   304,   306,
   284,   350,   348,    45,    43,    44,    46,    47,    48,    49,
    50,    51,    52,    53,    54,    55,   154,   155,    14,    15,
    16,    17,    18,   346,   345,    19,    20,    21,    22,    23,
    24,    25,    42,    26,    27,    28,    29,   343,    30,    31,
    32,    33,   342,    34,    35,    36,    37,    38,    39,    40,
    41,   339,    10,    11,   232,   235,   186,    45,    43,    44,
    46,    47,    48,    49,    50,    51,    52,    53,    54,    55,
   189,   233,   234,   188,   338,   335,   334,   332,   331,   108,
   321,   319,   318,   298,   295,   272,   220,   187,    94,   219,
   326,   164,   217,    12,    13,    14,    15,    16,    17,    18,
   239,   293,    19,    20,    21,    22,    23,    24,    25,    42,
    26,    27,    28,    29,   279,    30,    31,    32,    33,   245,
    34,    35,    36,    37,    38,    39,    40,    41,     4,    10,
    11,   226,   225,   224,    45,    43,    44,    46,    47,    48,
    49,    50,    51,    52,    53,    54,    55,   223,   222,   221,
   196,   211,   181,   273,   139,   328,   154,   155,    14,    15,
    16,    17,    18,   135,   134,    19,    20,    21,    22,    23,
    24,    25,    42,    26,    27,    28,    29,   133,    30,    31,
    32,    33,    56,    34,    35,    36,    37,    38,    39,    40,
    41,   149,    10,    11,   168,   171,   195,    45,    43,    44,
    46,    47,    48,    49,    50,    51,    52,    53,    54,    55,
    87,   147,     9,    69,    71,    78,    65,   199,   142,    98,
    66,   203,   169,   194,   192,    88,    89,    93,    90,    91,
   101,   100,   102,   231,   186,   162,   200,   256,    84,    85,
   101,   100,   102,    96,   202,    94,   198,   228,   189,    99,
   185,   188,    70,    72,    73,    74,    76,    77,    75,    68,
   252,    97,    92,    95,    87,   187,    94,    69,    71,    78,
    65,   324,     6,    98,    66,    81,   158,   323,   322,    88,
    89,   229,    90,    91,   315,   314,   313,   232,   235,   312,
   311,   206,    84,    85,   101,   100,   102,    96,   310,    94,
   309,   229,   308,    99,   233,   234,    70,    72,    73,    74,
    76,    77,    75,    68,   229,    97,    92,    95,   292,   290,
   126,   127,    87,   289,   164,    69,    71,    78,   251,   248,
   216,    98,   128,   129,   130,   106,   107,   215,   205,   204,
   140,   159,   163,   278,   277,   276,   161,   275,   266,   258,
    84,    85,   101,   100,   102,    96,   255,    94,   160,   132,
   349,    99,   249,   131,    70,    72,    73,    74,    76,    77,
    75,   203,   254,    97,    92,    95,    87,   244,   164,    69,
    71,    78,   246,   153,   156,    98,   165,   166,   243,   106,
   107,   242,   241,   104,   199,   227,   237,   213,   212,   210,
   209,   208,   180,   152,    84,    85,   101,   100,   102,    96,
   151,    94,   190,   200,   172,    99,   236,   148,    70,    72,
    73,    74,    76,    77,    75,   159,   163,    97,    92,    95,
   161,   145,   144,   143,   207,   240,   138,   137,   136,   247,
   122,   250,   160,   121,   253,   120,   257,   119,   118,   117,
   116,   170,   115,   114,   112,   111,   110,    61,    60,    57,
   190,    64,   164,   141,   274,   268,   236,    63,   347,   336,
   344,   341,   283,   340,   337,   320,   297,   270,   238,   214,
   218,   259,   260,   183,   113,   333,   299,   267,   271,   351,
   305,   330,   329,   317,   296,   269,   303,   150,   307,   302,
   265,   262,   201,   197,   316,   294,   157,    67,   182,   125,
    86,   193,   191,   184,   280,   281,   282,    83,    82,    80,
   124,    79,   230,   105,   123,   179,   178,   286,   287,   177,
   109,   288,   176,   175,   291,   174,   173,   167,   103,    59,
    62,    58,     3,     8,     7,     2,     1,     0,   300,   301,
     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,
     0,   146,     0,     0,     0,   105 };
int yypact[]={

  -115, -1000,  -124,   -70,   376, -1000, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000, -1000, -1000,   375,   374,
    48,   160,    -6, -1000, -1000,   373,   372,   371,   484,   370,
   369,   367,   366,   365,   364,   362,   360,   357, -1000,    93,
   -75,   -88,   -89,   355,   354,   353,   -98,   255, -1000, -1000,
 -1000, -1000,    -7,   350,   349,   348, -1000,    -7,   -38,   334,
 -1000, -1000, -1000,   106, -1000, -1000,   327,   320, -1000, -1000,
  -201,  -201,   122,  -201,  -201,   198, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000,   319,  -100,   483, -1000, -1000, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000,  -163,     8,     7, -1000,
 -1000,   -53, -1000, -1000, -1000, -1000,   -99, -1000, -1000, -1000,
 -1000,    -2,     2,   254, -1000, -1000,   253,   206, -1000,   318,
   317,   316,  -101,   315,   314,   478,   252,   245,  -161,   480,
  -164,  -167, -1000,  -103,  -104,  -105,  -119,  -120,  -121,   310,
  -165, -1000,   313,   477,    15, -1000,   309,   308,   305,   294,
  -133,   297,   244,   277,   243, -1000, -1000,   175, -1000,   289,
   273,   152, -1000,   266, -1000, -1000, -1000, -1000,  -201,   -61,
  -284, -1000, -1000, -1000,   265, -1000, -1000,   469,   500,   470,
   491, -1000, -1000, -1000, -1000, -1000, -1000, -1000,  -168, -1000,
    68, -1000,   264,   262,   261,   260,  -138, -1000, -1000, -1000,
 -1000,  -201,  -201,  -201,   -16, -1000, -1000,  -254, -1000, -1000,
  -257, -1000, -1000, -1000,  -201,  -201, -1000, -1000,  -201,   238,
   234,  -201,   233, -1000, -1000,  -151, -1000,  -169,   498,   475,
  -170,   489, -1000, -1000, -1000,  -201,  -201,  -255, -1000, -1000,
   217,   215,   213,   205, -1000, -1000,   204,   201,   200, -1000,
 -1000,   199, -1000, -1000, -1000,   497,  -171,  -172,   468,  -173,
   193,   192,   186,  -266, -1000, -1000, -1000,  -162, -1000, -1000,
 -1000, -1000, -1000, -1000, -1000, -1000,  -102,   496,   495,  -175,
  -176,   488, -1000, -1000, -1000, -1000, -1000, -1000, -1000,  -177,
 -1000,  -178, -1000, -1000,   451,   467,  -179,  -202,   466,   464,
  -211,  -216,   463, -1000,  -229,  -230,   450,  -251,   342,  -252,
   493, -1000 };
int yypgo[]={

     0,   586,   585,   312,   584,   583,   582,   581,   580,   579,
   578,   507,   433,   501,   577,   576,   575,   573,   572,   569,
   566,   565,   564,   562,   561,   560,   559,   315,   558,   557,
   553,   552,   551,   550,   503,   549,   548,   547,   546,   545,
   544,   290,   267,   543,   542,   316,   541,   275,   540,   273,
   539,   538,   536,   252,   286,   284,   287 };
int yyr1[]={

     0,     1,     1,     1,     2,     2,     2,     2,     6,     7,
     4,     9,     5,     8,     8,    10,    10,    13,    13,    15,
    13,    16,    13,    17,    13,    18,    13,    19,    13,    20,
    13,    13,    22,    13,    13,    13,    13,    13,    13,    13,
    13,    13,    13,    13,    13,    29,    29,    29,    29,    24,
    24,    34,    34,    25,    35,    35,    35,    35,    35,    35,
    35,    14,    14,    14,    14,    14,    14,    11,    11,    11,
    11,    39,    11,    40,    40,    33,    30,    30,    36,    36,
    41,    41,    41,    41,    41,    27,    28,    26,    42,    42,
    42,    12,    12,    12,    37,    37,    37,    37,    38,    38,
    45,    45,    45,    45,    45,    48,    45,    23,    23,    49,
    49,    49,    51,    49,    49,    47,    50,    50,    50,    52,
    52,    46,    46,    43,    43,    54,    54,    44,    44,    55,
    31,    31,    21,    21,    32,    32,    56,    56,     3,     3,
     3,    53,    53,    53,    53,    53,    53,    53,    53,    53,
    53,    53,    53,    53,    53,    53,    53,    53,    53,    53,
    53,    53,    53,    53,    53,    53,    53,    53,    53,    53,
    53,    53,    53,    53,    53,    53,    53,    53,    53,    53,
    53,    53,    53,    53,    53 };
int yyr2[]={

     0,     2,     5,     5,     0,     4,     4,     5,     5,     1,
    11,     1,    11,     2,     4,     2,     4,     9,     9,     1,
     9,     1,     9,     1,     9,     1,     9,     1,     9,     1,
     9,     9,     1,    11,     7,     5,     5,     5,     9,     9,
     8,     9,     9,     5,     5,     3,     7,     5,     7,     1,
     3,     1,     3,     7,     3,     3,     3,     3,     3,     3,
     3,    39,    13,    23,    13,    13,     3,     2,     9,     9,
     9,     1,    17,     0,     5,     5,     2,     4,     0,     4,
     9,     9,     9,     9,     5,     5,     5,     5,     2,     2,
     2,     2,     9,     9,     3,     3,     3,     3,     2,     4,
     9,     9,    11,     9,     5,     1,     9,     2,     4,     9,
     9,     9,     1,     9,     5,     5,     3,     3,     3,     3,
     5,     3,     3,     2,     4,     9,     9,     2,     4,     9,
     0,     7,     0,     7,     0,     7,     0,     2,     2,     2,
     3,     2,     2,     2,     2,     2,     2,     2,     2,     2,
     2,     2,     2,     2,     2,     2,     2,     2,     2,     2,
     2,     2,     2,     2,     2,     2,     2,     2,     2,     2,
     2,     2,     2,     2,     2,     2,     2,     2,     2,     2,
     2,     2,     2,     2,     2 };
int yychk[]={

 -1000,    -1,    -2,    -6,   283,   125,    -3,    -4,    -5,   -53,
   293,   294,   257,   258,   259,   260,   261,   262,   263,   266,
   267,   268,   269,   270,   271,   272,   274,   275,   276,   277,
   279,   280,   281,   282,   284,   285,   286,   287,   288,   289,
   290,   291,   273,   299,   300,   298,   301,   302,   303,   304,
   305,   306,   307,   308,   309,   310,   292,   123,    -7,    -9,
   123,   123,    -8,   -11,   -13,   262,   266,   -37,   305,   259,
   298,   260,   299,   300,   301,   304,   302,   303,   261,   -24,
   -26,   -27,   -28,   -29,   284,   285,   -33,   256,   271,   272,
   274,   275,   308,   -42,   291,   309,   289,   307,   265,   295,
   287,   286,   288,   -10,   -12,   -13,   269,   270,   125,   -11,
   123,   123,   123,    40,   123,   123,   123,   123,   123,   123,
   123,   123,   123,   -22,   -25,   -35,   267,   268,   279,   280,
   281,   310,   306,   292,   292,   292,   123,   123,   123,   292,
   125,   -34,   265,   123,   123,   123,   -34,   289,   123,   125,
   -12,   123,   123,    -3,   257,   258,    -3,   -38,   -45,   259,
   276,   264,   -47,   260,   296,    -3,    -3,   -14,    36,    64,
   293,    37,   256,   -15,   -16,   -17,   -18,   -19,   -20,   -21,
   123,   292,   -36,    40,   -30,   -41,   259,   290,   276,   273,
   -27,   -31,   256,   -32,   256,   289,   289,   -43,   -54,   259,
   278,   -44,   -55,   259,   125,   125,   125,   -45,   123,   123,
   123,   292,   123,   123,    41,   125,   125,   293,    40,   293,
   293,   292,   292,   292,   292,   292,   292,   125,   -56,    44,
   -23,   -49,   259,   276,   277,   260,   -47,   123,    41,   125,
   -41,   123,   123,   123,   123,   292,   125,   -56,   125,   125,
   -56,   125,   125,   -54,   123,   123,   125,   -55,   123,    -3,
    -3,    61,   -46,   293,   294,   -48,   123,    58,    36,    35,
    47,    37,   293,   125,   -49,   123,   123,   123,   123,   292,
    -3,    -3,    -3,   -42,   294,   294,    -3,    -3,    -3,   125,
   125,    -3,   125,   292,   -39,   293,    36,    41,   293,    37,
    -3,    -3,   -50,   -52,   293,   -53,   294,   -51,   125,   125,
   125,   125,   125,   125,   125,   125,   -40,    36,   293,   293,
    47,   293,   125,   125,   125,   294,   292,   125,   297,    36,
    36,   293,   293,    37,   293,   293,    58,    47,   293,   293,
    47,    47,   293,   293,    47,   293,   293,    58,   293,    58,
   293,    36 };
int yydef[]={

     4,    -2,     1,     0,     0,     2,     3,     5,     6,   138,
   139,   140,    -2,    -2,   143,   144,   145,   146,   147,   148,
   149,   150,   151,   152,   153,   154,   155,   156,   157,   158,
   159,   160,   161,   162,   163,   164,   165,   166,   167,   168,
   169,   170,   171,   172,   173,   174,   175,   176,   177,   178,
   179,   180,   181,   182,   183,   184,     7,     8,     0,     0,
    -2,    -2,    -2,    13,    67,     0,     0,     0,     0,     0,
     0,     0,     0,     0,     0,     0,     0,     0,    32,     0,
     0,     0,     0,     0,     0,     0,     0,     0,    94,    95,
    96,    97,    -2,     0,     0,     0,    45,    51,     0,     0,
    88,    89,    90,    -2,    15,    91,     0,     0,    10,    14,
     0,     0,     0,     0,     0,     0,    19,    21,    23,    25,
    27,    29,   132,     0,     0,    78,    54,    55,    56,    57,
    58,    59,    60,    35,    36,    37,     0,    -2,    -2,    43,
    44,     0,    52,    87,    85,    86,     0,    47,    75,    12,
    16,     0,     0,     0,   141,   142,     0,     0,    98,     0,
     0,     0,     0,     0,     0,     0,     0,     0,     0,     0,
     0,     0,    66,     0,     0,     0,     0,     0,     0,   136,
     0,    34,     0,     0,     0,    76,     0,     0,     0,     0,
     0,   136,     0,   136,     0,    48,    46,     0,   123,     0,
     0,     0,   127,     0,    68,    69,    70,    99,     0,     0,
     0,   104,   105,   115,     0,    17,    18,     0,     0,     0,
     0,    20,    22,    24,    26,    28,    30,    31,     0,   137,
     0,   107,     0,     0,     0,     0,     0,    53,    79,    38,
    77,     0,     0,     0,     0,    84,    39,     0,    41,    40,
     0,    42,    92,   124,     0,     0,    93,   128,     0,     0,
     0,     0,     0,   121,   122,     0,    71,     0,     0,     0,
     0,     0,   133,    33,   108,     0,     0,     0,   112,   114,
     0,     0,     0,     0,   131,   135,     0,     0,     0,   100,
   101,     0,   103,   106,    73,     0,     0,     0,     0,     0,
     0,     0,     0,   116,   117,   118,   119,     0,    80,    81,
    82,    83,   125,   126,   129,   102,     0,     0,     0,     0,
     0,     0,   109,   110,   111,   120,   113,    72,    74,     0,
    62,     0,    64,    65,     0,     0,     0,     0,     0,     0,
     0,     0,     0,    63,     0,     0,     0,     0,     0,     0,
     0,    61 };
typedef struct { char *t_name; int t_val; } yytoktype;
#ifndef YYDEBUG
#	define YYDEBUG	0	/* don't allow debugging */
#endif

#if YYDEBUG

yytoktype yytoks[] =
{
	"DEFSTAR",	257,
	"GALAXY",	258,
	"NAME",	259,
	"DESC",	260,
	"DEFSTATE",	261,
	"DOMAIN",	262,
	"NUMPORTS",	263,
	"NUM",	264,
	"VIRTUAL",	265,
	"DERIVED",	266,
	"CONSTRUCTOR",	267,
	"DESTRUCTOR",	268,
	"STAR",	269,
	"ALIAS",	270,
	"OUTPUT",	271,
	"INPUT",	272,
	"ACCESS",	273,
	"OUTMULTI",	274,
	"INMULTI",	275,
	"TYPE",	276,
	"DEFAULT",	277,
	"CLASS",	278,
	"SETUP",	279,
	"GO",	280,
	"WRAPUP",	281,
	"CONNECT",	282,
	"ID",	283,
	"CCINCLUDE",	284,
	"HINCLUDE",	285,
	"PROTECTED",	286,
	"PUBLIC",	287,
	"PRIVATE",	288,
	"METHOD",	289,
	"ARGLIST",	290,
	"CODE",	291,
	"BODY",	292,
	"IDENTIFIER",	293,
	"STRING",	294,
	"CONSCALLS",	295,
	"ATTRIB",	296,
	"LINE",	297,
	"VERSION",	298,
	"AUTHOR",	299,
	"ACKNOWLEDGE",	300,
	"COPYRIGHT",	301,
	"EXPLANATION",	302,
	"SEEALSO",	303,
	"LOCATION",	304,
	"CODEBLOCK",	305,
	"EXECTIME",	306,
	"PURE",	307,
	"INLINE",	308,
	"HEADER",	309,
	"INITCODE",	310,
	"-unknown-",	-1	/* ends search */
};

char * yyreds[] =
{
	"-no such reduction-",
	"full_file : file",
	"full_file : file '}'",
	"full_file : file ident",
	"file : /* empty */",
	"file : file stardef",
	"file : file galdef",
	"file : id BODY",
	"id : ID '{'",
	"stardef : DEFSTAR",
	"stardef : DEFSTAR '{' starlist '}'",
	"galdef : GALAXY",
	"galdef : GALAXY '{' gallist '}'",
	"starlist : staritem",
	"starlist : starlist staritem",
	"gallist : galitem",
	"gallist : gallist galitem",
	"sgitem : NAME '{' ident '}'",
	"sgitem : VERSION '{' version '}'",
	"sgitem : DESC '{'",
	"sgitem : DESC '{' BODY",
	"sgitem : AUTHOR '{'",
	"sgitem : AUTHOR '{' BODY",
	"sgitem : ACKNOWLEDGE '{'",
	"sgitem : ACKNOWLEDGE '{' BODY",
	"sgitem : COPYRIGHT '{'",
	"sgitem : COPYRIGHT '{' BODY",
	"sgitem : LOCATION '{'",
	"sgitem : LOCATION '{' BODY",
	"sgitem : EXPLANATION '{'",
	"sgitem : EXPLANATION '{' BODY",
	"sgitem : SEEALSO '{' seealso '}'",
	"sgitem : DEFSTATE",
	"sgitem : DEFSTATE '{' dstatelist '}'",
	"sgitem : inl stdmethkey BODY",
	"sgitem : members BODY",
	"sgitem : code BODY",
	"sgitem : header BODY",
	"sgitem : method '{' methlist '}'",
	"sgitem : CCINCLUDE '{' cclist '}'",
	"sgitem : HINCLUDE '{' hlist '}'",
	"sgitem : CCINCLUDE '{' error '}'",
	"sgitem : HINCLUDE '{' error '}'",
	"sgitem : conscalls BODY",
	"sgitem : error '}'",
	"method : METHOD",
	"method : PURE vopt METHOD",
	"method : VIRTUAL METHOD",
	"method : INLINE vopt METHOD",
	"inl : /* empty */",
	"inl : INLINE",
	"vopt : /* empty */",
	"vopt : VIRTUAL",
	"stdmethkey : stdkey2 optp '{'",
	"stdkey2 : CONSTRUCTOR",
	"stdkey2 : DESTRUCTOR",
	"stdkey2 : SETUP",
	"stdkey2 : GO",
	"stdkey2 : WRAPUP",
	"stdkey2 : INITCODE",
	"stdkey2 : EXECTIME",
	"version : '$' IDENTIFIER ':' IDENTIFIER '$' '$' IDENTIFIER ':' IDENTIFIER '/' IDENTIFIER '/' IDENTIFIER IDENTIFIER ':' IDENTIFIER ':' IDENTIFIER '$'",
	"version : '$' IDENTIFIER '$' '$' IDENTIFIER '$'",
	"version : '@' '(' '#' ')' IDENTIFIER IDENTIFIER IDENTIFIER '/' IDENTIFIER '/' IDENTIFIER",
	"version : IDENTIFIER IDENTIFIER '/' IDENTIFIER '/' IDENTIFIER",
	"version : '%' IDENTIFIER '%' '%' IDENTIFIER '%'",
	"version : error",
	"staritem : sgitem",
	"staritem : DOMAIN '{' ident '}'",
	"staritem : DERIVED '{' ident '}'",
	"staritem : portkey '{' portlist '}'",
	"staritem : CODEBLOCK '(' ident ')' '{'",
	"staritem : CODEBLOCK '(' ident ')' '{' lines '}'",
	"lines : /* empty */",
	"lines : lines LINE",
	"conscalls : CONSCALLS '{'",
	"methlist : methitem",
	"methlist : methlist methitem",
	"optp : /* empty */",
	"optp : '(' ')'",
	"methitem : NAME '{' ident '}'",
	"methitem : ARGLIST '{' ident '}'",
	"methitem : TYPE '{' ident '}'",
	"methitem : ACCESS '{' protkey '}'",
	"methitem : code BODY",
	"code : CODE '{'",
	"header : HEADER '{'",
	"members : protkey '{'",
	"protkey : PUBLIC",
	"protkey : PROTECTED",
	"protkey : PRIVATE",
	"galitem : sgitem",
	"galitem : STAR '{' starinstlist '}'",
	"galitem : ALIAS '{' aliaslist '}'",
	"portkey : OUTPUT",
	"portkey : INPUT",
	"portkey : OUTMULTI",
	"portkey : INMULTI",
	"portlist : portitem",
	"portlist : portlist portitem",
	"portitem : NAME '{' ident '}'",
	"portitem : TYPE '{' ident '}'",
	"portitem : TYPE '{' '=' ident '}'",
	"portitem : NUM '{' expval '}'",
	"portitem : attrib BODY",
	"portitem : DESC '{'",
	"portitem : DESC '{' BODY",
	"dstatelist : dstateitem",
	"dstatelist : dstatelist dstateitem",
	"dstateitem : NAME '{' ident '}'",
	"dstateitem : TYPE '{' ident '}'",
	"dstateitem : DEFAULT '{' defval '}'",
	"dstateitem : DESC '{'",
	"dstateitem : DESC '{' BODY",
	"dstateitem : attrib BODY",
	"attrib : ATTRIB '{'",
	"defval : stringseq",
	"defval : IDENTIFIER",
	"defval : keyword",
	"stringseq : STRING",
	"stringseq : stringseq STRING",
	"expval : IDENTIFIER",
	"expval : STRING",
	"starinstlist : starinstitem",
	"starinstlist : starinstlist starinstitem",
	"starinstitem : NAME '{' ident '}'",
	"starinstitem : CLASS '{' ident '}'",
	"aliaslist : aliasitem",
	"aliaslist : aliaslist aliasitem",
	"aliasitem : NAME '{' ident '}'",
	"cclist : /* empty */",
	"cclist : cclist optcomma STRING",
	"seealso : /* empty */",
	"seealso : seealso optcomma IDENTIFIER",
	"hlist : /* empty */",
	"hlist : hlist optcomma STRING",
	"optcomma : /* empty */",
	"optcomma : ','",
	"ident : keyword",
	"ident : IDENTIFIER",
	"ident : STRING",
	"keyword : DEFSTAR",
	"keyword : GALAXY",
	"keyword : NAME",
	"keyword : DESC",
	"keyword : DEFSTATE",
	"keyword : DOMAIN",
	"keyword : NUMPORTS",
	"keyword : DERIVED",
	"keyword : CONSTRUCTOR",
	"keyword : DESTRUCTOR",
	"keyword : STAR",
	"keyword : ALIAS",
	"keyword : OUTPUT",
	"keyword : INPUT",
	"keyword : OUTMULTI",
	"keyword : INMULTI",
	"keyword : TYPE",
	"keyword : DEFAULT",
	"keyword : SETUP",
	"keyword : GO",
	"keyword : WRAPUP",
	"keyword : CONNECT",
	"keyword : CCINCLUDE",
	"keyword : HINCLUDE",
	"keyword : PROTECTED",
	"keyword : PUBLIC",
	"keyword : PRIVATE",
	"keyword : METHOD",
	"keyword : ARGLIST",
	"keyword : CODE",
	"keyword : ACCESS",
	"keyword : AUTHOR",
	"keyword : ACKNOWLEDGE",
	"keyword : VERSION",
	"keyword : COPYRIGHT",
	"keyword : EXPLANATION",
	"keyword : SEEALSO",
	"keyword : LOCATION",
	"keyword : CODEBLOCK",
	"keyword : EXECTIME",
	"keyword : PURE",
	"keyword : INLINE",
	"keyword : HEADER",
	"keyword : INITCODE",
};
#endif /* YYDEBUG */
#line 1 "/usr/lib/yaccpar"
/*	@(#)yaccpar 1.10 89/04/04 SMI; from S5R3 1.10	*/

/*
** Skeleton parser driver for yacc output
*/

/*
** yacc user known macros and defines
*/
#define YYERROR		goto yyerrlab
#define YYACCEPT	{ free(yys); free(yyv); return(0); }
#define YYABORT		{ free(yys); free(yyv); return(1); }
#define YYBACKUP( newtoken, newvalue )\
{\
	if ( yychar >= 0 || ( yyr2[ yytmp ] >> 1 ) != 1 )\
	{\
		yyerror( "syntax error - cannot backup" );\
		goto yyerrlab;\
	}\
	yychar = newtoken;\
	yystate = *yyps;\
	yylval = newvalue;\
	goto yynewstate;\
}
#define YYRECOVERING()	(!!yyerrflag)
#ifndef YYDEBUG
#	define YYDEBUG	1	/* make debugging available */
#endif

/*
** user known globals
*/
int yydebug;			/* set to 1 to get debugging */

/*
** driver internal defines
*/
#define YYFLAG		(-1000)

/*
** static variables used by the parser
*/
static YYSTYPE *yyv;			/* value stack */
static int *yys;			/* state stack */

static YYSTYPE *yypv;			/* top of value stack */
static int *yyps;			/* top of state stack */

static int yystate;			/* current state */
static int yytmp;			/* extra var (lasts between blocks) */

int yynerrs;			/* number of errors */

int yyerrflag;			/* error recovery flag */
int yychar;			/* current input token number */


/*
** yyparse - return 0 if worked, 1 if syntax error not recovered from
*/
int
yyparse()
{
	register YYSTYPE *yypvt;	/* top of value stack for $vars */
	unsigned yymaxdepth = YYMAXDEPTH;

	/*
	** Initialize externals - yyparse may be called more than once
	*/
	yyv = (YYSTYPE*)malloc(yymaxdepth*sizeof(YYSTYPE));
	yys = (int*)malloc(yymaxdepth*sizeof(int));
	if (!yyv || !yys)
	{
		yyerror( "out of memory" );
		return(1);
	}
	yypv = &yyv[-1];
	yyps = &yys[-1];
	yystate = 0;
	yytmp = 0;
	yynerrs = 0;
	yyerrflag = 0;
	yychar = -1;

	goto yystack;
	{
		register YYSTYPE *yy_pv;	/* top of value stack */
		register int *yy_ps;		/* top of state stack */
		register int yy_state;		/* current state */
		register int  yy_n;		/* internal state number info */

		/*
		** get globals into registers.
		** branch to here only if YYBACKUP was called.
		*/
	yynewstate:
		yy_pv = yypv;
		yy_ps = yyps;
		yy_state = yystate;
		goto yy_newstate;

		/*
		** get globals into registers.
		** either we just started, or we just finished a reduction
		*/
	yystack:
		yy_pv = yypv;
		yy_ps = yyps;
		yy_state = yystate;

		/*
		** top of for (;;) loop while no reductions done
		*/
	yy_stack:
		/*
		** put a state and value onto the stacks
		*/
#if YYDEBUG
		/*
		** if debugging, look up token value in list of value vs.
		** name pairs.  0 and negative (-1) are special values.
		** Note: linear search is used since time is not a real
		** consideration while debugging.
		*/
		if ( yydebug )
		{
			register int yy_i;

			(void)printf( "State %d, token ", yy_state );
			if ( yychar == 0 )
				(void)printf( "end-of-file\n" );
			else if ( yychar < 0 )
				(void)printf( "-none-\n" );
			else
			{
				for ( yy_i = 0; yytoks[yy_i].t_val >= 0;
					yy_i++ )
				{
					if ( yytoks[yy_i].t_val == yychar )
						break;
				}
				(void)printf( "%s\n", yytoks[yy_i].t_name );
			}
		}
#endif /* YYDEBUG */
		if ( ++yy_ps >= &yys[ yymaxdepth ] )	/* room on stack? */
		{
			/*
			** reallocate and recover.  Note that pointers
			** have to be reset, or bad things will happen
			*/
			int yyps_index = (yy_ps - yys);
			int yypv_index = (yy_pv - yyv);
			int yypvt_index = (yypvt - yyv);
			yymaxdepth += YYMAXDEPTH;
			yyv = (YYSTYPE*)realloc((char*)yyv,
				yymaxdepth * sizeof(YYSTYPE));
			yys = (int*)realloc((char*)yys,
				yymaxdepth * sizeof(int));
			if (!yyv || !yys)
			{
				yyerror( "yacc stack overflow" );
				return(1);
			}
			yy_ps = yys + yyps_index;
			yy_pv = yyv + yypv_index;
			yypvt = yyv + yypvt_index;
		}
		*yy_ps = yy_state;
		*++yy_pv = yyval;

		/*
		** we have a new state - find out what to do
		*/
	yy_newstate:
		if ( ( yy_n = yypact[ yy_state ] ) <= YYFLAG )
			goto yydefault;		/* simple state */
#if YYDEBUG
		/*
		** if debugging, need to mark whether new token grabbed
		*/
		yytmp = yychar < 0;
#endif
		if ( ( yychar < 0 ) && ( ( yychar = yylex() ) < 0 ) )
			yychar = 0;		/* reached EOF */
#if YYDEBUG
		if ( yydebug && yytmp )
		{
			register int yy_i;

			(void)printf( "Received token " );
			if ( yychar == 0 )
				(void)printf( "end-of-file\n" );
			else if ( yychar < 0 )
				(void)printf( "-none-\n" );
			else
			{
				for ( yy_i = 0; yytoks[yy_i].t_val >= 0;
					yy_i++ )
				{
					if ( yytoks[yy_i].t_val == yychar )
						break;
				}
				(void)printf( "%s\n", yytoks[yy_i].t_name );
			}
		}
#endif /* YYDEBUG */
		if ( ( ( yy_n += yychar ) < 0 ) || ( yy_n >= YYLAST ) )
			goto yydefault;
		if ( yychk[ yy_n = yyact[ yy_n ] ] == yychar )	/*valid shift*/
		{
			yychar = -1;
			yyval = yylval;
			yy_state = yy_n;
			if ( yyerrflag > 0 )
				yyerrflag--;
			goto yy_stack;
		}

	yydefault:
		if ( ( yy_n = yydef[ yy_state ] ) == -2 )
		{
#if YYDEBUG
			yytmp = yychar < 0;
#endif
			if ( ( yychar < 0 ) && ( ( yychar = yylex() ) < 0 ) )
				yychar = 0;		/* reached EOF */
#if YYDEBUG
			if ( yydebug && yytmp )
			{
				register int yy_i;

				(void)printf( "Received token " );
				if ( yychar == 0 )
					(void)printf( "end-of-file\n" );
				else if ( yychar < 0 )
					(void)printf( "-none-\n" );
				else
				{
					for ( yy_i = 0;
						yytoks[yy_i].t_val >= 0;
						yy_i++ )
					{
						if ( yytoks[yy_i].t_val
							== yychar )
						{
							break;
						}
					}
					(void)printf( "%s\n", yytoks[yy_i].t_name );
				}
			}
#endif /* YYDEBUG */
			/*
			** look through exception table
			*/
			{
				register int *yyxi = yyexca;

				while ( ( *yyxi != -1 ) ||
					( yyxi[1] != yy_state ) )
				{
					yyxi += 2;
				}
				while ( ( *(yyxi += 2) >= 0 ) &&
					( *yyxi != yychar ) )
					;
				if ( ( yy_n = yyxi[1] ) < 0 )
					YYACCEPT;
			}
		}

		/*
		** check for syntax error
		*/
		if ( yy_n == 0 )	/* have an error */
		{
			/* no worry about speed here! */
			switch ( yyerrflag )
			{
			case 0:		/* new error */
				yyerror( "syntax error" );
				goto skip_init;
			yyerrlab:
				/*
				** get globals into registers.
				** we have a user generated syntax type error
				*/
				yy_pv = yypv;
				yy_ps = yyps;
				yy_state = yystate;
				yynerrs++;
			skip_init:
			case 1:
			case 2:		/* incompletely recovered error */
					/* try again... */
				yyerrflag = 3;
				/*
				** find state where "error" is a legal
				** shift action
				*/
				while ( yy_ps >= yys )
				{
					yy_n = yypact[ *yy_ps ] + YYERRCODE;
					if ( yy_n >= 0 && yy_n < YYLAST &&
						yychk[yyact[yy_n]] == YYERRCODE)					{
						/*
						** simulate shift of "error"
						*/
						yy_state = yyact[ yy_n ];
						goto yy_stack;
					}
					/*
					** current state has no shift on
					** "error", pop stack
					*/
#if YYDEBUG
#	define _POP_ "Error recovery pops state %d, uncovers state %d\n"
					if ( yydebug )
						(void)printf( _POP_, *yy_ps,
							yy_ps[-1] );
#	undef _POP_
#endif
					yy_ps--;
					yy_pv--;
				}
				/*
				** there is no state on stack with "error" as
				** a valid shift.  give up.
				*/
				YYABORT;
			case 3:		/* no shift yet; eat a token */
#if YYDEBUG
				/*
				** if debugging, look up token in list of
				** pairs.  0 and negative shouldn't occur,
				** but since timing doesn't matter when
				** debugging, it doesn't hurt to leave the
				** tests here.
				*/
				if ( yydebug )
				{
					register int yy_i;

					(void)printf( "Error recovery discards " );
					if ( yychar == 0 )
						(void)printf( "token end-of-file\n" );
					else if ( yychar < 0 )
						(void)printf( "token -none-\n" );
					else
					{
						for ( yy_i = 0;
							yytoks[yy_i].t_val >= 0;
							yy_i++ )
						{
							if ( yytoks[yy_i].t_val
								== yychar )
							{
								break;
							}
						}
						(void)printf( "token %s\n",
							yytoks[yy_i].t_name );
					}
				}
#endif /* YYDEBUG */
				if ( yychar == 0 )	/* reached EOF. quit */
					YYABORT;
				yychar = -1;
				goto yy_newstate;
			}
		}/* end if ( yy_n == 0 ) */
		/*
		** reduction by production yy_n
		** put stack tops, etc. so things right after switch
		*/
#if YYDEBUG
		/*
		** if debugging, print the string that is the user's
		** specification of the reduction which is just about
		** to be done.
		*/
		if ( yydebug )
			(void)printf( "Reduce by (%d) \"%s\"\n",
				yy_n, yyreds[ yy_n ] );
#endif
		yytmp = yy_n;			/* value to switch over */
		yypvt = yy_pv;			/* $vars top of value stack */
		/*
		** Look in goto table for next state
		** Sorry about using yy_state here as temporary
		** register variable, but why not, if it works...
		** If yyr2[ yy_n ] doesn't have the low order bit
		** set, then there is no action to be done for
		** this reduction.  So, no saving & unsaving of
		** registers done.  The only difference between the
		** code just after the if and the body of the if is
		** the goto yy_stack in the body.  This way the test
		** can be made before the choice of what to do is needed.
		*/
		{
			/* length of production doubled with extra bit */
			register int yy_len = yyr2[ yy_n ];

			if ( !( yy_len & 01 ) )
			{
				yy_len >>= 1;
				yyval = ( yy_pv -= yy_len )[1];	/* $$ = $1 */
				yy_state = yypgo[ yy_n = yyr1[ yy_n ] ] +
					*( yy_ps -= yy_len ) + 1;
				if ( yy_state >= YYLAST ||
					yychk[ yy_state =
					yyact[ yy_state ] ] != -yy_n )
				{
					yy_state = yyact[ yypgo[ yy_n ] ];
				}
				goto yy_stack;
			}
			yy_len >>= 1;
			yyval = ( yy_pv -= yy_len )[1];	/* $$ = $1 */
			yy_state = yypgo[ yy_n = yyr1[ yy_n ] ] +
				*( yy_ps -= yy_len ) + 1;
			if ( yy_state >= YYLAST ||
				yychk[ yy_state = yyact[ yy_state ] ] != -yy_n )
			{
				yy_state = yyact[ yypgo[ yy_n ] ];
			}
		}
					/* save until reenter driver code */
		yystate = yy_state;
		yyps = yy_ps;
		yypv = yy_pv;
	}
	/*
	** code supplied by user is placed in this switch
	*/
	switch( yytmp )
	{
		
case 2:
# line 233 "ptlang.y"
{ yyerror("Too many closing curly braces");
			  exit(1);
			} break;
case 3:
# line 236 "ptlang.y"
{ mismatch(yypvt[-0]);} break;
case 7:
# line 244 "ptlang.y"
{ idBlock = yypvt[-0]; bodyMode = 0;} break;
case 8:
# line 247 "ptlang.y"
{ bodyMode = 1;} break;
case 9:
# line 251 "ptlang.y"
{ clearDefs(0);} break;
case 10:
# line 252 "ptlang.y"
{ genDef();} break;
case 11:
# line 255 "ptlang.y"
{ clearDefs(1);} break;
case 12:
# line 256 "ptlang.y"
{ genDef();} break;
case 17:
# line 269 "ptlang.y"
{ objName = yypvt[-1];} break;
case 18:
# line 270 "ptlang.y"
{ } break;
case 19:
# line 271 "ptlang.y"
{ descMode = 1; docMode = 1;} break;
case 20:
# line 272 "ptlang.y"
{ objDesc = yypvt[-0];
					  docMode = 0;
					  descMode = 0;} break;
case 21:
# line 275 "ptlang.y"
{ bodyMode = 1; docMode = 1;} break;
case 22:
# line 276 "ptlang.y"
{ objAuthor = yypvt[-0];
					  docMode = 0;
					  bodyMode = 0;} break;
case 23:
# line 279 "ptlang.y"
{ bodyMode = 1; docMode = 1;} break;
case 24:
# line 280 "ptlang.y"
{ objAcknowledge = yypvt[-0];
					  docMode = 0;
					  bodyMode = 0;} break;
case 25:
# line 283 "ptlang.y"
{ bodyMode = 1; docMode = 1;} break;
case 26:
# line 284 "ptlang.y"
{ objCopyright = yypvt[-0];
					  docMode = 0;
					  bodyMode = 0;} break;
case 27:
# line 287 "ptlang.y"
{ bodyMode = 1; docMode = 1;} break;
case 28:
# line 288 "ptlang.y"
{ objLocation = yypvt[-0];
					  docMode = 0;
					  bodyMode = 0;} break;
case 29:
# line 291 "ptlang.y"
{ bodyMode = 1; docMode = 1;} break;
case 30:
# line 292 "ptlang.y"
{ objExpl = yypvt[-0];
					  bodyMode = 0;
					  docMode = 0;} break;
case 31:
# line 295 "ptlang.y"
{ } break;
case 32:
# line 296 "ptlang.y"
{ clearStateDefs();} break;
case 33:
# line 297 "ptlang.y"
{ genState(); describeState();} break;
case 34:
# line 300 "ptlang.y"
{ inlineFlag[methKey] = yypvt[-2] ? 1 : 0;
					  codeBody[methKey] = yypvt[-0];
					  bodyMode = 0;
					} break;
case 35:
# line 305 "ptlang.y"
{ addMembers (yypvt[-1], yypvt[-0]); bodyMode = 0;} break;
case 36:
# line 306 "ptlang.y"
{ strcat (ccCode, yypvt[-0]); 
					  strcat (ccCode, "\n\n");
					  bodyMode = 0;
					} break;
case 37:
# line 310 "ptlang.y"
{ strcat (hCode, yypvt[-0]);
					  strcat (hCode, "\n");
					  bodyMode = 0;
					} break;
case 38:
# line 314 "ptlang.y"
{ genMethod();} break;
case 39:
# line 315 "ptlang.y"
{ } break;
case 41:
# line 317 "ptlang.y"
{ yyerror("Error in ccinclude list");} break;
case 42:
# line 318 "ptlang.y"
{ yyerror("Error in hinclude list");} break;
case 43:
# line 319 "ptlang.y"
{ if(consCalls[0]) {
					     strcat(consCalls,", ");
					     strcat(consCalls,yypvt[-0]);
					  } else {
					     strcpy(consCalls,yypvt[-0]);
					  }
					  bodyMode = 0; 
					} break;
case 44:
# line 327 "ptlang.y"
{ yyerror ("Illegal item");} break;
case 45:
# line 332 "ptlang.y"
{ clearMethodDefs(0);} break;
case 46:
# line 333 "ptlang.y"
{ clearMethodDefs(M_PURE);} break;
case 47:
# line 334 "ptlang.y"
{ clearMethodDefs(M_VIRTUAL);} break;
case 48:
# line 335 "ptlang.y"
{ int mode = M_INLINE;
					  if (yypvt[-1]) mode |= M_VIRTUAL;
					  clearMethodDefs(mode);
					} break;
case 49:
# line 342 "ptlang.y"
{ yyval = 0;} break;
case 50:
# line 343 "ptlang.y"
{ yyval = yypvt[-0];} break;
case 51:
# line 347 "ptlang.y"
{ yyval = 0;} break;
case 52:
# line 348 "ptlang.y"
{ yyval = yypvt[-0];} break;
case 53:
# line 353 "ptlang.y"
{ bodyMode = 1;} break;
case 54:
# line 357 "ptlang.y"
{ methKey = C_CONS;} break;
case 55:
# line 358 "ptlang.y"
{ methKey = C_DEST;} break;
case 56:
# line 359 "ptlang.y"
{ methKey = C_SETUP;} break;
case 57:
# line 360 "ptlang.y"
{ methKey = C_GO;} break;
case 58:
# line 361 "ptlang.y"
{ methKey = C_WRAPUP;} break;
case 59:
# line 362 "ptlang.y"
{ methKey = C_INITCODE;} break;
case 60:
# line 363 "ptlang.y"
{ methKey = C_EXECTIME;} break;
case 61:
# line 371 "ptlang.y"
{ char b[SMALLBUFSIZE];
                  objVer = yypvt[-15];
                  sprintf(b, "\"%s/%s/%s %s:%s:%s\"", yypvt[-10], yypvt[-8], yypvt[-6], yypvt[-5], yypvt[-3], yypvt[-1]);
                  objDate = save(b);
                } break;
case 62:
# line 378 "ptlang.y"
{
                  char b[SMALLBUFSIZE];
                  long t;
                  objVer = "?.?";
                  t = time((time_t *)0);
                  b[0] = QUOTE;
                  strncat(b,ctime(&t),24);
                  strcat(b,"\"");
                  objDate = save(b);
                } break;
case 63:
# line 392 "ptlang.y"
{ char b[SMALLBUFSIZE];
		  objVer = yypvt[-5];
		  sprintf(b, "\"%s/%s/%s\"", yypvt[-4], yypvt[-2], yypvt[-0]);
		  objDate = save(b);
		} break;
case 64:
# line 398 "ptlang.y"
{ char b[SMALLBUFSIZE];
		  objVer = yypvt[-5];
		  sprintf(b, "\"%s/%s/%s\"", yypvt[-4], yypvt[-2], yypvt[-0]);
		  objDate = save(b);
		} break;
case 65:
# line 404 "ptlang.y"
{
		  char b[SMALLBUFSIZE];
		  long t;
		  objVer = "?.?";
		  t = time((time_t *)0);
		  b[0] = QUOTE;
		  strncat(b,ctime(&t),24);
		  strcat(b,"\"");
		  objDate = save(b);
		  /* objDate = "\"checked out\""; */
		} break;
case 66:
# line 415 "ptlang.y"
{ yyerror("Illegal version format");} break;
case 68:
# line 422 "ptlang.y"
{ domain = yypvt[-1];} break;
case 69:
# line 423 "ptlang.y"
{ derivedFrom = yypvt[-1];} break;
case 70:
# line 424 "ptlang.y"
{ genPort();
					  describePort(); } break;
case 71:
# line 427 "ptlang.y"
{ char* b = malloc(SMALLBUFSIZE);
					  blockID = yypvt[-2];
					  strcpy(b,blockID);
					  blockNames[numBlocks]=b;
					  codeMode = 1;
					  codeModeBraceCount = 1;
					} break;
case 72:
# line 434 "ptlang.y"
{ char* b = save(codeBlock);
					  codeBlocks[numBlocks++]=b;
					  codeMode = 0; 
					  codeBlock[0] = 0;
					} break;
case 74:
# line 442 "ptlang.y"
{ char b[SMALLBUFSIZE];
		sprintf(b,"\"%s\\n\"\n",yypvt[-0]);
		strcat(codeBlock,b);
				} break;
case 75:
# line 449 "ptlang.y"
{ bodyMode = 1;} break;
case 80:
# line 463 "ptlang.y"
{ methodName = yypvt[-1];} break;
case 81:
# line 464 "ptlang.y"
{ methodArgs = checkArgs(yypvt[-1]);} break;
case 82:
# line 465 "ptlang.y"
{ methodType = yypvt[-1];} break;
case 83:
# line 466 "ptlang.y"
{ methodAccess = yypvt[-1];} break;
case 84:
# line 467 "ptlang.y"
{ methodCode = yypvt[-0]; bodyMode = 0;} break;
case 85:
# line 472 "ptlang.y"
{ bodyMode = 1;} break;
case 86:
# line 477 "ptlang.y"
{ bodyMode = 1;} break;
case 87:
# line 482 "ptlang.y"
{ bodyMode = 1; yyval = yypvt[-1];} break;
case 92:
# line 495 "ptlang.y"
{ genInstance();} break;
case 93:
# line 496 "ptlang.y"
{ genAlias();} break;
case 94:
# line 504 "ptlang.y"
{ initPort(1,0);} break;
case 95:
# line 505 "ptlang.y"
{ initPort(0,0);} break;
case 96:
# line 506 "ptlang.y"
{ initPort(1,1);} break;
case 97:
# line 507 "ptlang.y"
{ initPort(0,1);} break;
case 100:
# line 516 "ptlang.y"
{ portName = yypvt[-1];} break;
case 101:
# line 517 "ptlang.y"
{ portType = portDataType(yypvt[-1]);} break;
case 102:
# line 518 "ptlang.y"
{ portInherit = yypvt[-1];} break;
case 103:
# line 519 "ptlang.y"
{ portNum = yypvt[-1];} break;
case 104:
# line 520 "ptlang.y"
{ portAttrib = yypvt[-0]; bodyMode = 0;} break;
case 105:
# line 521 "ptlang.y"
{ descMode = 1; docMode = 1;} break;
case 106:
# line 521 "ptlang.y"
{ portDesc = yypvt[-0]; docMode = 0;
					  descMode = 0;} break;
case 109:
# line 532 "ptlang.y"
{ stateName = yypvt[-1];} break;
case 110:
# line 533 "ptlang.y"
{ int tc = stateTypeClass(yypvt[-1]);
					  stateMarks[tc]++;
					  stateClass = stateClasses[tc];
					} break;
case 111:
# line 537 "ptlang.y"
{ stateDef = yypvt[-1];} break;
case 112:
# line 538 "ptlang.y"
{ descMode = 1; docMode = 1;} break;
case 113:
# line 539 "ptlang.y"
{ stateDesc = yypvt[-0];
					  docMode = 0;
					  descMode = 0;} break;
case 114:
# line 542 "ptlang.y"
{ stateAttrib = yypvt[-0]; bodyMode=0;} break;
case 115:
# line 545 "ptlang.y"
{ bodyMode=1;} break;
case 116:
# line 548 "ptlang.y"
{ yyval = yypvt[-0];} break;
case 117:
# line 549 "ptlang.y"
{ char b[SMALLBUFSIZE];
					  sprintf (b, "\"%s\"", yypvt[-0]);
					  yyval = save(b);
					} break;
case 118:
# line 553 "ptlang.y"
{ char b[SMALLBUFSIZE];
					  sprintf (b, "\"%s\"", yypvt[-0]);
					  yyval = save(b);
					} break;
case 119:
# line 559 "ptlang.y"
{ yyval = save(yypvt[-0]);} break;
case 120:
# line 560 "ptlang.y"
{ char* b = malloc(MEDBUFSIZE);
					  strcpy(b,yypvt[-1]);
					  strcat(b," ");
					  strcat(b,yypvt[-0]);
					  yyval = b; } break;
case 121:
# line 568 "ptlang.y"
{ yyval = yypvt[-0];} break;
case 122:
# line 569 "ptlang.y"
{ char b[SMALLBUFSIZE];
					  strcpy (b, yypvt[-0]+1);
					  b[strlen(yypvt[-0])-2] = 0;
					  yyval = save(b);
					} break;
case 125:
# line 582 "ptlang.y"
{ instName = yypvt[-1];} break;
case 126:
# line 583 "ptlang.y"
{ instClass = yypvt[-1];} break;
case 129:
# line 592 "ptlang.y"
{ galPortName = yypvt[-1];} break;
case 131:
# line 597 "ptlang.y"
{ ccInclude[nCcInclude++] = yypvt[-0];} break;
case 133:
# line 602 "ptlang.y"
{ seeAlsoList[nSeeAlso++] = yypvt[-0];} break;
case 135:
# line 606 "ptlang.y"
{ hInclude[nHInclude++] = yypvt[-0];} break;
case 140:
# line 618 "ptlang.y"
{ yyval = stripQuotes (yypvt[-0]);} break;
	}
	goto yystack;		/* reset registers in driver code */
}
