static const char file_id[] = "Interpreter.cc";
/******************************************************************
Version identification:
@(#)Interpreter.cc	2.20	11/25/92

Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
							COPYRIGHTENDKEY

 Programmer:  J. Buck
 Date of creation: 3/19/90
 Revisions:
	5/26/90 - I. Kuroda 
		add Interpreter::state
		add Interpreter::setstate

	6/4/90 - Make a table of functions

Interpreter is the Ptolemy interpreter.  It allows the user to
 dynamically build galaxies composed of the stars known to the system
 and to schedule and run them.  It is limited to SDF universes.

*******************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "Tokenizer.h"
#include "Interpreter.h"
#include "InterpGalaxy.h"
#include "KnownBlock.h"
#include "KnownTarget.h"
#include "miscFuncs.h"
#include "Linker.h"
#include "Domain.h"
#include "textAnimate.h"
#include "SimControl.h"
#include <ACG.h>
#include <stream.h>

extern ACG* gen;

const char* programName = "PTOLEMY";

const char* startupFile = ".ptrc";

// Warning: if you're editing, the following line should look like
// "Version  % I %   % G %" without the spaces.  If it contains an
// actual version and date, you're attempting to edit an SCCS-controlled
// copy!!
static const char* version = "   Version 2.20 11/25/92";

// Construct a new interpreter.
Interpreter :: Interpreter() {
	depth=0;		// on the top level
	quitFlag = 0;
	newUniverse ();
	welcome();
	// process startup file if present.
	if (tin.fromFile(startupFile))
		say("Processing startup file ",startupFile);
	terminalInput = isatty(0);
	prompt();
};

// Construct an interpreter that reads all input from a file
// and writes all output to a stream.
// The "bottom level" input for the tokenizer is a string with
// a (quit) command in it.

Interpreter :: Interpreter (const char* fileName) : tin("(quit)\n","()") {
	depth=0;
	quitFlag = 0;
	terminalInput = 0;
	currentTarget = 0;
	if (!tin.fromFile(fileName))
		say("Interpreter: error opening file ", fileName);
	newUniverse ();
};

// Destructor
Interpreter :: ~Interpreter () {
	LOG_DEL; delete universe;
}

// Attach the interpreter to a new universe
void Interpreter :: newUniverse () {
	LOG_NEW; universe = new InterpUniverse;
	currentGalaxy = universe;
}

// Delete the universe and make another
void Interpreter :: resetUniverse () {
	LOG_DEL; delete universe;
	newUniverse ();
}

// Welcome message
void Interpreter :: welcome() {
	say("Welcome to ",programName, version);
	say("Help available: Just type \"(help)\"");
}

void Interpreter :: prompt() {
	// skip any cruft in include files
	tin.skipwhite();

	// don't prompt if reading from file
	if (tin.readingFromFile() || !terminalInput) return;

	if (depth == 0)
		cout << programName << ": ";
	else cout << "> ";
	cout.flush();
}

void Interpreter :: say(const char* s1) {
	cout << s1 << "\n";
}

void Interpreter :: say(const char* s1,const char* s2) {
	cout << s1 << s2 << "\n";
}

void Interpreter :: say(const char* s1,const char* s2,const char* s3) {
	cout << s1 << s2 << s3 << "\n";
}

static char token[TOKSIZE];

// main command loop.  We return TRUE if the last command was processed
// successfully, otherwise false.
int Interpreter :: commandRouter()
{
    while (!tin.eof()) {
	errorFlag = FALSE;
	tin >> token;
	if (*token == ')') {
// close parenth: either pop to previous level, or complain if at the top
		if (depth > 0) return !errorFlag;
		say(where(),"Unexpected ) character: on the top level");
		tin.flush();
	}
	else if (*token == '(') {
// get the verb -- look it up in the table
		tin >> token;
		dispatch (token);
		// Check for error during galaxy definition
		// also quit if (quit) executed
		if ((depth && errorFlag) || quitFlag) return !errorFlag;
	}
	// on EOF, tin >> token gives a null token
	else if (*token == 0) break;
	else {
		say (where(), "Syntax: (verb args).  Type (help) for help\n");
		tin.flush();
	}
	prompt();
    }
// EOF on input: exit the function
    return !errorFlag;
}

// descriptor: set or print the descriptor of the current galaxy

void Interpreter::descriptor() {
	char buf[512];		// allow a big descriptor
	tin >> buf;
// (descriptor) prints the descriptor
	if (*buf == ')') {
		say (currentGalaxy->descriptor());
		return;
	}
// (descriptor "text") sets the descriptor.
	else currentGalaxy->setDescriptor (hashstring(buf));
	tin >> token;
	(void)expect (")");
}

// star: create a new instance of star within galaxy
void Interpreter::star() {
	char starname[TOKSIZE], starclass[TOKSIZE];
	tin >> starname;
	if (badname(starname)) return;
	tin >> starclass;
	if (badname(starclass)) return;
	tin >> token;
	(void)expect(")");
	if(! currentGalaxy->addStar(starname,starclass))
		recover();
	return;
}

// delstar: delete a star from the galaxy
void Interpreter::delstar() {
	char starname[TOKSIZE];
	tin >> starname;
	if (badname(starname)) return;
	tin >> token;
	(void)expect(")");
	if (! currentGalaxy->delStar(starname))
		recover();
	return;
}

// connect:
void Interpreter::connect() {
	char srcstar[TOKSIZE], dststar[TOKSIZE], srcpipe[TOKSIZE], dstpipe[TOKSIZE];
	int match = 0;
	char delay[TOKSIZE];
	tin >> srcstar;
	if (*srcstar == '(') {
		tin >> srcstar;
		match=1;
	}
	if (badname(srcstar)) return;
	tin >> srcpipe;
	if (badname(srcpipe)) return;
	if (match) {
		tin >> token;
		if (!expect(")")) return;
	}
	tin >> dststar;
	if (*dststar == '(') {
		tin >> dststar;
		match=1;
	}
	if (badname(dststar)) return;
	tin >> dstpipe;
	if (badname(dstpipe)) return;
	if (match) {
		tin >> token;
		if (!expect(")")) return;
	}
// now there may be a delay argument and there may not be
	tin >> token;
	if (*token != ')') {
		strcpy (delay, token);
		tin >> token;
		if (!expect(")")) return;
	}
	else delay[0] = 0;
	if (!currentGalaxy->connect(srcstar,srcpipe,dststar,dstpipe,delay))
		recover();
}

// busconnect
void Interpreter::busconnect() {
	char srcstar[TOKSIZE], dststar[TOKSIZE], srcpipe[TOKSIZE], dstpipe[TOKSIZE];
	char delay[TOKSIZE], width[TOKSIZE];
	int match = 0;
	tin >> srcstar;
	if (*srcstar == '(') {
		tin >> srcstar;
		match=1;
	}
	if (badname(srcstar)) return;
	tin >> srcpipe;
	if (badname(srcpipe)) return;
	if (match) {
		tin >> token;
		if (!expect(")")) return;
	}
	tin >> dststar;
	if (*dststar == '(') {
		tin >> dststar;
		match=1;
	}
	if (badname(dststar)) return;
	tin >> dstpipe;
	if (badname(dstpipe)) return;
	if (match) {
		tin >> token;
		if (!expect(")")) return;
	}
// get width argument
	tin >> width;

// now there may be a delay argument and there may not be
	tin >> delay;
	if (*delay == ')') delay[0] = 0;
	else {
		tin >> token;
		if (!expect(")")) return;
	}
	if (!currentGalaxy->busConnect(srcstar,srcpipe,dststar,dstpipe,width,delay))
		recover();
}

void Interpreter::alias() {
	int match = 0;
	char galportname[TOKSIZE], starname[TOKSIZE], portname[TOKSIZE];
	tin >> galportname;
	if (badname(galportname)) return;
	tin >> starname;
	if (*starname == '(') {
		match = 1;
		tin >> starname;
	}
	if (badname (starname)) return;
	tin >> portname;
	if (badname (portname)) return;
	if (match) {
		tin >> token;
		if (!expect(")")) return;
	}
	tin >> token;
	(void)expect (")");
	if (!currentGalaxy->alias(galportname,starname,portname))
		recover();
	return;
}

void Interpreter::numports() {
	char starname[TOKSIZE], portname[TOKSIZE];
	tin >> starname;
	if (badname(starname)) return;
	tin >> portname;
	if (badname(portname)) return;
	tin >> token;
	int num = atoi (token);
	tin >> token;
	if (!expect(")")) return;
	if (!currentGalaxy->numPorts (starname, portname, num))
		recover();
}

// node: create a new node for netlist-style connections
void Interpreter::node() {
	char node[TOKSIZE];
	tin >> node;
	if (badname(node)) return;
	tin >> token;
	if (!expect(")")) return;
	if (!currentGalaxy->addNode (node))
		recover();
}

// delnode: delete a node
void Interpreter::delnode() {
	char node[TOKSIZE];
	tin >> node;
	if (badname(node)) return;
	tin >> token;
	if (!expect(")")) return;
	if (!currentGalaxy->delNode (node))
		recover();
}

// nodeconnect: connect a porthole to a node
void Interpreter::nodeconnect() {
	char srcstar[TOKSIZE], srcpipe[TOKSIZE], node[TOKSIZE], delay[TOKSIZE];
	int match = 0;
	tin >> srcstar;
	if (*srcstar == '(') {
		tin >> srcstar;
		match=1;
	}
	if (badname(srcstar)) return;
	tin >> srcpipe;
	if (badname(srcpipe)) return;
	if (match) {
		tin >> token;
		if (!expect(")")) return;
	}
	tin >> node;
	if (badname(node)) return;
// now there may be a delay argument and there may not be
	tin >> delay;
	if (*delay != ')') {
		tin >> token;
		if (!expect(")")) return;
	}
	else delay[0] = 0;
	if (!currentGalaxy->nodeConnect(srcstar,srcpipe,node,delay))
		recover();
}

// disconnect: disconnect a porthole
void Interpreter::disconnect() {
	char starname[TOKSIZE], portname[TOKSIZE];
	tin >> starname;
	if (badname(starname)) return;
	tin >> portname;
	if (badname(portname)) return;
	tin >> token;
	(void)expect(")");
	if(! currentGalaxy->disconnect(starname,portname))
		recover();
	return;
}

// state: create a new instance of state within galaxy
void Interpreter::state() {
        char statename[TOKSIZE], stateclass[TOKSIZE], statevalue[TOKSIZE*4];
        tin >> statename;
        if (badname(statename)) return;
        tin >> stateclass;
        if (badname(stateclass)) return;
        tin >> statevalue;
        if (strcmp (statevalue, ")") == 0) {
                say (where(), "Syntax error: unexpected ",statevalue);
                tin.flush();
                return ;
        }

        tin >> token;
        (void)expect(")");
	if (!currentGalaxy->addState(statename,stateclass,statevalue))
		recover();
        return;
}

// setstate: set (change) a state within galaxy
void Interpreter::setstate() {
        char blockname[TOKSIZE], statename[TOKSIZE], statevalue[TOKSIZE*4];
	tin >> blockname;
        if (badname(blockname)) return;
	tin >> statename;
        if (badname(statename)) return;
        tin >> statevalue;
        if (strcmp (statevalue, ")") == 0) {
                say (where(), "Syntax error: unexpected ",statevalue);
                tin.flush();
                return ;
        }
        
	tin >> token;
        (void)expect(")");
	if (!currentGalaxy->setState(blockname,statename,statevalue))
		recover();
        return;
}

void Interpreter::defgalaxy() {
	char galname[TOKSIZE];
	const char* outerDomain = KnownBlock::domain();
	InterpGalaxy *saveGalaxy = currentGalaxy;
	Target *saveTarget = currentTarget;
	currentTarget = 0;
	tin >> galname;
	if (badname (galname)) return;
	errorFlag = FALSE;
	LOG_NEW; currentGalaxy = new InterpGalaxy;
	currentGalaxy->setBlock (hashstring(galname), saveGalaxy);
// set my domain for currentGalaxy
	currentGalaxy->setDomain(saveGalaxy->domain());
	depth++;
// read commands, until a ')' is seen to pop us up
	commandRouter();
	if (quitFlag) return;
// now register the new galaxy definition, if no error occurred
	if (errorFlag) {
		say ("Aborting definition of galaxy ", galname);
		LOG_DEL; delete currentGalaxy;
		KnownBlock::setDomain (outerDomain);
	}
	else
		currentGalaxy->addToKnownList(outerDomain, currentTarget);
	depth--;
	currentGalaxy = saveGalaxy;
	currentTarget = saveTarget;
	return;
}

int Interpreter :: genSched() {
	universe->initTarget();
	if (SimControl::haltRequested()) {
		say ("Error setting up the schedule.");
		recover();
		return FALSE;
	}
	return TRUE;
}

void Interpreter :: schedule() {
	tin >> token;
	(void)expect (")");
	if (genSched())
		cout << universe->displaySchedule();
}

void Interpreter :: run() {
	if (genSched()) {
		stopTime = 0.0;
		lastTime = 1.0;
		cont();
	}
}

void Interpreter :: cont() {
	tin >> token;
	if (*token != ')') {
		lastTime = atof (token);
		tin >> token;
		if (!expect(")")) return;
	}
	stopTime += lastTime;
	universe->setStopTime(stopTime);
	universe->run();
}

void Interpreter :: wrapup() {
	universe->wrapup();
	tin >> token;
	(void)expect (")");
}

// quit: exit from the command router.
void Interpreter :: quit() {
	quitFlag = TRUE;
}

// domains: list domains
void Interpreter :: domains() {
	int n = Domain::number();
	for (int i = 0; i < n; i++)
		say (Domain::nthName(i));
	tin >> token;
	(void)expect (")");
}

// read the seed of the random number generation
void Interpreter :: seed() {
	int val;
	tin >> token;
	if (*token != ')') {
		val = atoi (token);
		tin >> token;
		if (!expect(")")) return;
	} else {
		val = 1;		// default seed.
	}
	LOG_DEL; delete gen;
	LOG_NEW; gen = new ACG(val);
}

void Interpreter :: animation() {
	tin >> token;
	if (*token != ')') {
		if(strcmp(token,"on") == 0)
			textAnimationOn();
		else if(strcmp(token,"off") == 0)
			textAnimationOff();
		else {
			say (where(), "Syntax error: set animation \"on\" or \"off\".");
			tin.flush();
			return;
		}
		tin >> token;
		if (!expect(")")) return;
	}
	if (textAnimationState())
		say("Animation is enabled");
	else
		say("Animation is disabled");
}
	
// initialize the states
void Interpreter :: initstate() {
	currentGalaxy->initState();
	tin >> token;
	(void)expect (")");
}

// Return a string saying where we are (for error messages)

char*
Interpreter :: where() {
	static char buf[128];
	if (tin.readingFromFile()) {
		sprintf (buf, "\"%s\", line %d: ",
			 tin.current_file(), tin.current_line());
	}
	else buf[0] = 0;
	return buf;
}

// syntax check functions

int Interpreter :: expect(const char* expect) {
	if (strcmp (token, expect) != 0) {
		say (where(), "Syntax error: expected ",expect);
		tin.flush();
		return 0;
	}
	return 1;
}

// return true, and print an error, if name is not a possible identifier
int Interpreter :: badname(const char* name) {
// for now, we'll allow wild identifiers
	if (*name == ')') {
		say (where(), "expected more arguments");
		tin.flush();
		return 1;
	}
	else if (*name == '(') {
		say (where(), name, " is not a legal identifier");
		tin.flush();
		return 1;
	}
	return 0;
}

// recover from an InterpGalaxy error.  If in a load file, we say where;
// we flush the rest of the line (or command file).
void Interpreter::recover() {
	char* msg = where();
	if (*msg) say ("Error in command file ", msg, " aborting");
	tin.flush();
	errorFlag = TRUE;
}


// Information-showing functions.

// knownlist shows the available blocks in the current domain, or (if an
// argument is given) the supplied domain.
void Interpreter::knownlist () {
	tin >> token;
	if (*token == ')') {
		cout << KnownBlock::nameList();
		return;
	}
	else {
		cout << KnownBlock::nameList(token);
		tin >> token;
		(void)expect (")");
	}
}

// (show) -- show current galaxy
// (show name) -- show indicated star
// (dump) -- show current galaxy, recursively
// (dump name) -- show indicated star or subgalaxy, recursively
void Interpreter::dump() {
	showfunc(1);
}

void Interpreter::show() {
	showfunc(0);
}

// This function is used by showfunc
static void showIt(int recursive,const Block* bp) {
	cout << bp->print(recursive);
}

// function to do show and dump commands
void Interpreter::showfunc(int recursive) {
	tin >> token;
	if (*token == ')')
		showIt (recursive, currentGalaxy);
	else {
		const Block* b = currentGalaxy->blockWithDottedName (token);
		if (!b) b = KnownBlock::find (token);
		if (b)
			showIt (recursive, b);
		else
			say (where(), "No such star: ", token);
		tin >> token;
		(void)expect (")");
	}
}

void Interpreter::reset() {
// eat the closing parenth
	tin >> token;
	if (!expect (")")) return;
	if (depth != 0)
		say (where(), "reset can only be invoked from the top level");
	else
		resetUniverse();
}

// display or change the domain.  If on top level, target always changes
// to the default for that domain.  Thus, if both a domain and a target
// are specified, domain must be first.
void Interpreter::domain() {
	char domname[TOKSIZE];
        tin >> domname;
        if (*domname == ')') {
                say (KnownBlock::domain());
                return;
        }
        tin >> token;
        if (!expect (")")) return;
	if (! currentGalaxy->setDomain (domname)) {
		recover();
		return;
	}
// make a new target if we're at the top level
	if (depth == 0) universe->newTarget ();
}

// display or change the target
void Interpreter::target() {
	char tarname[TOKSIZE];
	tin >> tarname;
	if (*tarname == ')') {
		Target* tp = depth ? currentTarget : universe->myTarget();
		cout << tp->print();
		return;
	}
	tin >> token;
	if (!expect (")")) return;

	if (depth == 0) {
		if (!universe->newTarget(hashstring(tarname)))
			recover();
		}
	else {
		// shouldn't need this test. Compiler bug?
		if (currentTarget) { LOG_DEL; delete currentTarget; }
		currentTarget = KnownTarget::clone(tarname);
		if (!currentTarget) recover();
	}
}

const int MAX_NAMES = 20;

void Interpreter::targets() {
	tin >> token;
	if (!expect(")")) return;
	const char *names[MAX_NAMES];
	int n = KnownTarget::getList (KnownBlock::domain(), names, MAX_NAMES);
	for (int i = 0; i < n; i++) say (names[i]);
}

// change target states
void Interpreter::targetparam() {
	char name[TOKSIZE], value[TOKSIZE];
	tin >> name;
	if (badname(name)) return;
	Target* t = universe->myTarget();
	if (depth && currentTarget) t = currentTarget;
	tin >> value;
	State* s = t->stateWithName(name);
	if (!s) {
		say ("No such target-state: ", name);
		recover();
	}
	if (*value == ')') {
		StringList msg = s->print(0);
		say (msg);
		return;
	}
	tin >> token;
	if (!expect(")")) return;
	// set the value.
	s->setInitValue(hashstring(value));
}

// load a file of interpreter commands
void Interpreter::load() {
// get the file argument
	char filename[TOKSIZE];
	tin >> filename;
	if (*filename == ')') {
		say (where(), "File argument required");
		tin.flush();
		return;
	}
// eat the closing parenth (must do this BEFORE tin.fromFile)
	tin >> token;
	if(!expect(")")) return;
	if (!tin.fromFile (expandPathName(filename)))
		say (where(), "Cannot open file: ", filename);
	return;
}

// Run dynamic linker to load a file
void Interpreter::link() {
	char filename[TOKSIZE];
	tin >> filename;
	if (*filename == ')') {
		say (where(), "File argument required");
		tin.flush();
		return;
	}
	tin >> token;
	if(!expect(")")) return;
	if (!Linker::linkObj (filename))
		recover();
	return;
}

// change the working directory
void Interpreter::chdir() {
	char dirname[TOKSIZE];
	tin >> dirname;
	if (*dirname == ')') {
		say (where(), "chdir requires an argument");
		tin.flush();
		return;
	}
// eat closing parenth
	tin >> token;
	if (!expect(")")) return;
	if (::chdir (expandPathName(dirname)) != 0) {
		say (where(), "can't change directory to ", dirname);
		tin.flush();
		return;
	}
}

// Print the arguments (allow messages from a file)
void Interpreter::echo() {
	char msg[TOKSIZE];
	while (1) {
		tin >> msg;
		if (*msg == ')') break;
		cout << msg << " ";
	}
	cout << "\n";
}

// Print the help file.  HELPDIR is passed to this file from the
// makefile and says where the interpreter is.
#ifndef HELPDIR
#define HELPDIR "$PTOLEMY/lib"
#endif

// Allow a different pager
#ifndef PAGER
#define PAGER "more"
#endif
void Interpreter::help() {
// eat the closing parenth
	tin >> token;
	if (!expect (")")) return;
	StringList cmd;
	cmd = PAGER;
	cmd += " ";
	cmd += expandPathName(HELPDIR);
	cmd += "/interpreter.help";
	system (cmd);
}

void Interpreter::exec() {
	char cmd[256];
	tin >> cmd;
	if (*cmd == ')') {
		say (where(), "Exec: argument required");
		tin.flush();
	}
	tin >> token;
	if (!expect (")")) return;
	system (cmd);
}

void Interpreter::unknown() {
	say (where(), "Unrecognized command: ", token);
	tin.flush();
	return;
}

// An InterpFuncP is a pointer to an Interpreter function that returns void

typedef void (Interpreter::*InterpFuncP)();

struct InterpTableEntry {
	const char* name;
	InterpFuncP func;
};

// Here is the function table and dispatcher function.
// These macros define entries for the table

#define ENTRY(verb) { quote(verb), Interpreter::verb }
#define ENTRY2(verb,action) { quote(verb), Interpreter::action }

void Interpreter::dispatch(const char* verb) {

// Here is the table.  Make sure it ends with "0,Interpreter::unknown"
	static InterpTableEntry funcTable[] = {
	ENTRY(alias),
	ENTRY(animation),
	ENTRY(busconnect),
	ENTRY2(cd,chdir),
	ENTRY(chdir),
	ENTRY(connect),
	ENTRY(cont),
	ENTRY(defgalaxy),
	ENTRY(delnode),
	ENTRY(delstar),
	ENTRY2(desc,descriptor),
	ENTRY(descriptor),
	ENTRY(disconnect),
	ENTRY(domain),
	ENTRY(domains),
	ENTRY(dump),
	ENTRY(echo),
	ENTRY(exec),
	ENTRY2(galaxy,star),
	ENTRY(help),
	ENTRY(initstate),
	ENTRY2(input,alias),
	ENTRY(knownlist),
	ENTRY(link),
	ENTRY(load),
	ENTRY(node),
	ENTRY2(nodecon,nodeconnect),
	ENTRY(nodeconnect),
	ENTRY(numports),
	ENTRY2(output,alias),
	ENTRY(quit),
	ENTRY(reset),
	ENTRY(run),
	ENTRY(schedule),
	ENTRY(seed),
	ENTRY(setstate),
	ENTRY(show),
	ENTRY(star),
	ENTRY(state),
	ENTRY(target),
	ENTRY(targetparam),
	ENTRY(targets),
	ENTRY(wrapup),
	0, Interpreter::unknown
	};

	InterpTableEntry* p = funcTable;
	while (p->name && strcmp(verb,p->name) != 0)
		p++;
	// call the function.  The last function catches
	// errors and prints "Unrecognized command".
	(this->*(p->func))();
}
