%{

/**************************************************************************
 * Version identification:
 * @(#)pass2Lex.l	1.5	7/16/92
 *
 *  Copyright (c) 1990,1991 The Regents of the University of California.
 *                        All Rights Reserved.
 *
 *  Programmer:  Anders Wass
 *  Date of creation: 11/02/90
 *
 *
 *  Description:
 *
 * This file performs the vector macro expansion supported by thor
 * preprocessor epp. The algorithm is this: Everything that don't match is
 * copied to output. Code within C-comments are ignored. First match to
 * occur should be `name['. If `name' is declared a vector in the model,
 * pattern matching continues. Otherwise `name' is copied to output and the
 * search for next `name[' begins.
 *
 * When a valid name is found, `name' is temporarly stored and lex switch
 * state to skip over `['. After that lex switch state again to look for
 * four different cases: `]', `] =', `:', and an `index'.
 * In the two first cases a pair of empty brackets was found and
 * fpack/funpack translation should be performed and vector pattern matching
 * should be started from the beginning.
 * The third case indicate that default first index should be used and the
 * fourth case an index symbol was found. The two last cases forces lex to
 * switch state again.
 *
 * In this state lex looks for `[', `]', and `:'. The first case indicate
 * that `index' found actually is a vector name and should be handled
 * according to that. Therefor `name[' is written to the output and `index'
 * is stored as `name' instead. Lex is then switched back to its previous state.
 * The second case indicates that a plain vector is found and no translation
 * should be prerformed. `name' and `index' are written to output to form a
 * plain vector notation.
 * The third case indicates that a second index is to be expected and a
 * translation should be performed later on.
 *
 * Next state looks for a `]' or an `index2'. If `index2' is found, it will
 * be temporarly stored. Both cases will have lex to switch to the CLOSING
 * state.
 * This state share its actions partly with INDEX1. When we get here
 * there is now doubt about that translation should be performed.
 *
 * This pattern matching algorithm is in no way waterproof but should
 * perform as well as the algorithm in epp. The more complex the expressions
 * inside the brackets become, the more likely is this algorithm to fail.
 * However, the purpose of this program is NOT to compile the model. And
 * clean well written code should not create any trouble with translations.
 *
 *	-- Anders Wass 1990/11/02
 *
 ********************************************************************************/


static char identification[] = "@(#)pass2Lex.l	1.5\t7/16/92  (c) UC Berkeley";

#ifndef YYSTACKSIZE
#define YYSTACKSIZE 32
#endif

static struct yysvf *yysvfstack[YYSTACKSIZE];
static int yystackptr=0;
static char lstack_id[] =
    "@(#)lexststack.Pc\tv1.00\t1990/10/23  (c) UCB,  A Wass";

static void inityysvfstack()
{
    yystackptr=0;
    yysvfstack[0] = yybgin + 1;
}

static void yylexpush(state)
    struct yysvf *state;
{
    if (yystackptr>=YYSTACKSIZE) {
	fprintf(stderr,"lex: state stack overflow. (>=%d)\n",YYSTACKSIZE);
	exit(1);
    }
    yysvfstack[yystackptr++] = state;
}

static struct yysvf *yylexpop()
{
    if (yystackptr==0) {
	fprintf(stderr,"lex: state stack underflow.\n");
	exit(1);
    }
    return yysvfstack[--yystackptr];
}

extern char* strcpy();

#define POP yybgin = yylexpop()
#define PUSH yylexpush(yybgin)
#define CLEARSTSTACK inityysvfstack()

typedef struct vector {
    char name[128];
    char i1[512];
    char i2[512];
} VECTOR;

#include "comdefs.h"

	/* File from pass1 and resulting C++ file from pass2 */
static FILE *pass1fp = NULL, *pass2fp = NULL;

static VECTOR vec;

static NAMETABLE *NameTable = NULL;
static int funpackflg=0, firstind=1, lpar=0;
static int numOfUnpack=0, numOfPack=0, fstlbrack=0, errflg=0;
void fpack(), funpack();
%}

D		[0-9]
ST		[ \t]
STF		[ \t\14]
FF		\14

NAME		[A-Za-z_][A-Za-z0-9_\$]*

%START VECT INDEX1 INDEX2 INDEX3 CLOSING COMMENT

%%
    CLEARSTSTACK;
    yyin=pass1fp;
    yyout=pass2fp;
    firstind=1;
    lpar=0;
    vec.name[0] = vec.i1[0] = vec.i2[0] = NULL;
    BEGIN 0;



<COMMENT>"*/"		{
			    ECHO;
			    POP;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<COMMENT> `*/': Popping state. [_flg=%d.\n",
				       fstlbrack);
#endif
			}
<COMMENT>\n		|
<COMMENT>[^*\n]*	|
<COMMENT>"*"		{
			    ECHO;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<COMMENT> `\\n|[^*\\n]*|*': [_flg=%d.\n", fstlbrack);
#endif
			}
"/*"			{
			    ECHO;
			    PUSH; BEGIN COMMENT;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<?> `/*': Pushing to state COMMENT. [_flg=%d.\n",
				       fstlbrack);
#endif
			}


";"			{
			    while (lpar) putc(')',yyout), lpar--;
			    putc(';',yyout);
			}

{STF}			|
\n			{
			    ECHO;
			}

^[ \t\n]*\n		{   fputc('\n',yyout); /* remove mult. blank lines */ }

{NAME}/({ST}*)\[	{
			    if (getvecinfo(yytext,NameTable)) {
				if (fstlbrack) {
				    fprintf(yyout,"%s[",vec.name);
				    fstlbrack=0;
				}
				strcpy(vec.name,yytext);
				BEGIN VECT;
#ifndef NO_DEBUG
				if (debug>2)
				    dbgmsg(__FILE__, __LINE__,
					   "%s\n%s [_flg=%d, text = \"%s\".\n",
					   "<0> `{NAME}/({ST}*)\\[':",
					   "Going to state VECT.", fstlbrack, yytext);
#endif
			    }
			    else {
				ECHO;
				BEGIN 0;
#ifndef NO_DEBUG
				if (debug>2)
				    dbgmsg(__FILE__, __LINE__,
					   "%s\n%s [_flg=%d, text = \"%s\".\n",
					   "<0> `{NAME}/({ST}*)\\[':",
					   "Going to state 0.", fstlbrack, yytext);
#endif
			    }
			}



<VECT>\[{ST}*		{
			    fstlbrack=1;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<VECT> `\\[{ST}*': %s [_flg=%d.\n",
				       "Going to state INDEX1.", fstlbrack);
#endif
			    BEGIN INDEX1;
			}
<VECT>{ST}+		{
			    if (fstlbrack) fprintf(yyout,yytext);
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<VECT> `{ST}+': [_flg=%d.\n", fstlbrack);
#endif
			}

<INDEX1,CLOSING>\]{ST}*	{
			    fstlbrack=0;
			    fpack(yyout,&vec,NameTable);
			    firstind=1;
			    BEGIN 0;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX1,CLOSING> `\\]{ST}*': %s [_flg=%d.\n",
				       "Going to state 0.", fstlbrack);
#endif
			}
<INDEX1,CLOSING>\]({ST}*)={ST}*	{
			    fstlbrack=0;
			    funpack(yyout,&vec,NameTable);
			    lpar++;
			    firstind=1;
			    BEGIN 0;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX1,CLOSING> `\\]({ST}*)={ST}*': %s [_flg=%d.\n",
				       "Going to state 0.",fstlbrack);
#endif
			}
<INDEX1>:		{
			    unput(':');
			    BEGIN INDEX3;
			    firstind=0;
			    fstlbrack=0;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX1> `:': Going to state INDEX3. [_flg=%d.\n",
				       fstlbrack);
#endif
			}
<INDEX1>{ST}+		{
			    if (fstlbrack) ECHO;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX1> `{ST}+': [_flg=%d.\n", fstlbrack);
#endif
			}
<INDEX1>[^\[\]:]*	{
			    strcpy(firstind ? vec.i1 : vec.i2,yytext);
			    BEGIN INDEX2;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX1> `[^\\[\\]:]*': %s [_flg=%d.\n",
				       "Going to State INDEX2.", fstlbrack);
#endif
			}

<INDEX2>\[{ST}*		{
			    fprintf(yyout,"%s[%s",vec.name,(yytext+1));
			    strcpy(vec.name,vec.i1);
			    vec.i1[0] = NULL;
			    BEGIN INDEX1;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX2> `\\[{ST}*': %s [_flg=%d.\n",
				       "Going to state INDEX1.", fstlbrack);
#endif
			}
<INDEX2>\]		{
			    fprintf(yyout,"%s[%s]",vec.name,vec.i1);
			    fstlbrack=0;
			    vec.name [0] = vec.i1[0] = NULL;
			    BEGIN 0;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX2> `\\]': Going to state 0. [_flg=%d.\n",
				       fstlbrack);
#endif
			}
<INDEX2>:{ST}*		{
			    BEGIN INDEX3;
			    firstind=0;
			    fstlbrack=0;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX2> `:{ST}*': %s [_flg=%d.\n",
				       "Going to state INDEX3.", fstlbrack);
#endif
			}

<INDEX3>\]		{
			    unput(']');
			    BEGIN CLOSING;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX3> `\\]': %s [_flg=%d.\n",
				       "Going to state CLOSING.", fstlbrack);
#endif
			}
<INDEX3>{ST}+		{
			    if (fstlbrack) ECHO;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX3> `{ST}+': [_flg=%d.\n", fstlbrack);
#endif
			}
<INDEX3>[^\]]*		{
			    strcpy(firstind ? vec.i1 : vec.i2,yytext);
			    BEGIN CLOSING;
#ifndef NO_DEBUG
			    if (debug>2)
				dbgmsg(__FILE__, __LINE__,
				       "<INDEX3> `[^\\]]*': %s [_flg=%d.\n",
				       "Going to state CLOSING.", fstlbrack);
#endif
			}


%%

int yywrap() { return 1; }

static void fpack(fp,vector,tbl)
    FILE *fp;
    VECTOR *vector;
    NAMETABLE *tbl;
{
    char *ind1, *ind2, *s;
    NAMETABLE *infop = getvecinfo(vector->name,tbl);

    if (!infop) {
	errmsg("s","\"%s\": Macro error. No vector with name \"%s\" is declared.\n",
	       FileIParse, vector->name);
	errflg++;
	gsignal(1);
    }

    switch (infop->type) {
    case BIG_END:
	ind1 = vector->i1[0] ? vector->i1 :
	    (sprintf(vector->i1,"(%s)-1",infop->size), vector->i1);
	ind2 = vector->i2[0] ? vector->i2 : strcpy(vector->i2,"0");
	break;
    case LIT_END:
	ind1 = vector->i1[0] ? vector->i1 : strcpy(vector->i1,"0");
	ind2 = vector->i2[0] ? vector->i2 :
	    (sprintf(vector->i2,"(%s)-1",infop->size), vector->i2);
	break;
    default:
	errmsg("f","Bad endian value to vector \"%s\" in function `%s'.\n",
	       vector->name, "fpack");
	errmsg("s","\t\"%s\" declared in file \"%s\" on line %d.\n",
	       vector->name, FileIParse, infop->lineno);
	errmsg("S","\tShouldn't occur. BIG_END=%d, LIT_END=%d, value found = %d.\n%s",
	       BIG_END, LIT_END, infop->type, "\t(Data have been corrupted ???)\n");
	gsignal(1);
	break;
    }

    fprintf(yyout,"fpack(%s,%s,%s)", infop->name, ind1, ind2);
    vector->name[0] = vector->i1[0] = vector->i2[0] = NULL;
    numOfPack++;
}

static void funpack(fp,vector,tbl)
    FILE *fp;
    VECTOR *vector;
    NAMETABLE *tbl;
{
    char *ind1, *ind2, *s;
    NAMETABLE *infop = getvecinfo(vector->name,tbl);

    if (!infop) {
	errmsg("s","\"%s\": Macro error. No vector with name \"%s\" is declared.\n",
	       FileIParse, vector->name);
	errflg++;
	gsignal(1);
    }

    switch (infop->type) {
    case BIG_END:
	ind1 = vector->i1[0] ? vector->i1 :
	    (sprintf(vector->i1,"(%s)-1",infop->size), vector->i1);
	ind2 = vector->i2[0] ? vector->i2 : strcpy(vector->i2,"0");
	break;
    case LIT_END:
	ind1 = vector->i1[0] ? vector->i1 : strcpy(vector->i1,"0");
	ind2 = vector->i2[0] ? vector->i2 :
	    (sprintf(vector->i2,"(%s)-1",infop->size), vector->i2);
	break;
    default:
	errmsg("f","Bad endian value to vector \"%s\" in function `%s'.\n",
	       vector->name, "funpack");
	errmsg("s","\t\"%s\" declared in file \"%s\" on line %d.\n",
	       vector->name, FileIParse, infop->lineno);
	errmsg("S","\tShouldn't occur. BIG_END=%d, LIT_END=%d, value found = %d.\n%s",
	       BIG_END, LIT_END, infop->type, "\t(Data have been corrupted ???)\n");
	gsignal(1);
	break;
    }
    
    fprintf(yyout,"funpack(%s,%s,%s,", infop->name, ind1, ind2);
    vector->name[0] = vector->i1[0] = vector->i2[0] = NULL;
    numOfUnpack++;
}

int pass2(ifp,ofp,ofname,mfname,nametbl)
    FILE *ifp, *ofp;
    char *ofname, *mfname;
    NAMETABLE *nametbl;
{
    int retval = 0;
    errflg=0;

    if (!ifp) {
	errmsg("f","No file opened to read from.\n");
	return 3;
    }
    pass1fp = ifp;
    if (!ofp) {
	errmsg("f","No file opened to write to.\n");
	return 4;
    }
    pass2fp = ofp;
    FileIParse = mfname ? mfname : "(stdin)"; /* name that might be used by */
				/* external functions called within this block. */
    NameTable = nametbl;

    yylex();
    if (errflg)
	retval = 2;
    if (!(numOfPack || numOfUnpack))
	retval = retval ? retval : 1; /* don't overide error flag */
    else {
	errmsg(verbflg?"o":"O","Expansion statistics:");
	if (numOfPack)
	    errmsg(verbflg?"s":"S","   fpack   %3d.", numOfPack);
	if (numOfUnpack)
	    errmsg(verbflg?"s":"S","   funpack %3d.", numOfUnpack);
	errmsg(verbflg?"s":"S","\n");
    }
    pass1fp = pass2fp = NULL;
    FileIParse = NULL;
    return retval;
}

/* <EOF> pass2Lex.l */
