/* 
Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
							COPYRIGHTENDKEY
*/
static const char file_id[] = "ThorNode.cc";

///////////////////////////////////////////////////////////////////////////
//
// Version identification:
// @(#)ThorNode.cc	1.15 11/25/92
//
// Copyright  (c) 1990	The Regents of the University of California,
//				All Rights Reserved.
//
// Programmer:	Seungjun Lee
// Created:	08/03/90
// Revisions:
//
// Description:
// ------------
//	This file contains the implementatins of some of routines in
//	the ThorNode class.
//	This is a modification of NodeStar.cc by Anders Wass
//
///////////////////////////////////////////////////////////////////////////

#ifdef __GNUG__
#pragma implementation
#endif
#include "ThorNode.h"
#include "ThorPortHole.h"
#include "KnownBlock.h"

// Logic value definition: should be consistent with Thor.h
// FLOAT is renamed to FLOAT_  to avoid the conflict with Ptolemy dataType
// Need the initialization value _INIT_VAL_ so the node value can propagate
// correctly from the node to the inports that reads the node. (Added by
// Anders Wass 5/3/91)
#define ZERO 0
#define ONE  1
#define UNDEF 2
#define FLOAT_ 3
#define _INIT_VAL_ -1

static int table[] = {
    ZERO,			// 0,0 => 00 00
    UNDEF,			// 0,1 => 00 01
    UNDEF,			// 0,U => 00 10
    ZERO,			// 0,Z => 00 11
    UNDEF,			// 1,0 => 01 00
    ONE,			// 1,1 => 01 01
    UNDEF,			// 1,U => 01 10
    ONE,			// 1,Z => 01 11
    UNDEF,			// U,0 => 10 00
    UNDEF,			// U,1 => 10 01
    UNDEF,			// U,U => 10 10
    UNDEF,			// U,Z => 10 11
    ZERO,			// Z,0 => 11 00
    ONE,			// Z,1 => 11 01
    UNDEF,			// Z,U => 11 10
    FLOAT_			// Z,Z => 11 11
    };

ThorNode :: ThorNode (DataType type)
{

    if (type == INT) {
	LOG_NEW; nodeValue = new IntParticle(_INIT_VAL_);
    }

    // Warning!!! 
    // FLOAT is redefined in THOR domain

    else if (type == FLOAT) {
	LOG_NEW; nodeValue = new FloatParticle(0);
    }
    else
	nodeValue = NULL;

    clrNodeHasChanged();
    resetScheduled();
}

ThorNode :: ~ThorNode()
{
    LOG_DEL; delete nodeValue;
    // disconnect any connected portholes.  0 argument says
    // not to have PortHole::disconnect try to do things to geodesic
    // The way it is done below is both wrong and unreliable since ThorNode is
    // a shared Geodesic with multi destionations and multi originations.
//    if (originatingPort) originatingPort->disconnect(0);
//    if (destinationPort) destinationPort->disconnect(0);

    // The code starting here and until the end of this method is added by
    // Anders Wass 5/12/91.

    ListIter NextInPort(inputs);
    ThorPortHole* php;
    int size = inputs.size();	// make sure all ports are disconnected,
    while (size--) {		// therefore use size of list.
	php = (ThorPortHole*)NextInPort++;
	if (php) php->disconnect(0);
    }
    ListIter NextOutPort(outputs); // It would have been nice to use
				   // reconnect instead.
    size = outputs.size();
    while (size--) {
	php = (ThorPortHole*)NextOutPort++;
	if (php) php->disconnect(0);
    }
    originatingPort = NULL;	// disconnect these single referrences so
    destinationPort = NULL;	// no other methods will be fooled.
}

int ThorNode :: isItPersistent () const { return TRUE; }

void ThorNode :: initialize() {
    Geodesic::initialize();
    setNodeValue(_INIT_VAL_);	// was FLOAT_, changed by A Wass 5/3/91
}

PortHole* ThorNode :: setSourcePort (GenericPort& sp, int delay)
{
    Geodesic::setSourcePort(sp, delay);
    inputs.put(originatingPort);
    ((ThorPortHole*)originatingPort)->delay = delay;

    // Check if biput
    if (sp.isItInput())
	return setDestPort(sp);

    return originatingPort;
}

PortHole* ThorNode :: setDestPort (GenericPort& sp)
{
    Geodesic::setDestPort(sp);
    outputs.put(destinationPort);
    return destinationPort;
}

void ThorNode :: resolve ()
{
    register int val;

    clrNodeHasChanged();
    ThorPortHole *phptr;

    // added next if to allow arbitrary integers on single output nets
    if (inputs.size() == 1) {
	phptr = (ThorPortHole*) inputs.head();
	val = (int) phptr->getBuffParticle();
    }
    else {
	// evaluate node value from inputs
	register int index;
	ListIter nexti(inputs);
	val = FLOAT_;

	while ((phptr = (ThorPortHole*)nexti++) != 0) {
	    index = (val << 2) | (int) phptr->getBuffParticle();
	    val = (index < 16) ? table[index] : UNDEF;
	}
    }
   
    if (val != getNodeValue()) { // if this node got a new value,
	setNodeHasChanged();	 // notify the scheduler ...
	setNodeValue(val);	 // save the node value ...
    }
}


// The comparison function below is safe only if the incoming block
// refers to a ThorNode. It also just support integer data type
// and must therefor be modified when other data types will occur in
// the model descriptions. However, the scheduler does not have to know
// about the data type the particles are carrying, only the rules for
// how the ports are connected. (An element port is always connected to
// a node, never to a element port. The same for node.)

// renamed as a function
//int operator != (PortHole& ph, ThorNode& ns)
int
notEqual(PortHole& ph, ThorNode& ns)
{
    enum equality {eq=FALSE, neq=TRUE};
    static equality LogicValTable[] = {
		// 0==ZERO,  1==ONE,
		// U==UNDEF, Z==FLOAT_
		// elementport | nodeport
	eq,	// 0,0  <=>  00 00
	neq,	// 0,1  <=>  00 01
	eq,	// 0,U  <=>  00 10
	neq,	// 0,Z  <=>  00 11
	neq,	// 1,0  <=>  01 00
	eq,	// 1,1  <=>  01 01
	eq,	// 1,U  <=>  01 10
	neq,	// 1,Z  <=>  01 11
	neq,	// U,0  <=>  10 00
	neq,	// U,1  <=>  10 01
	eq,	// U,U  <=>  10 10
	neq,	// U,Z  <=>  10 11
	eq,	// Z,0  <=>  11 00
	eq,	// Z,1  <=>  11 01
	eq,	// Z,U  <=>  11 10
	eq	// Z,Z  <=>  11 11
	};

    ThorPortHole& tph = *(ThorPortHole*) &ph;

    if (tph.getBuffParticle().type() != INT) {
	Error::warn("Thor domain : Can only deal with INT so far.");
//	Error::abortRun("Thor domain : Can only deal with INT so far.");
//	return neq;
    }

    if (ns.getNodeValue() == _INIT_VAL_)
	return neq;
    else if (ns.inputs.size() == 1)
	return (ns.getNodeValue() != (int) tph.getBuffParticle());
    else
	return LogicValTable[(ns.getNodeValue() << 2) | (int) tph.getBuffParticle()];
}

//int operator == (PortHole& ph, ThorNode& ns)
//{  return !(ph != ns);  }
int
equal(PortHole& ph, ThorNode& ns)
{  return !notEqual(ph, ns); }

//	<EOF: ThorNode.cc>
