
/*
 *
 * Version identification:
 * @(#)window.c	1.3 11/23/92
 *
 */

#ifndef lint
static char rcsid[] = "$Header: window.c,v 1.7 87/05/17 17:46:17 rokicki Exp $";
#endif
/*
 * Window Manager for logic analyzer
 *
 */
#include <stdio.h>
#include "defs.h"

#define	CHARHEIGHT	metrics.charHeight
#define	CHARWIDTH	metrics.charWidth
#define	XTraceSize	(trace.right - trace.left - 2)
#define	YTraceSize	(trace.top - trace.bot)
#define LastStep	(times.dispStart + times.dispSteps)


char *wname = "Thor Analyzer";

Display *display;
Screen  *screen;
Window window = 0;
int descent;
XFontStruct *fontinfo ;
int xblack, xwhite ;


/*
 * Convert a time to its corresponding x position.
 */
int TimeToX( t )
  TimeType t;
  {
    return((t - times.dispStart) * XTraceSize/times.dispSteps + trace.left+1);
  }



/*
 * Convert an x position to the closest time-step to the left.  Return -1 if
 * the point lies outside the traces window.
 */
TimeType XToTime( x )
  XCoord  x;
  {
    float  tmp;

    if( (x < trace.left) || (x > trace.right) )
	return(-1);
    tmp = (float) times.dispSteps / XTraceSize;
    return( times.dispStart + round( (x - trace.left - 1) * tmp ) );
  }


/*
 * return 1 if the 2 bounding boxes intersect, otherwise 0
 */
int Intersect( b1, b2 )
  Box  b1;
  Box  *b2;
  {
    if( (b1.bot > b2->top) || (b1.top < b2->bot) ||
	(b1.left > b2->right) || (b1.right < b2->left ) )
	return( 0 );
    return( 1 );
  }


/*
 * Initialize and display the windows.   Called ONLY once at the begining.
 */


void InitWindows()
  {
    XEvent rep ;

    if (display == NULL)
       if ((display = XOpenDisplay(NULL))==NULL)
          ANALYZER_Crash("could not open display") ;

    if ((fontinfo=XLoadQueryFont(display, default_font))==NULL) {
       PrintFonts();
       ANALYZER_Crash("Could not open font") ;

     }
    screen = ScreenOfDisplay( display, DefaultScreen( display ) );
    metrics.charHeight = fontinfo->max_bounds.ascent + fontinfo->max_bounds.descent ;
    metrics.charWidth = fontinfo->max_bounds.width ;
    descent = fontinfo->max_bounds.descent;
    InitGraphics(fontinfo->fid);

    if (window == 0)
      {
        InitWindow( TRUE, NormalState, 0, 0, 0, 0, 10, 10);
      }
    if (window == 0) {
       ANALYZER_Crash("could not open window") ;
    }
    XMapWindow(display, window) ;
    XFlush(display);
    while (1) {
      XNextEvent(display,&rep) ;
      if (rep.type == Expose) {
           break;
         }
      else if (rep.type == ConfigureNotify) {
        XWINDOWSIZE = rep.xconfigure.width ;
        YWINDOWSIZE = rep.xconfigure. height ;
        break;
      }
    }
    traces.maxDigits = MaxTraceDigits();
    traces.maxName = MaxTraceName();
    InitTextWindow( DEFTEXTLINES );
    SetWindowParms();
    GetSignalPos();
    mySetPattern( clearPat );
    mySetRendering( PAINTFGBG );
    RedrawWindow( (Box*)NULL );
  }



int InitWindow( firstTime, state, x, y, w, h, ix, iy )
  int    firstTime;
  int    state;
  Coord  x, y, w, h;
  Coord  ix, iy;
  {
    int                   spec, u_spec;
    static int            b;

    XSizeHints            shint;
    XWMHints              wmh;
    XSetWindowAttributes  att;
    XClassHint            class;

    if( firstTime )
      {

	b = DEFBORDERWIDTH;
	spec = XParseGeometry( DEFGEOMETRY, &x, &y, &w, &h );

	if( (spec & (XValue | XNegative)) == (XValue | XNegative) )
	    x += WidthOfScreen( screen ) - w - 2 * b;

	if( (spec & (YValue | YNegative)) == (YValue | YNegative) )
	    y += HeightOfScreen( screen ) - h - 2 * b;

	XWINDOWSIZE = w;
	YWINDOWSIZE = h;
      }
    else
	u_spec = TRUE;

    att.background_pixel = colors.white;
    att.border_pixel = colors.black;
    att.backing_planes = colors.black | colors.white | colors.hilite |
     colors.traces | colors.banner_bg | colors.banner_fg;
    att.cursor = cursors.deflt;

    window = XCreateWindow( display, RootWindowOfScreen( screen ),
     x, y, w, h, b, DefaultDepthOfScreen( screen ), InputOutput,
     (Visual *) CopyFromParent,
     (CWBackPixel | CWBorderPixel | CWBackingPlanes | CWCursor), &att );

    XStoreName( display, window, wname );
    XSetIconName( display, window, wname );
    class.res_name = "thor";
    class.res_class = wname;
    XSetClassHint( display, window, &class );

    wmh.input = True;
    wmh.initial_state = state;
    wmh.icon_pixmap = pix.icon;

    wmh.icon_x = ix;
    wmh.icon_y = iy;
    wmh.flags = InputHint | StateHint | IconPixmapHint | IconWindowHint |
     IconPositionHint;
    XSetWMHints( display, window, &wmh );

    shint.x = x;
    shint.y = y;
    shint.width = w;
    shint.height = h;
    shint.max_width = shint.max_height = 16000;    /* any big number */
    shint.width_inc = shint.height_inc = 1;
    /*GetMinWsize( &shint.min_width, &shint.min_height );*/
    shint.min_width = 50;
    shint.min_height = 50;
    shint.flags = ( u_spec ) ?
      ( PMinSize | PMaxSize | PResizeInc | USPosition | USSize ) :
      ( PMinSize | PMaxSize | PResizeInc | PPosition | PSize );
    XSetNormalHints( display, window, &shint );

    XSelectInput( display, window, XDefaultSelects);
    XFlush( display );
  }

/*
 * Redraw any region that overlaps the redraw-box.
 * If the redraw-box is NULL then redraw everything.
 */
void RedrawWindow( redrawBox )
  Box  *redrawBox;
  {
    TimeType  t1, t2;
    mySetRendering( PAINTFGBG );
    mySetPattern( clearPat );
    if( redrawBox == NULL )			/* force redraw everything */
      {
	RedrawBanner();
	RedrawTimes();
	RedrawNames();
	DrawScrollBar();
	DrawCursVal();
	RedrawTraces( times.dispStart, LastStep, 1 );
	RedrawText();
      }
    else
      {
	if( Intersect( banner.box, redrawBox ) )
	    RedrawBanner();
	if( Intersect( times.box, redrawBox ) )
	    RedrawTimes();
	if( Intersect( sigNames, redrawBox ) )
	    RedrawNames();
	if( Intersect( scrollbar, redrawBox ) )
	    DrawScrollBar();
	if( Intersect( cursVal, redrawBox ) )
	    DrawCursVal();
	if( Intersect( trace, redrawBox ) )
	  {
	    if( redrawBox->left < trace.left )
		t1 = times.dispStart;
	    else
		t1 = XToTime( redrawBox->left ) - 1;
	    if( redrawBox->right > trace.right )
		t2 = LastStep;
	    else
		t2 = XToTime( redrawBox->right ) + 1;
	    RedrawTraces( t1, t2, 1 );
	  }
	if( Intersect( text, redrawBox ) )
	    RedrawText();
      }
    mySetColors( BLACK, WHITE );
    myRectangle( 0, 0, XWINDOWSIZE - 1, YWINDOWSIZE - 1 ) ;
  }


/*
 * Displays the window banner.  Invert colors when selected.
 */
void RedrawBanner()
  {
    char  *stat;

    if( windowState.isSelected )
      {
	mySetColors( WHITE, BLACK );
	myFilledRectangle( banner.box.left, banner.box.bot,
			     banner.box.right, banner.box.top );
      }
    else
      {
	mySetColors( BLACK, WHITE );
	myFilledRectangle( banner.box.left, banner.box.bot,
			     banner.box.right, banner.box.top );
	myRectangle( banner.box.left, banner.box.bot,
			     banner.box.right, banner.box.top );
      }

    switch( windowState.simulStatus )
     {
       case RUNNING :
         stat = "simulator status: running ";
	 break;
       case STOPPED :
       case STOP_NO_MEM :
         stat = "simulator status: stopped ";
	 break;
       case DEAD :
         stat = "simulator status: finished ";
     }
    StrRight( stat, banner.box.right, banner.box.bot + 2 );
    StrLeft( banner.left, banner.box.left, banner.box.bot + 2 );
  }


void RedrawTimes()
  {
    char  s[ 20 ];
    mySetColors( BLACK, WHITE );
    myFilledRectangle(  times.box.left + 1, times.box.bot,
			  times.box.right - 1, times.box.top );
   if( times.first < 0 )
      {
	(void) sprintf( s, "--" );
	StrCenter( s, (trace.left + times.box.left) / 2, times.box.bot + 1 );
	StrCenter( s, (times.box.right + trace.right)/ 2, times.box.bot + 1 );
      }
    else
      {
	(void) sprintf( s, "%d", times.first );
	StrCenter( s, (trace.left + times.box.left) / 2, times.box.bot + 1 );
	(void) sprintf( s, "%d", times.last );
	StrCenter( s, (times.box.right + trace.right)/ 2, times.box.bot + 1 );
      }
    mySetColors( WHITE, BLACK );
    (void) sprintf( s, "%d", times.dispStart );
    StrLeft( s, trace.left, times.box.bot );
    (void) sprintf( s, "%d", LastStep );
    StrRight( s, trace.right, times.box.bot );
    if( cursor.time >= 0 )
      {
	mySetColors( BLACK, WHITE );
	(void) sprintf( s, "   %d   ", cursor.time );
	StrCenter( s, (trace.left + trace.right) / 2, times.box.bot + 2 );
      }
  }


void UpdateLastTime()
  {
    char  s[ 20 ];

    mySetColors( BLACK, WHITE );
    myFilledRectangle(  trace.right + 1, times.box.bot, times.box.right - 1,
			  times.box.top);
    (void) sprintf( s, "%d", times.last );
    StrCenter( s, (times.box.right + trace.right)/ 2, times.box.bot + 1 );
  }



/*
 * Redraw signal names.
 */
void RedrawNames()
  {
    int       nBus, rlen;
    char      s[100];
    XCoord    x;
    YCoord    y;
    TraceEnt  *t, *lastT;
    char      *r;

    mySetColors( BLACK, WHITE );
    myFilledRectangle(  sigNames.left, sigNames.bot,
			  sigNames.right, sigNames.top);
    x = sigNames.right - 2;
    nBus = 0;
    lastT = &(traces.traces[ traces.dispTraces ]);
    for( t = traces.traces; t != lastT; t++ )
      {
	y = (t->bot + t->top - CHARHEIGHT) / 2;
	if( t->nsig == 1 )
	    StrRight( inputs.names[ t->signals[0] ], x, y );
	else
	  {
	    int  n1, n2;
	    if( t->sameRoot )
	      {
		r = Root( inputs.names[ t->signals[0] ]);
		rlen = strlen( r );
		if( inputs.names[ t->signals[0] ][ rlen ] == '[' )
		    rlen++;
		n1 = atoi( &(inputs.names[ t->signals[0] ][ rlen ]) );
		n2 = atoi( &(inputs.names[ t->signals[ t->nsig-1 ] ][rlen]) );
		(void) sprintf( s, "%s[%d-%d]", r, n1, n2 );
	      }
	    else
	      {
		sprintf( s, "GRP %d [%d]", nBus++, t->nsig );
	      }
	    StrRight( s, x, y );
	  }
      }
  }



/*
 * Return true if all bits in a bus are tri-stated.
 */
int IsTriState( bus, signals, nbits )
  Levels  bus;
  int     *signals;
  int     nbits;
  {
    register int     i;
    register Levels  p = bus;
    register int     *index = signals;
    
    for( i = 0; i < nbits; i++, index++ )
      {
	if( p[ *index ] != FLOAT )
	    return( 0 );
      }
    return( 1 );
  }


/*
 * Redraw the traces horizontally from time1 to time2.
 */
void RedrawTraces( time1, time2, clr_bg )
  TimeType  time1, time2;
  int       clr_bg;
  {
    int       j, change;
    TraceEnt  *t, *lastT;
    HistRef   endH, startH, begH;
    TimeType  startT, endT, prevT, lookBack, lookAhead;
    YCoord    base, top, mid;
    XCoord    x0, x1, x2, x3, len;

    mySetColors( WHITE, BLACK );
    if( time1 <= times.dispStart )
	time1 = times.dispStart;
    if( clr_bg )			/* clear the backgound */
      {
	if( time1 == times.dispStart )
	    x1 = trace.left;
	else
	    x1 = TimeToX( time1 ) + 1;

	if( time2 >= LastStep )
	    x2 = trace.right;
	else
	    x2 = TimeToX( time2 );
	myFilledRectangle( x1, trace.bot, x2, trace.top );
	if( (cursor.exists >= time1) && (cursor.exists < time2) )
	    cursor.exists = -1;		/* just erased it */
      }

	/* Erase the cursor while we draw the traces */
    if( (cursor.exists >= times.dispStart) && (cursor.exists < time2) )
	EraseCursor();

    if( times.first < 0 )
	return;			/* done, no traces yet */

    if( time2 > times.last )
	time2 = times.last;

    begH = GetHist( time1 );
    if( begH == NULL )
      return;

    lookBack = (time1 == times.dispStart) ? -1 : time1;
    lookAhead = ( LastStep > times.last ) ? times.last : LastStep;

    lastT = &(traces.traces[ traces.dispTraces ]);
    for( t = traces.traces; t != lastT; t++ )
      {
	base = t->bot;
	top = t->top;
	mid = (base + top) / 2;
	startT = time1;
	startH = begH;
	
	if( t->nsig != 1 )		/* bus */
	    len = CHARWIDTH * ( (t->nsig + t->bits - 1)/ t->bits );

	while( startT < time2 )			/* Draw a single trace */
	  {
	    x1 = TimeToX( startT );
	    endH = FindChanged( t, startH, lookAhead );

	    if( endH == startH )	/* No change before lookAhead */
	      {
		endT = time2;
		change = 0;
		x2 = TimeToX( time2 );
		x3 = TimeToX( lookAhead );
	      }
	    else
	      {
		endT = endH->time;
		if( endT > time2 )		/* change after time2 */
		  {
		    x2 = TimeToX( time2 );
		    x3 = TimeToX( endT );
		    change = 0;
		  }
		else				/* change before time2 */
		  {
		    x2 = x3 = TimeToX( endT );
		    change = 1;
		  }
	      }


	    if( t->nsig == 1 )		/* not a bus */
	      {
	        if( change )
		  {
		    j = ( (startH->levels[ t->signals[0] ]) << 2 ) |
			  (endH->levels[ t->signals[0] ]);
		  }
		else
		  {
		    j = ( (startH->levels[ t->signals[0] ]) << 2 ) |
			  (startH->levels[ t->signals[0] ]);
		  }
		switch( j )
		  {
		    case 0:				/* 0 -> 0 */
		      myLine( x1, base, x2, base );
		      break;
		    case 1:				/* 0 -> 1 */
		      myLine( x1, base, x2, base );
		      myLine( x2, base, x2, top );
		      break;
		    case 2:				/* 0 -> U */
		      myLine( x1, base, x2, base );
		      break;
		    case 3:				/* 0 -> Z */
		      myLine( x1, base, x2, base );
		      myLine( x2, base, x2, mid );
		      break;
		    case 4:				/* 1 -> 0 */
		      myLine( x1, top, x2, top );
		      myLine( x2, base, x2, top );
		      break;
		    case 5:				/* 1 -> 1 */
		      myLine( x1, top, x2, top );
		      break;
		    case 6:				/* 1 -> U */
		      myLine( x1, top, x2, top );
		      break;
		    case 7:				/* 1 -> Z */
		      myLine( x1, top, x2, top );
		      myLine( x2, top, x2, mid );
		      break;
		    case 8:				/* U -> 0 */
		    case 9:				/* U -> 1 */
		    case 10:				/* U -> U */
		    case 11:				/* U -> Z */
		      mySetPattern( majorDiagPat );
		      myFilledRectangle( x1, base+1, x2, top-1 );
		      mySetPattern( clearPat );
		      break;
		    case 12:				/* Z -> 0 */
		      myLine( x1, mid, x2, mid );
		      myLine( x2, mid, x2, base );
		      break;
		    case 13:				/* Z -> 1 */
		      myLine( x1, mid, x2, mid );
		      myLine( x2, mid, x2, top );
		      break;
		    case 14:				/* Z -> U */
		    case 15:				/* Z -> Z */
		      myLine( x1, mid, x2, mid );
		      break;
		  }
	      }
	    else			/* bus */
	      {
		if( IsTriState( startH->levels, t->signals, t->nsig ) )
		  {
		    myLine( x1, mid, x2, mid );
		  }
		else
		  {
		    if( lookBack == startT )
		      {
			prevT = FindPrevChange( t, startH );
			x0 = TimeToX( prevT );
			if( (x1 - x0) > len )
			    myFilledRectangle( x0+1, mid - CHARHEIGHT/2,
						 x1-1, mid + CHARHEIGHT/2 );
			
			if( (x3 - x2) > len )
			    myFilledRectangle( x2+1, mid - CHARHEIGHT/2,
						 x3-1, mid + CHARHEIGHT/2 );
			if( (x3 - x0) > len )
			    StrCenter(  Convert( startH->levels, t ),
					(x0 + x3)/2, mid - CHARHEIGHT/2 );
		      }
		    else
		      {
			if( (x3 - x2) > len )
			    myFilledRectangle( x2+1, mid - CHARHEIGHT/2,
						 x3-1, mid + CHARHEIGHT/2 );
			if( (x3 - x1) > len )
			    StrCenter(  Convert( startH->levels, t ),
					(x1 + x3)/2, mid - CHARHEIGHT/2 );
		      }
		    myLine( x1, base, x2, base );
		    myLine( x1, top, x2, top );
		  }
		if( (x2 == x3) && change )	/* there is a transition */
		  {					/* before time2  */
		    myLine( x2, base, x2, top );
		  }
	      }

	    startT = endT;
	    startH = endH;
	  }			/* end of while */
      }				/* end of for all traces */

	/* Restore the cursor */
    if( (cursor.time >= times.dispStart) && (cursor.time < time2) )
	RedrawCursor();
  }


/*
 * center traces around time.  Bit-blit if possible.
 */
void ScrollTracesLeft( time )
  TimeType  time;
  {
    TimeType  newStart;
    XCoord    x;

    newStart = time - times.dispSteps / 2;
    if( (newStart > times.dispStart) && (time <= LastStep) )
      {

	  {
	    times.dispStart = newStart;
	    RedrawTraces( newStart, time, 1 );
	  }
      }
    else
      {
	times.dispStart = newStart;
	RedrawTraces( newStart, time, 1 );
      }
    mySetColors( WHITE, BLACK );
    x = TimeToX( time );
    myFilledRectangle( x + 1, trace.bot, trace.right, trace.top );
  }



/*
 * FindChanged returns a pointer to the first history after startH, and
 * before lastT in which the trace changes, or returns startH if there
 * are no changed histories.
 */
HistRef FindChanged( tp, startH, lastT )
  TraceEnt	 	 *tp;
  HistRef 		 startH;
  TimeType		 lastT;
  {
    register HistRef   currentH;
    register int       i, *index;
    register Levels    startL, currL;

    startL = startH->levels;
    currentH = startH->next;

    while( (currentH != NULL) && (currentH->time <= lastT) )
      {
	index = tp->signals;
	currL = currentH->levels;
	for( i = 0; i < tp->nsig; i++, index++ )
	  {
	    if( startL[ *index ] != currL[ *index ] )
		return( currentH );
	  }
	currentH = currentH->next;
      }
    return( startH );
  }


/*
 * Search back in the history for a change in trace tp.  Return the time
 * step at which the change occurs, no earlier than the 1st displayed time.
 */
TimeType FindPrevChange( tp, thisH )
  TraceEnt  *tp;
  HistRef   thisH;
  {
    register  HistRef  prevH;
    register  int      i, *index;
    register  Levels   lev, prevLev;
    
    prevH = thisH->prev;
    if( prevH == NULL )
	return( thisH->time );
    lev = thisH->levels;
    while( (prevH != NULL) && (prevH->time >= times.dispStart) )
      {
	prevLev = prevH->levels;
	index = tp->signals;
	for( i = 0; i < tp->nsig; i++, index++ )
	  {
	    if( prevLev[ *index ] != lev[ *index ] )
		return( prevH->next->time );
	  }
	prevH = prevH->prev;
      }
    return( times.dispStart );
  }


/*
 * Draw cursor at specified time step.  Set cursor location and erase previous
 * cursor if it is still visible.
 */
void MoveCursor( time )
  TimeType  time;
  {
    char    s[20];
    XCoord  xPos;

    mySetColors( BLACK, WHITE );
    if( cursor.exists != -1 )			/* still exists -> erase it */
      {
	xPos = TimeToX( cursor.exists ) + 1;
	if( (xPos >= trace.left) && (xPos <= trace.right) )
	  {
	    mySetRendering( INVERT );
	    myLine( xPos, trace.bot, xPos, trace.top );
	  }
      }
    cursor.exists = -1;
    if( (time >= times.dispStart) && (time <= LastStep) )	/* visible? */
      {
	xPos = TimeToX( time ) + 1;
	mySetRendering( INVERT );
	myLine( xPos, trace.bot, xPos, trace.top );
	cursor.exists = time;
      }
    mySetRendering( PAINTFGBG );
    if( time >= 0 )
      {
	(void) sprintf( s, "   %d   ", time );
	StrCenter( s, (trace.left + trace.right) / 2, times.box.bot + 2 );
      }
    if( time != cursor.time )
      {
	cursor.time = time;
	DrawCursVal();
      }
    else
	cursor.time = time;
  }



/*
 *  Erase the cursor.
 */
void EraseCursor()
  {
    XCoord  xPos;

    xPos = TimeToX( cursor.time ) + 1;
    mySetRendering( INVERT );
    myLine( xPos, trace.bot, xPos, trace.top );
    mySetRendering( PAINTFGBG );
    cursor.exists = -1;
  }


RedrawCursor()
  {
    XCoord  xPos;
    
    xPos = TimeToX( cursor.time ) + 1;
    mySetRendering( INVERT );
    myLine( xPos, trace.bot, xPos, trace.top );
    mySetRendering( PAINTFGBG );
    cursor.exists = cursor.time;
  }


/*
 * Display signal values under cursor.
 */
void DrawCursVal()
  {
    XCoord       x;
    YCoord       y;
    int          i;
    HistRef      hist;
    TraceEnt     *t;
    static char  *Map[] = { "0", "1", "U", "Z" };

    mySetColors( BLACK, WHITE );
    myFilledRectangle(  cursVal.left, cursVal.bot,
			  cursVal.right - 1, cursVal.top );

    hist = GetHist( cursor.time );
    if( (hist == NULL) || (cursor.time > times.last) )
	return;
    x = 2 + (cursVal.left + cursVal.right) / 2;
    for( i = 0, t = traces.traces; i < traces.dispTraces; i++, t++ )
      {
	y = ( t->bot + t->top - CHARHEIGHT ) / 2;
	if( traces.traces[ i ].nsig == 1 )
	  StrCenter( Map[ hist->levels[ t->signals[0] ] ], x, y );
	else
	  StrCenter( Convert( hist->levels, t ), x, y );
      }
  }

/*
 * Display a string left justified at (x,y).
 */
StrLeft( s, x, y )
  char    *s;
  XCoord  x;
  YCoord  y;
  {
    myMoveTo( x, y );
    myCharStr( s, 1000 );
  }


/*
 * Display a string centered at (x,y).
 */
StrCenter( s, x, y )
  char    *s;
  XCoord  x;
  YCoord  y;
  {
    myMoveTo( x - ( (strlen( s ) * CHARWIDTH ) / 2  ), y );
    myCharStr( s, 1000 );
  }



/*
 * Display a string right justified at (x,y).
 */
StrRight( s, x, y )
  char    *s;
  XCoord  x;
  YCoord  y;
  {
    myMoveTo( x - (strlen( s ) * CHARWIDTH), y );
    myCharStr( s, 1000 );
  }


PrintFonts()
  {
    char **fonts, **XListFonts();
    int len,i;

    fprintf(stderr, "Available fonts are: \n");
    fonts = XListFonts(display, "*", 256, &len);
    for (i = 0; i < len; i++) 
      fprintf(stderr,"%s\n",fonts[i]);
  }



    
    
