
/*
 *
 * Version identification:
 * @(#)history.c	1.1 8/26/91
 *
 */

#ifndef lint
static char rcsid[] = "$Header: history.c,v 1.1 86/12/08 20:39:48 salz Exp $";
#endif

/*
 * Add / Retrieve a history entry.
 */

#include <stdio.h>
#include "defs.h"


#define TABLESIZE	1024			/* # of HistEnt per table */


typedef struct		/* Every HistTable contains TABLESIZE of these */
  {
    TimeType  time;			/* time of corresponding history */
    HistRef   hist;			/* ptr. to actual history entry */
  } TableEntry;


typedef struct HistTable	/* linked list of tables, last time at head */
  {
    TimeType          first;		/* first time recorded in table */
    struct HistTable  *pred;		/* table with previous time entries */
    int		      nElem;		/* no. of entries in the table */
    TableEntry        table[ TABLESIZE ];
  } HistTable;


typedef struct				/* dummy record to allocate memory */
  {
    HistEnt  hist;
    char     data[ MAXINPUTS ];		/* this field is variable */
  } MemElem;

#define	LastStep	(times.dispStart + times.dispSteps)

static HistTable* tables = NULL;	/* head of table list */

/*
 * Return the history entry that corresponds to the time, or the preceding
 * one (in time) if such an entry exists.  If no such entry can be found
 * return NULL.
 * First scan through the linked list of HistTable's to find in which one
 * the corresponding time might be found, then perform a binary search on
 * the table itself.  
 */
HistRef GetHist( time )
  TimeType  time;
  {
    register int	 t;
    register int	 bot, index;
    register TableEntry  *table;
    
    t = time;
    {
      register HistTable  *tbl;
      
      tbl = tables;
      while( (tbl != NULL) && (tbl->first > t) )
	 tbl = tbl->pred;
      if( tbl == NULL )
	  return( NULL );		/* No such entry in the tables */
      table = tbl->table;
      bot = tbl->nElem - 1;
    }
    {
      register int top = 0;

      while( bot >= top )
	{
	  index = (top + bot) / 2;
	  if( t < table[ index ].time )
	      bot = index - 1;
	  else if( t > table[ index ].time )
	      top = index + 1;
	  else
	      return( table[ index ].hist );	/* found entry for time */
	}
    }
    return( table[ bot ].hist );		/* return preceding entry */
  }


#define CopyLevels( from, to, n )			\
  {							\
    register char  *src;				\
    register char  *dest;				\
    register int   i;					\
							\
    src = from;						\
    dest = to;						\
    for( i = n; i != 0; i-- )				\
	*dest++ = *src++;				\
  }							\


#define NewTable()						\
  {								\
    HistTable  *newTable;					\
								\
    newTable = (HistTable *) Malloc( sizeof( HistTable ) );	\
    if( newTable == NULL )					\
      {								\
	PRINT( "No more memory for history.  " );		\
	return( -1 );						\
      }								\
    newTable->pred = tables;					\
    newTable->nElem = 0;					\
    tables = newTable;						\
  }								\


#define NewHistory( p, n )					\
  {								\
    MemElem	   *new;					\
								\
    new = (MemElem *) Malloc( sizeof( HistEnt ) + n );		\
    if( new == NULL )						\
      {								\
	PRINT( "No more memory for history.  " );		\
	return( -1 );						\
      }								\
    p = &(new->hist);						\
    p->levels = new->data;					\
  }								\



/*
 * Update history and refresh (scroll) the screen as appropriate.
 * Note that we only enter into the tables those entries in which at least
 * one signal differs from its predecessor (in time, of course).  The
 * entry corresponding to the first times will always be in the table.
 * We assume that following entries will have a time greater (or equal) to
 * the last one received.
 */
int AddHist( commH )
  CommHist		 *commH;
  {
    HistRef	lastH, newH;
    int		triggFound, dupT;
    TimeType	time, prevT;
    
    time = commH->time;
    if( time < times.last )
	return( 0 );

    if( time == times.last )		/* duplicate time => backtrack */
      {
	newH = tables->table[ tables->nElem - 1 ].hist;
	if( newH->time == time )
	  {
	    lastH = newH->prev;
	    if( lastH != NULL )
		lastH->next = NULL;
	    Mfree( newH, sizeof( HistEnt ) + inputs.numInputs );
	    if( (--tables->nElem) == 0 )
	      {
		HistTable *tmp = tables;
		tables = tables->pred;
		Mfree( tmp, sizeof( HistTable ) );
	      }
	  }
	else
	    lastH = newH;
	if( lastH != NULL )
	    prevT = lastH->time;
	dupT = 1;
      }
    else
      {
	dupT = 0;
	lastH = ( tables ) ? tables->table[ tables->nElem-1 ].hist : NULL;
	prevT = times.last;
      }


    if( lastH == NULL || bcmp(lastH->levels, commH->levels, inputs.numInputs))
      {
	if( (tables == NULL) || tables->nElem == TABLESIZE )
	  {
	    NewTable();
	    tables->first = time;
	  }
	NewHistory( newH, inputs.numInputs );
	newH->time = time;
	newH->prev = lastH;
	newH->next = NULL;
	CopyLevels( commH->levels, newH->levels, inputs.numInputs );
	if( lastH != NULL )
	    lastH->next = newH;
	tables->table[ tables->nElem ].time = newH->time;
	tables->table[ tables->nElem ].hist = newH;
	tables->nElem += 1;
	triggFound = ( trigger.first != NULL ) ? TriggerMatch( newH ) : 0;
      }
    else
      {
	triggFound = 0;
      }

    if( time > times.last )
	times.last = time;

    if( times.first < 0 )
      {
	times.first = time;
	RedrawTimes();
      }

	/* Check if finished a screen */
    if( (time > LastStep) || (time < times.dispStart) )
      {
	ScrollTracesLeft( time - 1 );		/* scroll half a screen */
	RedrawTraces( prevT, time, 0 );
	UpdateScrollBar();
	RedrawTimes();
      }
    else
      {
	if( !dupT )
	    UpdateLastTime();
	if( time > times.first )
	    RedrawTraces( prevT, time, dupT );
      }
    if( triggFound )
	MoveCursor( time );			/* Move Cursor to new end */
    return( triggFound & trigger.autostop );
  }


/*
 * Add a history read from a file.  This procedure is similar to AddHist,
 * except that no screen update takes place.
 */
int AddReadHist( rHist, nInputs )
  CommHist  *rHist;
  int       nInputs;
  {
    HistRef	   hist, lastH;
    
    if( tables != NULL )
	lastH = tables->table[ tables->nElem - 1 ].hist;
    else
	lastH = NULL;

    if( (tables == NULL) || (tables->nElem == TABLESIZE ) )
      {
	NewTable();
	tables->first = rHist->time;
      }
    NewHistory( hist, nInputs );
    hist->next = NULL;
    hist->time = rHist->time;
    hist->prev = lastH;
    if( lastH != NULL )
	lastH->next = hist;
    CopyLevels( rHist->levels, hist->levels, nInputs );

    tables->table[ tables->nElem ].time = hist->time;
    tables->table[ tables->nElem ].hist = hist;
    tables->nElem += 1;
    times.last = rHist->time;
    return( 0 );
  }



/*
 * Recover all memory used by history up to (not including) time.  If cmd
 * is set then this was issued by a user 'reclaim' command, otherwise it
 * was called automatically when changing the net-list.  Make sure that the
 * first time is always left in the table, this may require re-mapping the
 * corresponding entry.
 */
RecoverMem( time, cmd )
  TimeType  time;
  int       cmd;
  {
    HistRef   first, newFirst;
    
    first = GetHist( times.first );
    if( first == NULL )
	return;			/* nothing to recover */

    if( cmd )
      {
	PRINT( "Reclaiming memory..." );
	FLUSH();
      }

    newFirst = GetHist( time );
    if( newFirst != NULL && time < times.last )
      {
	newFirst->time = time;			/* re-map this entry */
	newFirst->prev = NULL;
      }

    {
      register HistTable   *tbl, *pred;
      
      for( tbl = tables; tbl != NULL && tbl->first >= time; tbl = tbl->pred );

      if( tbl != NULL )
	{
	  register TableEntry  *tab;
	  register int	   i;

	  tab = tbl->table;
	  for( i = tbl->nElem; i != 0 && tab->hist->time < time; i--, tab++ )
	      tab->time = -2;
	  if( i > 0 )			/* some live entries */
	    {
	      tbl->first = time;
	      tbl = tbl->pred;
	    }
	}
      while( tbl != NULL )
	{
	  pred = tbl->pred;
	  Mfree( tbl, sizeof( HistTable ) );
	  tbl = pred;
	}
    }

		/* Free the entries in [times.first..time-1] */
    {
      register HistRef  h, next;

      h = first;
      while( h->time < time )
	{
	  next = h->next;
	  Mfree( h, sizeof( HistEnt ) + inputs.numInputs );
	  h = next;
	}
    }

    if( time > times.last )		/* freed everything */
      {
	times.first = -1;
	times.last = -1;
	cursor.time = cursor.exists = -1;
	times.dispStart = 0;
      }
    else
      {
	times.first = time;
	if( cursor.time < time )
	    cursor.time = cursor.exists = -1;
	if( times.dispStart < time )
	    times.dispStart = time;
      }
    RedrawTimes();
    RedrawTraces( times.dispStart, LastStep, 1 );
    UpdateScrollBar();
    DrawCursVal();
    if( cmd )
	PRINT( "done\n" );
    if( windowState.simulStatus == STOP_NO_MEM )
      {
	CommHist  *oldH;

	GetLastHist( &oldH );		/* get the missing data */
	if( AddHist( oldH ) < 0 )
	    PRINT( "Not enough, try reclaiming more\n" );
	else
	    windowState.simulStatus = STOPPED;
      }
  }
