defstar {
	name {Lattice}
	domain {SDF}
	desc {
A Forward Lattice filter.  Default coefficients are the optimal
predictor for a particular 4th order AR random process.
To read reflection coefficients from a file, replace the default
coefficients with "<fileName", preferably specifying a complete path.
	}
	version {@(#)SDFLattice.pl	1.5 12/8/92}
	author { Alan Kamas}
	copyright {
Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.
See the file ~ptolemy/copyright for copyright notice,
limitation of liability, and disclaimer of warranty provisions.
	}
	location { SDF dsp library }
	explanation {
.pp
This star implements a lattice filter.
To load reflection coefficients from a file,
simply replace the default coefficients with the string "<filename".
It is advisable not to use an absolute path name as part of the file name,
especially if you are using the graphical interface.
This will allow the Lattice filter to work as expected regardless of
the directory in which the ptolemy process actually runs.
It is best to use tilde's in the filename to reference them to user's
home directory.  This way, future filesystem reorganizations
will have minimal effect.
.pp
The default reflection coefficients correspond to the optimal linear
predictor for an AR process generated by filtering white noise with
the following filter:
.EQ
H(z) ~=~ 1 over { 1 ~-~ 2 z sup -1 ~+~ 1.91z sup -2 ~-~ 0.91z sup -3 ~+~
0.205z sup -4 } ~.
.EN
Since this filter is minimum phase, the transfer function of the lattice
filter is $H sup -1 (z)$.
.pp
Note that the definition of reflection coefficients is not quite universal
in the literature.  The reflection coefficients in references [2] and [3]
are the negative of the ones used by this star, which
correspond to the definition in most other texts,
and to the definition of partial-correlation (PARCOR)
coefficients in the statistics literature.
The sign of the coefficients used in this star is appropriate for values
given by the LevDur or Burg stars.
.UH REFERENCES
.ip [1]
J. Makhoul, "Linear Prediction: A Tutorial Review",
\fIProc. IEEE\fR, Vol. 63, pp. 561-580, Apr. 1975.
.ip [2]
S. M. Kay, \fIModern Spectral Estimation: Theory & Application\fR,
Prentice-Hall, Englewood Cliffs, NJ, 1988.
.ip [3]
S. Haykin, \fIModern Filters\fR, MacMillan Publishing Company,
New York, 1989.
	}
	seealso { RLattice, BlockLattice, BlockRLattice, FIR, FIRCx, Biquad}
	input {
		name {signalIn}
		type {float}
	}
	output {
		name {signalOut}
		type {float}
	}
	defstate {
		name {reflectionCoefs}
		type {floatarray}
		default { "0.804534 -0.820577 0.521934 -0.205" }
		desc { Reflection or PARCOR coefficients. }
	}
	protected {
		double *b;
		double *f;
		int M;	
	}
	constructor {
		b = 0; f = 0; M = -1;
	}
	destructor {
		LOG_DEL; delete b;
		LOG_DEL; delete f;
	}
	setup {
		// reallocate arrays only if size has changed,
		// or this is the first run.
		if (M != reflectionCoefs.size()) {
			M = reflectionCoefs.size();
			LOG_DEL; delete b;
			LOG_NEW; b = new double[M];
			LOG_DEL; delete f;
			LOG_NEW; f = new double[M+1];
		}
		for (int i=0; i < M; i++)  b[i]=0.0 ;
	}
	go {
		double k;
		
		// Forward prediction error
		f[0] = double (signalIn%0);   // f(0)=x(n)
		for (int i=1; i <= M; i++) {
			k = - reflectionCoefs[i-1];
			f[i] = k * b[i-1] + f[i-1];
		}
		signalOut%0 << f[M];

		// Backward:  Compute the w's for the next round
		for (i = M-1; i >0 ; i--) {
			k = - reflectionCoefs[i-1];
			b[i] = k * f[i-1] + b[i-1];	
		}
		b[0] = double (signalIn%0);   // b[0]=x[n] 
			 
	}	
}
